var LocalViewer;
(function (LocalViewer) {
    LocalViewer.settings;
    LocalViewer.study;

    /**
    * Returns true if the viewer is running locally
    */
    function isLocalViewer() {
        return this.isNodeWebkitViewer() || this.isFileSystemViewer() || this.isPersonalAccelerator();
    }
    LocalViewer.isLocalViewer = isLocalViewer;

    /**
    * Returns true if the viewer is running in node-webkit
    */
    function isNodeWebkitViewer() {
        return typeof process !== "undefined";
    }
    LocalViewer.isNodeWebkitViewer = isNodeWebkitViewer;

    /**
    * Returns true if the viewer is running from the filesystem
    */
    function isFileSystemViewer() {
        return !isNodeWebkitViewer() && window.location.protocol.toLowerCase().indexOf("file") === 0;
    }
    LocalViewer.isFileSystemViewer = isFileSystemViewer;

    /**
    * Returns true if the viewer is running from the personal accelerator
    */
    function isPersonalAccelerator() {
        return this.personalAccelerator === true;
    }
    LocalViewer.isPersonalAccelerator = isPersonalAccelerator;

    /**
    * Returns true if the viewer is running as a local viewer, but not the personal accelerator
    */
    function isStandardLocalViewer() {
        return this.isLocalViewer() && !this.isPersonalAccelerator();
    }
    LocalViewer.isStandardLocalViewer = isStandardLocalViewer;

    /**
    * A helper function to mock the getStudyList call
    */
    function makePrior(study) {
        return {
            study_uid: study.studyAttributes.queryObject.studyUid.value,
            phi_namespace: study.studyAttributes.queryObject.phiNamespace.value,
            storage_namespace: study.studyAttributes.queryObject.storageNamespace.value,
            accession_number: study.studyAttributes.accessionNumber.value,
            study_description: study.studyAttributes.studyDescription,
            study_date: "",
            study_date_value: study.studyAttributes.studyCreateDate,
            prior_number: 0,
            thin: 0,
            uuid: ""
        };
    }
    LocalViewer.makePrior = makePrior;
})(LocalViewer || (LocalViewer = {}));
///<reference path="LocalViewer.ts" />
///<reference path="Query.ts" />
var Rendering;
(function (Rendering) {
    /**
    * Rendering mode types
    */
    (function (RenderingMode) {
        RenderingMode[RenderingMode["Canvas"] = 0] = "Canvas";
        RenderingMode[RenderingMode["WebGL"] = 1] = "WebGL";
        RenderingMode[RenderingMode["Simple"] = 2] = "Simple";
    })(Rendering.RenderingMode || (Rendering.RenderingMode = {}));
    var RenderingMode = Rendering.RenderingMode;

    /**
    * Get the rendering mode based on the option in the URL hash
    */
    function getRenderingMode(size) {
        if (typeof size === "undefined") { size = 0; }
        var mode = Query.findParameter(window.location, "mode");

        if (!mode) {
            return getDefaultRenderingMode(size);
        } else if (mode === "webgl" && !isWebGLSupported(size)) {
            return getDefaultRenderingMode();
        }

        return toRenderingMode(mode);
    }
    Rendering.getRenderingMode = getRenderingMode;

    function toRenderingMode(s) {
        switch (s) {
            case "webgl":
                return 1 /* WebGL */;
            case "canvas":
                return 0 /* Canvas */;
            case "simple":
                return 2 /* Simple */;
            default:
                return getDefaultRenderingMode();
        }
    }
    Rendering.toRenderingMode = toRenderingMode;

    /**
    * Get a default rendering mode for the environment
    */
    function getDefaultRenderingMode(size) {
        if (typeof size === "undefined") { size = 0; }
        if (LocalViewer.isLocalViewer()) {
            if (LocalViewer.isNodeWebkitViewer()) {
                return 0 /* Canvas */;
            } else if (LocalViewer.isPersonalAccelerator()) {
                // removing personal accelerator webgl support until issues resolved
                // if (Browser.isIE() || Browser.isEdge()) {  // performance issues in IE11 & Edge
                return 0 /* Canvas */;
                // } else {
                //     return isWebGLSupported(size) ? RenderingMode.WebGL : RenderingMode.Canvas;
                // }
            } else {
                return 2 /* Simple */;
            }
        } else {
            return 0 /* Canvas */;
        }
    }
    Rendering.getDefaultRenderingMode = getDefaultRenderingMode;

    var webGLSupported;

    /**
    * Check if WebGL is supported
    */
    function isWebGLSupported(size) {
        if (typeof size === "undefined") { size = 0; }
        if (size > 0) {
            return isWebGLSupportedImpl(size);
        }

        if (webGLSupported === undefined) {
            return webGLSupported = isWebGLSupportedImpl(size);
        }

        return webGLSupported;
    }
    Rendering.isWebGLSupported = isWebGLSupported;

    function isWebGLSupportedImpl(size) {
        var canvas = document.createElement("canvas");

        var gl = null;

        try  {
            gl = canvas.getContext("webgl") || canvas.getContext("experimental-webgl");

            if (gl) {
                if (gl.getParameter(gl.MAX_TEXTURE_SIZE) >= size) {
                    // Destroy webgl context explicitly. Not wait for GC cleaning up.
                    var ext = gl.getExtension('WEBGL_lose_context');
                    if (ext) {
                        ext.loseContext();
                    }

                    gl = null;
                    canvas = null;
                    return true;
                } else {
                    console.error("MAX_TEXTURE_SIZE too small, cannot use WebGL mode.");
                }
            }
        } catch (e) {
        }
        gl = null;
        canvas = null;
        return false;
    }
    Rendering.isWebGLSupportedImpl = isWebGLSupportedImpl;
})(Rendering || (Rendering = {}));
/**
* This module defines optional values
*/
var Maybe;
(function (_Maybe) {
    

    /**
    * A missing value
    */
    function Nothing() {
        return {
            ctor: "Maybe.Nothing"
        };
    }
    _Maybe.Nothing = Nothing;

    /**
    * Nothing if the argument is null, Just otherwise
    */
    function fromNull(t) {
        return t !== null ? Just(t) : Nothing();
    }
    _Maybe.fromNull = fromNull;

    /**
    * A provided value
    */
    function Just(t) {
        return {
            ctor: "Maybe.Just",
            value: t
        };
    }
    _Maybe.Just = Just;

    /**
    * Deconstruct an optional value
    */
    function maybe(m, r, f) {
        if (m.ctor === "Maybe.Nothing") {
            return r;
        } else {
            return f(m.value);
        }
    }
    _Maybe.maybe = maybe;

    /**
    * Test if an optional value is specified
    */
    function hasValue(m) {
        return maybe(m, false, function (_) {
            return true;
        });
    }
    _Maybe.hasValue = hasValue;

    /**
    * Map a function over optional values
    */
    function fmap(m, f) {
        return maybe(m, Nothing(), function (s) {
            return Just(f(s));
        });
    }
    _Maybe.fmap = fmap;

    /**
    * Unwrap an optional value, providing a default in case the value is missing
    */
    function fromMaybe(m, t) {
        return maybe(m, t, function (value) {
            return value;
        });
    }
    _Maybe.fromMaybe = fromMaybe;

    /**
    * Unwrap an optional value unsafely
    */
    function fromJust(m) {
        if (m.ctor === "Maybe.Nothing") {
            throw new Error("fromJust(Nothing)");
        }
        return m.value;
    }
    _Maybe.fromJust = fromJust;

    /**
    * Collect Justs from an array of Maybes.
    */
    function catMaybes(ms) {
        var result = [];
        _.each(ms, function (m) {
            if (hasValue(m)) {
                result.push(m.value);
            }
        });
        return result;
    }
    _Maybe.catMaybes = catMaybes;
})(Maybe || (Maybe = {}));
///<reference path="../libs/Maybe.ts" />
var HangingProtocols;
(function (HangingProtocols) {
    (function (USFilter) {
        USFilter[USFilter["TwoDImaging"] = 0x001] = "TwoDImaging";
        USFilter[USFilter["MMode"] = 0x002] = "MMode";
        USFilter[USFilter["CWDoppler"] = 0x004] = "CWDoppler";
        USFilter[USFilter["PWDoppler"] = 0x008] = "PWDoppler";
        USFilter[USFilter["ColorDoppler"] = 0x010] = "ColorDoppler";
        USFilter[USFilter["ColorMMode"] = 0x020] = "ColorMMode";
        USFilter[USFilter["ThreeDRendering"] = 0x040] = "ThreeDRendering";
        USFilter[USFilter["ColorPowerMode"] = 0x100] = "ColorPowerMode";
        USFilter[USFilter["TissueCharacterization"] = 0x20] = "TissueCharacterization";
        USFilter[USFilter["SpatiallyRelatedFrames"] = 0x400] = "SpatiallyRelatedFrames";
    })(HangingProtocols.USFilter || (HangingProtocols.USFilter = {}));
    var USFilter = HangingProtocols.USFilter;

    HangingProtocols.CURRENT_HP_VERSION = 2;

    HangingProtocols.MIN_HP_VERSION = 1;
})(HangingProtocols || (HangingProtocols = {}));
///<reference path="../libs/LocalViewer.ts" />
///<reference path="../libs/RenderingMode.ts" />
///<reference path="HangingProtocols.ts" />
var __extends = this.__extends || function (d, b) {
    for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p];
    function __() { this.constructor = d; }
    __.prototype = b.prototype;
    d.prototype = new __();
};
/**
* Types used throughout the application. Note many single values are wrapped in their own types. This is to
* prevent errors like passing a study UID when a series UID is expected. Methods using these values should
* try to use the strongly-typed versions of the data.
*/
var Classes;
(function (Classes) {
    /**
    * Owner ID, identifies the owning window
    */
    var OwnerId = (function () {
        function OwnerId(value) {
            this.value = value;
        }
        return OwnerId;
    })();
    Classes.OwnerId = OwnerId;

    /**
    * Session ID, aka. authentication token
    */
    var SessionId = (function () {
        function SessionId(value) {
            this.value = value;
        }
        return SessionId;
    })();
    Classes.SessionId = SessionId;

    /**
    * Instance UID
    */
    var InstanceUid = (function () {
        function InstanceUid(value) {
            this.value = value;
        }
        return InstanceUid;
    })();
    Classes.InstanceUid = InstanceUid;

    /**
    * Series UID
    */
    var SeriesUid = (function () {
        function SeriesUid(value) {
            this.value = value;
        }
        return SeriesUid;
    })();
    Classes.SeriesUid = SeriesUid;

    /**
    * Attachment ID
    */
    var AttachmentID = (function () {
        function AttachmentID(value) {
            this.value = value;
        }
        return AttachmentID;
    })();
    Classes.AttachmentID = AttachmentID;

    var AttachmentVersion = (function () {
        function AttachmentVersion(value) {
            this.value = value;
        }
        return AttachmentVersion;
    })();
    Classes.AttachmentVersion = AttachmentVersion;

    var AttachmentPhiNamespace = (function () {
        function AttachmentPhiNamespace(value) {
            this.value = value;
        }
        return AttachmentPhiNamespace;
    })();
    Classes.AttachmentPhiNamespace = AttachmentPhiNamespace;

    /**
    * Radiology Report ID
    */
    var ReportID = (function () {
        function ReportID(value) {
            this.value = value;
        }
        return ReportID;
    })();
    Classes.ReportID = ReportID;

    /**
    * UUID for a study action (routing rule)
    */
    var StudyActionID = (function () {
        function StudyActionID(value) {
            this.value = value;
        }
        return StudyActionID;
    })();
    Classes.StudyActionID = StudyActionID;

    /**
    * An email address
    */
    var EmailAddress = (function () {
        function EmailAddress(value) {
            this.value = value;
        }
        return EmailAddress;
    })();
    Classes.EmailAddress = EmailAddress;

    /**
    * Key Image ID
    */
    var KeyImageId = (function () {
        function KeyImageId(value) {
            this.value = value;
        }
        return KeyImageId;
    })();
    Classes.KeyImageId = KeyImageId;

    /**
    * Study UID
    */
    var StudyUid = (function () {
        function StudyUid(value) {
            this.value = value;
        }
        return StudyUid;
    })();
    Classes.StudyUid = StudyUid;

    /**
    * Study UUID
    */
    var StudyUUID = (function () {
        function StudyUUID(value) {
            this.value = value;
        }
        return StudyUUID;
    })();
    Classes.StudyUUID = StudyUUID;

    /**
    * Storage Namespace
    */
    var StorageNamespace = (function () {
        function StorageNamespace(value) {
            this.value = value;
        }
        return StorageNamespace;
    })();
    Classes.StorageNamespace = StorageNamespace;

    /**
    * PHI Namespace
    */
    var PhiNamespace = (function () {
        function PhiNamespace(value) {
            this.value = value;
        }
        return PhiNamespace;
    })();
    Classes.PhiNamespace = PhiNamespace;

    /**
    * Image Version
    */
    var ImageVersion = (function () {
        function ImageVersion(value) {
            this.value = value;
        }
        return ImageVersion;
    })();
    Classes.ImageVersion = ImageVersion;

    /**
    * Frame Number
    */
    var FrameNumber = (function () {
        function FrameNumber(value) {
            this.value = value;
        }
        return FrameNumber;
    })();
    Classes.FrameNumber = FrameNumber;

    /**
    * Patient ID
    */
    var PatientID = (function () {
        function PatientID(value) {
            this.value = value;
        }
        return PatientID;
    })();
    Classes.PatientID = PatientID;

    /**
    * Accession Number
    */
    var AccessionNumber = (function () {
        function AccessionNumber(value) {
            this.value = value;
        }
        return AccessionNumber;
    })();
    Classes.AccessionNumber = AccessionNumber;

    /**
    * A key image
    */
    var KeyImage = (function () {
        function KeyImage() {
        }
        return KeyImage;
    })();
    Classes.KeyImage = KeyImage;

    /**
    * Terminology code
    */
    var Term = (function () {
        function Term(code, defaultValue) {
            this.code = code;
            this.defaultValue = defaultValue;
        }
        return Term;
    })();
    Classes.Term = Term;

    /**
    * Language code
    */
    var Language = (function () {
        function Language(code) {
            this.code = code;
        }
        return Language;
    })();
    Classes.Language = Language;

    /**
    * Services annotation id
    */
    var AnnotationId = (function () {
        function AnnotationId(value) {
            this.value = value;
        }
        return AnnotationId;
    })();
    Classes.AnnotationId = AnnotationId;

    /**
    * Base type for all exceptions used in the application
    */
    var Exception = (function () {
        function Exception(message) {
            this.message = message;
        }
        return Exception;
    })();
    Classes.Exception = Exception;

    /**
    * The query string was invalid
    */
    var InvalidQueryStringException = (function (_super) {
        __extends(InvalidQueryStringException, _super);
        function InvalidQueryStringException() {
            _super.apply(this, arguments);
        }
        return InvalidQueryStringException;
    })(Exception);
    Classes.InvalidQueryStringException = InvalidQueryStringException;

    /**
    * No SID was found
    */
    var ExpectedSIDException = (function (_super) {
        __extends(ExpectedSIDException, _super);
        function ExpectedSIDException() {
            _super.apply(this, arguments);
        }
        return ExpectedSIDException;
    })(Exception);
    Classes.ExpectedSIDException = ExpectedSIDException;

    /**
    * An argument was invalid
    */
    var InvalidArgumentException = (function (_super) {
        __extends(InvalidArgumentException, _super);
        function InvalidArgumentException() {
            _super.apply(this, arguments);
        }
        return InvalidArgumentException;
    })(Exception);
    Classes.InvalidArgumentException = InvalidArgumentException;

    /**
    * Represents a study as described by storage - namespaces and UID.
    */
    var QueryObject = (function () {
        function QueryObject(storageNamespace, phiNamespace, studyUid) {
            this.storageNamespace = storageNamespace;
            this.phiNamespace = phiNamespace;
            this.studyUid = studyUid;
        }
        QueryObject.prototype.toString = function () {
            return this.storageNamespace.value + "/" + this.studyUid.value + "/" + this.phiNamespace.value;
        };
        return QueryObject;
    })();
    Classes.QueryObject = QueryObject;

    /**
    * The various document types supported by storage
    */
    (function (DocumentType) {
        DocumentType[DocumentType["PDF"] = 0] = "PDF";
        DocumentType[DocumentType["Video"] = 1] = "Video";
    })(Classes.DocumentType || (Classes.DocumentType = {}));
    var DocumentType = Classes.DocumentType;

    /**
    * The various image types supported by storage
    */
    (function (ImageType) {
        ImageType[ImageType["Thumbnail"] = 0] = "Thumbnail";
        ImageType[ImageType["FullResolution"] = 1] = "FullResolution";
        ImageType[ImageType["Diagnostic"] = 2] = "Diagnostic";
        ImageType[ImageType["FullResolutionHD"] = 3] = "FullResolutionHD";
    })(Classes.ImageType || (Classes.ImageType = {}));
    var ImageType = Classes.ImageType;

    (function (ImageTypeLoaded) {
        ImageTypeLoaded[ImageTypeLoaded["None"] = 0] = "None";
        ImageTypeLoaded[ImageTypeLoaded["Attributes"] = 1] = "Attributes";
        ImageTypeLoaded[ImageTypeLoaded["Thumbnail"] = 2] = "Thumbnail";
        ImageTypeLoaded[ImageTypeLoaded["Diagnostic"] = 3] = "Diagnostic";
    })(Classes.ImageTypeLoaded || (Classes.ImageTypeLoaded = {}));
    var ImageTypeLoaded = Classes.ImageTypeLoaded;

    /**
    * Helper methods for working with image types
    */
    var ImageTypes = (function () {
        function ImageTypes() {
        }
        /**
        * Convert an image type into a value which can be used in a storage request URI
        */
        ImageTypes.toUriComponent = function (type) {
            switch (type) {
                case 0 /* Thumbnail */:
                    return "/thumbnail";
                case 2 /* Diagnostic */:
                    return "/diagnostic";
                case 3 /* FullResolutionHD */:
                    return "/diagnostic";
                case 1 /* FullResolution */:
                default:
                    if (LocalViewer.isLocalViewer() && Rendering.getRenderingMode() === 2 /* Simple */) {
                        return "/hd";
                    } else {
                        return "";
                    }
            }
        };
        return ImageTypes;
    })();
    Classes.ImageTypes = ImageTypes;

    Classes.CURRENT_SETTINGS_VERSION = 1;

    Classes.MIN_SETTINGS_VERSION = 1;

    

    

    /**
    * Toolbar button types
    */
    (function (ToolbarButton) {
        ToolbarButton[ToolbarButton["Transform"] = 0] = "Transform";
        ToolbarButton[ToolbarButton["Zoom"] = 1] = "Zoom";
        ToolbarButton[ToolbarButton["Move"] = 2] = "Move";
        ToolbarButton[ToolbarButton["Flip"] = 3] = "Flip";
        ToolbarButton[ToolbarButton["Rotate"] = 4] = "Rotate";
        ToolbarButton[ToolbarButton["Scroll"] = 5] = "Scroll";
        ToolbarButton[ToolbarButton["WindowLevel"] = 6] = "WindowLevel";
        ToolbarButton[ToolbarButton["Annotations"] = 7] = "Annotations";
        ToolbarButton[ToolbarButton["Probe"] = 8] = "Probe";
        ToolbarButton[ToolbarButton["Fit"] = 9] = "Fit";
        ToolbarButton[ToolbarButton["Reset"] = 10] = "Reset";
        ToolbarButton[ToolbarButton["ThreeD"] = 11] = "ThreeD";
        ToolbarButton[ToolbarButton["Layout"] = 12] = "Layout";
        ToolbarButton[ToolbarButton["Print"] = 13] = "Print";
        ToolbarButton[ToolbarButton["Thumbs"] = 14] = "Thumbs";
        ToolbarButton[ToolbarButton["TextAnnotations"] = 15] = "TextAnnotations";
        ToolbarButton[ToolbarButton["Measurements"] = 16] = "Measurements";
        ToolbarButton[ToolbarButton["Invert"] = 17] = "Invert";
        ToolbarButton[ToolbarButton["Export"] = 18] = "Export";
        ToolbarButton[ToolbarButton["Cine"] = 19] = "Cine";
        ToolbarButton[ToolbarButton["Record"] = 20] = "Record";
        ToolbarButton[ToolbarButton["KeyImage"] = 21] = "KeyImage";
        ToolbarButton[ToolbarButton["Settings"] = 22] = "Settings";
        ToolbarButton[ToolbarButton["Actions"] = 23] = "Actions";
        ToolbarButton[ToolbarButton["SavePreset"] = 24] = "SavePreset";
        ToolbarButton[ToolbarButton["DeleteImage"] = 25] = "DeleteImage";
        ToolbarButton[ToolbarButton["DeleteSeries"] = 26] = "DeleteSeries";
        ToolbarButton[ToolbarButton["FlipV"] = 27] = "FlipV";
        ToolbarButton[ToolbarButton["ResetWindowLevel"] = 28] = "ResetWindowLevel";
        ToolbarButton[ToolbarButton["WindowLevelPresets"] = 29] = "WindowLevelPresets";
        ToolbarButton[ToolbarButton["SelectAnnotation"] = 30] = "SelectAnnotation";
        ToolbarButton[ToolbarButton["DeleteAnnotation"] = 31] = "DeleteAnnotation";
        ToolbarButton[ToolbarButton["FillAnnotation"] = 32] = "FillAnnotation";
        ToolbarButton[ToolbarButton["ExportGSPS"] = 33] = "ExportGSPS";
        ToolbarButton[ToolbarButton["Line"] = 34] = "Line";
        ToolbarButton[ToolbarButton["Arrow"] = 35] = "Arrow";
        ToolbarButton[ToolbarButton["Cobb"] = 36] = "Cobb";
        ToolbarButton[ToolbarButton["Rectangle"] = 37] = "Rectangle";
        ToolbarButton[ToolbarButton["Ellipse"] = 38] = "Ellipse";
        ToolbarButton[ToolbarButton["Text"] = 39] = "Text";
        ToolbarButton[ToolbarButton["ReferenceLines"] = 40] = "ReferenceLines";
        ToolbarButton[ToolbarButton["LinkedSeries"] = 41] = "LinkedSeries";
        ToolbarButton[ToolbarButton["PlaneLocalization"] = 42] = "PlaneLocalization";
        ToolbarButton[ToolbarButton["Maximize"] = 43] = "Maximize";
        ToolbarButton[ToolbarButton["NewWindow"] = 44] = "NewWindow";
        ToolbarButton[ToolbarButton["LayoutButtons"] = 45] = "LayoutButtons";
        ToolbarButton[ToolbarButton["ExportPNG"] = 46] = "ExportPNG";
        ToolbarButton[ToolbarButton["SecondaryCapture"] = 47] = "SecondaryCapture";
        ToolbarButton[ToolbarButton["Metadata"] = 48] = "Metadata";
        ToolbarButton[ToolbarButton["Play"] = 49] = "Play";
        ToolbarButton[ToolbarButton["Faster"] = 50] = "Faster";
        ToolbarButton[ToolbarButton["Slower"] = 51] = "Slower";
        ToolbarButton[ToolbarButton["FPSLabel"] = 52] = "FPSLabel";
        ToolbarButton[ToolbarButton["Angle"] = 53] = "Angle";
        ToolbarButton[ToolbarButton["FreeRotate"] = 54] = "FreeRotate";
        ToolbarButton[ToolbarButton["GSPSLayers"] = 55] = "GSPSLayers";
        ToolbarButton[ToolbarButton["Magnify"] = 56] = "Magnify";
        ToolbarButton[ToolbarButton["Anonymize"] = 57] = "Anonymize";
        ToolbarButton[ToolbarButton["Radius"] = 58] = "Radius";
        ToolbarButton[ToolbarButton["AnonymizeStudy"] = 59] = "AnonymizeStudy";
        ToolbarButton[ToolbarButton["AnonymizeImage"] = 60] = "AnonymizeImage";
        ToolbarButton[ToolbarButton["PreviousSeriesSet"] = 61] = "PreviousSeriesSet";
        ToolbarButton[ToolbarButton["NextSeriesSet"] = 62] = "NextSeriesSet";
        ToolbarButton[ToolbarButton["CineAll"] = 63] = "CineAll";
        ToolbarButton[ToolbarButton["DuplicateAnnotation"] = 64] = "DuplicateAnnotation";
        ToolbarButton[ToolbarButton["CopyAnnotation"] = 65] = "CopyAnnotation";
        ToolbarButton[ToolbarButton["PasteAnnotation"] = 66] = "PasteAnnotation";
        ToolbarButton[ToolbarButton["StartMeeting"] = 67] = "StartMeeting";
        ToolbarButton[ToolbarButton["UseDiagnosticQuality"] = 68] = "UseDiagnosticQuality";
        ToolbarButton[ToolbarButton["RecordAudio"] = 69] = "RecordAudio";
        ToolbarButton[ToolbarButton["MPR"] = 70] = "MPR";
        ToolbarButton[ToolbarButton["Enhance"] = 71] = "Enhance";
        ToolbarButton[ToolbarButton["ExportAllPNG"] = 72] = "ExportAllPNG";
        ToolbarButton[ToolbarButton["DetectWindowLevel"] = 73] = "DetectWindowLevel";
        ToolbarButton[ToolbarButton["AnnotationsDetailToggle"] = 74] = "AnnotationsDetailToggle";
        ToolbarButton[ToolbarButton["Blank"] = 75] = "Blank";
        ToolbarButton[ToolbarButton["WindowLevelDirect"] = 76] = "WindowLevelDirect";
        ToolbarButton[ToolbarButton["MouseToolSettings"] = 77] = "MouseToolSettings";
        ToolbarButton[ToolbarButton["ShowOnlyKeyImages"] = 78] = "ShowOnlyKeyImages";
        ToolbarButton[ToolbarButton["ExportAVI"] = 79] = "ExportAVI";
        ToolbarButton[ToolbarButton["ExportMP4"] = 80] = "ExportMP4";
        ToolbarButton[ToolbarButton["ExportSeries"] = 81] = "ExportSeries";
        ToolbarButton[ToolbarButton["NextStudy"] = 82] = "NextStudy";
        ToolbarButton[ToolbarButton["PreviousStudy"] = 83] = "PreviousStudy";
        ToolbarButton[ToolbarButton["Ruler"] = 84] = "Ruler";
        ToolbarButton[ToolbarButton["Circle"] = 85] = "Circle";
        ToolbarButton[ToolbarButton["Square"] = 86] = "Square";
        ToolbarButton[ToolbarButton["CoLocalization"] = 87] = "CoLocalization";
        ToolbarButton[ToolbarButton["OrthoAxes"] = 88] = "OrthoAxes";
        ToolbarButton[ToolbarButton["Label"] = 89] = "Label";
        ToolbarButton[ToolbarButton["NextStudyByMRN"] = 90] = "NextStudyByMRN";
        ToolbarButton[ToolbarButton["PreviousStudyByMRN"] = 91] = "PreviousStudyByMRN";
        ToolbarButton[ToolbarButton["StorePNG"] = 92] = "StorePNG";
        ToolbarButton[ToolbarButton["FemoralHead"] = 93] = "FemoralHead";
        ToolbarButton[ToolbarButton["ArrowAnnotate"] = 94] = "ArrowAnnotate";
        ToolbarButton[ToolbarButton["CircleAnnotate"] = 95] = "CircleAnnotate";
        ToolbarButton[ToolbarButton["EllipseAnnotate"] = 96] = "EllipseAnnotate";
        ToolbarButton[ToolbarButton["LineAnnotate"] = 97] = "LineAnnotate";
        ToolbarButton[ToolbarButton["RectangleAnnotate"] = 98] = "RectangleAnnotate";
        ToolbarButton[ToolbarButton["SquareAnnotate"] = 99] = "SquareAnnotate";
        ToolbarButton[ToolbarButton["PreviousImage"] = 100] = "PreviousImage";
        ToolbarButton[ToolbarButton["NextImage"] = 101] = "NextImage";
        ToolbarButton[ToolbarButton["LayoutSingleSeries"] = 102] = "LayoutSingleSeries";
        ToolbarButton[ToolbarButton["ColorTablePresets"] = 103] = "ColorTablePresets";
        ToolbarButton[ToolbarButton["UltrasoundRegions"] = 104] = "UltrasoundRegions";
        ToolbarButton[ToolbarButton["Polygon"] = 105] = "Polygon";
        ToolbarButton[ToolbarButton["Trace"] = 106] = "Trace";
        ToolbarButton[ToolbarButton["CalibrateLine"] = 107] = "CalibrateLine";
        ToolbarButton[ToolbarButton["SplitStudy"] = 108] = "SplitStudy";
        ToolbarButton[ToolbarButton["PlayRecording"] = 109] = "PlayRecording";
        ToolbarButton[ToolbarButton["StopPlayback"] = 110] = "StopPlayback";
        ToolbarButton[ToolbarButton["RewindPlayback"] = 111] = "RewindPlayback";
        ToolbarButton[ToolbarButton["FastForwardPlayback"] = 112] = "FastForwardPlayback";
        ToolbarButton[ToolbarButton["ExportStudy"] = 113] = "ExportStudy";
        ToolbarButton[ToolbarButton["ExportISO"] = 114] = "ExportISO";
        ToolbarButton[ToolbarButton["ExportLocalViewer"] = 115] = "ExportLocalViewer";
        ToolbarButton[ToolbarButton["ShowRecordings"] = 116] = "ShowRecordings";
        ToolbarButton[ToolbarButton["LoadReport"] = 117] = "LoadReport";
        ToolbarButton[ToolbarButton["Subtraction"] = 118] = "Subtraction";
        ToolbarButton[ToolbarButton["Propagate"] = 119] = "Propagate";
        ToolbarButton[ToolbarButton["PropagateAll"] = 120] = "PropagateAll";
        ToolbarButton[ToolbarButton["Stamp"] = 121] = "Stamp";
        ToolbarButton[ToolbarButton["PixelSpacingUser"] = 122] = "PixelSpacingUser";
        ToolbarButton[ToolbarButton["SliceSpacingUser"] = 123] = "SliceSpacingUser";
        ToolbarButton[ToolbarButton["RemoveImages"] = 124] = "RemoveImages";
        ToolbarButton[ToolbarButton["CropSeries"] = 125] = "CropSeries";
        ToolbarButton[ToolbarButton["SaveKeyImageLayout"] = 126] = "SaveKeyImageLayout";
        ToolbarButton[ToolbarButton["Area"] = 127] = "Area";
        ToolbarButton[ToolbarButton["ThresholdToArea"] = 128] = "ThresholdToArea";
        ToolbarButton[ToolbarButton["ShrinkWrapToArea"] = 129] = "ShrinkWrapToArea";
        ToolbarButton[ToolbarButton["ColorButtons"] = 130] = "ColorButtons";
        ToolbarButton[ToolbarButton["ShowAttachments"] = 131] = "ShowAttachments";
        ToolbarButton[ToolbarButton["ThresholdRangeToArea"] = 132] = "ThresholdRangeToArea";
        ToolbarButton[ToolbarButton["ReverseSeries"] = 133] = "ReverseSeries";
        ToolbarButton[ToolbarButton["UnweaveSeries"] = 134] = "UnweaveSeries";
        ToolbarButton[ToolbarButton["RearrangeSeries"] = 135] = "RearrangeSeries";
        ToolbarButton[ToolbarButton["PartSeries"] = 136] = "PartSeries";
        ToolbarButton[ToolbarButton["ResetStudy"] = 137] = "ResetStudy";
        ToolbarButton[ToolbarButton["MergeSeries"] = 138] = "MergeSeries";
        ToolbarButton[ToolbarButton["AnnotationsCreatedByOthersToggle"] = 139] = "AnnotationsCreatedByOthersToggle";
        ToolbarButton[ToolbarButton["ProstateTool"] = 140] = "ProstateTool";
        ToolbarButton[ToolbarButton["ShowAllGSPS"] = 141] = "ShowAllGSPS";
    })(Classes.ToolbarButton || (Classes.ToolbarButton = {}));
    var ToolbarButton = Classes.ToolbarButton;

    /**
    * Types of item which can appear on the toolbar
    */
    (function (ToolbarItemType) {
        ToolbarItemType[ToolbarItemType["Button"] = 0] = "Button";
        ToolbarItemType[ToolbarItemType["Group"] = 1] = "Group";
    })(Classes.ToolbarItemType || (Classes.ToolbarItemType = {}));
    var ToolbarItemType = Classes.ToolbarItemType;

    

    

    

    

    /**
    * Types of text annotations
    */
    (function (TextAnnotationType) {
        TextAnnotationType[TextAnnotationType["PatientName"] = 0] = "PatientName";
        TextAnnotationType[TextAnnotationType["PatientDOB"] = 1] = "PatientDOB";
        TextAnnotationType[TextAnnotationType["StudyDate"] = 2] = "StudyDate";
        TextAnnotationType[TextAnnotationType["PriorNumber"] = 3] = "PriorNumber";
        TextAnnotationType[TextAnnotationType["ImageNumber"] = 4] = "ImageNumber";
        TextAnnotationType[TextAnnotationType["ImageType"] = 5] = "ImageType";
        TextAnnotationType[TextAnnotationType["MeasurementCreator"] = 6] = "MeasurementCreator";
        TextAnnotationType[TextAnnotationType["ReferringPhysician"] = 7] = "ReferringPhysician";
        TextAnnotationType[TextAnnotationType["StudyDescription"] = 8] = "StudyDescription";
        TextAnnotationType[TextAnnotationType["SeriesDescription"] = 9] = "SeriesDescription";
        TextAnnotationType[TextAnnotationType["Zoom"] = 10] = "Zoom";
        TextAnnotationType[TextAnnotationType["Quality"] = 11] = "Quality";
        TextAnnotationType[TextAnnotationType["WindowLevel"] = 12] = "WindowLevel";
        TextAnnotationType[TextAnnotationType["RadiationMachineName"] = 13] = "RadiationMachineName";
        TextAnnotationType[TextAnnotationType["KVP"] = 14] = "KVP";
        TextAnnotationType[TextAnnotationType["Exposure"] = 15] = "Exposure";
        TextAnnotationType[TextAnnotationType["SliceThickness"] = 16] = "SliceThickness";
        TextAnnotationType[TextAnnotationType["PixelSpacingMeaning"] = 17] = "PixelSpacingMeaning";
        TextAnnotationType[TextAnnotationType["AccessionNumber"] = 18] = "AccessionNumber";
        TextAnnotationType[TextAnnotationType["CustomField"] = 19] = "CustomField";
        TextAnnotationType[TextAnnotationType["ImageLaterality"] = 20] = "ImageLaterality";
        TextAnnotationType[TextAnnotationType["ViewPosition"] = 21] = "ViewPosition";
        TextAnnotationType[TextAnnotationType["StationName"] = 22] = "StationName";
        TextAnnotationType[TextAnnotationType["OperatorsName"] = 23] = "OperatorsName";
        TextAnnotationType[TextAnnotationType["InstitutionName"] = 24] = "InstitutionName";
        TextAnnotationType[TextAnnotationType["InstitutionAddress"] = 25] = "InstitutionAddress";
        TextAnnotationType[TextAnnotationType["DetectorID"] = 26] = "DetectorID";
        TextAnnotationType[TextAnnotationType["PatientMRN"] = 27] = "PatientMRN";
        TextAnnotationType[TextAnnotationType["TagSeries"] = 28] = "TagSeries";
        TextAnnotationType[TextAnnotationType["TagImage"] = 29] = "TagImage";
        TextAnnotationType[TextAnnotationType["Subtraction"] = 30] = "Subtraction";
        TextAnnotationType[TextAnnotationType["CureMetrix"] = 31] = "CureMetrix";
        TextAnnotationType[TextAnnotationType["CADSRDensityFindings"] = 32] = "CADSRDensityFindings";
        TextAnnotationType[TextAnnotationType["CADSRCalcificationFindings"] = 33] = "CADSRCalcificationFindings";
        TextAnnotationType[TextAnnotationType["PaintMeasurementArea"] = 34] = "PaintMeasurementArea";
        TextAnnotationType[TextAnnotationType["PaintMeasurementMin"] = 35] = "PaintMeasurementMin";
        TextAnnotationType[TextAnnotationType["PaintMeasurementMax"] = 36] = "PaintMeasurementMax";
        TextAnnotationType[TextAnnotationType["PaintMeasurementMean"] = 37] = "PaintMeasurementMean";
        TextAnnotationType[TextAnnotationType["PaintMeasurementStDev"] = 38] = "PaintMeasurementStDev";
        TextAnnotationType[TextAnnotationType["PaintMeasurementLabel"] = 39] = "PaintMeasurementLabel";
        TextAnnotationType[TextAnnotationType["ProstateToolMeasurementLabel"] = 40] = "ProstateToolMeasurementLabel";
        TextAnnotationType[TextAnnotationType["ProstateToolMeasurementLength"] = 41] = "ProstateToolMeasurementLength";
        TextAnnotationType[TextAnnotationType["ProstateToolMeasurementWidth"] = 42] = "ProstateToolMeasurementWidth";
        TextAnnotationType[TextAnnotationType["ProstateToolMeasurementHeight"] = 43] = "ProstateToolMeasurementHeight";
        TextAnnotationType[TextAnnotationType["ProstateToolGroupMeasurement"] = 44] = "ProstateToolGroupMeasurement";
    })(Classes.TextAnnotationType || (Classes.TextAnnotationType = {}));
    var TextAnnotationType = Classes.TextAnnotationType;

    /**
    * Meaning of the pixel spacing
    */
    (function (PixelSpacingMeaning) {
        PixelSpacingMeaning[PixelSpacingMeaning["PatientGeometry"] = 0] = "PatientGeometry";
        PixelSpacingMeaning[PixelSpacingMeaning["AtImagingPlate"] = 1] = "AtImagingPlate";
    })(Classes.PixelSpacingMeaning || (Classes.PixelSpacingMeaning = {}));
    var PixelSpacingMeaning = Classes.PixelSpacingMeaning;

    /**
    * Positions of text annotations
    */
    (function (TextAnnotationPosition) {
        TextAnnotationPosition[TextAnnotationPosition["TopLeft"] = 0] = "TopLeft";
        TextAnnotationPosition[TextAnnotationPosition["BottomLeft"] = 1] = "BottomLeft";
        TextAnnotationPosition[TextAnnotationPosition["BottomRight"] = 2] = "BottomRight";
        TextAnnotationPosition[TextAnnotationPosition["TopRight"] = 3] = "TopRight";
    })(Classes.TextAnnotationPosition || (Classes.TextAnnotationPosition = {}));
    var TextAnnotationPosition = Classes.TextAnnotationPosition;

    Classes.CURRENT_MODALITY_SETTINGS_VERSION = 1;

    Classes.MIN_MODALITY_SETTINGS_VERSION = 1;

    /**
    * Positions of stacked series
    */
    (function (StackSeriesPosition) {
        StackSeriesPosition[StackSeriesPosition["None"] = 0] = "None";
        StackSeriesPosition[StackSeriesPosition["First"] = 1] = "First";
        StackSeriesPosition[StackSeriesPosition["Last"] = 2] = "Last";
    })(Classes.StackSeriesPosition || (Classes.StackSeriesPosition = {}));
    var StackSeriesPosition = Classes.StackSeriesPosition;

    

    

    

    

    

    

    

    

    

    

    

    /**
    * Left mouse button tools
    */
    (function (MouseTool) {
        MouseTool[MouseTool["Move"] = 0] = "Move";
        MouseTool[MouseTool["Scroll"] = 1] = "Scroll";
        MouseTool[MouseTool["Zoom"] = 2] = "Zoom";
        MouseTool[MouseTool["Window"] = 3] = "Window";
        MouseTool[MouseTool["Select"] = 4] = "Select";
        MouseTool[MouseTool["Measure"] = 5] = "Measure";
        MouseTool[MouseTool["Rectangle"] = 6] = "Rectangle";
        MouseTool[MouseTool["CobbAngle"] = 7] = "CobbAngle";
        MouseTool[MouseTool["Ellipse"] = 8] = "Ellipse";
        MouseTool[MouseTool["Localization"] = 9] = "Localization";
        MouseTool[MouseTool["Text"] = 10] = "Text";
        MouseTool[MouseTool["Probe"] = 11] = "Probe";
        MouseTool[MouseTool["Arrow"] = 12] = "Arrow";
        MouseTool[MouseTool["Angle"] = 13] = "Angle";
        MouseTool[MouseTool["FreeRotate"] = 14] = "FreeRotate";
        MouseTool[MouseTool["Magnify"] = 15] = "Magnify";
        MouseTool[MouseTool["Circle"] = 16] = "Circle";
        MouseTool[MouseTool["MPRRotate"] = 17] = "MPRRotate";
        MouseTool[MouseTool["DropCircle"] = 18] = "DropCircle";
        MouseTool[MouseTool["DropSquare"] = 19] = "DropSquare";
        MouseTool[MouseTool["OrthoAxes"] = 20] = "OrthoAxes";
        MouseTool[MouseTool["FemoralHead"] = 21] = "FemoralHead";
        MouseTool[MouseTool["ArrowAnnotate"] = 22] = "ArrowAnnotate";
        MouseTool[MouseTool["CircleAnnotate"] = 23] = "CircleAnnotate";
        MouseTool[MouseTool["EllipseAnnotate"] = 24] = "EllipseAnnotate";
        MouseTool[MouseTool["LineAnnotate"] = 25] = "LineAnnotate";
        MouseTool[MouseTool["RectangleAnnotate"] = 26] = "RectangleAnnotate";
        MouseTool[MouseTool["SquareAnnotate"] = 27] = "SquareAnnotate";
        MouseTool[MouseTool["Polygon"] = 28] = "Polygon";
        MouseTool[MouseTool["Trace"] = 29] = "Trace";
        MouseTool[MouseTool["CalibrateLine"] = 30] = "CalibrateLine";
        MouseTool[MouseTool["Stamp"] = 31] = "Stamp";
        MouseTool[MouseTool["Area"] = 32] = "Area";
        MouseTool[MouseTool["ProstateTool"] = 33] = "ProstateTool";
        MouseTool[MouseTool["None"] = 34] = "None";
    })(Classes.MouseTool || (Classes.MouseTool = {}));
    var MouseTool = Classes.MouseTool;

    /**
    * Helper methods for working with mouse tools
    */
    var MouseTools = (function () {
        function MouseTools() {
        }
        /**
        * Check if a tool is a measurement tool
        */
        MouseTools.isMeasurementTool = function (tool) {
            switch (tool) {
                case 4 /* Select */:
                case 5 /* Measure */:
                case 7 /* CobbAngle */:
                case 6 /* Rectangle */:
                case 8 /* Ellipse */:
                case 10 /* Text */:
                case 31 /* Stamp */:
                case 12 /* Arrow */:
                case 13 /* Angle */:
                case 16 /* Circle */:
                case 18 /* DropCircle */:
                case 19 /* DropSquare */:
                case 20 /* OrthoAxes */:
                case 21 /* FemoralHead */:
                case 22 /* ArrowAnnotate */:
                case 23 /* CircleAnnotate */:
                case 24 /* EllipseAnnotate */:
                case 25 /* LineAnnotate */:
                case 30 /* CalibrateLine */:
                case 26 /* RectangleAnnotate */:
                case 27 /* SquareAnnotate */:
                case 28 /* Polygon */:
                case 29 /* Trace */:
                case 32 /* Area */:
                case 33 /* ProstateTool */:
                    return true;
            }

            return false;
        };

        MouseTools.isAnnotationTool = function (tool, showTextOnDirected) {
            switch (tool) {
                case 10 /* Text */:
                case 31 /* Stamp */:
                case 22 /* ArrowAnnotate */:
                case 23 /* CircleAnnotate */:
                case 24 /* EllipseAnnotate */:
                case 25 /* LineAnnotate */:
                case 26 /* RectangleAnnotate */:
                case 27 /* SquareAnnotate */:
                    return true;
            }

            if (tool == 12 /* Arrow */ && !showTextOnDirected) {
                return true;
            }

            return false;
        };

        MouseTools.isCalibrationTool = function (tool) {
            switch (tool) {
                case 30 /* CalibrateLine */:
                    return true;
            }

            return false;
        };

        MouseTools.isGroupMeasurementTool = function (tool) {
            switch (tool) {
                case 33 /* ProstateTool */:
                    return true;
            }

            return false;
        };

        MouseTools.isPaintTool = function (tool) {
            switch (tool) {
                case 32 /* Area */:
                    return true;
            }

            return false;
        };

        /**
        * These tools "drop" a ROI at a clicked location
        * @param {Classes.MouseTool} tool
        * @returns {boolean}
        */
        MouseTools.isDropMeasurementTool = function (tool) {
            switch (tool) {
                case 18 /* DropCircle */:
                case 23 /* CircleAnnotate */:
                case 19 /* DropSquare */:
                case 27 /* SquareAnnotate */:
                    return true;
            }

            return false;
        };

        /**
        * These tools are sensitive to a click after an initial stroke
        * @param {Classes.MouseTool} tool
        * @returns {boolean}
        */
        MouseTools.isMultiClickMeasurementTool = function (tool) {
            switch (tool) {
                case 13 /* Angle */:
                case 21 /* FemoralHead */:
                case 28 /* Polygon */:
                    return true;
            }

            return false;
        };
        return MouseTools;
    })();
    Classes.MouseTools = MouseTools;

    /**
    * An immutable semaphore
    *
    * Useful in combination with a Subject when multiple parts of the application affect whether a condition is true.
    *
    * @see Subjects.Subject
    */
    var Resource = (function () {
        function Resource(value) {
            this.value = value;
        }
        /**
        * A resource with no consumers
        */
        Resource.unused = function () {
            return new Resource(0);
        };

        /**
        * Increment the number of consumers, and return the new resource
        */
        Resource.prototype.take = function () {
            return new Resource(this.value + 1);
        };

        /**
        * Decrement the number of consumers, and return the new resource
        */
        Resource.prototype.release = function () {
            return new Resource(this.value - 1);
        };

        /**
        * Test if the resource is being used
        */
        Resource.prototype.used = function () {
            return this.value > 0;
        };
        return Resource;
    })();
    Classes.Resource = Resource;
})(Classes || (Classes = {}));
///<reference path="../classes/Types.ts" />
/**
* Helper methods for working with query strings
*/
var Query;
(function (Query) {
    /**
    * Find a parameter in the hash or query portions of a URL
    *
    * @param {key} The name of the parameter to find
    * @returns The corresponding URI-decoded value, or null if the key is not found
    */
    function findParameter(location, key) {
        return findParameterIn(location.hash, key) || findParameterIn(location.search, key);
    }
    Query.findParameter = findParameter;

    /**
    * Find a parameter in a query string
    *
    * @param {key} The name of the parameter to find
    * @returns The corresponding URI-decoded value, or null if the key is not found
    */
    function findParameterIn(search, key) {
        var index = Math.max(search.indexOf("?" + key + "="), search.indexOf("&" + key + "="));

        if (index < 0) {
            return null;
        }

        var lastIndex = search.indexOf("&", index + 1);

        if (lastIndex < 0) {
            lastIndex = search.length;
        }

        return decodeURIComponent(search.substr(index + 2 + key.length, lastIndex - index - 2 - key.length));
    }
    Query.findParameterIn = findParameterIn;

    /**
    * Get the session ID from the query string, or null if the "sid" parameter is missing.
    */
    function getSessionId(location) {
        var querySid = Query.findParameter(location, "sid");

        if (querySid !== null) {
            return new Classes.SessionId(querySid);
        }

        return null;
    }
    Query.getSessionId = getSessionId;

    /**
    * Parse a hash into a route and its parameters
    *
    * The hash is of the form {route}/{storageNamespace}/{studyUid}/{phiNamespace}?param1=value&param2=value...
    * where {route} is one of "study" or "print".
    */
    function parseQueryString(location) {
        var hash = location.hash;

        var routeLength = hash.indexOf("/");

        if (routeLength < 0) {
            throw new Classes.InvalidQueryStringException("Invalid hash");
        }

        var route = hash.substring(1, routeLength);

        if (route === "study") {
            // Support ?route=... format
            route = findParameter(location, "route") || route;
        }

        if (!_.contains([
            "study",
            "print",
            "script",
            "meeting",
            "report",
            "keyimages",
            "mobile",
            "mpr"
        ], route)) {
            throw new Classes.InvalidQueryStringException("Unknown route");
        }

        var stop = hash.indexOf("?");

        if (stop < 0) {
            stop = hash.length;
        }

        var tailIndex = window.location.hash.indexOf("/");

        if (tailIndex < 0 || hash.charAt(tailIndex) !== '/') {
            throw new Classes.InvalidQueryStringException("Invalid hash: missing study identifier");
        }

        var study = decodeURIComponent(hash.substring(tailIndex + 1, stop));
        var parts = study.split("/", 3);

        if (parts.length != 3) {
            throw new Classes.InvalidQueryStringException("Invalid study ID");
        }

        var storageNamespace = new Classes.StorageNamespace(parts[0]);
        var studyUid = new Classes.StudyUid(parts[1]);
        var phiNamespace = new Classes.PhiNamespace(parts[2]);

        return {
            route: route,
            queryObject: new Classes.QueryObject(storageNamespace, phiNamespace, studyUid)
        };
    }
    Query.parseQueryString = parseQueryString;

    /**
    * Parse the path used by the personal accelerator.
    *
    * The path is of the form /session/{sid}/{storageNamespace}/{studyUID}/local.html?phinamespace=value&sid=value
    */
    function parsePersonalAcceleratorPath(location) {
        var path = location.pathname;
        var parts = path.split("/", 5);

        if (parts.length != 5) {
            throw new Classes.InvalidQueryStringException("Invalid study ID");
        }

        var storageNamespace = new Classes.StorageNamespace(parts[3]);
        var studyUid = new Classes.StudyUid(parts[4]);
        var phiNamespace = new Classes.PhiNamespace(Query.findParameter(location, "phinamespace"));

        return {
            route: "study",
            queryObject: new Classes.QueryObject(storageNamespace, phiNamespace, studyUid)
        };
    }
    Query.parsePersonalAcceleratorPath = parsePersonalAcceleratorPath;
})(Query || (Query = {}));
var Classes;
(function (Classes) {
    

    

    

    /**
    * Meeting data attached to a study
    */
    var StudyMeeting = (function () {
        function StudyMeeting() {
        }
        return StudyMeeting;
    })();
    Classes.StudyMeeting = StudyMeeting;

    var MeetingLink = (function () {
        function MeetingLink() {
        }
        return MeetingLink;
    })();
    Classes.MeetingLink = MeetingLink;

    /**
    * Meeting ID, identifies a meeting in services
    */
    var MeetingId = (function () {
        function MeetingId(value) {
            this.value = value;
        }
        return MeetingId;
    })();
    Classes.MeetingId = MeetingId;

    /**
    * Link ID
    */
    var MeetingLinkId = (function () {
        function MeetingLinkId(value) {
            this.value = value;
        }
        return MeetingLinkId;
    })();
    Classes.MeetingLinkId = MeetingLinkId;

    var RecordingLink = (function (_super) {
        __extends(RecordingLink, _super);
        function RecordingLink() {
            _super.apply(this, arguments);
        }
        return RecordingLink;
    })(MeetingLink);
    Classes.RecordingLink = RecordingLink;

    /**
    * Link ID
    */
    var RecordingLinkId = (function (_super) {
        __extends(RecordingLinkId, _super);
        function RecordingLinkId() {
            _super.apply(this, arguments);
        }
        return RecordingLinkId;
    })(MeetingLinkId);
    Classes.RecordingLinkId = RecordingLinkId;

    /**
    * User UUID
    */
    var UserUUID = (function () {
        function UserUUID(value) {
            this.value = value;
        }
        return UserUUID;
    })();
    Classes.UserUUID = UserUUID;

    

    
})(Classes || (Classes = {}));
///<reference path='../classes/Types.ts' />
var Models;
(function (Models) {
    /**
    * Study information from the /study/get call
    */
    var StudyStorageInfo = (function () {
        function StudyStorageInfo() {
            /**
            * A list of custom fields
            */
            this.customfields = [];
            /**
            * True if this study is cached in a local accelerator
            */
            this.localAccelerator = false;
        }
        return StudyStorageInfo;
    })();
    Models.StudyStorageInfo = StudyStorageInfo;

    

    

    /**
    * Attributes defined at the study level
    */
    var StudyAttributes = (function () {
        function StudyAttributes() {
            /**
            * A list of custom fields
            */
            this.customfields = [];
        }
        return StudyAttributes;
    })();
    Models.StudyAttributes = StudyAttributes;

    /**
    * A study and its series
    */
    var Study = (function () {
        function Study() {
            /**
            * The series in this study
            */
            this.series = [];
            /**
            * The original series in this study (before splitting multiframes)
            */
            this.originalSeries = [];
            /**
            * Study attachments
            */
            this.attachments = [];
            /**
            * Radiology reports
            */
            this.reports = [];
            /**
            * Structured report instances
            */
            this.structuredReports = [];
            /**
            * Study actions (routing rules)
            */
            this.actions = [];
            /**
            * A list of meetings in progress
            */
            this.meetings = [];
        }
        return Study;
    })();
    Models.Study = Study;

    /**
    * Attachment metadata
    */
    var Attachment = (function () {
        function Attachment() {
        }
        return Attachment;
    })();
    Models.Attachment = Attachment;

    /**
    * Customfield metadata
    */
    var CustomField = (function () {
        function CustomField() {
        }
        return CustomField;
    })();
    Models.CustomField = CustomField;

    /**
    * Report metadata
    */
    var Report = (function () {
        function Report() {
        }
        return Report;
    })();
    Models.Report = Report;

    /**
    * Study action
    */
    var StudyAction = (function () {
        function StudyAction() {
        }
        return StudyAction;
    })();
    Models.StudyAction = StudyAction;

    /**
    * Structured report metadata
    */
    var StructuredReport = (function () {
        function StructuredReport() {
        }
        return StructuredReport;
    })();
    Models.StructuredReport = StructuredReport;

    

    /**
    * Attributes defined at the series level
    */
    var SeriesAttributes = (function () {
        function SeriesAttributes() {
        }
        return SeriesAttributes;
    })();
    Models.SeriesAttributes = SeriesAttributes;

    

    /**
    * A series and its instances
    */
    var Series = (function () {
        function Series() {
            /**
            * The instances in this series
            */
            this.instances = [];
            this.loadedStatus = 0 /* None */;
            this.splitSelected = false;
        }
        return Series;
    })();
    Models.Series = Series;

    /**
    * Attributes defined at the instance level
    */
    var InstanceAttributes = (function () {
        function InstanceAttributes() {
            /**
            * A list of related presentation states
            */
            this.presentationStates = [];
            /**
            * A list of CADSR annotations
            */
            this.mesaurementsCADSR = [];
            /**
            * Indicates whether the most recent thumbnail load attempt failed for this image
            */
            this.mostRecentThumbnailLoadFailed = false;
            /**
            * Indicates whether the most recent hi-res load attempt failed for this image
            */
            this.mostRecentHiResLoadFailed = false;
            /**
            * True if this instance contains overlay data
            */
            this.containsOverlayData = false;
            /**
            * True if an overlay plane contains data
            */
            this.containsOverlayDataPlane = [];
        }
        return InstanceAttributes;
    })();
    Models.InstanceAttributes = InstanceAttributes;

    /**
    * An image along with its frames
    */
    var Instance = (function () {
        function Instance() {
        }
        return Instance;
    })();
    Models.Instance = Instance;

    var SeriesDocumentTypes = (function () {
        function SeriesDocumentTypes() {
        }
        return SeriesDocumentTypes;
    })();
    Models.SeriesDocumentTypes = SeriesDocumentTypes;
})(Models || (Models = {}));

/**
* Methods not provided on the String prototype
* @see String
*/
var Strings;
(function (Strings) {
    String.prototype.truncate = function (maxLength) {
        var str = this;
        if (str.length <= maxLength) {
            return str;
        } else {
            return str.substr(0, maxLength) + "...";
        }
    };
})(Strings || (Strings = {}));
///<reference path="../classes/Types.ts" />
///<reference path="../models/Study.ts" />
///<reference path="Strings.ts" />
/**
* Routes for service calls
*/
var Routes;
(function (Routes) {
    /**
    * Encode parameters as a query string
    */
    function encodeParameters(params) {
        if (params) {
            return "?" + _.map(params, function (p) {
                if (p.length == 1) {
                    return p[0];
                } else if (p.length == 2) {
                    return p[0] + "=" + encodeURIComponent(p[1]);
                } else {
                    throw new Error("encodeParameters: length");
                }
            }).join("&");
        }

        return "";
    }
    Routes.encodeParameters = encodeParameters;

    /**
    * Converts an object's properties to a list of parameter pairs.
    * @param params
    * @returns {string[][]}
    */
    function covertObjectToParams(params) {
        if (params) {
            var result = [];
            _.each(params, function (value, key) {
                result.push([key, value]);
            });

            return result;
        }

        return [];
    }
    Routes.covertObjectToParams = covertObjectToParams;

    /**
    * Session ID query parameter
    */
    function sid(sessionId) {
        return ["sid", sessionId.value];
    }
    Routes.sid = sid;

    /**
    * Study triplet query parameters
    */
    function studyTriplet(query) {
        return [
            ["study_uid", query.studyUid.value],
            ["storage_namespace", query.storageNamespace.value],
            ["phi_namespace", query.phiNamespace.value]
        ];
    }
    Routes.studyTriplet = studyTriplet;

    /**
    * Get user info
    */
    function GetUser(sessionId) {
        return "/api/v3/session/user" + encodeParameters([sid(sessionId)]);
    }
    Routes.GetUser = GetUser;

    /**
    * Get user permissions
    */
    function GetPermissions(sessionId, phiNamespace) {
        return "/api/v3/session/permissions" + encodeParameters([
            sid(sessionId),
            ["namespace_id", phiNamespace.value]
        ]);
    }
    Routes.GetPermissions = GetPermissions;

    function GetStoragePermissions(sessionId, namespace) {
        return "/api/v3/session/permissions" + encodeParameters([
            sid(sessionId),
            ["namespace_id", namespace.value]
        ]);
    }
    Routes.GetStoragePermissions = GetStoragePermissions;

    /**
    * Get user settings
    */
    function GetSettings(sessionId, key) {
        return "/api/v3/setting/get" + encodeParameters([
            sid(sessionId),
            ["key", key]
        ]);
    }
    Routes.GetSettings = GetSettings;

    /**
    * Save user settings
    */
    Routes.SaveSettings = "/api/v3/setting/set";

    /**
    * Load terminology
    */
    function Terminology(sessionId, tags, language, query) {
        return "/api/v3/terminology/tags" + encodeParameters([
            ["tags", tags.join(",")],
            ["language", language.code],
            sid(sessionId)
        ].concat(studyTriplet(query)));
    }
    Routes.Terminology = Terminology;

    /**
    * Load all terminology
    */
    function TerminologyAll(sessionId, language, query) {
        return "/api/v3/terminology/tags" + encodeParameters([["language", language.code], sid(sessionId)].concat(studyTriplet(query)));
    }
    Routes.TerminologyAll = TerminologyAll;

    /**
    * Get account settings
    */
    function GetAccountSettings(sessionId, query, fqdn) {
        var uri = "/api/v3/study/viewer/settings" + encodeParameters([
            sid(sessionId)
        ].concat(studyTriplet(query)));

        if (fqdn) {
            return "//" + fqdn + uri;
        } else {
            return uri;
        }
    }
    Routes.GetAccountSettings = GetAccountSettings;

    /**
    * Get a list of studies for a patient
    */
    function GetStudyList(sessionId, phiNamespace, patientId, accessionNumber, studyUuid, notStudyUuid, worklistFilter) {
        return "/api/v3/study/list" + encodeParameters([
            sid(sessionId),
            ["filter.phi_namespace.equals", phiNamespace.value],
            ["filter.patientid.equals", patientId.value]
        ].concat(accessionNumber ? [["filter.accession_number.equals", accessionNumber.value]] : []).concat(studyUuid ? [["filter.uuid.equals", studyUuid.value]] : []).concat(notStudyUuid ? [["filter.uuid.not_equals", notStudyUuid.value]] : []).concat(worklistFilter ? worklistFilter : []));
    }
    Routes.GetStudyList = GetStudyList;

    /**
    * Get a list of studies by uuid
    */
    function GetStudyListByUuid(sessionId, studyUuidList) {
        return "/api/v3/study/list" + encodeParameters([sid(sessionId)].concat([["filter.uuid.in", JSON.stringify(_.map(studyUuidList, function (uuid) {
                    return uuid.value;
                }))]]));
    }
    Routes.GetStudyListByUuid = GetStudyListByUuid;

    /**
    * study/get
    */
    function StudyStorageInfo(sessionId, query) {
        return "/api/v3/study/get" + encodeParameters([
            sid(sessionId)
        ].concat(studyTriplet(query)));
    }
    Routes.StudyStorageInfo = StudyStorageInfo;

    /**
    * study/get (with study UUID)
    * @param sessionId
    * @param uuid
    * @constructor
    */
    function StudyInfo(sessionId, uuid) {
        return "/api/v3/study/get" + encodeParameters([sid(sessionId), ["uuid", uuid.value]]);
    }
    Routes.StudyInfo = StudyInfo;

    /**
    * study/set
    */
    function StudySet(sessionId, query, tags) {
        return "/api/v3/study/set" + encodeParameters([
            sid(sessionId)
        ].concat(studyTriplet(query)).concat(_.map(tags, function (t) {
            return [
                "customfield-({g},{e})".replace("{g}", t.group.toString(16).padStart(4, "0")).replace("{e}", t.element.toString(16).padStart(4, "0")),
                t.value
            ];
        })));
    }
    Routes.StudySet = StudySet;

    /**
    * study/retrieve
    */
    function RetrieveStudy(sessionId, uuid) {
        return "/api/v3/study/retrieve" + encodeParameters([
            sid(sessionId),
            ["uuid", uuid.value]
        ]);
    }
    Routes.RetrieveStudy = RetrieveStudy;

    /**
    * study/audit
    */
    function StudyAudit(sessionId, uuid, action) {
        return "/api/v3/study/audit" + encodeParameters([
            sid(sessionId),
            ["uuid", uuid.value],
            ["action", action]
        ]);
    }
    Routes.StudyAudit = StudyAudit;

    /**
    * viewer/annotations/list
    */
    function ListAnnotations(sessionId, query) {
        var uri = "/api/v3/annotation/list" + encodeParameters([
            sid(sessionId)
        ].concat(studyTriplet(query)));

        return uri;
    }
    Routes.ListAnnotations = ListAnnotations;

    /**
    * viewer/annotations/get
    */
    function GetAnnotation(sessionId, query, id) {
        var uri = "/api/v3/annotation/get" + encodeParameters([
            sid(sessionId),
            ["uuid", id.value]
        ].concat(studyTriplet(query)));

        return uri;
    }
    Routes.GetAnnotation = GetAnnotation;

    /**
    * viewer/annotations/delete
    */
    function DeleteAnnotation(sessionId, query, id) {
        return "/api/v3/annotation/delete" + encodeParameters([
            sid(sessionId),
            ["uuid", id.value]
        ].concat(studyTriplet(query)));
    }
    Routes.DeleteAnnotation = DeleteAnnotation;

    /**
    * viewer/keyimage/list
    */
    function ListKeyImages(sessionId, query) {
        var uri = "/api/v3/keyimage/list" + encodeParameters([
            sid(sessionId)
        ].concat(studyTriplet(query)));

        return uri;
    }
    Routes.ListKeyImages = ListKeyImages;

    /**
    * viewer/keyimage/get
    */
    function GetKeyImage(sessionId, id) {
        return "/api/v3/keyimage/get" + encodeParameters([
            sid(sessionId),
            ["uuid", id.value]
        ]);
    }
    Routes.GetKeyImage = GetKeyImage;

    /**
    * viewer/keyimage/add
    */
    function AddKeyImage(sessionId, query, series, instance, frameNumber, version) {
        return "/api/v3/keyimage/add" + encodeParameters([
            sid(sessionId),
            ["series_uid", series.value],
            ["instance_uid", instance.value],
            ["frame_number", frameNumber.value.toString()],
            ["version", version.value]
        ].concat(studyTriplet(query)));
    }
    Routes.AddKeyImage = AddKeyImage;

    /**
    * viewer/keyimage/delete
    */
    function DeleteKeyImage(sessionId, id) {
        return "/api/v3/keyimage/delete" + encodeParameters([
            sid(sessionId),
            ["uuid", id.value]
        ]);
    }
    Routes.DeleteKeyImage = DeleteKeyImage;

    /**
    * study/manual/route
    */
    function RunRoutingRule(sessionId, studyId, routeId, email, message) {
        return "/api/v3/study/manual/route" + encodeParameters([
            sid(sessionId),
            ["uuid", studyId.value],
            ["route_id", routeId.value]
        ].concat(email ? [["email", email.value]] : []).concat(message ? [["message", message]] : []));
    }
    Routes.RunRoutingRule = RunRoutingRule;

    /**
    * log message
    */
    function AuditLog(error, bucket, userAgent, source, sessionId, studyId, seriesId) {
        return "/api/v3/audit/log" + encodeParameters([
            sid(sessionId),
            ["error", error],
            ["bucket", bucket],
            ["userAgent", userAgent],
            ["source", source]
        ].concat(studyId ? [["studyId", studyId.value]] : []).concat(seriesId ? [["seriesId", seriesId.value]] : []));
    }
    Routes.AuditLog = AuditLog;

    /**
    * log metric
    */
    function AuditMetric(name, metric, bucket, userAgent) {
        return "/api/v3/audit/log" + encodeParameters([
            ["name", name],
            ["bucket", bucket],
            ["userAgent", userAgent]
        ].concat(covertObjectToParams(metric)));
    }
    Routes.AuditMetric = AuditMetric;

    /**
    * meeting/add
    */
    Routes.StartMeeting = "/api/v3/meeting/add";

    /**
    * meeting/delete
    */
    Routes.EndMeeting = "/api/v3/meeting/delete";

    /**
    * meeting/set
    */
    Routes.UpdateMeeting = "/api/v3/meeting/set";

    /**
    * meeting/events/add
    */
    Routes.BroadcastEvent = "/api/v3/meeting/events/add";

    /**
    * meeting/ping
    */
    Routes.PingMeeting = "/api/v3/meeting/ping";

    /**
    * meeting/join
    */
    Routes.JoinMeeting = "/api/v3/meeting/join";

    /**
    * meeting/leave
    */
    Routes.LeaveMeeting = "/api/v3/meeting/leave";

    /**
    * meeting/presenter
    */
    Routes.ChangeMeetingPresenter = "/api/v3/meeting/presenter";

    /**
    * link/add
    */
    Routes.CreateMeetingLink = "/api/v3/link/add";

    /**
    * link/email
    * @type {string}
    */
    Routes.ShareLink = "/api/v3/link/mail";

    /**
    * accelerator/used
    */
    Routes.AcceleratorUsed = "/api/v3/accelerator/used";

    /**
    * annotation/add
    * @type {string}
    */
    Routes.AddAnnotation = "/api/v3/annotation/add";

    /**
    * annotation/set
    * @type {string}
    */
    Routes.SetAnnotation = "/api/v3/annotation/set";

    Routes.CreateScriptLink = "/api/v3/link/add";

    Routes.StudyPHIExtended = "/api/v3/study/phi/extended";

    /**
    * meeting/roster
    */
    function MeetingRoster(sessionId, meetingId) {
        return "/api/v3/meeting/roster" + encodeParameters([
            sid(sessionId),
            ["uuid", meetingId.value]
        ]);
    }
    Routes.MeetingRoster = MeetingRoster;

    /**
    * meeting/get
    */
    function GetMeeting(sessionId, meetingId) {
        return "/api/v3/meeting/get" + encodeParameters([
            sid(sessionId),
            ["uuid", meetingId.value]
        ]);
    }
    Routes.GetMeeting = GetMeeting;

    /**
    * meeting/list
    */
    function ListMeetings(sessionId, query) {
        return "/api/v3/meeting/list" + encodeParameters([
            sid(sessionId)
        ].concat(studyTriplet(query)));
    }
    Routes.ListMeetings = ListMeetings;

    /**
    * Storage address
    */
    function Storage(storageInfo, useServices, useMultipleHosts, canUseLocalAccelerator) {
        if (typeof useMultipleHosts === "undefined") { useMultipleHosts = false; }
        if (typeof canUseLocalAccelerator === "undefined") { canUseLocalAccelerator = false; }
        if (LocalViewer.isLocalViewer()) {
            return "./";
        } else if (canUseLocalAccelerator && storageInfo.localAccelerator) {
            return LocalAccelerator.LOCAL_ACCELERATOR_PROTOCOL + LocalAccelerator.LOCAL_ACCELERATOR_HOSTNAME + ":" + LocalAccelerator.LOCAL_ACCELERATOR_PORT + "/api/v3/storage";
        } else if (useServices) {
            return "/api/v3/cac";
        } else if (!ImageHosts.acceleratorUsed() && 'withCredentials' in new XMLHttpRequest() && !Browser.isIE()) {
            // CORS is supported, connect directly to engine.
            return (ImageHosts.getImageHost(storageInfo.engine_fqdn) || "//" + storageInfo.engine_fqdn) + "/api/v3/storage";
        } else {
            // CORS is not supported, use /host proxy.
            if (useMultipleHosts) {
                if (ImageHosts.acceleratorUsed()) {
                    return ImageHosts.getImageHost(location.hostname) + "/host/" + storageInfo.engine_fqdn + "/api/v3/storage";
                } else {
                    return ImageHosts.getImageHost() + "/host/" + storageInfo.engine_fqdn + "/api/v3/storage";
                }
            } else {
                return "/host/" + storageInfo.engine_fqdn + "/api/v3/storage";
            }
        }
    }
    Routes.Storage = Storage;

    var LocalAccelerator = (function () {
        function LocalAccelerator() {
        }
        LocalAccelerator.LOCAL_ACCELERATOR_HOSTNAME = "local.ambrahealth.com";
        LocalAccelerator.LOCAL_ACCELERATOR_PROTOCOL = "https://";
        LocalAccelerator.LOCAL_ACCELERATOR_PORT = "8021";
        LocalAccelerator.LOCAL_ACCELERATOR_TRUE = 1;
        return LocalAccelerator;
    })();
    Routes.LocalAccelerator = LocalAccelerator;

    /**
    * Private counter which tracks the next host to use for an image request
    */
    var ImageHosts = (function () {
        function ImageHosts() {
        }
        ImageHosts.nextHost = function (imageMaxHosts, engine_fqdn) {
            ImageHosts.currentImageHostIndex = (ImageHosts.currentImageHostIndex + 1) % imageMaxHosts;

            // image_host_index is 1-based
            return "//image" + (ImageHosts.currentImageHostIndex + 1).toString() + "-" + engine_fqdn;
        };

        ImageHosts.acceleratorUsed = function () {
            if (ImageHosts.isAcceleratorUsed === undefined) {
                if (Main && Main.Main && Main.Main.acceleratorUsed !== undefined) {
                    ImageHosts.isAcceleratorUsed = Main.Main.acceleratorUsed;
                }
            }
            return ImageHosts.isAcceleratorUsed;
        };

        ImageHosts.corsSupported = function () {
            if (ImageHosts.isCorsSupported === undefined) {
                var image = new window.Image();
                var corsBrowser = image.crossOrigin !== undefined;
                var isIE = (window.navigator.userAgent.indexOf("MSIE") != -1) || (navigator.userAgent.match(/Trident\/7\./) != null);
                ImageHosts.isCorsSupported = (corsBrowser && !isIE);
            }
            return ImageHosts.isCorsSupported;
        };

        ImageHosts.getImageHost = function (engine_fqdn) {
            if (typeof engine_fqdn === "undefined") { engine_fqdn = ImageHosts.imageBaseHost; }
            if (ImageHosts.isHttp2 === undefined) {
                ImageHosts.isHttp2 = false;

                if (performance && (typeof performance.getEntries === "function")) {
                    var entries = performance.getEntries();
                    for (var i = 0; i < entries.length; i++) {
                        if (entries[i].nextHopProtocol === 'h2') {
                            ImageHosts.isHttp2 = true;
                            break;
                        }
                    }
                }
            }

            if (ImageHosts.isHttp2)
                return '//' + engine_fqdn;

            if (!isNaN(ImageHosts.imageMaxHostsString)) {
                var imageMaxHosts = parseInt(ImageHosts.imageMaxHostsString);

                if (ImageHosts.corsSupported() && imageMaxHosts > 0 && typeof (ImageHosts.imageBaseHost) === "string" && ImageHosts.imageBaseHost.length > 0 && ImageHosts.imageBaseHost !== "(none)") {
                    return ImageHosts.nextHost(imageMaxHosts, engine_fqdn);
                }
            }

            return "";
        };
        ImageHosts.currentImageHostIndex = 0;
        ImageHosts.imageMaxHostsString = $("meta[name=imagemaxhosts]").attr("content");
        ImageHosts.imageBaseHost = $("meta[name=imagebasehost]").attr("content");
        return ImageHosts;
    })();

    /**
    * Get study metadata
    */
    function Study(storageInfo, queryObject, useServices, useMultipleHosts, canUseLocalAccelerator) {
        if (typeof useMultipleHosts === "undefined") { useMultipleHosts = false; }
        if (typeof canUseLocalAccelerator === "undefined") { canUseLocalAccelerator = false; }
        if (LocalViewer.isStandardLocalViewer()) {
            return Storage(storageInfo, useServices, useMultipleHosts, canUseLocalAccelerator);
        } else {
            return Storage(storageInfo, useServices, useMultipleHosts, canUseLocalAccelerator) + "/study/" + encodeURIComponent(queryObject.storageNamespace.value) + "/" + encodeURIComponent(queryObject.studyUid.value);
        }
    }
    Routes.Study = Study;

    /**
    * Get series metadata
    */
    function Series(storageInfo, queryObject, useServices, useMultipleHosts, seriesUid) {
        if (typeof useMultipleHosts === "undefined") { useMultipleHosts = false; }
        if (typeof seriesUid === "undefined") { seriesUid = null; }
        return Study(storageInfo, queryObject, useServices, useMultipleHosts) + "/series/" + encodeURIComponent(seriesUid.value);
    }
    Routes.Series = Series;

    /**
    * Get study schema information
    */
    function StudySchema(sessionId, storageInfo, queryObject, useServices) {
        return Study(storageInfo, queryObject, useServices, false, false) + "/schema" + encodeParameters([
            ["phi_namespace", queryObject.phiNamespace.value],
            sid(sessionId)
        ]);
    }
    Routes.StudySchema = StudySchema;

    /**
    * Get study PHI information
    */
    function StudyPhi(sessionId, storageInfo, queryObject) {
        return Study(storageInfo, queryObject, false, false, true) + "/phi" + encodeParameters([
            ["phi_namespace", queryObject.phiNamespace.value],
            sid(sessionId)
        ]);
    }
    Routes.StudyPhi = StudyPhi;

    /**
    * Get study tags
    */
    function StudyTag(sessionId, storageInfo, queryObject) {
        return Study(storageInfo, queryObject, false, false, true) + "/tag" + encodeParameters([
            ["phi_namespace", queryObject.phiNamespace.value],
            sid(sessionId)
        ]);
    }
    Routes.StudyTag = StudyTag;

    /**
    * Root URI for an image
    */
    function Image(storageInfo, queryObject, instanceUid, version, useServices, useMultipleHosts, canUseLocalAccelerator) {
        if (typeof useMultipleHosts === "undefined") { useMultipleHosts = false; }
        if (typeof canUseLocalAccelerator === "undefined") { canUseLocalAccelerator = false; }
        var uri;

        if (LocalViewer.isStandardLocalViewer()) {
            uri = Study(storageInfo, queryObject, useServices, useMultipleHosts);

            if (version) {
                uri += encodeURIComponent(version.value);
            }
        } else {
            uri = Study(storageInfo, queryObject, useServices, useMultipleHosts, canUseLocalAccelerator) + "/image/" + encodeURIComponent(instanceUid.value);

            if (version) {
                uri += "/version/" + encodeURIComponent(version.value);
            }
        }

        return uri;
    }
    Routes.Image = Image;

    /**
    * Get image data for a single frame
    */
    function ImageData(sessionId, storageInfo, queryObject, instanceUid, version, frameNumber, type, depth, useServices, useMultipleHosts, size) {
        if (typeof useMultipleHosts === "undefined") { useMultipleHosts = false; }
        if (typeof size === "undefined") { size = 0; }
        return Image(storageInfo, queryObject, instanceUid, version, useServices, useMultipleHosts, true) + (LocalViewer.isStandardLocalViewer() ? "/f/" : "/frame/") + frameNumber.value + Classes.ImageTypes.toUriComponent(type) + encodeParameters([
            sid(sessionId),
            ["phi_namespace", queryObject.phiNamespace.value],
            ["depth", depth.toString()]
        ]) + ((size > 0) ? "&size=" + size : "");
    }
    Routes.ImageData = ImageData;

    /**
    * Get PDF data for encapsulated PDF
    */
    function PDFData(sessionId, storageInfo, queryObject, instanceUid, version, useServices, useMultipleHosts) {
        if (typeof useMultipleHosts === "undefined") { useMultipleHosts = false; }
        return Image(storageInfo, queryObject, instanceUid, version, useServices, useMultipleHosts) + '/pdf' + encodeParameters([
            sid(sessionId),
            ["phi_namespace", queryObject.phiNamespace.value]
        ]);
    }
    Routes.PDFData = PDFData;

    /**
    * Get Video data for encapsulated Video
    */
    function VideoData(sessionId, storageInfo, queryObject, instanceUid, version, useServices, useMultipleHosts) {
        if (typeof useMultipleHosts === "undefined") { useMultipleHosts = false; }
        return Image(storageInfo, queryObject, instanceUid, version, useServices, useMultipleHosts) + '/video' + encodeParameters([
            sid(sessionId),
            ["phi_namespace", queryObject.phiNamespace.value]
        ]);
    }
    Routes.VideoData = VideoData;

    /**
    * Get Video download endpoint
    */
    function VideoDownload(sessionId, storageInfo, queryObject, seriesUid, format) {
        return Series(storageInfo, queryObject, false, false, seriesUid) + "/video" + encodeParameters([
            sid(sessionId),
            ["phi_namespace", queryObject.phiNamespace.value],
            ["format", format]
        ]);
    }
    Routes.VideoDownload = VideoDownload;

    /**
    * Download series
    */
    function SeriesDownload(sessionId, storageInfo, queryObject, seriesUid) {
        return Study(storageInfo, queryObject, false) + "/download" + encodeParameters([
            sid(sessionId),
            ["phi_namespace", queryObject.phiNamespace.value],
            ["series_uid", seriesUid.value],
            ["bundle", "dicom"]
        ]);
    }
    Routes.SeriesDownload = SeriesDownload;

    /**
    * Download study
    * @param sessionId
    * @param storageInfo
    * @param queryObject
    * @param seriesUid
    * @constructor
    */
    function StudyDownload(sessionId, storageInfo, queryObject) {
        return Study(storageInfo, queryObject, false) + "/download" + encodeParameters([
            sid(sessionId),
            ["phi_namespace", queryObject.phiNamespace.value],
            ["bundle", "dicom"]
        ]);
    }
    Routes.StudyDownload = StudyDownload;

    /**
    * Download ISO
    * @param sessionId
    * @param storageInfo
    * @param queryObject
    * @param seriesUid
    * @constructor
    */
    function ISODownload(sessionId, storageInfo, queryObject) {
        return Study(storageInfo, queryObject, false) + "/download" + encodeParameters([
            sid(sessionId),
            ["phi_namespace", queryObject.phiNamespace.value],
            ["bundle", "iso"]
        ]);
    }
    Routes.ISODownload = ISODownload;

    /**
    * Download local viewer
    * @param sessionId
    * @param storageInfo
    * @param queryObject
    * @param seriesUid
    * @constructor
    */
    function LocalViewerDownload(sessionId, storageInfo, queryObject) {
        return Study(storageInfo, queryObject, false) + "/download" + encodeParameters([
            sid(sessionId),
            ["phi_namespace", queryObject.phiNamespace.value],
            ["bundle", Browser.viewerOS()]
        ]);
    }
    Routes.LocalViewerDownload = LocalViewerDownload;

    /**
    * Get attributes for a study (when the accelerator is present)
    */
    function StudyAttributes(sessionId, storageInfo, queryObject) {
        return Study(storageInfo, queryObject, false) + "/attributes" + encodeParameters([
            sid(sessionId),
            ["phi_namespace", queryObject.phiNamespace.value]
        ]);
    }
    Routes.StudyAttributes = StudyAttributes;

    /**
    * Get attributes for an instance
    */
    function ImageAttributes(sessionId, storageInfo, queryObject, instanceUid, version) {
        return Image(storageInfo, queryObject, instanceUid, version, false, false, true) + "/attribute" + encodeParameters([
            sid(sessionId),
            ["phi_namespace", queryObject.phiNamespace.value]
        ]);
    }
    Routes.ImageAttributes = ImageAttributes;

    /**
    * Get all attributes for an instance
    */
    function ImageJSON(sessionId, storageInfo, queryObject, instanceUid, version) {
        return Image(storageInfo, queryObject, instanceUid, version, false, false, true) + "/json" + encodeParameters([
            sid(sessionId),
            ["phi_namespace", queryObject.phiNamespace.value],
            ["all_dicom_values", "1"]
        ]);
    }
    Routes.ImageJSON = ImageJSON;

    /**
    * Get CAD SR annotations
    * @param sessionId
    * @param storageInfo
    * @param queryObject
    * @param instanceUid
    * @param version
    * @constructor
    */
    function CADSR(sessionId, storageInfo, queryObject, instanceUid, version) {
        return Image(storageInfo, queryObject, instanceUid, version, false, false, false) + "/cadsr" + encodeParameters([
            sid(sessionId),
            ["phi_namespace", queryObject.phiNamespace.value]
        ]);
    }
    Routes.CADSR = CADSR;

    /**
    * Delete URI for an image
    */
    function DeleteImage(sessionId, storageInfo, queryObject, instanceUid, version) {
        var uri = Image(storageInfo, queryObject, instanceUid, null, false);

        return uri + encodeParameters([sid(sessionId)].concat(version ? [["version", version.value]] : []));
    }
    Routes.DeleteImage = DeleteImage;

    function DeleteAttachment(sessionId, storageInfo, queryObject, attachment) {
        return Attachment(sessionId, storageInfo, queryObject, attachment.id, attachment.phiNamespace, attachment.version);
    }
    Routes.DeleteAttachment = DeleteAttachment;

    /**
    * Get GSPS data
    */
    function GetGSPS(sessionId, storageInfo, queryObject, instanceUid, version) {
        return Image(storageInfo, queryObject, instanceUid, version, false, false, true) + "/gsps" + encodeParameters([
            sid(sessionId),
            ["phi_namespace", queryObject.phiNamespace.value]
        ]);
    }
    Routes.GetGSPS = GetGSPS;

    /**
    * Video for a multiframe instance
    */
    function Multiframe(sessionId, storageInfo, queryObject, instanceUid, version) {
        return Image(storageInfo, queryObject, instanceUid, version, false) + "/multiframe" + encodeParameters([
            sid(sessionId),
            ["phi_namespace", queryObject.phiNamespace.value],
            ["millisPerFrame=-1"]
        ]);
    }
    Routes.Multiframe = Multiframe;

    /**
    * Post a secondary capture image
    */
    function SecondaryCapture(sessionId, storageInfo, queryObject) {
        return Study(storageInfo, queryObject, false) + "/capture" + encodeParameters([
            sid(sessionId),
            ["phi_namespace", queryObject.phiNamespace.value]
        ]);
    }
    Routes.SecondaryCapture = SecondaryCapture;

    /**
    * Anonymize a series
    */
    function AnonymizeStudy(sessionId, storageInfo, queryObject) {
        return Study(storageInfo, queryObject, false) + "/anonymize" + encodeParameters([
            sid(sessionId),
            ["phi_namespace", queryObject.phiNamespace.value]
        ]);
    }
    Routes.AnonymizeStudy = AnonymizeStudy;

    /**
    * Split series out of study.
    * @param sessionId
    * @param storageInfo
    * @param queryObject
    * @param seriesUIDs
    * @constructor
    */
    function SplitStudy(sessionId, storageInfo, queryObject, seriesUIDs) {
        return Study(storageInfo, queryObject, false) + "/split" + encodeParameters([
            sid(sessionId),
            ["phi_namespace", queryObject.phiNamespace.value],
            ["delete_series_from_original", "1"]
        ]) + "&series_uid=" + seriesUIDs.join(',');
    }
    Routes.SplitStudy = SplitStudy;

    function CropStudy(sessionId, storageInfo, queryObject) {
        return Study(storageInfo, queryObject, false) + "/crop" + encodeParameters([
            sid(sessionId),
            ["phi_namespace", queryObject.phiNamespace.value]]);
    }
    Routes.CropStudy = CropStudy;

    /**
    * Post GSPS data
    */
    function PutGSPS(sessionId, storageInfo, queryObject) {
        return Study(storageInfo, queryObject, false) + "/gsps" + encodeParameters([
            sid(sessionId),
            ["phi_namespace", queryObject.phiNamespace.value]
        ]);
    }
    Routes.PutGSPS = PutGSPS;

    /**
    * Download an attachment
    */
    function Attachment(sessionId, storageInfo, queryObject, attachmentId, phiNamespace, version) {
        return Study(storageInfo, queryObject, false) + "/attachment/" + attachmentId.value + "/version/" + version.value + encodeParameters([
            sid(sessionId),
            ["phi_namespace", queryObject.phiNamespace.value]
        ]);
    }
    Routes.Attachment = Attachment;

    /**
    * Upload an attachment
    */
    function PostAttachment(sessionId, storageInfo, queryObject, contentType, uploadedBy, filename) {
        if (typeof filename === "undefined") { filename = "recorder.json"; }
        return Study(storageInfo, queryObject, false) + "/attachment/blob" + encodeParameters([
            sid(sessionId),
            ["phi_namespace", queryObject.phiNamespace.value],
            ["content_type", contentType],
            ["uploaded_by", uploadedBy],
            ["filename", filename]
        ]);
    }
    Routes.PostAttachment = PostAttachment;

    /**
    * Upload an image attachment
    * @param {Classes.SessionId} sessionId
    * @param {Models.StudyStorageInfo} storageInfo
    * @param {Classes.QueryObject} queryObject
    * @param {string} uploadedBy
    * @returns {string}
    */
    function PostImageAttachment(sessionId, storageInfo, queryObject, uploadedBy, filename) {
        return Study(storageInfo, queryObject, false) + "/attachment" + encodeParameters([
            sid(sessionId),
            ["phi_namespace", queryObject.phiNamespace.value],
            ["content_type", "image/png"],
            ["uploaded_by", uploadedBy],
            ["filename", filename]
        ]);
    }
    Routes.PostImageAttachment = PostImageAttachment;

    /**
    * Download an attachment
    */
    function HL7Report(sessionId, studyId, reportId) {
        return "/report.html" + encodeParameters([
            sid(sessionId),
            ["study_uuid", studyId.value],
            ["uuid", reportId.value]
        ]);
    }
    Routes.HL7Report = HL7Report;

    /**
    * Download a structured report
    */
    function StructuredReport(sessionId, studyStorage, queryObject, id, version) {
        return Image(studyStorage, queryObject, id, version, false) + "/sr" + encodeParameters([
            sid(sessionId),
            ["phi_namespace", queryObject.phiNamespace.value]
        ]);
    }
    Routes.StructuredReport = StructuredReport;
})(Routes || (Routes = {}));
/**
* A library for working with composable updatable values, aka. Subjects
*/
var Subjects;
(function (Subjects) {
    

    

    /**
    * A trivial updatable value which never changes
    */
    function ret(t) {
        return {
            read: function () {
                return t;
            },
            updates: Observable.empty()
        };
    }
    Subjects.ret = ret;

    /**
    * Like ret, but for writable subjects
    */
    function retW(t) {
        var sub1 = Subjects.ret(t);

        return {
            read: sub1.read,
            updates: sub1.updates,
            write: function (t) {
                throw new Error("Tried to write to read-only subject.");
            }
        };
    }
    Subjects.retW = retW;

    /**
    * Apply a value when reading the latest value
    */
    function map(s, f) {
        return {
            read: function () {
                return f(s.read());
            },
            updates: Observable.map(s.updates, f)
        };
    }
    Subjects.map = map;

    /**
    * Composition of updatable values. The choice of second updatable value may depend on the value of the first.
    */
    function bind(sub, f, g) {
        return {
            read: function () {
                var s = sub.read();
                var t = f(s).read();
                return g(s, t);
            },
            updates: Observable.bind2(sub.updates, function (s) {
                var tSub = f(s);
                var initialValue = tSub.read();
                return Observable.cons(initialValue, tSub.updates);
            }, g)
        };
    }
    Subjects.bind = bind;

    /**
    * Like bind, but for writable subjects
    */
    function bindW(sub, f) {
        var sub1 = Subjects.bind(sub, f, function (s, t) {
            return t;
        });

        return {
            read: sub1.read,
            updates: sub1.updates,
            write: function (t) {
                var s = sub.read();
                var tSub = f(s);
                tSub.write(t);
            }
        };
    }
    Subjects.bindW = bindW;

    /**
    * Parallel composition of independent updatable values.
    */
    function zip(s1, s2, f) {
        var s0 = s1.read();
        var t0 = s2.read();

        return {
            read: function () {
                var s = s1.read();
                var t = s2.read();
                return f(s, t);
            },
            updates: Observable.zipLatest(s1.updates, s2.updates, function (s, t) {
                return f(s === undefined ? s0 : s, t === undefined ? t0 : t);
            })
        };
    }
    Subjects.zip = zip;

    /**
    * Apply a function when reading a WritableSubject's value, and another function when writing its value.
    */
    function dimap(sub, t, s) {
        return {
            read: function () {
                var s = sub.read();
                return t(s);
            },
            updates: Observable.map(sub.updates, t),
            write: function (t) {
                sub.write(s(t));
            }
        };
    }
    Subjects.dimap = dimap;

    /**
    * Perform an action when a value is written
    */
    function onWrite(sub, action) {
        return {
            read: sub.read,
            updates: sub.updates,
            write: function (t) {
                sub.write(t);
                action(t);
            }
        };
    }
    Subjects.onWrite = onWrite;

    /**
    * Apply a getter/setter pair to a Subject to get a WritableSubject
    */
    function lens(sub, _get, _set) {
        return {
            read: function () {
                var s = sub.read();
                return _get(s);
            },
            updates: Observable.map(sub.updates, _get),
            write: function (t) {
                Subjects.modify(sub, function (s) {
                    return _set(s, t);
                });
            }
        };
    }
    Subjects.lens = lens;

    /**
    * Modify a writable subject by applying a function
    */
    function modify(sub, f) {
        sub.write(f(sub.read()));
    }
    Subjects.modify = modify;

    /**
    * Modify a writable subject by applying a function
    */
    function listen(sub, onNext) {
        return sub.updates.subscribe({
            next: onNext,
            done: function () {
            },
            fail: function (_) {
            }
        });
    }
    Subjects.listen = listen;

    /**
    * A default implementation of a Subject
    */
    var SubjectBase = (function () {
        function SubjectBase() {
            var _this = this;
            this.updates = {
                subscribe: function (ob) {
                    _this.observers.push(ob);
                    return {
                        cancel: function () {
                            for (var i = 0; i < _this.observers.length; i++) {
                                if (_this.observers[i] === ob) {
                                    _this.observers.splice(i, 1);
                                    break;
                                }
                            }
                        }
                    };
                }
            };
            this.observers = [];
        }
        SubjectBase.prototype.raiseChangedEvent = function (value) {
            for (var i = 0; i < this.observers.length; i++) {
                this.observers[i].next(value);
            }
        };
        return SubjectBase;
    })();
    Subjects.SubjectBase = SubjectBase;

    /**
    * A default implementation of a WritableSubject
    */
    var ObservableValue = (function (_super) {
        __extends(ObservableValue, _super);
        function ObservableValue(value) {
            _super.call(this);
            this.value = value;
        }
        ObservableValue.prototype.read = function () {
            return this.value;
        };

        ObservableValue.prototype.write = function (value) {
            this.value = value;
            this.raiseChangedEvent(this.value);
        };
        return ObservableValue;
    })(SubjectBase);
    Subjects.ObservableValue = ObservableValue;
})(Subjects || (Subjects = {}));
/// <reference path="Subject.ts" />
/**
* Priority queues
*/
var Queue;
(function (_Queue) {
    

    

    

    

    

    /**
    * A default implementation of an observable priority queue, implemented using a sorted array of priority/item pairs.
    */
    var ObservablePriorityQueue = (function () {
        function ObservablePriorityQueue() {
            this.items = [];
            this.isEmpty = new Subjects.ObservableValue(true);
        }
        ObservablePriorityQueue.prototype.clear = function () {
            this.items = [];
            this.isEmpty.write(true);
        };

        ObservablePriorityQueue.prototype.dequeue = function () {
            var t = this.items.pop();

            if (this.items.length == 0) {
                this.isEmpty.write(true);
            }

            return {
                key: t.key,
                value: t.value
            };
        };

        ObservablePriorityQueue.prototype.enqueue = function (key, t, priority, isKey, isPriority) {
            var index = this.items.indexWhere(function (item) {
                return isKey.equals(item.key, key);
            });

            if (index >= 0 && this.items[index].priority > priority) {
                return;
            }

            var item;

            if (index >= 0) {
                item = { key: key, value: this.items[index].value, priority: priority };
                this.items.splice(index, 1);
            } else {
                item = { key: key, value: t, priority: priority };
            }

            var insertIndex = this.items.lastSortedIndex(item, function (item) {
                return item.priority;
            }, isPriority.compare);

            this.items.splice(insertIndex + 1, 0, item);

            if (this.items.length > 0) {
                this.isEmpty.write(false);
            }
        };
        return ObservablePriorityQueue;
    })();
    _Queue.ObservablePriorityQueue = ObservablePriorityQueue;
})(Queue || (Queue = {}));
/// <reference path="Queue.ts" />
/**
* A library for working with asychronous values (aka. futures, aka. observables)
*/
var Observable;
(function (_Observable) {
    

    

    /**
    * Monoidal unit for subscriptions
    */
    _Observable.memptySubscription = {
        cancel: function () {
        }
    };

    /**
    * Monoidal composition for subscriptions
    */
    function mappendSubscription(s1, s2) {
        return {
            cancel: function () {
                s1.cancel();
                s2.cancel();
            }
        };
    }
    _Observable.mappendSubscription = mappendSubscription;

    /**
    * A cancellation token
    */
    var CancellationToken = (function () {
        function CancellationToken() {
            this.cancelled = false;
        }
        /**
        * Cancel the subscription
        */
        CancellationToken.prototype.cancel = function () {
            this.cancelled = true;
        };
        return CancellationToken;
    })();
    _Observable.CancellationToken = CancellationToken;

    

    /**
    * A trivial computation which returns a value immediately and then terminates.
    */
    function ret(t) {
        return {
            subscribe: function (ob) {
                ob.next(t);
                ob.done();
                return _Observable.memptySubscription;
            }
        };
    }
    _Observable.ret = ret;

    /**
    * A trivial computation which fails immediately with an error message.
    */
    function fail(msg) {
        return {
            subscribe: function (ob) {
                ob.fail(msg);
                return _Observable.memptySubscription;
            }
        };
    }
    _Observable.fail = fail;

    /**
    * A trivial computation which terminates immediately.
    */
    function empty() {
        return {
            subscribe: function (ob) {
                ob.done();
                return _Observable.memptySubscription;
            }
        };
    }
    _Observable.empty = empty;

    /**
    * Apply a function when an asynchronous computation successfully completes.
    */
    function map(o, f) {
        return {
            subscribe: function (ob) {
                return o.subscribe({
                    next: function (s) {
                        return ob.next(f(s));
                    },
                    done: ob.done,
                    fail: ob.fail
                });
            }
        };
    }
    _Observable.map = map;

    /**
    * Flatten a stream-of-streams
    */
    function join(o) {
        return Observable.bind(o, function (ts) {
            return ts;
        });
    }
    _Observable.join = join;

    /**
    * Sequential composition of asynchronous computations.
    *
    * For example, the result of one AJAX call may determine the input to a second AJAX request.This function
    * allows the result of the first to be bound to a variable which can be used to compute a second Observable.
    * Subscription to the resulting Observable will then trigger the sequential computation.
    */
    function bind(o, f) {
        return {
            subscribe: function (ob) {
                var latestSubscription = null;

                var subscription = o.subscribe({
                    done: function () {
                        ob.done();
                    },
                    fail: ob.fail,
                    next: function (s) {
                        if (latestSubscription) {
                            latestSubscription.cancel();
                        }
                        latestSubscription = f(s).subscribe({
                            done: function () {
                            },
                            next: function (t) {
                                ob.next(t);
                            },
                            fail: function (msg) {
                                subscription.cancel();
                                ob.fail(msg);
                                latestSubscription = null;
                            }
                        });
                    }
                });

                return {
                    cancel: function () {
                        if (latestSubscription) {
                            latestSubscription.cancel();
                            latestSubscription = null;
                        }
                        subscription.cancel();
                    }
                };
            }
        };
    }
    _Observable.bind = bind;

    /**
    * Cons a result onto the front of a stream
    */
    function cons(t0, o) {
        return {
            subscribe: function (ob) {
                ob.next(t0);
                return o.subscribe(ob);
            }
        };
    }
    _Observable.cons = cons;

    /**
    * A variation on bind which applies a function after successful completion of both asynchronous computations.
    *
    * @see bind
    */
    function bind2(o, f, g) {
        return bind(o, function (s) {
            return map(f(s), function (t) {
                return g(s, t);
            });
        });
    }
    _Observable.bind2 = bind2;

    /**
    * An asynchronous computation which returns a value if a condition is met
    */
    function filter(o, p) {
        return {
            subscribe: function (ob) {
                return o.subscribe({
                    done: ob.done,
                    next: function (t) {
                        if (p(t)) {
                            ob.next(t);
                        }
                    },
                    fail: ob.fail
                });
            }
        };
    }
    _Observable.filter = filter;

    /**
    * An asynchronous computation which only starts if a condition is met
    *
    * @param {test} The condition which must be met before computation starts
    * @returns A computation which either starts o and returns its result, or succeeds immediately with an empty value.
    */
    function ifThenElse(test, tr, fa) {
        return {
            subscribe: function (ob) {
                if (test()) {
                    return tr.subscribe(ob);
                } else {
                    return fa.subscribe(ob);
                }
            }
        };
    }
    _Observable.ifThenElse = ifThenElse;

    /**
    * An asynchronous computation which runs a method on successful completion
    *
    * @param {f} The method to call on successful completion
    */
    function invoke(o, f) {
        return {
            subscribe: function (ob) {
                return o.subscribe({
                    done: ob.done,
                    next: function (t) {
                        f(t);
                        ob.next(t);
                    },
                    fail: ob.fail
                });
            }
        };
    }
    _Observable.invoke = invoke;

    /**
    * Run a method on each event
    */
    function on(o, onNext, onFail, onDone) {
        return {
            subscribe: function (ob) {
                return o.subscribe({
                    done: function () {
                        onDone();
                        ob.done();
                    },
                    next: function (t) {
                        onNext(t);
                        ob.next(t);
                    },
                    fail: function (err) {
                        onFail(err);
                        ob.fail(err);
                    }
                });
            }
        };
    }
    _Observable.on = on;

    /**
    * An asynchronous computation which runs a method on subscription
    *
    * @param {f} The method to call on subscription
    */
    function invokeFirst(o, f) {
        return {
            subscribe: function (ob) {
                f();
                return o.subscribe(ob);
            }
        };
    }
    _Observable.invokeFirst = invokeFirst;

    /**
    * An asynchronous computation which runs a method on successful completion
    *
    * @param {f} The method to call on completion or failure
    */
    function _finally(o, f) {
        return {
            subscribe: function (ob) {
                return o.subscribe({
                    done: function () {
                        f();
                        ob.done();
                    },
                    next: function (t) {
                        ob.next(t);
                    },
                    fail: function (err) {
                        f();
                        ob.fail(err);
                    }
                });
            }
        };
    }
    _Observable._finally = _finally;

    /**
    * Parallel composition of asynchronous computations. The resulting Observable succeeds if both inputs succeed, and
    * applies a function to the two results.
    */
    function zip(o1, o2, f) {
        return {
            subscribe: function (ob) {
                var ss = [];
                var ts = [];
                var s1 = o1.subscribe({
                    next: function (s) {
                        if (ts.length == 0) {
                            ss.push(s);
                        } else {
                            var t = ts[0];
                            ob.next(f(s, t));
                            ts = ts.slice(1);
                        }
                    },
                    done: function () {
                        ob.done();
                    },
                    fail: function (err) {
                        s2.cancel();
                        ob.fail(err);
                    }
                });
                var s2 = o2.subscribe({
                    next: function (t) {
                        if (ss.length == 0) {
                            ts.push(t);
                        } else {
                            var s = ss[0];
                            ob.next(f(s, t));
                            ss = ss.slice(1);
                        }
                    },
                    done: function () {
                        ob.done();
                    },
                    fail: function (err) {
                        s1.cancel();
                        ob.fail(err);
                    }
                });
                return mappendSubscription(s1, s2);
            }
        };
    }
    _Observable.zip = zip;

    /**
    * Parallel composition of asynchronous computations. The resulting Observable succeeds whenever either input succeeds, and
    * applies a function to the two latest results.
    */
    function zipLatest(o1, o2, f) {
        return {
            subscribe: function (ob) {
                var latest_s;
                var latest_t;

                var s1 = o1.subscribe({
                    next: function (s) {
                        ob.next(f(s, latest_t));
                        latest_s = s;
                    },
                    done: ob.done,
                    fail: function (err) {
                        s2.cancel();
                        ob.fail(err);
                    }
                });
                var s2 = o2.subscribe({
                    next: function (t) {
                        ob.next(f(latest_s, t));
                        latest_t = t;
                    },
                    done: ob.done,
                    fail: function (err) {
                        s1.cancel();
                        ob.fail(err);
                    }
                });
                return mappendSubscription(s1, s2);
            }
        };
    }
    _Observable.zipLatest = zipLatest;

    /**
    * Applicative composition
    */
    function ap(f, x) {
        return zip(f, x, function (a, b) {
            return a(b);
        });
    }
    _Observable.ap = ap;

    /**
    * Parallel composition of multiple asynchronous computations.
    * The resulting Observable succeeds if all inputs succeed.
    */
    function sequenceA(os) {
        if (os.length == 0) {
            return ret([]);
        }

        var head = os[0];
        var tail = os.slice(1);

        return ap(map(head, function (t) {
            return function (ts) {
                return [t].concat(ts);
            };
        }), sequenceA(tail));
    }
    _Observable.sequenceA = sequenceA;

    /**
    * Sequential composition of multiple asynchronous computations.
    * The resulting Observable succeeds if all inputs succeed.
    */
    function sequenceM(os) {
        return {
            subscribe: function (ob) {
                var results = [];
                var currentSubscription = null;

                var fire = function (i) {
                    if (i < os.length) {
                        currentSubscription = os[i].subscribe({
                            next: function (t) {
                                results[i] = t;
                            },
                            done: function () {
                                fire(i + 1);
                            },
                            fail: function (err) {
                                ob.fail(err);
                            }
                        });
                    } else {
                        ob.done();
                    }
                };

                fire(0);

                return {
                    cancel: function () {
                        if (currentSubscription) {
                            currentSubscription.cancel();
                        }
                    }
                };
            }
        };
    }
    _Observable.sequenceM = sequenceM;

    /**
    * Forget the return type of a computation.
    */
    function forget(o) {
        return map(o, function (_) {
            return {};
        });
    }
    _Observable.forget = forget;

    /**
    * Handle the failure case by returning a specified value.
    *
    * @param {onError} The function to apply on failure
    */
    function catchError(o, onError) {
        return catchErrorWith(o, function (err) {
            return ret(onError(err));
        });
    }
    _Observable.catchError = catchError;

    /**
    * Create an Observable which only yields a finite number of results.
    */
    function take(n, o) {
        return bind(withIndex(o), function (s) {
            if (s.index >= n) {
                return empty();
            } else {
                return ret(s.value);
            }
        });
    }
    _Observable.take = take;

    /**
    * Create an Observable which also yields the indices of its results.
    */
    function withIndex(o) {
        return scan(o, null, function (s, t) {
            return {
                value: t,
                index: s !== null ? s.index + 1 : 0
            };
        });
    }
    _Observable.withIndex = withIndex;

    /**
    * Create an Observable which accumulates state based on its inputs.
    */
    function scan(o, s0, f) {
        return {
            subscribe: function (ob) {
                var s = s0;
                return o.subscribe({
                    done: ob.done,
                    fail: ob.fail,
                    next: function (t) {
                        s = f(s, t);
                        ob.next(s);
                    }
                });
            }
        };
    }
    _Observable.scan = scan;

    /**
    * Handle the failure case by returning the value from a second Observable.
    *
    * @param {onError} The observable to subscribe to on failure
    */
    function catchErrorWith(o, onError) {
        return {
            subscribe: function (ob) {
                var innerSubscription = null;

                var subscription = o.subscribe({
                    done: ob.done,
                    next: ob.next,
                    fail: function (err) {
                        innerSubscription = onError(err).subscribe(ob);
                    }
                });

                return {
                    cancel: function () {
                        if (innerSubscription) {
                            innerSubscription.cancel();
                        }
                        subscription.cancel();
                    }
                };
            }
        };
    }
    _Observable.catchErrorWith = catchErrorWith;

    /**
    * An observable which fires once every number of milliseconds
    */
    function interval(millis) {
        return {
            subscribe: function (ob) {
                var interval = window.setInterval(function () {
                    ob.next({});
                }, millis);

                return {
                    cancel: function () {
                        window.clearInterval(interval);
                    }
                };
            }
        };
    }
    _Observable.interval = interval;

    /**
    * Tries the observable, and fails if no answer is received after
    * the specified number of milliseconds.
    */
    function timeout(o, millis) {
        return {
            subscribe: function (ob) {
                var completed = false;
                var timeout = window.setTimeout(function () {
                    if (!completed) {
                        ob.fail("Timed out after " + millis + "ms");
                        completed = true;
                    }
                }, millis);
                var subscription = o.subscribe({
                    next: function (t) {
                        ob.next(t);
                    },
                    done: function () {
                        if (!completed) {
                            window.clearTimeout(timeout);
                            ob.done();
                            completed = true;
                        }
                    },
                    fail: function (err) {
                        if (!completed) {
                            ob.fail(err);
                            completed = true;
                        }
                    }
                });
                return {
                    cancel: function () {
                        if (!completed) {
                            window.clearTimeout(timeout);
                        }
                        subscription.cancel();
                    }
                };
            }
        };
    }
    _Observable.timeout = timeout;

    var QueueObservable = (function () {
        function QueueObservable(queue, maxConcurrentTasks, isKey) {
            this.concurrentTasks = 0;
            this.cancelled = false;
            this.queue = queue;
            this.maxConcurrentTasks = maxConcurrentTasks;
            this.isKey = isKey;
        }
        QueueObservable.prototype.tryFire = function () {
            var _this = this;
            while (!this.cancelled && this.concurrentTasks < this.maxConcurrentTasks && !this.queue.isEmpty.read()) {
                this.concurrentTasks++;

                var item = this.queue.dequeue();

                item.value.subscribe({
                    next: function (_) {
                    },
                    done: function () {
                        _this.concurrentTasks--;
                        setTimeout(function () {
                            return _this.tryFire();
                        }, 0);
                    },
                    fail: function (err) {
                        console.error("Queued job " + _this.isKey.show(item.key) + " failed: " + err);
                        _this.concurrentTasks--;
                        setTimeout(function () {
                            return _this.tryFire();
                        }, 0);
                    }
                });
            }
        };

        QueueObservable.prototype.subscribe = function (ob) {
            var _this = this;
            this.queue.isEmpty.updates.subscribe({
                done: function () {
                },
                next: function (_) {
                    _this.tryFire();
                },
                fail: function (err) {
                    console.error("Error waiting for updates from the queue: " + err);
                }
            });

            for (var i = 0; i < this.maxConcurrentTasks; i++) {
                this.tryFire();
            }

            return {
                cancel: function () {
                    _this.cancelled = true;
                }
            };
        };
        return QueueObservable;
    })();

    /**
    * Queued composition of many Observables.
    *
    * The queue is repeatedly polled as Observables succeed or fail. The Observer never receives updates directly, so
    * some other method of observation, such as invoke, should be used to respond to copmpletion of individual computations.
    *
    * @param {queue} The queue which provides new work
    * @param {maxConcurrentTasks} The maximum number of parallel tasks to spawn
    */
    function queue(queue, maxConcurrentTasks, isKey) {
        return new QueueObservable(queue, maxConcurrentTasks, isKey);
    }
    _Observable.queue = queue;

    /**
    * An observable which passes after a delay
    */
    function delay(millis) {
        return {
            subscribe: function (ob) {
                var timeout = window.setTimeout(function () {
                    ob.next({});
                    ob.done();
                }, millis);
                return {
                    cancel: function () {
                        window.clearTimeout(timeout);
                    }
                };
            }
        };
    }
    _Observable.delay = delay;

    /**
    * Try an observable a number of times with a delay
    */
    function retryWithDelay(o, millis, maxRetries) {
        if (maxRetries <= 1) {
            return o;
        }

        return catchErrorWith(o, function () {
            return bind(delay(millis), function (_) {
                return retryWithDelay(o, millis, maxRetries - 1);
            });
        });
    }
    _Observable.retryWithDelay = retryWithDelay;
})(Observable || (Observable = {}));
/// <reference path="Observable.ts" />
/**
* Helper functions for using jQuery's AJAX methods with Observables
*/
var AJAX;
(function (AJAX) {
    /**
    * An AJAX GET request as an Observable
    */
    function ajaxGet(uri, dataType) {
        if (typeof dataType === "undefined") { dataType = "json"; }
        return ajax(uri, { dataType: dataType });
    }
    AJAX.ajaxGet = ajaxGet;

    /**
    * An AJAX POST request as an Observable
    */
    function ajax(uri, options) {
        return {
            subscribe: function (ob) {
                var token = new Observable.CancellationToken();

                var ajax = $.ajax(uri, options);

                ajax.done(function (res) {
                    if (!token.cancelled) {
                        ob.next(res);
                        ob.done();
                    }
                }).fail(function (_1, _2, err) {
                    if (!token.cancelled) {
                        // Redirect on unauthorized
                        if (_1.status == 401) {
                            window.location.assign("/");
                        }

                        var errorMsg = err;
                        if (_1 && _1.responseText) {
                            try  {
                                var json = JSON.parse(_1.responseText);
                                if (json["error_type"]) {
                                    errorMsg += (" (" + json["error_type"] + ")");
                                } else {
                                    errorMsg += (" (" + _1.responseText + ")");
                                }
                            } catch (ex) {
                                errorMsg += (" (" + _1.status + ")");
                            }
                        }

                        ob.fail(errorMsg);
                    }
                });

                return token;
            }
        };
    }
    AJAX.ajax = ajax;
})(AJAX || (AJAX = {}));
var Models;
(function (Models) {
    

    

    

    

    

    

    

    
})(Models || (Models = {}));
var Models;
(function (Models) {
    /**
    * List of terminology returned by the terminology tags service call
    */
    var TerminologyTags = (function () {
        function TerminologyTags() {
        }
        return TerminologyTags;
    })();
    Models.TerminologyTags = TerminologyTags;
})(Models || (Models = {}));
var Models;
(function (Models) {
    

    

    

    

    

    

    

    

    

    

    

    

    

    /**
    * Post attachment result
    */
    var AttachmentResult = (function () {
        function AttachmentResult() {
        }
        return AttachmentResult;
    })();
    Models.AttachmentResult = AttachmentResult;
})(Models || (Models = {}));
/// <reference path="../classes/Types.ts" />
/// <reference path="../classes/Meeting.ts" />
/// <reference path="../typings/jquery/jquery.d.ts" />
/// <reference path="Routes.ts" />
/// <reference path="Observable.ts" />
/// <reference path="AJAX.ts" />
/// <reference path="../models/Annotations.ts" />
/// <reference path="../models/User.ts" />
/// <reference path="../models/Terminology.ts" />
/// <reference path="../models/StudySchema.ts" />
/// <reference path="../models/Permissions.ts" />
/// <reference path="../models/KeyImages.ts" />
/**
* Observables for service methods
*
* @see Observable
* @see Routes
*/
var Services;
(function (Services) {
    /**
    * A wrapper for the GetUser AJAX call
    */
    function getUser(sessionId) {
        if (!LocalViewer.isLocalViewer()) {
            return AJAX.ajaxGet(Routes.GetUser(sessionId));
        } else {
            if (LocalViewer.user) {
                return Observable.ret(LocalViewer.user);
            } else {
                return Observable.ret({
                    uuid: "",
                    name: "Local User",
                    first: "",
                    last: "",
                    email: "",
                    namespace_id: "",
                    sid_md5: ""
                });
            }
        }
    }
    Services.getUser = getUser;

    /**
    * A wrapper for the GetSettings AJAX call
    */
    function getPermissions(sessionId, phiNamespace) {
        if (!LocalViewer.isLocalViewer()) {
            return AJAX.ajaxGet(Routes.GetPermissions(sessionId, phiNamespace));
        } else {
            if (LocalViewer.permissions) {
                return Observable.ret(LocalViewer.permissions);
            } else {
                return Observable.ret({});
            }
        }
    }
    Services.getPermissions = getPermissions;

    function getStoragePermissions(sessionId, phiNamespace) {
        if (!LocalViewer.isLocalViewer()) {
            return AJAX.ajaxGet(Routes.GetStoragePermissions(sessionId, phiNamespace));
        } else {
            if (LocalViewer.permissions) {
                return Observable.ret(LocalViewer.permissions);
            } else {
                return Observable.ret({});
            }
        }
    }
    Services.getStoragePermissions = getStoragePermissions;

    /**
    * A wrapper for the GetSettings AJAX call
    */
    function getViewerSettings(sessionId) {
        if (!LocalViewer.isLocalViewer()) {
            return Observable.map(AJAX.ajaxGet(Routes.GetSettings(sessionId, "viewer")), function (res) {
                try  {
                    var settings = JSON.parse(res.value);

                    var version = settings.version || Classes.MIN_SETTINGS_VERSION;

                    if (version < Classes.CURRENT_SETTINGS_VERSION) {
                        return {
                            version: Classes.CURRENT_SETTINGS_VERSION
                        };
                    }

                    return settings;
                } catch (ex) {
                    return {
                        version: Classes.CURRENT_SETTINGS_VERSION
                    };
                }
            });
        } else if (LocalViewer.isPersonalAccelerator()) {
            return Observable.ret(LocalViewer.settings);
        } else {
            if (LocalViewer.settings) {
                return Observable.ret({
                    modalities: LocalViewer.settings.modalities,
                    version: Classes.CURRENT_SETTINGS_VERSION
                });
            } else {
                return Observable.ret({
                    version: Classes.CURRENT_SETTINGS_VERSION
                });
            }
        }
    }
    Services.getViewerSettings = getViewerSettings;

    /**
    * A wrapper for the GetSettings call.
    * @param sessionId
    */
    function getViewerKeyImageSettings(sessionId) {
        return Observable.map(AJAX.ajaxGet(Routes.GetSettings(sessionId, "viewerKeyImageLayout")), function (res) {
            if (res) {
                return JSON.parse(res.value);
            }

            return null;
        });
    }
    Services.getViewerKeyImageSettings = getViewerKeyImageSettings;

    /**
    * A wrapper for the GetSettings AJAX call
    */
    function getDateSettings(sessionId) {
        if (!LocalViewer.isLocalViewer()) {
            return Observable.map(AJAX.ajaxGet(Routes.GetSettings(sessionId, "user_default_date_format")), function (res) {
                return res.value || 'MM-DD-YYYY';
            });
        } else {
            if (LocalViewer.date && LocalViewer.date.value) {
                return Observable.ret(LocalViewer.date.value);
            } else {
                return Observable.ret('MM-DD-YYYY');
            }
        }
    }
    Services.getDateSettings = getDateSettings;

    /**
    * A wrapper for the GetSettings AJAX call
    */
    function getSettings(sessionId) {
        return Observable.zip(Services.getViewerSettings(sessionId), Services.getDateSettings(sessionId), function (settings, date) {
            settings.dateFormat = date;
            return settings;
        });
    }
    Services.getSettings = getSettings;

    /**
    * A wrapper for the SaveSettings AJAX call
    */
    function putSettings(sessionId, settings) {
        if (!LocalViewer.isLocalViewer()) {
            return AJAX.ajax(Routes.SaveSettings, {
                type: "POST",
                data: {
                    key: "viewer",
                    sid: sessionId.value,
                    value: JSON.stringify(settings)
                }
            });
        } else {
            return Observable.ret({});
        }
    }
    Services.putSettings = putSettings;

    /**
    * Returns the KeyImageHang setting.
    * @param sessionId
    */
    function getKeyImageSettings(sessionId) {
        return Services.getViewerKeyImageSettings(sessionId);
    }
    Services.getKeyImageSettings = getKeyImageSettings;

    /**
    * Stores KeyImageHang setting
    * @param sessionId
    * @param settings
    */
    function putKeyImageSettings(sessionId, settings) {
        return AJAX.ajax(Routes.SaveSettings, {
            type: "POST",
            data: {
                key: "viewerKeyImageLayout",
                sid: sessionId.value,
                value: JSON.stringify(settings)
            }
        });
    }
    Services.putKeyImageSettings = putKeyImageSettings;

    /**
    * A wrapper for the Terminology/Tags AJAX call
    */
    function loadTerminology(sessionId, tags, language, query) {
        if (!LocalViewer.isLocalViewer()) {
            return AJAX.ajaxGet(Routes.Terminology(sessionId, tags, language, query));
        } else {
            return Observable.ret({ values: {} });
        }
    }
    Services.loadTerminology = loadTerminology;

    /**
    * A wrapper for the Terminology/Tags AJAX call to get all tags
    */
    function loadAllTerminology(sessionId, language, query) {
        if (!LocalViewer.isLocalViewer()) {
            return AJAX.ajaxGet(Routes.TerminologyAll(sessionId, language, query));
        } else {
            if (LocalViewer.terminology) {
                return Observable.ret(LocalViewer.terminology);
            } else {
                return Observable.ret({ values: {} });
            }
        }
    }
    Services.loadAllTerminology = loadAllTerminology;

    /**
    * A wrapper for the GetAccountSettings AJAX call
    */
    function getAccountSettings(sessionId, query, fqdn) {
        if (!LocalViewer.isLocalViewer()) {
            return AJAX.ajaxGet(Routes.GetAccountSettings(sessionId, query, fqdn));
        } else {
            if (LocalViewer.account) {
                return Observable.ret(LocalViewer.account);
            } else {
                return Observable.ret({});
            }
        }
    }
    Services.getAccountSettings = getAccountSettings;

    /**
    * A wrapper for the study/get AJAX call
    */
    function getStudyStorageInfo(sessionId, query) {
        if (!LocalViewer.isLocalViewer()) {
            return AJAX.ajaxGet(Routes.StudyStorageInfo(sessionId, query));
        } else {
            if (LocalViewer.studyinfo) {
                return Observable.ret(LocalViewer.studyinfo);
            } else {
                var customfields = (LocalViewer && LocalViewer.study && LocalViewer.study.customfields) ? LocalViewer.study.customfields : [];
                return Observable.ret({
                    uuid: "",
                    engine_fqdn: "",
                    hl7: [],
                    customfields: customfields
                });
            }
        }
    }
    Services.getStudyStorageInfo = getStudyStorageInfo;

    /**
    * Study info, using study UUID
    * @param sessionId
    * @param uuid
    */
    function getStudyInfo(sessionId, uuid) {
        if (!LocalViewer.isLocalViewer()) {
            return AJAX.ajaxGet(Routes.StudyInfo(sessionId, uuid));
        } else {
            return Observable.ret({});
        }
    }
    Services.getStudyInfo = getStudyInfo;

    /**
    * A wrapper for the study/set AJAX call
    */
    function setStudy(sessionId, query, tags) {
        if (!LocalViewer.isLocalViewer()) {
            return AJAX.ajaxGet(Routes.StudySet(sessionId, query, tags));
        } else {
            return Observable.ret({});
        }
    }
    Services.setStudy = setStudy;

    /**
    * A wrapper for the study/retrieve AJAX call
    */
    function retrieveStudy(sessionId, uuid) {
        if (!LocalViewer.isLocalViewer()) {
            return AJAX.ajaxGet(Routes.RetrieveStudy(sessionId, uuid));
        } else {
            return Observable.ret({});
        }
    }
    Services.retrieveStudy = retrieveStudy;

    /**
    * A wrapper for the study/audit AJAX call
    */
    function setStudyAudit(sessionId, uuid, action) {
        if (!LocalViewer.isStandardLocalViewer()) {
            return AJAX.ajaxGet(Routes.StudyAudit(sessionId, uuid, action));
        } else {
            return Observable.ret({});
        }
    }
    Services.setStudyAudit = setStudyAudit;

    /**
    * A wrapper for the GetStudyList AJAX call
    */
    function getStudyList(sessionId, phiNamespace, patientId, accessionNumber, studyUuid, notStudyUuid) {
        if (!LocalViewer.isLocalViewer()) {
            return Observable.map(AJAX.ajaxGet(Routes.GetStudyList(sessionId, phiNamespace, patientId, accessionNumber, studyUuid, notStudyUuid)), function (response) {
                return response.studies;
            });
        } else {
            throw new Error("Not supported in the local viewer");
        }
    }
    Services.getStudyList = getStudyList;

    /**
    * A wrapper for the GetStudyList AJAX call
    */
    function getStudyListFiltered(sessionId, phiNamespace, patientId, accessionNumber, studyUuid, notStudyUuid, worklistFilter) {
        if (!LocalViewer.isLocalViewer()) {
            return Observable.map(AJAX.ajaxGet(Routes.GetStudyList(sessionId, phiNamespace, patientId, accessionNumber, studyUuid, notStudyUuid, worklistFilter)), function (response) {
                return response.studies;
            });
        } else {
            throw new Error("Not supported in the local viewer");
        }
    }
    Services.getStudyListFiltered = getStudyListFiltered;

    /**
    * A wrapper for the GetStudyList AJAX call
    */
    function getStudyListByUuid(sessionId, studyUuidList) {
        if (!LocalViewer.isLocalViewer()) {
            return Observable.map(AJAX.ajaxGet(Routes.GetStudyListByUuid(sessionId, studyUuidList)), function (response) {
                return response.studies;
            });
        } else {
            throw new Error("Not supported in the local viewer");
        }
    }
    Services.getStudyListByUuid = getStudyListByUuid;

    /**
    * A wrapper for the Image Annotations AJAX call
    */
    function listImageAnnotations(sessionId, query) {
        if (!LocalViewer.isLocalViewer()) {
            return AJAX.ajaxGet(Routes.ListAnnotations(sessionId, query));
        } else {
            if (LocalViewer.annotations) {
                return Observable.ret(LocalViewer.annotations);
            } else {
                return Observable.ret({
                    annotations: []
                });
            }
        }
    }
    Services.listImageAnnotations = listImageAnnotations;

    /**
    * A wrapper for the Image Annotation AJAX call
    */
    function getImageAnnotation(sessionId, query, id) {
        if (!LocalViewer.isStandardLocalViewer()) {
            return AJAX.ajaxGet(Routes.GetAnnotation(sessionId, query, id));
        } else {
            throw "Not supported in local viewer";
        }
    }
    Services.getImageAnnotation = getImageAnnotation;

    /**
    * A wrapper for the Add Image Annotation AJAX call
    */
    function addImageAnnotation(sessionId, query, series, instance, frameNumber, annotation) {
        if (!LocalViewer.isStandardLocalViewer()) {
            return AJAX.ajax(Routes.AddAnnotation, {
                type: "POST",
                data: {
                    sid: sessionId.value,
                    study_uid: query.studyUid.value,
                    storage_namespace: query.storageNamespace.value,
                    phi_namespace: query.phiNamespace.value,
                    series_uid: series.value,
                    instance_uid: instance.value,
                    frame_number: frameNumber.value.toString(),
                    stamp: annotation.type == Classes.MouseTool[31 /* Stamp */] ? 1 : 0,
                    json: JSON.stringify(annotation)
                } });
        } else {
            return Observable.ret({
                uuid: ""
            });
        }
    }
    Services.addImageAnnotation = addImageAnnotation;

    /**
    * A wrapper for the Delete Image Annotation AJAX call
    */
    function deleteImageAnnotation(sessionId, query, id) {
        if (!LocalViewer.isStandardLocalViewer()) {
            return AJAX.ajaxGet(Routes.DeleteAnnotation(sessionId, query, id));
        } else {
            return Observable.ret({});
        }
    }
    Services.deleteImageAnnotation = deleteImageAnnotation;

    /**
    * A wrapper for the Edit Image Annotation AJAX call
    */
    function editImageAnnotation(sessionId, query, id, annotation) {
        if (!LocalViewer.isStandardLocalViewer()) {
            return AJAX.ajax(Routes.SetAnnotation, {
                type: "POST",
                data: {
                    sid: sessionId.value,
                    study_uid: query.studyUid.value,
                    storage_namespace: query.storageNamespace.value,
                    phi_namespace: query.phiNamespace.value,
                    uuid: id.value,
                    json: JSON.stringify(annotation)
                } });
        } else {
            return Observable.ret({
                uuid: ""
            });
        }
    }
    Services.editImageAnnotation = editImageAnnotation;

    /**
    * List all key images in a study
    */
    function listKeyImages(sessionId, query) {
        if (!LocalViewer.isLocalViewer()) {
            return AJAX.ajaxGet(Routes.ListKeyImages(sessionId, query));
        } else {
            if (LocalViewer.keyimages) {
                return Observable.ret(LocalViewer.keyimages);
            } else {
                return Observable.ret({
                    images: []
                });
            }
        }
    }
    Services.listKeyImages = listKeyImages;

    /**
    * Add a key image
    */
    function getKeyImage(sessionId, id) {
        if (!LocalViewer.isStandardLocalViewer()) {
            return AJAX.ajaxGet(Routes.GetKeyImage(sessionId, id));
        } else {
            return Observable.ret({});
        }
    }
    Services.getKeyImage = getKeyImage;

    /**
    * Add a key image
    */
    function addKeyImage(sessionId, query, series, instance, frameNumber, version) {
        if (!LocalViewer.isStandardLocalViewer()) {
            return AJAX.ajaxGet(Routes.AddKeyImage(sessionId, query, series, instance, frameNumber, version));
        } else {
            return Observable.ret({});
        }
    }
    Services.addKeyImage = addKeyImage;

    /**
    * Delete a key image
    */
    function deleteKeyImage(sessionId, id) {
        if (!LocalViewer.isStandardLocalViewer()) {
            return AJAX.ajaxGet(Routes.DeleteKeyImage(sessionId, id));
        } else {
            return Observable.ret({});
        }
    }
    Services.deleteKeyImage = deleteKeyImage;

    /**
    * Run a routing rule
    */
    function runRoutingRule(sessionId, studyId, routeId, email, message) {
        if (!LocalViewer.isStandardLocalViewer()) {
            return AJAX.ajaxGet(Routes.RunRoutingRule(sessionId, studyId, routeId, email, message));
        } else {
            return Observable.ret({});
        }
    }
    Services.runRoutingRule = runRoutingRule;

    /**
    * meeting/add
    */
    function StartMeeting(sessionId, queryObject, meetingName, state) {
        if (!LocalViewer.isLocalViewer()) {
            return AJAX.ajax(Routes.StartMeeting, {
                type: "POST",
                data: {
                    sid: sessionId.value,
                    study_uid: queryObject.studyUid.value,
                    storage_namespace: queryObject.storageNamespace.value,
                    phi_namespace: queryObject.phiNamespace.value,
                    name: meetingName,
                    state: JSON.stringify(state)
                }
            });
        } else {
            return Observable.ret({});
        }
    }
    Services.StartMeeting = StartMeeting;

    /**
    * link/add
    */
    function CreateMeetingLink(sessionId, meetingId, queryObject, uuid, emailAddress) {
        if (!LocalViewer.isLocalViewer()) {
            return AJAX.ajax(Routes.CreateMeetingLink, {
                type: "POST",
                data: {
                    action: "STUDY_VIEW",
                    sid: sessionId.value,
                    study_id: uuid.value,
                    namespace_id: queryObject.storageNamespace.value,
                    meeting_id: meetingId.value,
                    email: emailAddress.value,
                    parameters: JSON.stringify([
                        "route", "meeting",
                        "meetingId", meetingId.value,
                        "phiNamespace", queryObject.phiNamespace.value
                    ])
                }
            });
        } else {
            throw "Not supported in local viewer";
        }
    }
    Services.CreateMeetingLink = CreateMeetingLink;

    function ShareMeetingLink(sessionId, meetingLinkId, emailAddress) {
        if (!LocalViewer.isLocalViewer()) {
            return AJAX.ajax(Routes.ShareLink, {
                type: "POST",
                data: {
                    sid: sessionId.value,
                    uuid: meetingLinkId.value,
                    email: emailAddress.value
                }
            });
        } else {
            throw "Not supported in local viewer";
        }
    }
    Services.ShareMeetingLink = ShareMeetingLink;

    function ShareRecordingLink(sessionId, linkId, emailAddress) {
        if (!LocalViewer.isLocalViewer()) {
            return AJAX.ajax(Routes.ShareLink, {
                type: "POST",
                data: {
                    sid: sessionId.value,
                    uuid: linkId.value,
                    email: emailAddress.value
                }
            });
        } else {
            throw "Not supported in local viewer";
        }
    }
    Services.ShareRecordingLink = ShareRecordingLink;

    /**
    * meeting/set
    */
    function UpdateMeeting(sessionId, meetingId, state) {
        if (!LocalViewer.isLocalViewer()) {
            return AJAX.ajax(Routes.UpdateMeeting, {
                type: "POST",
                data: {
                    sid: sessionId.value,
                    uuid: meetingId.value,
                    state: JSON.stringify(state)
                }
            });
        } else {
            return Observable.ret({});
        }
    }
    Services.UpdateMeeting = UpdateMeeting;

    /**
    * meeting/delete
    */
    function EndMeeting(sessionId, meetingId) {
        if (!LocalViewer.isLocalViewer()) {
            return AJAX.ajax(Routes.EndMeeting, {
                type: "POST",
                data: {
                    sid: sessionId.value,
                    uuid: meetingId.value
                }
            });
        } else {
            return Observable.ret({});
        }
    }
    Services.EndMeeting = EndMeeting;

    /**
    * meeting/events/add
    */
    function BroadcastEvent(sessionId, meetingId, event) {
        if (!LocalViewer.isLocalViewer()) {
            return AJAX.ajax(Routes.BroadcastEvent, {
                type: "POST",
                data: {
                    sid: sessionId.value,
                    uuid: meetingId.value,
                    event: JSON.stringify(event)
                }
            });
        } else {
            return Observable.ret({});
        }
    }
    Services.BroadcastEvent = BroadcastEvent;

    /**
    * meeting/ping
    */
    function PingMeeting(sessionId, meetingId) {
        if (!LocalViewer.isLocalViewer()) {
            return AJAX.ajax(Routes.PingMeeting, {
                type: "POST",
                data: {
                    sid: sessionId.value,
                    uuid: meetingId.value
                }
            });
        } else {
            return Observable.ret({});
        }
    }
    Services.PingMeeting = PingMeeting;

    /**
    * meeting/join
    */
    function JoinMeeting(sessionId, meetingId) {
        if (!LocalViewer.isLocalViewer()) {
            return AJAX.ajax(Routes.JoinMeeting, {
                type: "POST",
                data: {
                    sid: sessionId.value,
                    uuid: meetingId.value
                }
            });
        } else {
            return Observable.ret({});
        }
    }
    Services.JoinMeeting = JoinMeeting;

    /**
    * meeting/leave
    */
    function LeaveMeeting(sessionId, meetingId) {
        if (!LocalViewer.isLocalViewer()) {
            return AJAX.ajax(Routes.LeaveMeeting, {
                type: "POST",
                data: {
                    sid: sessionId.value,
                    uuid: meetingId.value
                }
            });
        } else {
            return Observable.ret({});
        }
    }
    Services.LeaveMeeting = LeaveMeeting;

    /**
    * meeting/presenter
    */
    function ChangeMeetingPresenter(sessionId, meetingId, userId) {
        if (!LocalViewer.isLocalViewer()) {
            return AJAX.ajax(Routes.ChangeMeetingPresenter, {
                type: "POST",
                data: {
                    sid: sessionId.value,
                    uuid: meetingId.value,
                    user_id: userId.value
                }
            });
        } else {
            return Observable.ret({});
        }
    }
    Services.ChangeMeetingPresenter = ChangeMeetingPresenter;

    /**
    * meeting/roster
    */
    function MeetingRoster(sessionId, meetingId) {
        if (!LocalViewer.isLocalViewer()) {
            return AJAX.ajaxGet(Routes.MeetingRoster(sessionId, meetingId));
        } else {
            return Observable.ret({});
        }
    }
    Services.MeetingRoster = MeetingRoster;

    /**
    * meeting/get
    */
    function GetMeeting(sessionId, meetingId) {
        if (!LocalViewer.isLocalViewer()) {
            return AJAX.ajaxGet(Routes.GetMeeting(sessionId, meetingId));
        } else {
            return Observable.ret({});
        }
    }
    Services.GetMeeting = GetMeeting;

    function StudyPHIExtended(sessionId, uuid, extended) {
        if (!LocalViewer.isLocalViewer()) {
            return AJAX.ajax(Routes.StudyPHIExtended, {
                type: "POST",
                data: {
                    sid: sessionId.value,
                    uuid: uuid.value,
                    extended: extended
                }
            });
        } else {
            throw new Error("Not supported in the local viewer");
        }
    }
    Services.StudyPHIExtended = StudyPHIExtended;

    function CreateScriptLink(sessionId, queryObject, uuid, attachmentId, attachmentVersion, attachmentPhiNamespace, emailAddress) {
        if (!LocalViewer.isLocalViewer()) {
            return AJAX.ajax(Routes.CreateScriptLink, {
                type: "POST",
                data: {
                    action: "STUDY_VIEW",
                    sid: sessionId.value,
                    study_id: uuid.value,
                    namespace_id: queryObject.storageNamespace.value,
                    email: emailAddress.value,
                    parameters: JSON.stringify([
                        "route", "script",
                        "attachmentId", attachmentId.value,
                        "attachmentVersion", attachmentVersion.value,
                        "attachmentPhiNs", attachmentPhiNamespace.value
                    ])
                }
            });
        } else {
            throw "Not supported in local viewer";
        }
    }
    Services.CreateScriptLink = CreateScriptLink;

    /**
    * A wrapper for the Accelerator Used AJAX call
    */
    function AcceleratorUsed(sessionId) {
        if (!LocalViewer.isLocalViewer()) {
            return AJAX.ajax(Routes.AcceleratorUsed, {
                type: "GET",
                data: {
                    sid: sessionId.value
                }
            });
        } else {
            return Observable.ret({});
        }
    }
    Services.AcceleratorUsed = AcceleratorUsed;

    /**
    * Log error message to /audit/log
    * @param {string} message
    * @param {Classes.SessionId} sessionId
    * @param {Classes.StudyUUID} studyId
    * @param {Classes.SeriesUid} seriesId
    * @returns {Observable.Observable<{}>}
    * @constructor
    */
    function AuditLog(message, source, sessionId, studyId, seriesId) {
        AJAX.ajaxGet(Routes.AuditLog(message, "viewererror", window.navigator.userAgent, source, sessionId, studyId, seriesId)).subscribe({
            next: function () {
                console.error(message);
            },
            done: function () {
            },
            fail: function (err) {
                console.log("Unable to send to /audit/log: " + err);
            }
        });
    }
    Services.AuditLog = AuditLog;

    /**
    * Log viewer metric to /audit/log
    * @param data
    * @param {string} source
    * @param {Classes.SessionId} sessionId
    * @constructor
    */
    function AuditMetric(name, data) {
        AJAX.ajaxGet(Routes.AuditMetric(name, data, "viewermetric", window.navigator.userAgent)).subscribe({
            next: function () {
            },
            done: function () {
            },
            fail: function (err) {
                console.log("Unable to send to /audit/log: " + err);
            }
        });
    }
    Services.AuditMetric = AuditMetric;
})(Services || (Services = {}));
///<reference path="../typings/jquery/jquery.d.ts" />
///<reference path="../classes/Types.ts" />
///<reference path="Services.ts" />
///<reference path="Observable.ts" />

/**
* Methods for working with the terminology and translation service
*/
var Terminology;
(function (Terminology) {
    /**
    * Enumerates customizable strings displayed in the viewer
    */
    Terminology.Terms = {
        /* Error Messages */
        ErrorLoadingSettings: new Classes.Term("webviewer:error-loading-settings", "Error loading settings"),
        ErrorLoadingStudy: new Classes.Term("webviewer:error-loading-study", "Error loading study"),
        NotForDiagnosticUse: new Classes.Term("webviewer:not-for-diagnostic-use", "Not intended for diagnostic use on mobile devices"),
        ErrorStudyNotFound: new Classes.Term("webviewer:error-study-not-found", "Study not found"),
        /* Thin Studies */
        CannotRetrieveThinStudy: new Classes.Term("webviewer:cannot-retrieve-this-study", "Unable to initiate retrieval of the study."),
        StudyBeingRetrieved: new Classes.Term("webviewer:study-being-retrieved", "The study is being retrieved from the PACS. It will be available in three to seven minutes."),
        /* General Terms */
        Study: new Classes.Term("webviewer:study", "Study"),
        StudyPage: new Classes.Term("webviewer:study-page", "Go to Study Page"),
        Series: new Classes.Term("webviewer:series", "Series"),
        NoImageSeries: new Classes.Term("webviewer:no-image-series", "No image series"),
        SeriesPlural: new Classes.Term("webviewer:series-plural", "Series"),
        Instance: new Classes.Term("webviewer:instance", "Instance"),
        Instances: new Classes.Term("webviewer:instances", "Instances"),
        Frame: new Classes.Term("webviewer:frame", "Frame"),
        Frames: new Classes.Term("webviewer:frames", "Frames"),
        /* Thumbnails */
        RelatedStudies: new Classes.Term("webviewer:related-studies", "Related Studies"),
        LoadImages: new Classes.Term("webviewer:load-images", "Load Images"),
        AccessionNumberShort: new Classes.Term("webviewer:accession-short", "Acc #"),
        Date: new Classes.Term("webviewer:date", "Date"),
        Attachments: new Classes.Term("webviewer:attachments", "Attachments"),
        Filename: new Classes.Term("webviewer:filename", "Filename"),
        Download: new Classes.Term("webviewer:download", "Download"),
        Reports: new Classes.Term("webviewer:reports", "Reports"),
        View: new Classes.Term("webviewer:view", "View"),
        HL7Report: new Classes.Term("webviewer:hl7-report", "HL7 Report"),
        StructuredReport: new Classes.Term("webviewer:structured-report", "Structured Report"),
        RecordedSession: new Classes.Term("webviewer:recorded-session", "Recorded Session"),
        MeetingInProgress: new Classes.Term("webviewer:meeting-in-progress", "Meeting in Progress"),
        StackSeries: new Classes.Term("webviewer:stack-series", "Stack Series"),
        KeyImageSeries: new Classes.Term("webviewer:keyimage-series", "Key Images"),
        /* Toolbar */
        Scroll: new Classes.Term("webviewer:scroll", "Scroll"),
        ScrollTooltip: new Classes.Term("webviewer:scroll-tooltip", "Scroll"),
        Transform: new Classes.Term("webviewer:transform", "View"),
        TransformTooltip: new Classes.Term("webviewer:transform-tooltip", "Image Transformation Tools"),
        Move: new Classes.Term("webviewer:move", "Move"),
        MoveTooltip: new Classes.Term("webviewer:move-tooltip", "Move"),
        Zoom: new Classes.Term("webviewer:zoom", "Zoom"),
        ZoomTooltip: new Classes.Term("webviewer:zoom-tooltip", "Zoom"),
        FreeRotate: new Classes.Term("webviewer:zoom-rotate", "Rotate"),
        FreeRotateTooltip: new Classes.Term("webviewer:zoom-rotate-tooltip", "Free Rotate"),
        Flip: new Classes.Term("webviewer:flip", "Flip H"),
        FlipTooltip: new Classes.Term("webviewer:flip-tooltip", "Flip Horizontally"),
        FlipV: new Classes.Term("webviewer:flip-v", "Flip V"),
        FlipVTooltip: new Classes.Term("webviewer:flip-v-tooltip", "Flip Vertically"),
        Rotate: new Classes.Term("webviewer:rotate", "Rotate"),
        RotateTooltip: new Classes.Term("webviewer:rotate-tooltip", "Rotate"),
        WindowLevel: new Classes.Term("webviewer:window-level", "W/L"),
        WindowLevelTooltip: new Classes.Term("webviewer:window-level-tooltip", "Window/Level"),
        SelectWindowLevel: new Classes.Term("webviewer:window-level", "Select"),
        SelectWindowLevelTooltip: new Classes.Term("webviewer:window-level-tooltip", "Select Window/Level"),
        SavePreset: new Classes.Term("webviewer:save-preset", "Save"),
        SavePresetTooltip: new Classes.Term("webviewer:save-preset-tooltip", "Save Window/Level Preset"),
        DetectWindowLevel: new Classes.Term("webviewer:detect-window-level", "Detect"),
        DetectWindowLevelTooltip: new Classes.Term("webviewer:detect-window-level-tooltip", "Detect Window/Level"),
        PleaseSelectAnAnnotation: new Classes.Term("webviewer:please-select-an-annotation", "Please select a rectangle or ellipse annotation to auto-detect the window level settings."),
        CannotDetectWindowLevel: new Classes.Term("webviewer:cannot-detect-window-level", "Unable to auto-detect the window level settings."),
        InvalidUltrasoundMeasurement: new Classes.Term("webviewer:invalid-us-measurement", "This measurement crosses regions with different or no pixel spacing."),
        Annotations: new Classes.Term("webviewer:annotations", "Annot."),
        AnnotationsTooltip: new Classes.Term("webviewer:annotations-tooltip", "Annotations"),
        SelectAnnotation: new Classes.Term("webviewer:select-annotation", "Select"),
        SelectAnnotationTooltip: new Classes.Term("webviewer:select-annotation-tooltip", "Select Annotation"),
        Filled: new Classes.Term("webviewer:filled", "Fill"),
        FilledTooltip: new Classes.Term("webviewer:filled-tooltip", "Fill Annotation"),
        DeleteAnnotation: new Classes.Term("webviewer:delete-annotation", "Delete"),
        DeleteAnnotationTooltip: new Classes.Term("webviewer:delete-annotation-tooltip", "Delete Selected Annotation"),
        ExportGSPS: new Classes.Term("webviewer:export-gsps", "GSPS"),
        ExportGSPSTooltip: new Classes.Term("webviewer:export-gsps-tooltip", "Export Annotations as Grayscale Presentation State"),
        GSPS: new Classes.Term("webviewer:gsps", "GSPS"),
        GSPSTooltip: new Classes.Term("webviewer:gsps-tooltip", "Show All GSPS"),
        Line: new Classes.Term("webviewer:line", "Line"),
        LineTooltip: new Classes.Term("webviewer:line-tooltip", "Measure Line"),
        Arrow: new Classes.Term("webviewer:arrow", "Arrow"),
        ArrowTooltip: new Classes.Term("webviewer:arrow-tooltip", "Draw Arrow"),
        Angle: new Classes.Term("webviewer:angle", "Angle"),
        AngleTooltip: new Classes.Term("webviewer:angle-tooltip", "Measure Angle"),
        Cobb: new Classes.Term("webviewer:cobb", "Cobb"),
        CobbTooltip: new Classes.Term("webviewer:cobb-tooltip", "Measure Cobb Angle"),
        Rectangle: new Classes.Term("webviewer:rectangle", "Rect"),
        RectangleTooltip: new Classes.Term("webviewer:rectangle-tooltip", "Measure Rectangle"),
        Ellipse: new Classes.Term("webviewer:ellipse", "Ellipse"),
        EllipseTooltip: new Classes.Term("webviewer:ellipse-tooltip", "Measure Ellipse"),
        Radius: new Classes.Term("webviewer:radius", "Radius"),
        RadiusTooltip: new Classes.Term("webviewer:radius-tooltip", "Measure Radius"),
        Circle: new Classes.Term("webviewer:circle", "Circle"),
        CircleTooltip: new Classes.Term("webviewer:circle-tooltip", "Measure Circle"),
        Square: new Classes.Term("webviewer:square", "Square"),
        SquareTooltip: new Classes.Term("webviewer:square-tooltip", "Measure Square"),
        CoLocalization: new Classes.Term("webviewer:colocate", "Co-Locate"),
        CoLocalizationTooltip: new Classes.Term("webviewer:colocate-tooltip", "Co-Locate Measurements"),
        OrthoAxes: new Classes.Term("webviewer:orthoaxes", "Orthogonal Axes"),
        OrthoAxesTooltip: new Classes.Term("webviewer:orthoaxes-tooltip", "Measure Orthogonal Axes"),
        FemoralHead: new Classes.Term("webviewer:femoralhead", "Femoral Head"),
        FemoralHeadTooltip: new Classes.Term("webviewer:femoralhead-tooltip", "Measure Femoral Head"),
        Polygon: new Classes.Term("webviewer:polygon", "Polygon"),
        PolygonTooltip: new Classes.Term("webviewer:polygon-tooltip", "Measure Polygon"),
        Trace: new Classes.Term("webviewer:trace", "Trace"),
        TraceTooltip: new Classes.Term("webviewer:trace-tooltip", "Measure Trace"),
        Paint: new Classes.Term("webviewer:paint", "Paint"),
        PaintTooltip: new Classes.Term("webviewer:paint-tooltip", "Measure Paint"),
        Label: new Classes.Term("webviewer:label", "Label"),
        LabelTooltip: new Classes.Term("webviewer:label-tooltip", "Label Measurement"),
        AssignPixelSpacing: new Classes.Term("webviewer:pixel-spacing", "Pixel Size"),
        AssignPixelSpacingTooltip: new Classes.Term("webviewer:assign-pixel-spacing-tooltip", "Assign Pixel Spacing"),
        AssignPixelSpacingDialog: new Classes.Term("webviewer:enter-pixel-spacing-value", "Enter Pixel Spacing Value (mm)"),
        AssignSliceSpacing: new Classes.Term("webviewer:slice-spacing", "Slice Size"),
        AssignSliceSpacingTooltip: new Classes.Term("webviewer:assign-slice-spacing-tooltip", "Assign Slice Spacing"),
        AssignSliceSpacingDialog: new Classes.Term("webviewer:enter-clie-spacing-value", "Enter Slice Spacing Value (mm)"),
        Text: new Classes.Term("webviewer:text", "Text"),
        TextTooltip: new Classes.Term("webviewer:text-tooltip", "Add Text Annotation"),
        ProstateTool: new Classes.Term("webviewer:prostate-tool", "Measure"),
        ProstateToolTooltip: new Classes.Term("webviewer:prostate-tool-tooltip", "Measure Prostate"),
        Stamp: new Classes.Term("webviewer:stamp", "Stamp"),
        StampTooltip: new Classes.Term("webviewer:stamp-tooltip", "Stamp Annotation"),
        ArrowAnnotate: new Classes.Term("webviewer:arrow", "Arrow"),
        ArrowAnnotateTooltip: new Classes.Term("webviewer:arrow-tooltip", "Draw Arrow"),
        CircleAnnotate: new Classes.Term("webviewer:circle", "Circle"),
        CircleAnnotateTooltip: new Classes.Term("webviewer:circle-tooltip", "Draw Circle"),
        EllipseAnnotate: new Classes.Term("webviewer:ellipse", "Ellipse"),
        EllipseAnnotateTooltip: new Classes.Term("webviewer:ellipse-tooltip", "Draw Ellipse"),
        LineAnnotate: new Classes.Term("webviewer:line", "Line"),
        LineAnnotateTooltip: new Classes.Term("webviewer:line-tooltip", "Draw Line"),
        LineCalibrate: new Classes.Term("webviewer:line-calibrate", "Calibrate"),
        LineCalibrateTooltip: new Classes.Term("webviewer:line-calibrate-tooltip", "Calibrate Line"),
        LineCalibratePrompt: new Classes.Term("webviewer:enter-length", "Enter length (mm)"),
        ThresholdPrompt: new Classes.Term("webviewer:enter-threshold", "Enter threshold"),
        ThresholdMinPrompt: new Classes.Term("webviewer:enter-threshold-min", "Enter minimum threshold"),
        ThresholdMaxPrompt: new Classes.Term("webviewer:enter-threshold-max", "Enter maximum threshold"),
        RectangleAnnotate: new Classes.Term("webviewer:rectangle", "Rect"),
        RectangleAnnotateTooltip: new Classes.Term("webviewer:rectangle-tooltip", "Draw Rectangle"),
        SquareAnnotate: new Classes.Term("webviewer:square", "Square"),
        SquareAnnotateTooltip: new Classes.Term("webviewer:square-tooltip", "Draw Square"),
        Propagate: new Classes.Term("webviewer:propagate", "Propagate"),
        PropagateTooltip: new Classes.Term("webviewer:propagate-tooltip", "Propagate Across Slices"),
        PropagateAllTooltip: new Classes.Term("webviewer:propagate-all-tooltip", "Propagate Across Series"),
        Threshold: new Classes.Term("webviewer:threshold", "Threshold"),
        ThresholdTooltip: new Classes.Term("webviewer:threshold-tooltip", "Threshold To Area"),
        Range: new Classes.Term("webviewer:range", "Range"),
        RangeTooltip: new Classes.Term("webviewer:range-tooltip", "Range To Area"),
        ShrinkWrap: new Classes.Term("webviewer:shrinkwrap", "Shrink"),
        ShrinkWrapTooltip: new Classes.Term("webviewer:shrinkwrap-tooltip", "Shrink Wrap To Area"),
        EnterText: new Classes.Term("webviewer:enter-text", "Enter Text"),
        Fit: new Classes.Term("webviewer:fit", "Fit"),
        FitTooltip: new Classes.Term("webviewer:fit-tooltip", "Fit To Viewport"),
        Reset: new Classes.Term("webviewer:reset", "Reset"),
        ResetTooltip: new Classes.Term("webviewer:reset-tooltip", "Reset"),
        ThreeD: new Classes.Term("webviewer:3d", "3D"),
        ThreeDTooltip: new Classes.Term("webviewer:3d-tooltip", "3D Tools"),
        Probe: new Classes.Term("webviewer:probe", "Probe"),
        ProbeTooltip: new Classes.Term("webviewer:probe-tooltip", "Probe Tool"),
        ReferenceLines: new Classes.Term("webviewer:ref-lines", "R/L"),
        ReferenceLinesTooltip: new Classes.Term("webviewer:ref-lines-tooltip", "Reference Lines"),
        LinkedSeries: new Classes.Term("webviewer:linked-series", "Link"),
        LinkedSeriesTooltip: new Classes.Term("webviewer:linked-series-tooltip", "Linked Series"),
        Layout: new Classes.Term("webviewer:layout", "Layout"),
        LayoutTooltip: new Classes.Term("webviewer:layout-tooltip", "Layout"),
        NewWindow: new Classes.Term("webviewer:new-window", "New"),
        NewWindowTooltip: new Classes.Term("webviewer:new-window-tooltip", "New Study Window"),
        Maximize: new Classes.Term("webviewer:maximize", "Max."),
        MaximizeTooltip: new Classes.Term("webviewer:maximize-tooltip", "Maximize/Minimize"),
        Print: new Classes.Term("webviewer:print", "Print"),
        PrintTooltip: new Classes.Term("webviewer:print-tooltip", "Print"),
        Thumbnails: new Classes.Term("webviewer:thumbnails", "Thumbs"),
        ThumbnailsTooltip: new Classes.Term("webviewer:thumbnails-tooltip", "Show/Hide Thumbnails"),
        Record: new Classes.Term("webviewer:record", "Record"),
        RecordTooltip: new Classes.Term("webviewer:record-tooltip", "Record"),
        Actions: new Classes.Term("webviewer:actions", "Actions"),
        ActionsTooltip: new Classes.Term("webviewer:actions-tooltip", "Study Actions"),
        Settings: new Classes.Term("webviewer:settings", "Settings"),
        SettingsTooltip: new Classes.Term("webviewer:settings-tooltip", "Edit User Settings"),
        PlaneLocalization: new Classes.Term("webviewer:plane-localization", "Local."),
        PlaneLocalizationTooltip: new Classes.Term("webviewer:plane-localization-tooltip", "Plane Localization"),
        MPR: new Classes.Term("webviewer:mpr", "MPR"),
        MPRTooltip: new Classes.Term("webviewer:mpr-tooltip", "Multi-planar Reconstruction"),
        MIP: new Classes.Term("webviewer:mip", "MIP"),
        MIPTooltip: new Classes.Term("webviewer:mip-tooltip", "Maximum Intensity Projection"),
        Magnify: new Classes.Term("webviewer:magnify", "Mag."),
        MagnifyTooltip: new Classes.Term("webviewer:magnify-tooltip", "Magnify Image"),
        Info: new Classes.Term("webviewer:info", "Info"),
        InfoTooltip: new Classes.Term("webviewer:info-tooltip", "Show/Hide Information"),
        ShowAnnotations: new Classes.Term("webviewer:show-annotations", "Annot."),
        ShowAnnotationsTooltip: new Classes.Term("webviewer:show-annotations-tooltip", "Show/Hide Annotations"),
        ShowAnnotationsDetail: new Classes.Term("webviewer:show-annotations-detail", "Detail"),
        ShowAnnotationsDetailTooltip: new Classes.Term("webviewer:show-annotations-detail-tooltip", "Show/Hide Annotations Detail"),
        ToggleCreatedByOthersAnnotationsTooltip: new Classes.Term("webviewer:toggle-created-by-others-annotations-tooltip", "Show/Hide Annotations created by others"),
        ShowRulerTooltip: new Classes.Term("webviewer:show-ruler-tooltip", "Show/Hide Ruler"),
        Invert: new Classes.Term("webviewer:invert", "Invert"),
        InvertTooltip: new Classes.Term("webviewer:invert-tooltip", "Invert Colors"),
        Subtract: new Classes.Term("webviewer:subtract", "Subtract"),
        SubtractTooltip: new Classes.Term("webviewer:subtract-tooltip", "Subtract"),
        Subtraction: new Classes.Term("webviewer:subtraction", "Subtraction"),
        Enhance: new Classes.Term("webviewer:enhance", "Enhance"),
        EnhanceTooltip: new Classes.Term("webviewer:enhance-tooltip", "Enhance Image"),
        Export: new Classes.Term("webviewer:export", "Export"),
        ExportTooltip: new Classes.Term("webviewer:export-tooltip", "Export"),
        ExportAVI: new Classes.Term("webviewer:export-png", "AVI"),
        ExportAVITooltip: new Classes.Term("webviewer:export-png-tooltip", "Export AVI"),
        ExportMP4: new Classes.Term("webviewer:export-png", "MP4"),
        ExportMP4Tooltip: new Classes.Term("webviewer:export-png-tooltip", "Export MP4"),
        ExportPNG: new Classes.Term("webviewer:export-png", "PNG"),
        ExportPNGTooltip: new Classes.Term("webviewer:export-png-tooltip", "Export PNG"),
        StorePNG: new Classes.Term("webviewer:store-png", "Store PNG"),
        StorePNGTooltip: new Classes.Term("webviewer:store-png-tooltip", "Store PNG as Study Attachment"),
        ExportAllPNG: new Classes.Term("webviewer:export-all-png", "All"),
        ExportAllPNGTooltip: new Classes.Term("webviewer:export-all-png-tooltip", "Export All as PNG"),
        ExportSeries: new Classes.Term("webviewer:export-series", "Series"),
        ExportSeriesTooltip: new Classes.Term("webviewer:export-series-tooltip", "Export Series"),
        ExportStudy: new Classes.Term("webviewer:export-study", "Study"),
        ExportStudyTooltip: new Classes.Term("webviewer:export-study-tooltip", "Export Study"),
        ExportISO: new Classes.Term("webviewer:export-iso", "ISO"),
        ExportISOTooltip: new Classes.Term("webviewer:export-iso-tooltip", "Export ISO"),
        ExportViewer: new Classes.Term("webviewer:export-viewer", "Viewer"),
        ExportViewerTooltip: new Classes.Term("webviewer:export-viewer-tooltip", "Export Viewer"),
        EditMetadata: new Classes.Term("webviewer:edit-metadata", "Meta"),
        EditMetadataTooltip: new Classes.Term("webviewer:edit-metadata-tooltip", "Edit Metadata"),
        SecondaryCapture: new Classes.Term("webviewer:secondary-capture", "DICOM"),
        SecondaryCaptureTooltip: new Classes.Term("webviewer:secondary-capture-tooltip", "Secondary Capture"),
        Anonymize: new Classes.Term("webviewer:anonymize", "Anon."),
        AnonymizeTooltip: new Classes.Term("webviewer:anonymize-tooltip", "Anonymize Series"),
        AnonymizeStudy: new Classes.Term("webviewer:anonymize-study", "Anon."),
        AnonymizeStudyTooltip: new Classes.Term("webviewer:anonymize-study-tooltip", "Anonymize Study"),
        AnonymizeImage: new Classes.Term("webviewer:anonymize-image", "Anon."),
        AnonymizeImageTooltip: new Classes.Term("webviewer:anonymize-image-tooltip", "Anonymize Image Only"),
        CropSeries: new Classes.Term("webviewer:crop-series", "Crop"),
        CropSeriesTooltip: new Classes.Term("webviewer:crop-series-tooltip", "Crop Series"),
        SplitStudy: new Classes.Term("webviewer:split-study", "Split"),
        SplitStudyTooltip: new Classes.Term("webviewer:split-study-tooltip", "Split Study"),
        ReverseSeries: new Classes.Term("webviewer:reverse-series", "Reverse"),
        ReverseSeriesTooltip: new Classes.Term("webviewer:reverse-series-tooltip", "Reverse Series"),
        UnweaveSeries: new Classes.Term("webviewer:unweave-series", "Unweave"),
        UnweaveSeriesTooltip: new Classes.Term("webviewer:unweave-series-tooltip", "Unweave Series"),
        RearrangeSeries: new Classes.Term("webviewer:rearrange-series", "Rearrange"),
        RearrangeSeriesTooltip: new Classes.Term("webviewer:rearrange-series-tooltip", "Rearrange Series"),
        PartSeries: new Classes.Term("webviewer:part-series", "Part"),
        PartSeriesTooltip: new Classes.Term("webviewer:part-series-tooltip", "Part Series"),
        MergeSeries: new Classes.Term("webviewer:merge-series", "Merge"),
        MergeSeriesTooltip: new Classes.Term("webviewer:merge-series-tooltip", "Merge Series"),
        ResetStudyArrangement: new Classes.Term("webviewer:reset-study-arrangement", "Reset"),
        ResetStudyArrangementTooltip: new Classes.Term("webviewer:reset-study-arrangement-tooltip", "Reset Arrangement"),
        StartMeeting: new Classes.Term("webviewer:start-meeting", "Meet"),
        StartMeetingTooltip: new Classes.Term("webviewer:start-meeting-tooltip", "Start New Meeting"),
        KeyImage: new Classes.Term("webviewer:key-image", "Key"),
        KeyImageTooltip: new Classes.Term("webviewer:key-image-tooltip", "Mark As Key Image"),
        ShowOnlyKeyImages: new Classes.Term("webviewer:show-only-key-image", "Key Only"),
        ShowOnlyKeyImagesTooltip: new Classes.Term("webviewer:show-only-key-image-tooltip", "Only Display Key Images"),
        SaveKeyImageLayout: new Classes.Term("webviewer:save-key-image-layout", "Save Layout"),
        SaveKeyImageLayoutTooltip: new Classes.Term("webviewer:save-key-image-layout-tooltip", "Save Key Image Layout"),
        Delete: new Classes.Term("webviewer:delete", "Delete"),
        DeleteTooltip: new Classes.Term("webviewer:delete-tooltip", "Delete Images and Series"),
        DeleteImage: new Classes.Term("webviewer:delete-image", "Image"),
        DeleteImageTooltip: new Classes.Term("webviewer:delete-image-tooltip", "Delete Selected Image"),
        RemoveImages: new Classes.Term("webviewer:remove-images", "Remove"),
        RemoveImagesTooltip: new Classes.Term("webviewer:remove-images-tooltip", "Remove Images"),
        DeleteSeries: new Classes.Term("webviewer:delete-series", "Series"),
        DeleteSeriesTooltip: new Classes.Term("webviewer:delete-series-tooltip", "Delete Entire Series"),
        PreviousSeriesSet: new Classes.Term("webviewer:prev-series", "Prev."),
        PreviousSeriesSetTooltip: new Classes.Term("webviewer:prev-series-tooltip", "Previous Series"),
        NextSeriesSet: new Classes.Term("webviewer:next-series", "Next"),
        NextSeriesSetTooltip: new Classes.Term("webviewer:next-series-tooltip", "Next Series"),
        CineAll: new Classes.Term("webviewer:cine-all", "Cine All"),
        CineAllTooltip: new Classes.Term("webviewer:cine-all-tooltip", "Play/Pause All Multiframe Series"),
        DuplicateAnnotation: new Classes.Term("webviewer:duplicate-anotation", "Dupl."),
        DuplicateAnnotationTooltip: new Classes.Term("webviewer:duplicate-anotation-tooltip", "Duplicate Annotation"),
        CopyAnnotation: new Classes.Term("webviewer:copy-anotation", "Copy"),
        CopyAnnotationTooltip: new Classes.Term("webviewer:copy-anotation-tooltip", "Copy Annotation"),
        PasteAnnotation: new Classes.Term("webviewer:paste-anotation", "Paste"),
        PasteAnnotationTooltip: new Classes.Term("webviewer:paste-anotation-tooltip", "Paste Annotation"),
        Cine: new Classes.Term("webviewer:cine", "Cine"),
        CineTooltip: new Classes.Term("webviewer:cine-tooltip", "Cine"),
        Play: new Classes.Term("webviewer:play", "Play"),
        PlayTooltip: new Classes.Term("webviewer:play-tooltip", "Play"),
        Pause: new Classes.Term("webviewer:pause", "Pause"),
        PauseTooltip: new Classes.Term("webviewer:pause-tooltip", "Pause"),
        Faster: new Classes.Term("webviewer:faster", "Faster"),
        FasterTooltip: new Classes.Term("webviewer:faster-tooltip", "Faster"),
        Slower: new Classes.Term("webviewer:slower", "Slower"),
        SlowerTooltip: new Classes.Term("webviewer:slower-tooltip", "Slower"),
        Stop: new Classes.Term("webviewer:stop", "Stop"),
        StopTooltip: new Classes.Term("webviewer:stop-tooltip", "Stop"),
        Recordings: new Classes.Term("webviewer:recordings", "Recordings"),
        RecordingsTooltip: new Classes.Term("webviewer:recordings-tooltip", "Show Recordings"),
        AttachedReports: new Classes.Term("webviewer:attached-reports", "Attached Reports"),
        ReportsTooltip: new Classes.Term("webviewer:reports-tooltip", "Show Attached Reports"),
        Rewind: new Classes.Term("webviewer:rewind", "Rewind"),
        RewindTooltip: new Classes.Term("webviewer:rewind-tooltip", "Rewind 5 Seconds"),
        Forward: new Classes.Term("webviewer:forward", "Forward"),
        ForwardTooltip: new Classes.Term("webviewer:forward-tooltip", "Fast Forward 5 Seconds"),
        FullScreen: new Classes.Term("webviewer:fullscreen", "Full"),
        FullScreenTooltip: new Classes.Term("webviewer:fullscreen-tooltip", "Show FullScreen"),
        UseDiagnosticQuality: new Classes.Term("webviewer:use-diagnostic-quality", "Diag"),
        UseDiagnosticQualityTooltip: new Classes.Term("webviewer:use-diagnostic-quality-tooltip", "Use Diagnostic Quality Always"),
        RecordAudio: new Classes.Term("webviewer:record-audio", "Record"),
        RecordAudioTooltip: new Classes.Term("webviewer:record-audio-tooltip", "Record Audio"),
        Show: new Classes.Term("webviewer:show", "Show"),
        Hide: new Classes.Term("webviewer:hide", "Hide"),
        Mouse: new Classes.Term("webviewer:mouse", "Mouse"),
        MouseTooltip: new Classes.Term("webviewer:mouse-tool-settings", "Mouse Tool Settings"),
        ToolDependent: new Classes.Term("webviewer:tool-dependent", "Tool Dependent"),
        Close: new Classes.Term("webviewer:close", "Close"),
        Done: new Classes.Term("webviewer:done", "Done"),
        Cancel: new Classes.Term("webviewer:cancel", "Cancel"),
        Save: new Classes.Term("webviewer:save", "Save"),
        NextStudy: new Classes.Term("webviewer:next", "Next"),
        PreviousStudy: new Classes.Term("webviewer:previous", "Previous"),
        NextStudyTooltip: new Classes.Term("webviewer:next-study", "Next Study"),
        PreviousStudyTooltip: new Classes.Term("webviewer:previous-study", "Previous Study"),
        NextStudyByMRN: new Classes.Term("webviewer:next-mrn", "Next"),
        PreviousStudyByMRN: new Classes.Term("webviewer:previous-mrn", "Previous"),
        NextStudyByMRNTooltip: new Classes.Term("webviewer:next-study-mrn", "Next Patient"),
        PreviousStudyByMRNTooltip: new Classes.Term("webviewer:previous-study-mrn", "Previous Patient"),
        NextImage: new Classes.Term("webviewer:next-image", "Next"),
        PreviousImage: new Classes.Term("webviewer:previous-image", "Previous"),
        NextImageTooltip: new Classes.Term("webviewer:next-image", "Next Image"),
        PreviousImageTooltip: new Classes.Term("webviewer:previous-image", "Previous Image"),
        LayoutSingleSeries: new Classes.Term("webviewer:layout-series", "Layout Series"),
        LayoutSingleSeriesTooltip: new Classes.Term("webviewer:layout-single-series", "Layout Single Series"),
        ShowUltrasoundRegion: new Classes.Term("webviewer:ultrasound-region", "Show US"),
        HideUltrasoundRegion: new Classes.Term("webviewer:ultrasound-region", "Hide US"),
        UltrasoundRegionTooltip: new Classes.Term("webviewer:paste-anotation-tooltip", "Show/Hide Ultrasound Regions"),
        OpenReports: new Classes.Term("webviewer:reports", "Reports"),
        OpenReportsTooltip: new Classes.Term("webviewer:reports-tooltip", "Open Reports"),
        NewReport: new Classes.Term("webviewer:new-report", "Report"),
        NewReportTooltip: new Classes.Term("webviewer:new-report-tooltip", "Create New Report"),
        OpenReport: new Classes.Term("webviewer:open-report", "Open Report"),
        DeleteReport: new Classes.Term("webviewer:delete-report", "Delete Report"),
        /* Text Annotations */
        Patient: new Classes.Term("webviewer:patient", "Patient"),
        DOB: new Classes.Term("webviewer:dob", "DOB"),
        Primary: new Classes.Term("webviewer:primary", "Primary"),
        Related: new Classes.Term("webviewer:related", "Related"),
        Current: new Classes.Term("webviewer:current", "Latest Study"),
        Prior: new Classes.Term("webviewer:prior", "Prior"),
        FPS: new Classes.Term("webviewer:fps", "FPS"),
        ReferringPhysician: new Classes.Term("webviewer:referring-physician", "Ref. Phys."),
        Quality: new Classes.Term("webviewer:quality", "Quality"),
        HD: new Classes.Term("webviewer:hd", "HD"),
        SD: new Classes.Term("webviewer:sd", "SD"),
        Diagnostic: new Classes.Term("webviewer:diagnostic", "Diagnostic"),
        CenterWidth: new Classes.Term("webviewer:center-width", "Center/Width"),
        Auto: new Classes.Term("webviewer:auto", "Auto"),
        ImageType: new Classes.Term("webviewer:image-type", "Image Type"),
        Measurement: new Classes.Term("webviewer:measurement", "Measurement"),
        Area: new Classes.Term("webviewer:area", "Area"),
        Min: new Classes.Term("webviewer:min", "Min"),
        Max: new Classes.Term("webviewer:max", "Max"),
        Mean: new Classes.Term("webviewer:mean", "Mean"),
        StandardDev: new Classes.Term("webviewer:stdev", "SD"),
        SliceThickness: new Classes.Term("webviewer:slice-thickness", "Slice Thickness"),
        mm: new Classes.Term("webviewer:mm", "mm"),
        PixelSpacing: new Classes.Term("webviewer:pixel-spacing", "Distance"),
        PatientGeometry: new Classes.Term("webviewer:patient-geometry", "Patient Geometry"),
        AtImagingPlate: new Classes.Term("webviewer:at-imaging-plate", "Imaging Plate"),
        kVp: new Classes.Term("webviewer:kVp", "KVp"),
        mA: new Classes.Term("webviewer:mA", "mAs"),
        Exposure: new Classes.Term("webviewer:exposure", "Exposure"),
        MachineName: new Classes.Term("webviewer:machine-name", "Machine Name"),
        ImageLaterality: new Classes.Term("webviewer:image-laterality", "Image Laterality"),
        ViewPosition: new Classes.Term("webviewer:view-position", "View Position"),
        StationName: new Classes.Term("webviewer:station-name", "Station Name"),
        OperatorsName: new Classes.Term("webviewer:operators-name", "Operator's Name"),
        InstitutionName: new Classes.Term("webviewer:institution-name", "Institution Name"),
        InstitutionAddress: new Classes.Term("webviewer:institution-address", "Institution Address"),
        DetectorID: new Classes.Term("webviewer:detector-id", "Detector ID"),
        PatientMRN: new Classes.Term("webviewer:mrn", "MRN"),
        /* Print Viewer */
        PatientDOB: new Classes.Term("webviewer:patient-dob", "Patient DOB"),
        StudyDescription: new Classes.Term("webviewer:study-description", "Study Description"),
        StudyCreatedOn: new Classes.Term("webviewer:study-created-on", "Study Created On"),
        Untitled: new Classes.Term("webviewer:untitled", "Untitled"),
        TrueSize: new Classes.Term("webviewer:truesize", "True Size"),
        FullSize: new Classes.Term("webviewer:fullsize", "Full Size"),
        FitToPage: new Classes.Term("webviewer:fit-to-page", "Fit to Page"),
        ShowAllTrueSize: new Classes.Term("webviewer:show-all-truesize", "Show All True Size"),
        ShowAllFullSize: new Classes.Term("webviewer:show-all-fullsize", "Show All Full Size"),
        ShowAllFitToPage: new Classes.Term("webviewer:show-all-fit-to-page", "Show All Fit To Page"),
        Clear: new Classes.Term("webviewer:clear", "Clear"),
        Prev: new Classes.Term("webviewer:prev", "Prev"),
        Previous: new Classes.Term("webviewer:previous", "Previous"),
        Next: new Classes.Term("webviewer:next", "Next"),
        Remove: new Classes.Term("webviewer:remove", "Remove"),
        BackToTop: new Classes.Term("webviewer:back-to-top", "Back To Top"),
        PrintViewerRedirectMessageLine1: new Classes.Term("webviewer:print-redirect-line1", "Your web browser does not meet the minimum requirements for the Ambra Viewer application. You have been redirected to the Basic Viewer."),
        PrintViewerRedirectMessageLine2: new Classes.Term("webviewer:print-redirect-line2", "Click here for more information"),
        PrintViewerInstructions: new Classes.Term("webviewer:print-viewer-instructions", "Select images below to include them in the printed study."),
        /* Window Level Presets */
        SoftTissue: new Classes.Term("webviewer:soft-tissue", "Soft Tissue"),
        Bone: new Classes.Term("webviewer:bone", "Bone"),
        Head: new Classes.Term("webviewer:head", "Head"),
        Lung: new Classes.Term("webviewer:lung", "Lung"),
        Default: new Classes.Term("webviewer:default", "Default"),
        EnterPresetName: new Classes.Term("webviewer:enter-preset-name", "Enter a name for this window level preset"),
        SelectPresetToReplace: new Classes.Term("webviewer:select-preset-to-replace", "Which preset (1-9) would you like to replace?"),
        /* Color Table Presets */
        HotIron: new Classes.Term("webviewer:hot-iron", "Hot Iron"),
        PET: new Classes.Term("webviewer:pet", "PET"),
        PET20: new Classes.Term("webviewer:pet20", "PET 20"),
        HotBlue: new Classes.Term("webviewer:hot-blue", "Hot Blue"),
        Fire: new Classes.Term("webviewer:fire", "Fire"),
        /* Cine */
        VideoLoading: new Classes.Term("webviewer:video-loading", "Preparing video, please wait..."),
        VideoLoadingError: new Classes.Term("webviewer:video-error", "Error loading video"),
        VideoNotSupported: new Classes.Term("webviewer:video-not-supported", "Sorry, your browser does not support MP4 video. Please use a recent version of Chrome, IE or Safari to use the multiframe cine functionality."),
        /* Secondary Capture */
        SecondaryCaptureSuccess: new Classes.Term("webviewer:secondary-capture-success", "The secondary capture image was stored successfully."),
        SecondaryCaptureFailed: new Classes.Term("webviewer:secondary-capture-failed", "An error occurred during storage of the secondary capture image."),
        ImageCaptureSuccess: new Classes.Term("webviewer:image-capture-success", "The capture image was stored successfully."),
        ImageCaptureFailed: new Classes.Term("webviewer:image-capture-failed", "An error occurred during storage of the capture image."),
        /* Anonymization */
        CannotAnonymize: new Classes.Term("webviewer:cannot-anonymize", "Please create at least one rectangular measurement before anonymizing a study."),
        AnonymizeWarning: new Classes.Term("webviewer:anonymize-warning", "Are you sure you would like to anonymize this study? This operation will create a new study."),
        AnonymizeSuccess: new Classes.Term("webviewer:anonymize-success", "The study was enqueued for anonymization. The anonymized study will be stored in this namespace when the process is complete."),
        AnonymizeFailed: new Classes.Term("webviewer:anonymize-failed", "An error occurred during storage of the anonymized image(s)."),
        /* Crop */
        CannotCrop: new Classes.Term("webviewer:cannot-crop-1", "Please create at least one rectangular measurement before cropping."),
        CropWarning: new Classes.Term("webviewer:crop-warning", "Are you sure you would like to crop these series? This operation will create a new study."),
        CropSuccess: new Classes.Term("webviewer:crop-success", "The study was enqueued for cropping."),
        CropFailed: new Classes.Term("webviewer:crop-failed", "An error occurred during storage of the cropped series."),
        /* Study Structure Modification */
        SplitWarningNoSeries: new Classes.Term("webviewer:no-series-found-to-split", "No series found to split."),
        SplitWarningAllSeries: new Classes.Term("webviewer:all-series-found-to-split", "Cannot select all series to split."),
        SplitWarning: new Classes.Term("webviewer:no-series-found-to-split", "The selected series will be split into a new study.  Proceed?"),
        SplitSuccess: new Classes.Term("webviewer:split-success", "The split study was enqueued. It will appear in the study list when the process is complete."),
        SplitFailed: new Classes.Term("webviewer:split-failed", "An error occurred during storage of the split study."),
        SplitInstructions: new Classes.Term("webviewer:split-instructions", "Click thumbnails to select series.  Then click Split again to finalize."),
        RemoveImagesInstructions: new Classes.Term("webviewer:remove-images-instructions", "Remove images (format: \"1-16,18,20,25-36\") from series:"),
        RemoveImagesWarning: new Classes.Term("webviewer:remove-images-warning", "The range of values was invalid."),
        RemoveImagesError: new Classes.Term("webviewer:remove-images-error", "There was an error removing one or more images."),
        RemoveImagesSuccess: new Classes.Term("webviewer:remove-images-success", "The specified images have been removed."),
        StudyPHIExtendedError: new Classes.Term("webviewer:study-phi-extended-error", "Could not update study information."),
        StudyPHIExtendedSuccess: new Classes.Term("webviewer:study-phi-extended-success", "Study has been updated."),
        RearrangeImagesInstructions: new Classes.Term("webviewer:rearrange-images-instructions", "Move images (format: \"1-16,18,20,25-36\") to just before current image location for series:"),
        RearrangeImagesWarning: new Classes.Term("webviewer:rearrange-images-warning", "The range of values was invalid."),
        PartImagesInstructions: new Classes.Term("webviewer:part-images-instructions", "Move images (format: \"1-16,18,20,25-36\") to new series from series:"),
        StudyPHIExtendedResetSuccess: new Classes.Term("webviewer:study-phi-reset-extended-success", "Please reload the study to viewer the original data."),
        MergeInstructions: new Classes.Term("webviewer:merge-instructions", "Click thumbnails to select series.  Then click Merge again to finalize."),
        /* GSPS */
        SaveGSPSSuccess: new Classes.Term("webviewer:save-gsps-success", "The presentation state was stored successfully."),
        SaveGSPSFailed: new Classes.Term("webviewer:save-gsps-failed", "An error occurred during storage of the presentation state data."),
        /* Recorder */
        RecordedScriptIntroText: new Classes.Term("webviewer:recorded-script-intro", "The following URL can be copied and shared to replay the recorded script in the browser."),
        RecordingInProgress: new Classes.Term("webviewer:recording-in-progress", "Recording in progress"),
        PlaybackInProgress: new Classes.Term("webviewer:playback-in-progress", "Playback in progress"),
        ScriptSaved: new Classes.Term("webviewer:script-saved", "The recorded script was saved as an attachment."),
        ScriptSaveError: new Classes.Term("webviewer:script-save-error", "An error occurred while trying to store the recorded script."),
        AudioRecordingNotSupported: new Classes.Term("webviewer:audio-recording-not-supported", "Audio recording is not supported."),
        RecordingName: new Classes.Term("webviewer:recording-name", "Recording name"),
        RecordingUntitled: new Classes.Term("webviewer:recording-untitled", "Untitled Recording"),
        SentRecordingLink: new Classes.Term("webviewer:recording-link-sent", "The recording link has been sent."),
        ErrorSharingRecording: new Classes.Term("webviewer:error-sharing-recording", "An error occurred while attempting to create a public link to this recording."),
        ShareRecordingLink: new Classes.Term("webviewer:share-recording-link", "Share Recording Link"),
        PlayRecording: new Classes.Term("webviewer:play-recording", "Play Recording"),
        DeleteRecording: new Classes.Term("webviewer:delete-recording", "Delete Recording"),
        ErrorDeletingRecording: new Classes.Term("webviewer:error-deleting-recording", "A problem occurred while trying to delete the recording."),
        DeleteRecordingWarning: new Classes.Term("webviewer:delete-recording-warning", "Are you sure you would like to delete this recording? This operation cannot be undone."),
        NoRecordingsFound: new Classes.Term("webviewer:no-recordings-found", "No recordings found"),
        DeleteReportWarning: new Classes.Term("webviewer:delete-report-warning", "Are you sure you would like to delete this report? This operation cannot be undone."),
        ErrorDeletingReport: new Classes.Term("webviewer:error-deleting-report", "A problem occurred while trying to delete the report."),
        NoReportsFound: new Classes.Term("webviewer:no-reports-found", "No reports found"),
        NoGSPSFound: new Classes.Term("webviewer:no-gsps-found", "No GSPS found"),
        /* Meeting */
        EnterMeetingName: new Classes.Term("webviewer:enter-meeting-name", "Enter a name for this meeting:"),
        DefaultMeetingName: new Classes.Term("webviewer:default-meeting-name", "Untitled Meeting"),
        CannotStartMeeting: new Classes.Term("webviewer:cannot-start-meeting", "An error occurred while trying to start the meeting."),
        CannotJoinMeeting: new Classes.Term("webviewer:cannot-join-meeting", "An error occurred while trying to join the meeting."),
        CannotEndMeeting: new Classes.Term("webviewer:cannot-end-meeting", "An error occurred while trying to end the meeting."),
        CannotLeaveMeeting: new Classes.Term("webviewer:cannot-leave-meeting", "An error occurred while trying to leave the meeting."),
        MeetingEnded: new Classes.Term("webviewer:meeting-ended", "The meeting has ended."),
        ShareMeetingLink: new Classes.Term("webviewer:share-meeting-link", "Share Meeting Link"),
        CreatePublicLink: new Classes.Term("webviewer:create-public-link", "Create Public Link"),
        MeetingShared: new Classes.Term("webviewer:meeting-shared", "You can invite users to this meeting by sharing the following link."),
        MeetingAttendees: new Classes.Term("webviewer:meeting-attendees", "Attendees"),
        ErrorSharingMeeting: new Classes.Term("webviewer:error-sharing-meeting", "An error occurred while attempting to create a public link to this meeting."),
        YouAreThePresenter: new Classes.Term("webviewer:you-are-the-presenter", "You have been made the meeting presenter."),
        LeaveMeeting: new Classes.Term("webviewer:leave-meeting", "Leave Meeting"),
        EndMeeting: new Classes.Term("webviewer:end-meeting", "End Meeting"),
        MakePresenter: new Classes.Term("webviewer:make-presenter", "Make Presenter"),
        ConfirmChangeOfPresenter: new Classes.Term("webviewer:confirm-change-of-presenter", "Are you sure you would like to change the meeting presenter?"),
        CannotChangePresenter: new Classes.Term("webviewer:cannot-change-presenter", "An error occurred while trying to change the meeting presenter."),
        SentMeetingLink: new Classes.Term("webviewer:meeting-link-sent", "The meeting link has been sent."),
        MeetingSharing: new Classes.Term("webviewer:sharing", "Sharing"),
        /* Delete Images */
        DeleteImageWarning: new Classes.Term("webviewer:delete-image-warning", "Are you sure you would like to delete this image? This operation cannot be undone."),
        DeleteSeriesWarning: new Classes.Term("webviewer:delete-series-warning", "Are you sure you would like to delete this series? This operation cannot be undone."),
        ErrorDeletingImage: new Classes.Term("webviewer:error-deleting-image", "A problem occurred while trying to delete the image."),
        ErrorDeletingSeries: new Classes.Term("webviewer:error-deleting-series", "A problem occurred while trying to delete the series."),
        /* Study Actions */
        EnterEmailAddress: new Classes.Term("webviewer:enter-email-address", "Please enter an email address to share the study:"),
        EnterShareMessage: new Classes.Term("webviewer:enter-share-message", "You may enter an optional message for the email recipient:"),
        /* Image loading error */
        ImageLoadingErrorLine1: new Classes.Term("webviewer:image-error-line1", "This image is currently unavailable."),
        ImageLoadingErrorLine2: new Classes.Term("webviewer:image-error-line2", "Please try again later."),
        /* End User Agreement */
        EndUserTitle: new Classes.Term("webviewer:enduser-title", "Not Intended For Diagnostic Use"),
        EndUserBody: new Classes.Term("webviewer:enduser-body", 'By clicking "I Agree" you acknowledge this viewer is not intended for diagnostic use.'),
        EndUserDecline: new Classes.Term("webviewer:enduser-decline", "Cancel"),
        EndUserAccept: new Classes.Term("webviewer:enduser-accept", "I Agree"),
        /* Mouse Tool Settings */
        MouseSettingsPanelInstructions: new Classes.Term("webviewer:mousepanel-instructions", "Click with left or right mouse button to assign a tool to a mouse button."),
        RightButton: new Classes.Term("webviewer:mouse-right-button", "Right Button"),
        LeftButton: new Classes.Term("webviewer:mouse-left-button", "Left Button"),
        WheelScroll: new Classes.Term("webviewer:mouse-wheel-scroll", "Wheel"),
        /* Local Viewer Instructions */
        LocalViewerInstructionsTitle: new Classes.Term("webviewer:localviewer-instructions-title", "Viewer Instructions"),
        LocalViewerInstructionsBody: new Classes.Term("webviewer:localviewer-instructions-body", "To run the viewer on your computer, please use the following instructions."),
        LocalViewerInstructionsWinHeader: new Classes.Term("webviewer:localviewer-instructions-win-head", "Windows"),
        LocalViewerInstructionsWinBody: new Classes.Term("webviewer:localviewer-instructions-win-body", "Navigate to your CD and find the file named viewer.exe. Double click this file to start the viewer."),
        LocalViewerInstructionsMacHeader: new Classes.Term("webviewer:localviewer-instructions-mac-head", "Mac OS X"),
        LocalViewerInstructionsMacBody: new Classes.Term("webviewer:localviewer-instructions-mac-body", "Navigate to your CD and find the file named viewer. Double click this file to start the viewer."),
        /* Key Images */
        ClearedKeyImageLayout: new Classes.Term("webviewer:cleared-key-image-layout", "The key image layout has been cleared."),
        SaveKeyImageLayoutSuccess: new Classes.Term("webviewer:save-key-image-layout-success", "The current key image layout has been saved."),
        SaveKeyImageLayoutError: new Classes.Term("webviewer:save-key-image-layout-error", "There was a problem saving the current key image layout."),
        /* Warning Banners */
        PropagateInstructions: new Classes.Term("webviewer:propagate-instructions", "Scroll to propagate the selected annotation to slice."),
        PropagateAllComplete: new Classes.Term("webviewer:propagate-all-complete", "The annotation has been propagated to all instances in this series."),
        WarningCalibrationUsed: new Classes.Term("webviewer:calibration-used", "Measurements on this image are calculated using manually entered data."),
        WarningInstanceStamped: new Classes.Term("webviewer:instance-stamped", "This instance has been stamped by another user.  A new annotation cannot be created at this time."),
        OperationNotSupported: new Classes.Term("webviewer:operation-not-supported", "This operation is not supported."),
        OperationExceededLimits: new Classes.Term("webviewer:operation-not-supported", "This operation could not be completed."),
        OperationNoResult: new Classes.Term("webviewer:operation-no-result", "No area was found based on that threshold."),
        FrozenStudyWarning: new Classes.Term("webviewer:frozen-study", "This study is locked and cannot be edited at this time.")
    };

    function allViewerTerms() {
        var terms = [];

        for (var name in Terminology.Terms) {
            if (Terminology.Terms.hasOwnProperty(name)) {
                terms.push(Terminology.Terms[name]);
            }
        }

        return terms;
    }

    /**
    * A map of default values
    */
    function defaultValues() {
        return { lookup: function (term) {
                return term.defaultValue;
            } };
    }
    Terminology.defaultValues = defaultValues;

    /**
    * Load terminology asynchronously
    */
    function loadTerminology(sessionId, query) {
        var tags = _.map(allViewerTerms(), function (t) {
            return t.code;
        });
        var language = currentLanguage();

        return Observable.catchError(Observable.map(Services.loadTerminology(sessionId, tags, language, query), function (tags) {
            return {
                lookup: function (term) {
                    if (tags.values[term.code]) {
                        return tags.values[term.code];
                    } else {
                        return term.defaultValue;
                    }
                }
            };
        }), function () {
            return defaultValues();
        });
    }
    Terminology.loadTerminology = loadTerminology;

    /**
    * Load terminology asynchronously
    */
    function loadAllTerminology(sessionId, query) {
        var language = currentLanguage();

        return Observable.catchError(Observable.map(Services.loadAllTerminology(sessionId, language, query), function (tags) {
            return {
                lookup: function (term) {
                    if (tags.values[term.code]) {
                        return tags.values[term.code];
                    } else {
                        return term.defaultValue;
                    }
                }
            };
        }), function () {
            return defaultValues();
        });
    }
    Terminology.loadAllTerminology = loadAllTerminology;

    /**
    * Get the current language from the query string or browser
    */
    function currentLanguage() {
        var language = Query.findParameter(window.location, "language");

        if (!language) {
            var iso6391Code;

            if (navigator.userLanguage) {
                iso6391Code = navigator.userLanguage;
            } else if (navigator.language) {
                iso6391Code = navigator.language;
            } else {
                iso6391Code = "en";
            }

            language = iso6391Code.split("-")[0];
        }

        return new Classes.Language(language);
    }
    Terminology.currentLanguage = currentLanguage;

    /**
    * Convert an image type to a term
    */
    function imageTypeToTerm(type) {
        switch (type) {
            case 0 /* Thumbnail */:
                return Terminology.Terms.SD;
            case 1 /* FullResolution */:
                return Terminology.Terms.HD;
            case 3 /* FullResolutionHD */:
                return Terminology.Terms.HD;
            case 2 /* Diagnostic */:
                return Terminology.Terms.Diagnostic;
        }

        return null;
    }
    Terminology.imageTypeToTerm = imageTypeToTerm;
})(Terminology || (Terminology = {}));
var Browser;
(function (Browser) {
    Browser.MAX_MOBILE_DIM = 4096;

    function isChrome() {
        return navigator.userAgent.indexOf('Chrome') >= 0;
    }
    Browser.isChrome = isChrome;

    function isFirefox() {
        return !!window.InstallTrigger;
    }
    Browser.isFirefox = isFirefox;

    function isSafari() {
        return navigator.userAgent.indexOf('Safari') >= 0;
    }
    Browser.isSafari = isSafari;

    function isIE() {
        return navigator.userAgent.indexOf('MSIE') >= 0 || navigator.userAgent.indexOf('Trident') >= 0;
    }
    Browser.isIE = isIE;

    function isIE11() {
        return !!window.MSInputMethodContext && !!document.documentMode;
    }
    Browser.isIE11 = isIE11;

    function isIE9() {
        return $('body').is('.ie9');
    }
    Browser.isIE9 = isIE9;

    function isEdge() {
        return navigator.userAgent.indexOf('Edge') >= 0;
    }
    Browser.isEdge = isEdge;

    function isiOS() {
        return navigator.userAgent.indexOf('iPad') >= 0 || navigator.userAgent.indexOf('iPhone') >= 0;
    }
    Browser.isiOS = isiOS;

    function isiPad() {
        return navigator.userAgent.indexOf('iPad') >= 0;
    }
    Browser.isiPad = isiPad;

    function isiPadRetina() {
        return window.devicePixelRatio > 1;
    }
    Browser.isiPadRetina = isiPadRetina;

    function isMobile() {
        var ua = navigator.userAgent;
        return ua.indexOf("Android") >= 0 || ua.indexOf("iPhone") >= 0 || ua.indexOf("iPad") >= 0 || ua.indexOf("Windows Phone") >= 0;
    }
    Browser.isMobile = isMobile;

    function isPhone() {
        return (window.screen.width <= 640) || (window.matchMedia && window.matchMedia("only screen and (max-width: 640px)").matches);
    }
    Browser.isPhone = isPhone;

    function OS() {
        var matches = ["Win", "Mac", "X11", "Linux"];
        return _.find(matches, function (os) {
            return navigator.appVersion.indexOf(os) != -1;
        });
    }

    function viewerOS() {
        var osMap = { Mac: "osx", Win: "win" };
        return osMap[OS()] || "Browser";
    }
    Browser.viewerOS = viewerOS;

    function uiFound() {
        if (window.self !== window.top) {
            return !!window.parent.DG;
        }

        return !!window.DG;
    }
    Browser.uiFound = uiFound;

    /**
    * Check if HD image needs to be resized, since maximum canvas size can be exceeded on mobile platforms.
    * @param instance
    */
    function needsResize(instance) {
        return instance && ((instance.instanceAttributes.columns > Browser.MAX_MOBILE_DIM) || (instance.instanceAttributes.rows > Browser.MAX_MOBILE_DIM)) && Browser.isMobile();
    }
    Browser.needsResize = needsResize;

    function resize(instance) {
        var longest = Math.max(instance.instanceAttributes.rows, instance.instanceAttributes.columns);
        var shortest = Math.min(instance.instanceAttributes.rows, instance.instanceAttributes.columns);
        return Math.floor(Browser.MAX_MOBILE_DIM * (shortest / longest));
    }
    Browser.resize = resize;
})(Browser || (Browser = {}));
/// <reference path="../classes/Types.ts" />
/// <reference path="../models/StudySchema.ts" />
/// <reference path='../models/KO.ts' />
/// <reference path="../typings/jquery/jquery.d.ts" />
/// <reference path="Routes.ts" />
/// <reference path="Observable.ts" />
/**
* Observable wrappers for AJAX calls to storage
*
* @see Observable
*/
var V3Storage;
(function (V3Storage) {
    /**
    * A wrapper for the Study Schema method
    */
    function getStudySchema(sessionId, storageInfo, queryObject, useServices) {
        if (LocalViewer.isLocalViewer()) {
            return Observable.ret(LocalViewer.schema);
        } else {
            return AJAX.ajaxGet(Routes.StudySchema(sessionId, storageInfo, queryObject, useServices));
        }
    }
    V3Storage.getStudySchema = getStudySchema;

    /**
    * A wrapper for the Study PHI method
    */
    function getStudyPhi(sessionId, storageInfo, queryObject) {
        if (LocalViewer.isLocalViewer()) {
            return Observable.ret(LocalViewer.phi);
        } else {
            return AJAX.ajaxGet(Routes.StudyPhi(sessionId, storageInfo, queryObject));
        }
    }
    V3Storage.getStudyPhi = getStudyPhi;

    /**
    * A wrapper for the Study Tags method
    */
    function getStudyTag(sessionId, storageInfo, queryObject) {
        if (LocalViewer.isLocalViewer()) {
            return Observable.ret(LocalViewer.tag);
        } else {
            return AJAX.ajaxGet(Routes.StudyTag(sessionId, storageInfo, queryObject));
        }
    }
    V3Storage.getStudyTag = getStudyTag;

    /**
    * A wrapper for the Study Attributes method
    */
    function getStudyAttributes(sessionId, storageInfo, queryObject) {
        return AJAX.ajaxGet(Routes.StudyAttributes(sessionId, storageInfo, queryObject));
    }
    V3Storage.getStudyAttributes = getStudyAttributes;

    /**
    * A wrapper for the Image Attributes method
    */
    function getImageAttributes(sessionId, storageInfo, queryObject, instanceUid, imageVersion) {
        if (LocalViewer.isPersonalAccelerator()) {
            if (LocalViewer.images[instanceUid.value]) {
                return Observable.ret(LocalViewer.images[instanceUid.value]);
            } else {
                return AJAX.ajaxGet(Routes.ImageAttributes(sessionId, storageInfo, queryObject, instanceUid, imageVersion));
            }
        } else if (LocalViewer.isLocalViewer()) {
            return Observable.ret(LocalViewer.images[instanceUid.value]);
        } else {
            return AJAX.ajaxGet(Routes.ImageAttributes(sessionId, storageInfo, queryObject, instanceUid, imageVersion));
        }
    }
    V3Storage.getImageAttributes = getImageAttributes;

    /**
    * A wrapper for the Image Delete method
    */
    function deleteImage(sessionId, storageInfo, queryObject, instanceUid, imageVersion) {
        if (LocalViewer.isFileSystemViewer()) {
            return Observable.ret({});
        } else {
            return AJAX.ajax(Routes.DeleteImage(sessionId, storageInfo, queryObject, instanceUid), { type: "DELETE" });
        }
    }
    V3Storage.deleteImage = deleteImage;

    /**
    * Delete an attachment
    * @param sessionId
    * @param storageInfo
    * @param queryObject
    * @param attachment
    */
    function deleteAttachment(sessionId, storageInfo, queryObject, attachment) {
        if (LocalViewer.isFileSystemViewer()) {
            return Observable.ret({});
        } else {
            return AJAX.ajax(Routes.DeleteAttachment(sessionId, storageInfo, queryObject, attachment), { type: "DELETE" });
        }
    }
    V3Storage.deleteAttachment = deleteAttachment;

    /**
    * A wrapper for the GSPS method
    */
    function getGSPS(sessionId, storageInfo, queryObject, instanceUid, imageVersion) {
        if (LocalViewer.isLocalViewer()) {
            return Observable.ret({
                identificationModule: {},
                relationshipModule: {
                    referencedSeriesSequence: []
                },
                graphicAnnotationModule: {
                    graphicAnnotationSequence: []
                }
            });
        } else {
            return AJAX.ajaxGet(Routes.GetGSPS(sessionId, storageInfo, queryObject, instanceUid, imageVersion));
        }
    }
    V3Storage.getGSPS = getGSPS;

    /**
    * A wrapper for the attributes JSON method
    */
    function getImageJSON(sessionId, storageInfo, queryObject, instanceUid, imageVersion) {
        if (LocalViewer.isLocalViewer()) {
            return Observable.ret({});
        } else {
            return AJAX.ajaxGet(Routes.ImageJSON(sessionId, storageInfo, queryObject, instanceUid, imageVersion));
        }
    }
    V3Storage.getImageJSON = getImageJSON;

    /**
    * A wrapper for the CADSR method.
    * @param sessionId
    * @param storageInfo
    * @param queryObject
    * @param instanceUid
    * @param imageVersion
    */
    function getCADSR(sessionId, storageInfo, queryObject, instanceUid, imageVersion) {
        return AJAX.ajaxGet(Routes.CADSR(sessionId, storageInfo, queryObject, instanceUid, imageVersion));
    }
    V3Storage.getCADSR = getCADSR;

    /**
    * A wrapper for the Secondary Capture method
    */
    function postSecondaryCaptureImage(sessionId, storageInfo, queryObject, base64ImageData) {
        return AJAX.ajax(Routes.SecondaryCapture(sessionId, storageInfo, queryObject), {
            type: "POST",
            data: JSON.stringify({ base64ImageData: base64ImageData }),
            dataType: "json",
            contentType: "application/json"
        });
    }
    V3Storage.postSecondaryCaptureImage = postSecondaryCaptureImage;

    /**
    * A wrapper for the Secondary Capture method
    */
    function anonymizeStudy(sessionId, storageInfo, queryObject, regions) {
        return AJAX.ajax(Routes.AnonymizeStudy(sessionId, storageInfo, queryObject), {
            type: "POST",
            data: JSON.stringify(regions),
            contentType: "application/json"
        });
    }
    V3Storage.anonymizeStudy = anonymizeStudy;

    /**
    * A wrapper for Study Split method.
    * @param sessionId
    * @param storageInfo
    * @param queryObject
    * @param seriesUIDs
    */
    function splitStudy(sessionId, storageInfo, queryObject, seriesUIDs) {
        return AJAX.ajax(Routes.SplitStudy(sessionId, storageInfo, queryObject, seriesUIDs), {
            type: "POST",
            contentType: "application/json"
        });
    }
    V3Storage.splitStudy = splitStudy;

    function cropStudy(sessionId, storageInfo, queryObject, regions) {
        return AJAX.ajax(Routes.CropStudy(sessionId, storageInfo, queryObject), {
            type: "POST",
            data: JSON.stringify(regions),
            contentType: "application/json"
        });
    }
    V3Storage.cropStudy = cropStudy;

    /**
    * A wrapper for the GSPS method
    */
    function postGSPS(sessionId, storageInfo, queryObject, gsps) {
        return AJAX.ajax(Routes.PutGSPS(sessionId, storageInfo, queryObject), {
            type: "POST",
            data: JSON.stringify(gsps),
            dataType: "json",
            contentType: "application/json"
        });
    }
    V3Storage.postGSPS = postGSPS;

    /**
    * Get an attachment via AJAX
    */
    function GetAttachment(sessionId, storageInfo, queryObject, attachmentId, phiNamespace, version) {
        return AJAX.ajaxGet(Routes.Attachment(sessionId, storageInfo, queryObject, attachmentId, phiNamespace, version));
    }
    V3Storage.GetAttachment = GetAttachment;

    /**
    * Get a structured report
    */
    function GetStructuredReport(sessionId, storageInfo, queryObject, imageUid, version) {
        return AJAX.ajaxGet(Routes.StructuredReport(sessionId, storageInfo, queryObject, imageUid, version));
    }
    V3Storage.GetStructuredReport = GetStructuredReport;

    /**
    * Post an attachment to storage
    */
    function PostAttachment(sessionId, storageInfo, queryObject, contentType, uploadedBy, filename, data) {
        return AJAX.ajax(Routes.PostAttachment(sessionId, storageInfo, queryObject, contentType, uploadedBy, filename), {
            type: "POST",
            data: JSON.stringify(data),
            dataType: "json",
            contentType: "application/json"
        });
    }
    V3Storage.PostAttachment = PostAttachment;

    /**
    * Post an image attachment to storage
    * @param {Classes.SessionId} sessionId
    * @param {Models.StudyStorageInfo} storageInfo
    * @param {Classes.QueryObject} queryObject
    * @param {string} uploadedBy
    * @param blob
    * @returns {Observable.Observable<Models.AttachmentResult>}
    * @constructor
    */
    function PostImageAttachment(sessionId, storageInfo, queryObject, uploadedBy, filename, blob) {
        var fd = new FormData();
        fd.append("data", blob, filename);

        return AJAX.ajax(Routes.PostImageAttachment(sessionId, storageInfo, queryObject, uploadedBy, filename), {
            type: "POST",
            data: fd,
            processData: false,
            contentType: false
        });
    }
    V3Storage.PostImageAttachment = PostImageAttachment;
})(V3Storage || (V3Storage = {}));
/// <reference path="Map.d.ts" />

/**
* Additional methods for dealing with arrays
* @see Array
*/
var Arrays;
(function (Arrays) {
    function toMap(ts, keySelector, valueSelector) {
        var obj = {};

        for (var i = 0; i < ts.length; i++) {
            var key = keySelector(ts[i]);
            var value = valueSelector(ts[i]);
            obj[key] = value;
        }

        return Maps.fromObject(obj);
    }

    function indexWhere(ts, filter) {
        for (var i = 0; i < ts.length; i++) {
            if (filter(ts[i])) {
                return i;
            }
        }
        return -1;
    }

    function only(ts) {
        if (ts.length === 1) {
            return ts[0];
        }
        return null;
    }

    function lastSortedIndex(ts, t, ord, compare) {
        var k = ord(t);
        for (var i = ts.length - 1; i >= 0; i--) {
            if (compare(ord(ts[i]), k) <= 0) {
                return i;
            }
        }
        return -1;
    }

    function sum(ts, f) {
        return _.foldl(ts, function (sum, t) {
            return sum + f(t);
        }, 0);
    }

    function foldl1(ts, f) {
        return _.foldl(_.tail(ts), function (x, a, index, list) {
            return f(x, a);
        }, _.head(ts));
    }

    Array.prototype.toMap = function (keySelector, valueSelector) {
        return toMap(this, keySelector, valueSelector);
    };

    Array.prototype.sum = function (f) {
        return sum(this, f);
    };

    Array.prototype.indexWhere = function (f) {
        return indexWhere(this, f);
    };

    Array.prototype.only = function () {
        return only(this);
    };

    Array.prototype.lastSortedIndex = function (t, f, c) {
        return lastSortedIndex(this, t, f, c);
    };

    Array.prototype.foldl1 = function (f) {
        return foldl1(this, f);
    };
})(Arrays || (Arrays = {}));

/**
* Methods for dealing with maps
* @see Map
*/
var Maps;
(function (Maps) {
    /**
    * Apply the toString function before looking up a key
    */
    function preMap(m) {
        return {
            lookup: function (k) {
                return m.lookup(k.toString());
            }
        };
    }
    Maps.preMap = preMap;

    /**
    * Convert an object to a map, where keys are proprties and values are property values
    */
    function fromObject(obj) {
        return {
            lookup: function (key) {
                return obj[key];
            }
        };
    }
    Maps.fromObject = fromObject;

    /**
    * Lookup a key, returning a default value if none is present
    */
    function lookupDef(m, k, def) {
        var val = m.lookup(k);

        if (val === undefined) {
            return def;
        }

        return val;
    }
    Maps.lookupDef = lookupDef;
})(Maps || (Maps = {}));
var Dicom;
(function (Dicom) {
    /**
    * Represents a DICOM tag - a group and an element
    */
    var Tag = (function () {
        function Tag(group, element, name) {
            this.group = group;
            this.element = element;
            this.name = name;
        }
        Tag.prototype.toString = function () {
            return "(" + this.group + "," + this.element + ")";
        };
        return Tag;
    })();
    Dicom.Tag = Tag;

    var Overlay = (function () {
        function Overlay() {
        }
        Overlay.MAX = 16;
        return Overlay;
    })();
    Dicom.Overlay = Overlay;

    var UltrasoundRegion = (function () {
        function UltrasoundRegion(index, tags) {
            this.index = index;
            this.typeCode = Dicom.findDecimalTagSafely(tags, ["(0018,6014)"]);
            this.unitsCode = {
                x: Dicom.findDecimalTagSafely(tags, ["(0018,6024)"]),
                y: Dicom.findDecimalTagSafely(tags, ["(0018,6026)"]) };
            this.spacing = {
                x: Dicom.findDecimalTagSafely(tags, ["(0018,602C)"]),
                y: Dicom.findDecimalTagSafely(tags, ["(0018,602E)"]) };
            this.bounds = {
                min: {
                    x: Dicom.findDecimalTagSafely(tags, ["(0018,6018)"]),
                    y: Dicom.findDecimalTagSafely(tags, ["(0018,601A)"])
                },
                max: {
                    x: Dicom.findDecimalTagSafely(tags, ["(0018,601C)"]),
                    y: Dicom.findDecimalTagSafely(tags, ["(0018,601E)"])
                }
            };
        }
        UltrasoundRegion.prototype.isDrawable = function () {
            return !isNaN(this.bounds.min.x) && !isNaN(this.bounds.min.y) && !isNaN(this.bounds.max.x) && !isNaN(this.bounds.max.y);
        };

        UltrasoundRegion.prototype.isAnnotatable = function () {
            return (this.unitsCode.x == UltrasoundRegion.UNIT_CM) && (this.unitsCode.y == UltrasoundRegion.UNIT_CM) && (this.spacing.x > 0) && (this.spacing.y > 0);
        };

        UltrasoundRegion.prototype.contains = function (pt) {
            var p1 = this.bounds.min;
            var p2 = this.bounds.max;

            return pt.x >= Math.min(p1.x, p2.x) && pt.x <= Math.max(p1.x, p2.x) && pt.y >= Math.min(p1.y, p2.y) && pt.y <= Math.max(p1.y, p2.y);
        };
        UltrasoundRegion.UNIT_CM = 0x0003;
        return UltrasoundRegion;
    })();
    Dicom.UltrasoundRegion = UltrasoundRegion;

    var Ultrasound = (function () {
        function Ultrasound(json) {
            this.tags = Dicom.findTag(json.tags, ["(0018,6011)"], -1);
        }
        Ultrasound.prototype.getNumRegions = function () {
            if (this.tags && this.tags.items) {
                return this.tags.items.length;
            }

            return 0;
        };

        Ultrasound.prototype.getRegion = function (index) {
            return new UltrasoundRegion(index, this.tags.items[index].tags);
        };

        Ultrasound.prototype.getAllRegions = function () {
            var regions = [];
            var num = this.getNumRegions();

            for (var ctr = 0; ctr < num; ctr += 1) {
                regions.push(this.getRegion(ctr));
            }

            return regions;
        };
        Ultrasound.isUltrasound = function (instance) {
            return instance.seriesAttributes.modality == "US";
        };

        Ultrasound.isUltrasoundSeries = function (seriesAttributes) {
            return seriesAttributes.modality == "US";
        };

        Ultrasound.findSpacing = function (measurementInProgress, instance, json) {
            var us = new Dicom.Ultrasound(json || instance.instanceAttributes.json);
            var regions = us.getAllRegions();
            var pts = measurementInProgress.endpoints();
            var spacingX, spacingY;
            var regionFound = false;
            var validMeasurement = true;

            loop:
            for (var ctrP = 0; ctrP < pts.length; ctrP += 1) {
                var loc = pts[ctrP].location;
                var ptFound = false;

                for (var ctrR = 0; ctrR < regions.length; ctrR += 1) {
                    var region = regions[ctrR];

                    if (region.isAnnotatable()) {
                        if (region.contains(loc)) {
                            ptFound = true;

                            if (regionFound) {
                                if (spacingX != region.spacing.x) {
                                    validMeasurement = false;
                                    break loop;
                                } else if (spacingY != region.spacing.y) {
                                    validMeasurement = false;
                                    break loop;
                                }
                            } else {
                                spacingX = region.spacing.x;
                                spacingY = region.spacing.y;
                                regionFound = true;
                            }
                        }
                    }
                }

                if (!ptFound) {
                    validMeasurement = false;
                    break;
                }
            }

            return { valid: validMeasurement, spacingX: spacingX * 10, spacingY: spacingY * 10 };
        };
        return Ultrasound;
    })();
    Dicom.Ultrasound = Ultrasound;

    var Subtraction = (function () {
        function Subtraction(json) {
            this.tags = Dicom.findTag(json.tags, ["(0028,6100)"], -1);
        }
        Subtraction.prototype.getMaskIndices = function () {
            if (this.tags && this.tags.length) {
                var originalIndices = Dicom.findDecimalsTagSafely(this.tags.items[0].tags, ["(0028,6110)"]);

                if (!originalIndices) {
                    originalIndices = [1];
                }

                return _.map(originalIndices, function (index) {
                    return index - 1;
                });
            }

            return [0];
        };

        Subtraction.prototype.isMask = function (instance, series) {
            var index = _.indexOf(series.instances, instance);
            return _.contains(this.getMaskIndices(), index);
        };

        Subtraction.prototype.allMasksLoaded = function (series) {
            var maskIndices = this.getMaskIndices();

            if (!maskIndices) {
                maskIndices = [0];
            }

            var allLoaded = true;

            _.each(maskIndices, function (index) {
                var maskInstance = series.series.instances[index];
                var imageType = series.getImageType(maskInstance);
                var instanceKey = series.getInstanceKey(index);
                var image = series.renderer.imageElements.get(instanceKey, imageType);

                allLoaded = (allLoaded && (!!image && image.imageElement.complete && typeof image.imageElement.naturalWidth !== "undefined" && image.imageElement.naturalWidth !== 0));
            });

            return allLoaded;
        };

        Subtraction.supportsSubtraction = function (seriesAttributes) {
            return seriesAttributes.modality == "XA";
        };

        Subtraction.buildXA = function (instance) {
            if (instance.instanceAttributes.json) {
                return new Subtraction(instance.instanceAttributes.json);
            } else {
                return null;
            }
        };
        return Subtraction;
    })();
    Dicom.Subtraction = Subtraction;

    /**
    * Standard DICOM tags used by the application
    */
    var Dictionary = (function () {
        function Dictionary() {
        }
        Dictionary.ImageType = new Tag("0008", "0008", "Image Type");
        Dictionary.SOPInstanceUID = new Tag("0008", "0018", "SOP Instance UID");
        Dictionary.StudyCreateDate = new Tag("0008", "0020", "Study Create Date");
        Dictionary.SeriesCreateDate = new Tag("0008", "0021", "Series Create Date");
        Dictionary.AcquisitionDateTime = new Tag("0008", "002A", "Acquisition Date/Time");
        Dictionary.StudyCreateTime = new Tag("0008", "0030", "Study Create Time");
        Dictionary.SeriesCreateTime = new Tag("0008", "0031", "Series Create Time");
        Dictionary.AccessionNumber = new Tag("0008", "0050", "Accession Number");
        Dictionary.Modality = new Tag("0008", "0060", "Modality");
        Dictionary.ReferringPhysicianName = new Tag("0008", "0090", "Referring Physician Name");
        Dictionary.StationName = new Tag("0008", "1010", "Station Name");
        Dictionary.StudyDescription = new Tag("0008", "1030", "Study Description");
        Dictionary.SeriesDescription = new Tag("0008", "103E", "Series Description");
        Dictionary.OperatorsName = new Tag("0008", "1070", "Operator's Name");
        Dictionary.InstitutionName = new Tag("0008", "0080", "Institution Name");
        Dictionary.InstitutionAddress = new Tag("0008", "0081", "Institution Address");
        Dictionary.ReferencedImageSequence = new Tag("0008", "1140", "Referenced Image Sequence");
        Dictionary.ReferencedInstanceUID = new Tag("0008", "1155", "Referenced Instance UID");
        Dictionary.RecommendedFrameRate = new Tag("0008", "2144", "Recommended Frame Rate");
        Dictionary.PatientName = new Tag("0010", "0010", "Patient Name");
        Dictionary.PatientID = new Tag("0010", "0020", "Patient ID");
        Dictionary.PatientBirthDate = new Tag("0010", "0030", "Patient Birth Date");
        Dictionary.PatientSex = new Tag("0010", "0040", "Patient Sex");
        Dictionary.OtherPatientIDs = new Tag("0010", "1000", "Other Patient IDs");
        Dictionary.PatientAge = new Tag("0010", "1010", "Patient Age");
        Dictionary.PatientSize = new Tag("0010", "1020", "Patient Size");
        Dictionary.PatientWeight = new Tag("0010", "1030", "Patient Weight");
        Dictionary.PatientAddress = new Tag("0010", "1040", "Patient Address");
        Dictionary.AdditionalPatientHistory = new Tag("0010", "21B0", "Additional Patient History");
        Dictionary.PatientComments = new Tag("0010", "4000", "Patient Comments");
        Dictionary.BodyPart = new Tag("0018", "0015", "Body Part");
        Dictionary.CineRate = new Tag("0018", "0040", "Cine Rate");
        Dictionary.HeartRate = new Tag("0018", "1088", "Heart Rate");
        Dictionary.RadiographicMagnificationFactor = new Tag("0018", "1114", "Radiographic Magnification Factor");
        Dictionary.ImagerPixelSpacing = new Tag("0018", "1164", "Imager Pixel Spacing");
        Dictionary.PatientPosition = new Tag("0018", "5100", "Patient Position");
        Dictionary.ViewPosition = new Tag("0018", "5101", "View Position");
        Dictionary.DetectorID = new Tag("0018", "700A", "Detector ID");
        Dictionary.StudyUID = new Tag("0020", "000D", "Study UID");
        Dictionary.SeriesUID = new Tag("0020", "000E", "Series UID");
        Dictionary.InstanceNumber = new Tag("0020", "0013", "Instance Number");
        Dictionary.PatientOrientation = new Tag("0020", "0020", "Patient Orientation");
        Dictionary.ImagePositionPatient = new Tag("0020", "0032", "Image Position Patient");
        Dictionary.ImageOrientationPatient = new Tag("0020", "0037", "Image Orientation Patient");
        Dictionary.ImageLaterality = new Tag("0020", "0062", "Image Laterality");
        Dictionary.PhotometricInterpretation = new Tag("0028", "0004", "Photometric Interpretation");
        Dictionary.Rows = new Tag("0028", "0010", "Rows");
        Dictionary.Columns = new Tag("0028", "0011", "Columns");
        Dictionary.PixelSpacing = new Tag("0028", "0030", "Pixel Spacing");
        Dictionary.PixelAspectRatio = new Tag("0028", "0034", "Pixel Aspect Ratio");
        Dictionary.BitsStored = new Tag("0028", "0101", "Bits Stored");
        Dictionary.PixelRepresentation = new Tag("0028", "0103", "Pixel Representation");
        Dictionary.PixelSpacingCalibrationDescription = new Tag("0028", "0A04", "Pixel Spacing Calibration Description");
        Dictionary.WindowCenter = new Tag("0028", "1050", "Window Center");
        Dictionary.WindowWidth = new Tag("0028", "1051", "Window Width");
        Dictionary.RescaleIntercept = new Tag("0028", "1052", "RescaleIntercept");
        Dictionary.RescaleSlope = new Tag("0028", "1053", "Rescale Slope");
        Dictionary.RescaleType = new Tag("0028", "1054", "Rescale Type");
        Dictionary.VOILUTFunction = new Tag("0028", "1056", "VOI LUT Function");
        Dictionary.FrameTime = new Tag("0018", "1063", "Frame Time");
        Dictionary.BreastImplantPresent = new Tag("0028", "1300", "Breast Implant Present");
        Dictionary.RequestedProcedureDescription = new Tag("0032", "1060", "Requested Procedure Description");
        Dictionary.SliceThickness = new Tag("0018", "0050", "Slice Thickness");
        Dictionary.KVP = new Tag("0018", "0060", "KVP");
        Dictionary.Exposure = new Tag("0018", "1152", "Exposure");
        Dictionary.PresentationLUTShape = new Tag("2050", "0020", "Presentation LUT Shape");
        Dictionary.RadiationMachineName = new Tag("3002", "0020", "Radiation Machine Name");
        Dictionary.OverlayDataTags = _.map(_.range(Overlay.MAX), function (num) {
            return new Dicom.Tag((0x6000 + (num * 2)).toString(16), "3000", "Overlay Data");
        });
        return Dictionary;
    })();
    Dicom.Dictionary = Dictionary;

    /**
    * Enhanced (multi-frame) DICOM sequences
    */
    var Multiframe = (function () {
        function Multiframe() {
        }
        Multiframe.SharedFunctionalGroupSeq = new Tag("5200", "9229", "Shared Functional Group Sequence");
        Multiframe.PerFrameFunctionalGroupSeq = new Tag("5200", "9230", "Per-frame Functional Group Sequence");
        Multiframe.PixelValueTransSeq = new Tag("0028", "9145", "Pixel Value Transformation Sequence");
        Multiframe.FrameVOILUTSeq = new Tag("0028", "9132", "Frame VOI LUT Sequence");
        Multiframe.PlaneOrientationSeq = new Tag("0020", "9116", "Plane Orientation Sequence");
        Multiframe.PlanePositionSeq = new Tag("0020", "9113", "Plane Position Sequence");
        Multiframe.PixelMeasuresSeq = new Tag("0028", "9110", "Pixel Measures Sequence");
        return Multiframe;
    })();
    Dicom.Multiframe = Multiframe;

    /**
    * A map of tags by group and element number
    */
    Dicom.tagsByGroupAndElement = Maps.preMap(Maps.fromObject((function () {
        var o = {};

        for (var prop in Dictionary) {
            if (Dictionary.hasOwnProperty(prop)) {
                var tag = Dictionary[prop];

                o[tag.toString()] = tag.name;
            }
        }

        return o;
    })()));

    function hexToDec(tag) {
        var tags = tag.replace(/\(|\)/g, '').split(/,|-/);
        return _.map(tags, function (item) {
            return parseInt('0x' + item);
        });
    }

    /**
    *
    * @param currentTags - structured metadata via /json
    * @param {string[]} tags - list of nested tags, e.g., (1234,1234) or a sequence with item index (1234,1234)[0]
    * @param indexDefault
    * @returns {any}
    */
    function findTag(currentTags, tags, indexDefault) {
        if (typeof indexDefault === "undefined") { indexDefault = 0; }
        var foundTag;

        _.each(tags, function (tag) {
            var itemIndex = indexDefault;
            if (tag.indexOf("[") != -1) {
                itemIndex = parseInt(tag.split("[")[1].replace(/\D/g, ''));
            }

            var tagDecimal = hexToDec(tag);
            foundTag = _.find(currentTags, function (item) {
                return ((item.group == tagDecimal[0]) && (item.element == tagDecimal[1]));
            });

            if (foundTag && foundTag.items && (foundTag.items.length > itemIndex) && (itemIndex != -1)) {
                currentTags = foundTag.items[itemIndex].tags;
            }
        });

        return foundTag;
    }
    Dicom.findTag = findTag;

    function findDecimalTagSafely(currentTags, tags, indexDefault) {
        if (typeof indexDefault === "undefined") { indexDefault = 0; }
        var tag = Dicom.findTag(currentTags, tags, indexDefault);
        if (tag && tag.value) {
            return parseFloat(tag.value);
        }

        return Number.NaN;
    }
    Dicom.findDecimalTagSafely = findDecimalTagSafely;

    function findDecimalsTagSafely(currentTags, tags, indexDefault) {
        if (typeof indexDefault === "undefined") { indexDefault = 0; }
        var tag = Dicom.findTag(currentTags, tags, indexDefault);
        if (tag && tag.value) {
            return Parsers.parseDS(tag.value);
        }

        return null;
    }
    Dicom.findDecimalsTagSafely = findDecimalsTagSafely;
})(Dicom || (Dicom = {}));
///<reference path="../typings/underscore/underscore.d.ts" />
/**
* Helper methods for working with DICOM-formatted strings
*/
var Parsers;
(function (Parsers) {
    /**
    * Parse a DS (double string)
    */
    function parseDS(ds) {
        if (ds) {
            return _.map(ds.split(/\\|\//), parseFloat);
        }
        return null;
    }
    Parsers.parseDS = parseDS;

    /**
    * Test whether a DS value is corrupt
    */
    function isCorruptDS(ds) {
        if (ds) {
            return _.any(ds.split(/\\|\//), function (s, index, list) {
                return isNaN(s);
            });
        }
        return false;
    }
    Parsers.isCorruptDS = isCorruptDS;

    /**
    * Parse a DICOM date string
    */
    function parseDicomDate(ds) {
        if (!ds) {
            return null;
        }

        ds = $.trim(ds);

        if (ds.length === 8 && !isNaN(ds)) {
            var year = ds.substr(0, 4);
            var month = ds.substr(4, 2);
            var day = ds.substr(6, 2);

            if (!isNaN(year) && !isNaN(month) && !isNaN(day)) {
                return new Date(parseInt(year, 10), parseInt(month, 10) - 1, parseInt(day, 10), 0, 0, 0, 0);
            }
        }

        return null;
    }
    Parsers.parseDicomDate = parseDicomDate;

    /**
    * Parse a DICOM time string
    */
    function parseDicomTime(ts) {
        if (!ts) {
            return null;
        }

        ts = $.trim(ts);

        var date = null;

        if (ts.length >= 2 && !isNaN(ts)) {
            var hourS = ts.substr(0, 2);
            var hour = 0;
            var minute = 0;
            var second = 0;

            if (!isNaN(hourS)) {
                hour = parseInt(hourS, 10);

                if (ts.length >= 4) {
                    var minuteS = ts.substr(2, 2);

                    if (!isNaN(minuteS)) {
                        minute = parseInt(minuteS, 10);

                        if (ts.length >= 6) {
                            var secondS = ts.substring(4, ts.length);

                            if (!isNaN(secondS)) {
                                second = parseFloat(secondS);
                            }
                        }
                    }
                }
            }

            var wholeSeconds = Math.floor(second);
            var millis = Math.round((second - wholeSeconds) * 1000);

            return new Date(0, 0, 0, hour, minute, wholeSeconds, millis);
        }

        return null;
    }
    Parsers.parseDicomTime = parseDicomTime;

    /**
    * Split a set of strings into an array
    */
    function parseCS(ss) {
        if (ss) {
            return ss.split(/\\|\//);
        }
        return [];
    }
    Parsers.parseCS = parseCS;

    /**
    * Fixes bit stored when it's stored as a bit mask.
    * @param {number} val
    * @returns {number}
    */
    function fixBitCountField(val) {
        if (val > 16) {
            switch (val) {
                case 0xffff:
                    val = 16;
                    break;
                case 0x0fff:
                    val = 12;
                    break;
                case 0x08ff:
                    val = 10;
                    break;
                case 0x00ff:
                    val = 8;
                    break;
            }
        }

        return val;
    }
    Parsers.fixBitCountField = fixBitCountField;
})(Parsers || (Parsers = {}));
///<reference path='../classes/Types.ts' />
///<reference path='../libs/Routes.ts' />
///<reference path='../models/StudySchema.ts' />
///<reference path='../models/Study.ts' />
///<reference path='Observable.ts' />
///<reference path='Map.d.ts' />
///<reference path='Array.ts' />
///<reference path='DicomDictionary.ts' />
///<reference path='Parsers.ts' />
///<reference path='../Main.ts' />
/**
* Observables for working with study metadata
*/
var Study;
(function (Study) {
    /**
    * Make all necessary AJAX calls to assemble the parts of a study object required for initial viewing.
    *
    * Loads top-level metadata and instance attributes for the first image in every series.
    */
    function loadStudy(sessionId, studyStorage, queryObject, keyImagesOnly, permissions, viewerSettings, settings) {
        var accountSettings = settings ? Observable.ret(settings) : Observable.map(Observable.catchError(Services.getAccountSettings(sessionId, queryObject), function () {
            return {};
        }), function (settings) {
            return Main.Main.applySettingsOverrides(settings);
        });

        return Observable.bind(accountSettings, function (accountSettings) {
            var useServices = accountSettings.cache == 1;
            var accelerated = accountSettings.is_accelerated == 1;

            var schema = V3Storage.getStudySchema(sessionId, studyStorage, queryObject, useServices);
            var phi = V3Storage.getStudyPhi(sessionId, studyStorage, queryObject);
            var tags = V3Storage.getStudyTag(sessionId, studyStorage, queryObject);
            var keyImages = permissions.keyimage_view !== 0 ? Services.listKeyImages(sessionId, queryObject) : Observable.ret({ images: [] });

            var loadInParallel = Observable.ap(Observable.ap(Observable.ap(Observable.ap(Observable.ret(function (s) {
                return function (p) {
                    return function (t) {
                        return function (k) {
                            return {
                                schema: s,
                                phi: p,
                                tags: t,
                                keyImages: k
                            };
                        };
                    };
                };
            }), schema), phi), tags), keyImages);

            return Observable.bind(loadInParallel, function (o) {
                return Study.convertStudySchema(sessionId, studyStorage, queryObject, o.schema, o.phi, o.tags, o.keyImages, keyImagesOnly, useServices, accelerated, viewerSettings, accountSettings);
            });
        });
    }
    Study.loadStudy = loadStudy;

    /**
    * Convert a list of tag value pairs into a typed dictionary
    */
    function tagsToMap(tags) {
        var dict = tags.toMap(function (t) {
            return t.tag;
        }, function (t) {
            return t.value;
        });

        return Maps.preMap(dict);
    }

    // order schema series by id and images by rank
    function preprocessSchema(schema) {
        schema.series = _.sortBy(schema.series, function (series) {
            return series.id;
        });

        _.each(schema.series, function (series) {
            series.images = _.sortBy(series.images, function (image) {
                return image.rank;
            });
        });

        return schema;
    }

    /**
    * Convert the results of AJAX calls into a study object.
    *
    * @returns An Observable which completes the computation by loading instance attributes
    */
    function convertStudySchema(sessionId, studyStorage, query, schema, phi, tags, keyImages, keyImagesOnly, useServices, accelerated, viewerSettings, accountSettings) {
        var _this = this;
        var showPDFs = (viewerSettings != null && viewerSettings.showPDFs != null && viewerSettings.showPDFs == true) ? true : false;
        var singleInstanceSeries = (accountSettings != null && accountSettings.viewer_single_instance_series > 0) ? true : false;

        schema = preprocessSchema(schema);

        var newStudy = new Models.Study();

        newStudy.studyAttributes = new Models.StudyAttributes();
        newStudy.studyAttributes.queryObject = query;
        newStudy.studyAttributes.studyStorage = studyStorage;
        newStudy.studyAttributes.accelerated = accelerated;

        var phiDict = tagsToMap(phi.attr);
        var typedDict = tagsToMap(tags.tags);

        newStudy.studyAttributes.patientId = new Classes.PatientID(phiDict.lookup(Dicom.Dictionary.PatientID));
        newStudy.studyAttributes.patientName = phiDict.lookup(Dicom.Dictionary.PatientName);
        newStudy.studyAttributes.patientBirthDate = Parsers.parseDicomDate(phiDict.lookup(Dicom.Dictionary.PatientBirthDate));
        newStudy.studyAttributes.patientSex = phiDict.lookup(Dicom.Dictionary.PatientSex);
        newStudy.studyAttributes.referringPhysicianName = phiDict.lookup(Dicom.Dictionary.ReferringPhysicianName);
        newStudy.studyAttributes.accessionNumber = new Classes.AccessionNumber(phiDict.lookup(Dicom.Dictionary.AccessionNumber));
        newStudy.studyAttributes.studyDescription = phiDict.lookup(Dicom.Dictionary.StudyDescription);
        newStudy.studyAttributes.studyCreateDate = Parsers.parseDicomDate(phiDict.lookup(Dicom.Dictionary.StudyCreateDate) || typedDict.lookup(Dicom.Dictionary.StudyCreateDate));
        newStudy.studyAttributes.studyCreateTime = Parsers.parseDicomTime(typedDict.lookup(Dicom.Dictionary.StudyCreateTime));
        newStudy.studyAttributes.uuid = new Classes.StudyUUID(studyStorage.uuid);

        newStudy.studyAttributes.customfields = studyStorage.customfields;

        newStudy.series = [];

        var futures = [];

        // if modality is missing, set as OT
        _.each(schema.series, function (series) {
            if (series.modality == null && series.images.length) {
                if (isImageSOPClass(series.images[0].sop_class)) {
                    series.modality = "OT";
                }
            }
        });

        var imageSeries = _.filter(schema.series, function (series) {
            return isImageModality(series.modality) && _.any(series.images, function (image) {
                return isImageSOPClass(image.sop_class) || (isPDFSOPClass(image.sop_class) && showPDFs) || isVideoSOPClass(image.sop_class);
            });
        });

        imageSeries = filterSeriesSchema(imageSeries, viewerSettings);

        var standAloneSeries = [];

        // Pull each video or PDF out and put it in a stand-alone series
        _.each(imageSeries, function (series) {
            var filteredImages = _.filter(series.images, function (image) {
                if ((isPDFSOPClass(image.sop_class) && showPDFs) || isVideoSOPClass(image.sop_class) || singleInstanceSeries) {
                    var newSeries = {
                        id: series.id,
                        images: [image],
                        modality: series.modality,
                        series_uid: series.series_uid
                    };

                    standAloneSeries.push(newSeries);

                    return false;
                } else {
                    return true;
                }
            });

            // Original series should only include images, not videos or PDFs
            series.images = filteredImages;
        });

        // Any series no longer containing images should be removed
        imageSeries = _.filter(_.flatten([imageSeries, standAloneSeries]), function (series) {
            return series.images.length > 0;
        });

        _.each(imageSeries, function (series) {
            var newSeries = new Models.Series();

            // Each Series or SeriesLike needs a unique id for the viewer to keep track of what is displayed in each
            // viewing pane and what data a thumbnail refers to. This is needed because individual series are now being
            // split into multiple thumbnails.
            newSeries.uuid = _this.generateUUID();

            newSeries.id = series.id;
            newSeries.studyAttributes = newStudy.studyAttributes;

            newSeries.seriesAttributes = new Models.SeriesAttributes();
            newSeries.seriesAttributes.seriesUid = new Classes.SeriesUid(series.series_uid);
            newSeries.seriesAttributes.modality = series.modality ? series.modality : "";
            newSeries.seriesAttributes.instanceCount = series.images.length;

            newSeries.seriesAttributes.imageAttributesLoaded = new Subjects.ObservableValue(0);
            newSeries.seriesAttributes.imageDataLoaded = new Subjects.ObservableValue(0);
            newSeries.seriesAttributes.description = series.description;

            if (_.any(series.images, function (image) {
                return isPDFSOPClass(image.sop_class);
            })) {
                newSeries.seriesAttributes.documentType = { type: 0 /* PDF */, thumbnailText: 'PDF', thumbnailIcon: null };
            }

            if (_.any(series.images, function (image) {
                return isVideoSOPClass(image.sop_class);
            })) {
                newSeries.seriesAttributes.documentType = { type: 1 /* Video */, thumbnailText: null, thumbnailIcon: 'fa-film' };
            }

            newSeries.instances = [];

            var images = _.filter(series.images, function (image) {
                return isImageSOPClass(image.sop_class) || (isPDFSOPClass(image.sop_class) && showPDFs) || isVideoSOPClass(image.sop_class);
            });

            _.each(images, function (image, index, list) {
                var newInstance = new Models.Instance();

                newInstance.frameNumber = new Classes.FrameNumber(0);

                newInstance.seriesAttributes = newSeries.seriesAttributes;
                newInstance.studyAttributes = newStudy.studyAttributes;

                newInstance.instanceAttributes = new Models.InstanceAttributes();

                newInstance.instanceAttributes.instanceIndex = index;

                newInstance.instanceAttributes.attributesLoaded = false;
                newInstance.instanceAttributes.attributesDownloadState = 0 /* None */;

                newInstance.id = new Classes.InstanceUid(image.id);
                newInstance.instanceAttributes.rank = image.rank;
                newInstance.instanceAttributes.frameCount = image.frame_count;
                newInstance.instanceAttributes.sopClass = image.sop_class;
                newInstance.instanceAttributes.version = new Classes.ImageVersion(image.version);
                newInstance.instanceAttributes.instanceNumber = index;
                newInstance.instanceAttributes.isKeyImage = new Subjects.ObservableValue(false);

                newSeries.instances.push(newInstance);
            });

            newStudy.series.push(newSeries);
        });

        if (keyImagesOnly && _.any(keyImages.images)) {
            newStudy.series = _.filter(newStudy.series, function (series) {
                return _.any(keyImages.images, function (k) {
                    return k.series_uid === series.seriesAttributes.seriesUid.value;
                });
            });

            _.each(newStudy.series, function (series) {
                series.instances = _.filter(series.instances, function (instance) {
                    return _.any(keyImages.images, function (k) {
                        return k.instance_uid === instance.id.value && k.version === instance.instanceAttributes.version.value;
                    });
                });
            });
        }

        if (accelerated) {
            futures.push(Study.loadStudyAttributes(sessionId, newStudy));
        } else {
            _.each(newStudy.series, function (series) {
                var isMultiframe = _.any(series.instances, function (instance) {
                    return instance.instanceAttributes.frameCount > 1;
                });

                _.each(series.instances, function (instance, index, list) {
                    if (accountSettings.viewer_multiframe_split_method == 2) {
                        futures.push(Study.loadImageAttributes(sessionId, instance));
                    } else if (isMultiframe || index == 0 || Multiframe.shouldSplitInstances(series.seriesAttributes.modality)) {
                        if (!Multiframe.shouldSplitInstances(series.seriesAttributes.modality)) {
                            futures.push(Study.loadImageAttributes(sessionId, instance));
                        } else {
                            futures.push(Observable.ret({}));
                        }
                    }
                });
            });
        }

        // load /json metadata, if necessary
        var seriesFilterNeedsJson = false;
        var hangingProtocolNeedsJson = false;

        if (viewerSettings && newStudy.series.length) {
            seriesFilterNeedsJson = filterSeriesNeedsJson(newStudy.series, viewerSettings);
            hangingProtocolNeedsJson = HangingProtocols.needsJSON(HangingProtocols.select(newStudy, viewerSettings));
        }

        _.each(newStudy.series, function (series) {
            var needsSeriesJSON = false;
            var needsImageJSON = false;

            if (viewerSettings) {
                needsSeriesJSON = TextAnnotations.usesCustomSeriesTag(series.seriesAttributes.modality, viewerSettings) || seriesFilterNeedsJson || hangingProtocolNeedsJson;
                needsImageJSON = TextAnnotations.usesCustomImageTag(series.seriesAttributes.modality, viewerSettings);
            }

            var isMultiframe = _.any(series.instances, function (instance) {
                return instance.instanceAttributes.frameCount > 1;
            });

            _.each(series.instances, function (instance, index, list) {
                if (isMultiframe) {
                    futures.push(Study.loadImageJSON(sessionId, instance, series, false)); // multiframe split into series downstream
                } else if (needsImageJSON) {
                    futures.push(Study.loadImageJSON(sessionId, instance, series, needsSeriesJSON));
                } else if ((needsSeriesJSON && index == 0)) {
                    futures.push(Study.loadImageJSON(sessionId, instance, series, true));
                }
            });
        });

        _.each(schema.attachments || [], function (attachment) {
            var newAttachment = new Models.Attachment();

            newAttachment.id = new Classes.AttachmentID(attachment.id);
            newAttachment.stored = attachment.stored;
            newAttachment.mime = attachment.mime;
            newAttachment.version = new Classes.ImageVersion(attachment.version);
            newAttachment.filename = attachment.filename;
            newAttachment.phiNamespace = new Classes.PhiNamespace(attachment.phi_namespace);

            newStudy.attachments.push(newAttachment);
        });

        _.each(studyStorage.hl7 || [], function (hl7) {
            var newReport = new Models.Report();

            newReport.id = new Classes.ReportID(hl7.uuid);

            newStudy.reports.push(newReport);
        });

        _.each(studyStorage.routes || [], function (route) {
            var newAction = new Models.StudyAction();

            newAction.name = route.name;
            newAction.id = new Classes.StudyActionID(route.uuid);
            newAction.requiresEmailAddress = route.capture_email == 1;

            newStudy.actions.push(newAction);
        });

        _.each(studyStorage.meetings || [], function (meeting) {
            var newMeeting = new Classes.StudyMeeting();

            newMeeting.id = new Classes.MeetingId(meeting.uuid);
            newMeeting.name = meeting.name;
            newMeeting.host = meeting.user_name;

            newStudy.meetings.push(newMeeting);
        });

        _.each(keyImages.images, function (keyImage) {
            var series = _.find(newStudy.series, function (s) {
                return s.seriesAttributes.seriesUid.value === keyImage.series_uid;
            });

            if (series) {
                var instance = _.find(series.instances, function (i) {
                    return i.id.value === keyImage.instance_uid;
                });

                if (instance) {
                    instance.instanceAttributes.isKeyImage.write(true);
                    instance.instanceAttributes.keyImageId = new Classes.KeyImageId(keyImage.uuid);
                }
            }
        });

        var presentationStateSeries = _.filter(schema.series, function (series) {
            if (series.images.length) {
                return isPresentationState(series.modality, series.images[0].sop_class);
            } else {
                return isPresentationState(series.modality);
            }
        });
        var allPresentationStates = _.flatten(_.map(presentationStateSeries, function (series) {
            return series.images;
        }));

        newStudy.studyAttributes.presentationStates = allPresentationStates;

        var structuredReportSeries = _.filter(schema.series, function (series) {
            return isStructuredReport(series.modality);
        });
        var allStructuredReports = _.flatten(_.map(structuredReportSeries, function (series) {
            return _.map(series.images, function (instance) {
                var report = new Models.StructuredReport();
                report.id = new Classes.InstanceUid(instance.id);
                report.version = new Classes.ImageVersion(instance.version);
                return report;
            });
        }));

        newStudy.structuredReports = allStructuredReports;

        return Observable.map(Observable.sequenceA(futures), function (_) {
            return newStudy;
        });
    }
    Study.convertStudySchema = convertStudySchema;

    /**
    * Update study from /get/study call
    */
    function updateStudyFromStorage(study, studyStorage) {
        study.studyAttributes.customfields = studyStorage.customfields;
        return study;
    }
    Study.updateStudyFromStorage = updateStudyFromStorage;

    /**
    * Utility function to create UUID
    */
    function generateUUID() {
        var d = new Date().getTime();
        var uuid = 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function (c) {
            var r = (d + Math.random() * 16) % 16 | 0;
            d = Math.floor(d / 16);
            return (c == 'x' ? r : (r & 0x3 | 0x8)).toString(16);
        });
        return uuid;
    }
    Study.generateUUID = generateUUID;

    /**
    * Create a "stack series" for a study
    */
    function createStackSeries(study) {
        var instances = _.chain(study.series).map(function (s) {
            return s.instances;
        }).flatten().value();

        var seriesAttributes = {
            seriesUid: new Classes.SeriesUid(generateUUID()),
            instanceCount: instances.length,
            modality: _.chain(study.series).map(function (s) {
                return s.seriesAttributes.modality;
            }).uniq().value().join(", "),
            imageAttributesLoaded: new Subjects.ObservableValue(0),
            imageDataLoaded: new Subjects.ObservableValue(0),
            documentType: null,
            description: null,
            json: null,
            multiframe: false
        };

        return {
            uuid: this.generateUUID(),
            seriesAttributes: seriesAttributes,
            studyAttributes: study.studyAttributes,
            instances: instances,
            loadedStatus: 0 /* None */
        };
    }
    Study.createStackSeries = createStackSeries;

    function orderSeries(allSeries, desc) {
        var sorted = _.sortBy(allSeries, function (series) {
            if (series.studyAttributes.studyCreateDate) {
                return series.studyAttributes.studyCreateDate.getTime();
            } else {
                return Number.NEGATIVE_INFINITY;
            }
        });

        if (desc) {
            sorted = sorted.reverse();
        }

        return sorted;
    }
    Study.orderSeries = orderSeries;

    function createKeyImageSeries(study, stackKeyImageSeries) {
        var _this = this;
        var keyImageSeries = [];

        if (stackKeyImageSeries) {
            var series = this.createStackSeries(study);
            series.instances = _.filter(series.instances, function (i) {
                return i.instanceAttributes.isKeyImage.read();
            });
            series.seriesAttributes.instanceCount = series.instances.length;
            return (series.instances.length > 0) ? [series] : [];
        } else {
            keyImageSeries = _.map(study.series, function (series) {
                var seriesAttributes = {
                    seriesUid: new Classes.SeriesUid(generateUUID()),
                    instanceCount: 0,
                    modality: _.chain(study.series).map(function (s) {
                        return s.seriesAttributes.modality;
                    }).uniq().value().join(", "),
                    imageAttributesLoaded: new Subjects.ObservableValue(0),
                    imageDataLoaded: new Subjects.ObservableValue(0),
                    documentType: null,
                    description: null,
                    json: null,
                    multiframe: false
                };

                var instances = _.filter(series.instances, function (i) {
                    return i.instanceAttributes.isKeyImage.read();
                });
                seriesAttributes.instanceCount = instances.length;

                var series = {
                    uuid: _this.generateUUID(),
                    seriesAttributes: seriesAttributes,
                    studyAttributes: study.studyAttributes,
                    instances: instances,
                    loadedStatus: 0 /* None */
                };

                return (series.instances.length > 0) ? series : null;
            });
        }

        return _.compact(keyImageSeries);
    }
    Study.createKeyImageSeries = createKeyImageSeries;

    function splitSeries(series) {
        var _this = this;
        var splitSeries = [];

        _.each(series.instances, function (i) {
            var seriesAttributes = {
                seriesUid: new Classes.SeriesUid(generateUUID()),
                instanceCount: 0,
                modality: series.seriesAttributes.modality,
                imageAttributesLoaded: new Subjects.ObservableValue(0),
                imageDataLoaded: new Subjects.ObservableValue(0),
                documentType: null,
                description: null,
                json: null,
                multiframe: false
            };

            var split = {
                uuid: _this.generateUUID(),
                seriesAttributes: seriesAttributes,
                studyAttributes: series.studyAttributes,
                instances: [i],
                parent: series,
                loadedStatus: 0 /* None */
            };

            seriesAttributes.instanceCount = 1;
            splitSeries.push(split);
        });

        return splitSeries;
    }
    Study.splitSeries = splitSeries;

    /**
    * Load GSPS data and apply it to a study
    */
    function loadAndApplyAllGSPSData(sessionId, study, allowSR) {
        var loadAllPresentationStates = Observable.sequenceA(_.map(study.studyAttributes.presentationStates, function (presentationState) {
            return V3Storage.getGSPS(sessionId, study.studyAttributes.studyStorage, study.studyAttributes.queryObject, new Classes.InstanceUid(presentationState.id), new Classes.ImageVersion(presentationState.version));
        }));

        if (allowSR) {
            var loadAllCADSR = Observable.sequenceA(_.map(study.structuredReports, function (sr) {
                return V3Storage.getCADSR(sessionId, study.studyAttributes.studyStorage, study.studyAttributes.queryObject, new Classes.InstanceUid(sr.id.value), new Classes.ImageVersion(sr.version.value));
            }));

            return Observable.zip(loadAllPresentationStates, loadAllCADSR, function (presentationStates, cadsrs) {
                GSPS.apply(presentationStates, study);
                CADSR.apply(cadsrs, study);
                return presentationStates;
            });
        } else {
            return Observable.invoke(loadAllPresentationStates, function (presentationStates) {
                GSPS.apply(presentationStates, study);
            });
        }
    }
    Study.loadAndApplyAllGSPSData = loadAndApplyAllGSPSData;

    // Changes to this should also be made to Storage study download service in DicomService::canTranscodeImage()
    function isImageSOPClass(sopClass) {
        switch (sopClass) {
            case "1.2.840.10008.5.1.4.1.1.104.1":
            case "1.2.840.10008.5.1.4.1.1.66":
            case "1.3.46.670589.11.0.0.12.2":
            case "1.3.46.670589.11.0.0.12.4":
            case "1.2.840.10008.5.1.4.1.1.66.1":
            case "1.2.840.10008.5.1.4.1.1.11.1":
            case "1.2.840.10008.5.1.4.1.1.11.2":
            case "1.2.840.10008.5.1.4.1.1.11.3":
            case "1.2.840.10008.5.1.4.1.1.11.4":
            case "1.2.826.0.1.3680043.2.93.1.0.1":
                return false;
            default:
                return true;
        }
    }
    Study.isImageSOPClass = isImageSOPClass;

    function isPDFSOPClass(sopClass) {
        switch (sopClass) {
            case "1.2.840.10008.5.1.4.1.1.104.1":
                return true;
            default:
                return false;
        }
    }
    Study.isPDFSOPClass = isPDFSOPClass;

    function isVideoSOPClass(sopClass) {
        switch (sopClass) {
            case "1.2.840.10008.5.1.4.1.1.77.1.4.1":
            case "1.2.840.10008.5.1.4.1.1.77.1.1.1":
                return true;
            default:
                return false;
        }
    }
    Study.isVideoSOPClass = isVideoSOPClass;

    // Changes to this should also be made to Storage study download service in DicomService::canTranscodeImage()
    function isImageModality(modality) {
        if (!modality) {
            return false;
        }

        switch (modality.toUpperCase()) {
            case "CC":
            case "KO":
            case "PR":
            case "RIS":
            case "RTSTRUCT":
            case "RTPLAN":
            case "RTRECORD":
            case "SR":
            case "REG":
                return false;
        }

        return true;
    }
    Study.isImageModality = isImageModality;

    function isPresentationState(modality, sopClass) {
        if (typeof sopClass === "undefined") { sopClass = ""; }
        return modality === "PR" || sopClass.indexOf("1.2.840.10008.5.1.4.1.1.11.") !== -1;
    }
    Study.isPresentationState = isPresentationState;

    function isStructuredReport(modality) {
        return modality === "SR";
    }
    Study.isStructuredReport = isStructuredReport;

    function isKeyObjectSelection(modality) {
        return modality === "KO";
    }
    Study.isKeyObjectSelection = isKeyObjectSelection;

    function isPDFSeries(series) {
        return (series && series.instances && series.instances.length > 0 && series.instances[0].instanceAttributes && series.instances[0].instanceAttributes.sopClass && series.instances[0].instanceAttributes.sopClass == '1.2.840.10008.5.1.4.1.1.104.1');
    }
    Study.isPDFSeries = isPDFSeries;

    function isImagedORUSeries(series) {
        return (series && series.instances && series.instances.length > 0 && series.instances[0].instanceAttributes && series.instances[0].instanceAttributes.sopClass && series.instances[0].instanceAttributes.sopClass == '1.2.840.10008.5.1.4.1.1.7' && series.instances[0].instanceAttributes.seriesDescription != null && (series.instances[0].instanceAttributes.seriesDescription == 'report' || series.instances[0].instanceAttributes.seriesDescription.indexOf('pdf') > -1 || series.instances[0].instanceAttributes.seriesDescription.indexOf('doc') > -1) && series.seriesAttributes.seriesUid.value.indexOf('1.3.6.1.4.1.34692') == 0);
    }
    Study.isImagedORUSeries = isImagedORUSeries;

    function loadStudyAttributes(sessionId, study) {
        var getStudyAttributes = V3Storage.getStudyAttributes(sessionId, study.studyAttributes.studyStorage, study.studyAttributes.queryObject);

        return Observable.forget(Observable.invoke(getStudyAttributes, function (attr) {
            var allImages = _.flatten(_.map(study.series, function (s) {
                return s.instances;
            }));

            _.each(attr.images, function (attribs) {
                var instance = _.find(allImages, function (image) {
                    return image.id.value === attribs.value;
                });

                if (instance) {
                    updateInstanceAttributes(attribs, instance);
                }
            });
        }));
    }
    Study.loadStudyAttributes = loadStudyAttributes;

    function loadImageAttributes(sessionId, instance, series) {
        // If this attribute is already loaded return immediately
        if (instance.instanceAttributes.attributesDownloadState == 3 /* Success */) {
            return Observable.ret({});
        }

        // Observable function - returns the image attributes from storage (AJAX call)
        var getImageAttributes = V3Storage.getImageAttributes(sessionId, instance.studyAttributes.studyStorage, instance.studyAttributes.queryObject, instance.id, instance.instanceAttributes.version);

        // Set the image attributes to the started state
        var setStartedState = function () {
            instance.instanceAttributes.attributesDownloadState = 2 /* Started */;
        };

        // Observable function - first set the state, then get the attributes from storage
        var processRequest = Observable.invokeFirst(getImageAttributes, setStartedState);

        // Only download the attributes if the download is not already started (or done)
        var shouldDownload = function (i) {
            var needToDownload = (i.instanceAttributes.attributesDownloadState == null || i.instanceAttributes.attributesDownloadState == 0 /* None */ || i.instanceAttributes.attributesDownloadState == 4 /* Failed */);

            // If this attribute file needs to download then mark it as being queued
            if (needToDownload) {
                i.instanceAttributes.attributesDownloadState = 1 /* Queued */;
            }
            return needToDownload;
        };

        return Observable.ifThenElse(function () {
            return shouldDownload(instance);
        }, Observable.invoke(processRequest, function (attribs) {
            updateInstanceAttributes(attribs, instance);
            if (series) {
                series.attributesLoaded(instance);
            }
        }), Observable.ret({}));
    }
    Study.loadImageAttributes = loadImageAttributes;

    function loadImageJSON(sessionId, instance, series, useSeries) {
        // If this attribute is already loaded return immediately
        if (instance.instanceAttributes.jsonDownloadState == 3 /* Success */) {
            return Observable.ret({});
        }

        // Observable function - returns the image attributes from storage (AJAX call)
        var getImageJSON = V3Storage.getImageJSON(sessionId, instance.studyAttributes.studyStorage, instance.studyAttributes.queryObject, instance.id, instance.instanceAttributes.version);

        // Set the image attributes to the started state
        var setStartedState = function () {
            instance.instanceAttributes.jsonDownloadState = 2 /* Started */;
        };

        // Observable function - first set the state, then get the attributes from storage
        var processRequest = Observable.invokeFirst(getImageJSON, setStartedState);

        // Only download the attributes if the download is not already started (or done)
        var shouldDownload = function (i) {
            var needToDownload = (i.instanceAttributes.jsonDownloadState == null || i.instanceAttributes.jsonDownloadState == 0 /* None */ || i.instanceAttributes.jsonDownloadState == 4 /* Failed */);

            // If this attribute file needs to download then mark it as being queued
            if (needToDownload) {
                i.instanceAttributes.jsonDownloadState = 1 /* Queued */;
            }
            return needToDownload;
        };

        return Observable.ifThenElse(function () {
            return shouldDownload(instance);
        }, Observable.invoke(processRequest, function (json) {
            instance.instanceAttributes.json = json;

            if (useSeries) {
                instance.seriesAttributes.json = json;
            }
        }), Observable.ret({}));
    }
    Study.loadImageJSON = loadImageJSON;

    function updateInstanceAttributes(attribs, instance) {
        var instanceDict = attribs.attr.toMap(function (t) {
            return t.tag;
        }, function (t) {
            return t.value;
        });
        var typedInstanceDict = Maps.preMap(instanceDict);

        instance.instanceAttributes.attributesLoaded = true;
        instance.instanceAttributes.attributesDownloadState = 3 /* Success */;

        instance.instanceAttributes.seriesDescription = typedInstanceDict.lookup(Dicom.Dictionary.SeriesDescription);

        if (typedInstanceDict.lookup(Dicom.Dictionary.PixelSpacing)) {
            instance.instanceAttributes.pixelSpacing = Parsers.parseDS(typedInstanceDict.lookup(Dicom.Dictionary.PixelSpacing));
            instance.instanceAttributes.pixelSpacingMeaning = 0 /* PatientGeometry */;
        } else if (typedInstanceDict.lookup(Dicom.Dictionary.ImagerPixelSpacing)) {
            instance.instanceAttributes.pixelSpacing = Parsers.parseDS(typedInstanceDict.lookup(Dicom.Dictionary.ImagerPixelSpacing));
            instance.instanceAttributes.pixelSpacingMeaning = 1 /* AtImagingPlate */;
        }

        instance.instanceAttributes.imagePositionPatient = Parsers.parseDS(typedInstanceDict.lookup(Dicom.Dictionary.ImagePositionPatient));
        instance.instanceAttributes.imageOrientationPatient = Parsers.parseDS(typedInstanceDict.lookup(Dicom.Dictionary.ImageOrientationPatient));
        instance.instanceAttributes.rows = parseInt(typedInstanceDict.lookup(Dicom.Dictionary.Rows));
        instance.instanceAttributes.columns = parseInt(typedInstanceDict.lookup(Dicom.Dictionary.Columns));
        instance.instanceAttributes.bitsStored = Parsers.fixBitCountField(parseInt(typedInstanceDict.lookup(Dicom.Dictionary.BitsStored)));
        instance.instanceAttributes.windowWidth = Parsers.parseDS(typedInstanceDict.lookup(Dicom.Dictionary.WindowWidth));
        instance.instanceAttributes.windowCenter = Parsers.parseDS(typedInstanceDict.lookup(Dicom.Dictionary.WindowCenter));
        instance.instanceAttributes.windowLevelIsCorrupt = Parsers.isCorruptDS(typedInstanceDict.lookup(Dicom.Dictionary.WindowWidth)) || Parsers.isCorruptDS(typedInstanceDict.lookup(Dicom.Dictionary.WindowCenter));

        instance.instanceAttributes.rescaleSlope = parseFloat(Maps.lookupDef(typedInstanceDict, Dicom.Dictionary.RescaleSlope, "1"));
        instance.instanceAttributes.rescaleIntercept = parseFloat(Maps.lookupDef(typedInstanceDict, Dicom.Dictionary.RescaleIntercept, "0"));

        instance.instanceAttributes.signed = typedInstanceDict.lookup(Dicom.Dictionary.PixelRepresentation) === "1";
        instance.instanceAttributes.presentationLUTShape = typedInstanceDict.lookup(Dicom.Dictionary.PresentationLUTShape);
        instance.instanceAttributes.photometricInterpretation = typedInstanceDict.lookup(Dicom.Dictionary.PhotometricInterpretation);

        instance.instanceAttributes.sliceThickness = parseFloat(typedInstanceDict.lookup(Dicom.Dictionary.SliceThickness));
        instance.instanceAttributes.kvp = parseFloat(typedInstanceDict.lookup(Dicom.Dictionary.KVP));
        instance.instanceAttributes.exposure = parseFloat(typedInstanceDict.lookup(Dicom.Dictionary.Exposure));
        instance.instanceAttributes.radiationMachineName = typedInstanceDict.lookup(Dicom.Dictionary.RadiationMachineName);

        instance.instanceAttributes.imageType = Parsers.parseCS(typedInstanceDict.lookup(Dicom.Dictionary.ImageType));
        instance.instanceAttributes.imageLaterality = typedInstanceDict.lookup(Dicom.Dictionary.ImageLaterality);
        instance.instanceAttributes.voiLutFunction = typedInstanceDict.lookup(Dicom.Dictionary.VOILUTFunction);
        instance.instanceAttributes.frameTime = Parsers.parseDS(typedInstanceDict.lookup(Dicom.Dictionary.FrameTime));

        _.each(Dicom.Dictionary.OverlayDataTags, function (tag, index) {
            instance.instanceAttributes.containsOverlayDataPlane[index] = typedInstanceDict.lookup(tag) === "1";
        });
        instance.instanceAttributes.containsOverlayData = _.find(instance.instanceAttributes.containsOverlayDataPlane, function (item) {
            return item;
        });

        instance.instanceAttributes.imageLaterality = typedInstanceDict.lookup(Dicom.Dictionary.ImageLaterality) || '';
        instance.instanceAttributes.viewPosition = typedInstanceDict.lookup(Dicom.Dictionary.ViewPosition) || '';
        instance.instanceAttributes.stationName = typedInstanceDict.lookup(Dicom.Dictionary.StationName) || '';
        instance.instanceAttributes.operatorsName = typedInstanceDict.lookup(Dicom.Dictionary.OperatorsName) || '';
        instance.instanceAttributes.institutionName = typedInstanceDict.lookup(Dicom.Dictionary.InstitutionName) || '';
        instance.instanceAttributes.institutionAddress = typedInstanceDict.lookup(Dicom.Dictionary.InstitutionAddress) || '';
        instance.instanceAttributes.detectorID = typedInstanceDict.lookup(Dicom.Dictionary.DetectorID) || '';

        if (typedInstanceDict.lookup(Dicom.Dictionary.SeriesCreateDate)) {
            instance.instanceAttributes.seriesCreateDate = Parsers.parseDicomDate(typedInstanceDict.lookup(Dicom.Dictionary.SeriesCreateDate));
        }

        if (typedInstanceDict.lookup(Dicom.Dictionary.SeriesCreateTime)) {
            instance.instanceAttributes.seriesCreateTime = Parsers.parseDicomTime(typedInstanceDict.lookup(Dicom.Dictionary.SeriesCreateTime));
        }

        if (instance.seriesAttributes.multiframe && instance.seriesAttributes.json) {
            Multiframe.loadSharedMultiframeTags(instance.seriesAttributes, [instance]);
            Multiframe.loadPerFrameMultiframeTags(instance.seriesAttributes, [instance]);
        }

        Subjects.modify(instance.seriesAttributes.imageAttributesLoaded, function (n) {
            return n + 1;
        });
    }
    Study.updateInstanceAttributes = updateInstanceAttributes;

    // tests whether this study will require a series-level /json call, based on series modality and modality settings
    function filterSeriesNeedsJson(allSeries, viewerSettings) {
        var modalitySettings = (viewerSettings != null && viewerSettings.modalities);
        var needsJson = false;

        if (modalitySettings) {
            _.each(allSeries, function (series) {
                if (series.instances.length) {
                    var modalitySetting = _.find(modalitySettings, function (m) {
                        return m.modality === series.seriesAttributes.modality;
                    });

                    if (modalitySetting && modalitySetting.seriesFilters) {
                        _.each(modalitySetting.seriesFilters, function (filter) {
                            if (filter.type == "tag") {
                                needsJson = true;
                            }
                        });
                    }
                }
            });
        }

        return needsJson;
    }
    Study.filterSeriesNeedsJson = filterSeriesNeedsJson;

    // filter series by series description or DICOM tag based on /attributes and /json calls
    function filterSeries(series, viewerSettings) {
        var modalitySettings = (viewerSettings != null && viewerSettings.modalities);
        var imageSeries = series;

        if (modalitySettings) {
            imageSeries = _.filter(imageSeries, function (series) {
                if (series.instances.length) {
                    var match = false;
                    var modalitySetting = _.find(modalitySettings, function (m) {
                        return m.modality === series.seriesAttributes.modality;
                    });

                    if (modalitySetting && modalitySetting.seriesFilters) {
                        _.each(modalitySetting.seriesFilters, function (filter) {
                            var filterVal = filter.value.toLowerCase();
                            var filterFilter = filter.filter;
                            var compareWith;

                            if (filter.type == "tag") {
                                if (series.instances[0].seriesAttributes.json) {
                                    var currentTags = series.instances[0].seriesAttributes.json.tags;
                                    var tags = filter.option.split('->');
                                    var foundTag = Dicom.findTag(currentTags, tags);

                                    if (foundTag && foundTag.value) {
                                        compareWith = foundTag.value.toLowerCase();
                                    }
                                }
                            } else if (series.instances[0].instanceAttributes.seriesDescription) {
                                compareWith = series.instances[0].instanceAttributes.seriesDescription.toLowerCase();
                            }

                            if (compareWith) {
                                if (filterFilter == "equals") {
                                    if (filterVal == compareWith) {
                                        match = true;
                                    }
                                } else {
                                    var contains = (compareWith.indexOf(filterVal) != -1);
                                    var containsFilter = (filterFilter != "not");
                                    if (contains == containsFilter) {
                                        match = true;
                                    }
                                }
                            }
                        });
                    }
                }

                return !match;
            });
        }

        return imageSeries;
    }
    Study.filterSeries = filterSeries;

    // filter series by description based on /schema call
    function filterSeriesSchema(series, viewerSettings) {
        var imageSeries = series;
        var modalitySettings = (viewerSettings != null && viewerSettings.modalities);

        if (modalitySettings) {
            imageSeries = _.filter(imageSeries, function (series) {
                if (series.description) {
                    var match = false;
                    var modalitySetting = _.find(modalitySettings, function (m) {
                        return m.modality === series.modality;
                    });

                    if (modalitySetting && modalitySetting.seriesFilters) {
                        _.each(modalitySetting.seriesFilters, function (filter) {
                            if (filter.type == null) {
                                var seriesDes = series.description.toLowerCase();
                                var filterVal = filter.value.toLowerCase();
                                var filterFilter = filter.filter;

                                if (filter.filter == "equals") {
                                    if (filterVal == seriesDes) {
                                        match = true;
                                    }
                                } else {
                                    var contains = (seriesDes.indexOf(filterVal) != -1);
                                    var containsFilter = (filterFilter != "not");
                                    if (contains == containsFilter) {
                                        match = true;
                                    }
                                }
                            }
                        });
                    }
                }

                return !match;
            });
        }

        return imageSeries;
    }
})(Study || (Study = {}));


/**
* Methods not provided on the Date prototype
* @see Date
*/
var Dates;
(function (Dates) {
    Date.prototype.defaultDateFormat = 'MM-DD-YYYY';
    Date.prototype.defaultTimeFormat = 'hh:mm A';

    Date.prototype.toShortDateString = function (format) {
        if (typeof format === "undefined") { format = Date.prototype.defaultDateFormat; }
        var date = this;

        /* Hack around node-webkit issue */
        if (typeof moment === 'undefined') {
            moment = global.moment;
        }

        return moment(date).format(format);
    };

    Date.prototype.toShortTimeString = function (format) {
        if (typeof format === "undefined") { format = Date.prototype.defaultTimeFormat; }
        var date = this;

        /* Hack around node-webkit issue */
        if (typeof moment === 'undefined') {
            moment = global.moment;
        }

        return moment(date).format(format);
    };

    Date.prototype.toShortDateTimeString = function (format, timeZone) {
        if (typeof format === "undefined") { format = Date.prototype.defaultDateFormat; }
        var date = this;

        /* Hack around node-webkit issue */
        if (typeof moment === 'undefined') {
            moment = global.moment;
        }

        format = format + ' ' + Date.prototype.defaultTimeFormat;

        var result = moment(date);

        if (timeZone) {
            // get a moment representing the current time
            var original = moment(date);

            // create a new moment based on the original one
            var shifted = original.clone();

            // change the time zone of the new moment
            shifted.tz(timeZone);

            // shift the moment by the difference in offsets
            shifted.add(original.utcOffset() - shifted.utcOffset(), 'minutes');

            // shift to user's current timezone
            shifted.tz(moment.tz.guess());
            result = shifted;
        }

        return result.format(format);
    };

    Date.prototype.toDicomDate = function () {
        var date = this;
        return date.getFullYear().padStart(4, "0") + (date.getMonth() + 1).padStart(2, "0") + date.getDate().padStart(2, "0");
    };

    Date.prototype.toDicomTime = function () {
        var date = this;
        return date.getHours().padStart(2, "0") + date.getHours().padStart(2, "0") + date.getSeconds().padStart(2, "0");
    };
})(Dates || (Dates = {}));


/**
* Methods not provided on the Number prototype
* @see Number
*/
var Numbers;
(function (Numbers) {
    Number.prototype.padStart = function (length, char) {
        return this.toString().padStart(length, char);
    };

    String.prototype.padStart = function (length, char) {
        var s = this;
        while (s.length < length) {
            s = char + s;
        }
        return s;
    };
})(Numbers || (Numbers = {}));
/**
* Helper methods for handling multiframe data
*/
var Multiframe;
(function (Multiframe) {
    /**
    * Test whether or not a series contains multiframe instances
    */
    function isMultiframe(series) {
        if (series) {
            return _.any(series.instances, function (instance) {
                return instance.instanceAttributes.frameCount > 1;
            });
        }

        return false;
    }
    Multiframe.isMultiframe = isMultiframe;

    /**
    * Determine whether or not a series should be split into instances, based on its
    * modality.
    */
    function shouldSplitInstances(modality) {
        return modality === "MG";
    }
    Multiframe.shouldSplitInstances = shouldSplitInstances;

    /**
    * Split a multiframe series into multiple collections of frames.
    * If the series is not a multiframe series, only the series itself is returned.
    *
    * @see Models.SeriesLike
    */
    function split(series, settings) {
        if (!settings.viewer_multiframe_split_method || settings.viewer_multiframe_split_method == 0) {
            // This method will put each multiframe instance in its own series
            if (isMultiframe(series) || shouldSplitInstances(series.seriesAttributes.modality)) {
                var multiframeInstances = [];

                _.each(series.instances, function (instance) {
                    if (instance.instanceAttributes.frameCount > 1) {
                        multiframeInstances.push(instance);
                    }
                });

                if (multiframeInstances.length) {
                    series.instances = _.difference(series.instances, multiframeInstances);

                    var originalSeries = [];
                    if (series.instances.length) {
                        // clean up instance indices and counts
                        _.each(series.instances, function (instance, index, list) {
                            instance.instanceAttributes.instanceIndex = index;
                            instance.instanceAttributes.instanceNumber = index;
                        });

                        series.seriesAttributes.instanceCount = series.instances.length;
                        originalSeries.push(series);
                    }

                    var splitMultiframe = _.map(multiframeInstances, function (instance) {
                        var seriesAttributes = copySeriesAttributes(instance.seriesAttributes);
                        seriesAttributes.json = instance.instanceAttributes.json;

                        return {
                            uuid: Study.generateUUID(),
                            instances: _.map(_.range(0, instance.instanceAttributes.frameCount), function (frame) {
                                var copy = new Models.Instance();
                                copy.frameNumber = new Classes.FrameNumber(frame);
                                copy.id = instance.id;
                                copy.instanceAttributes = copyInstanceAttributes(instance.instanceAttributes);
                                copy.instanceAttributes.measurements = [];
                                copy.instanceAttributes.instanceIndex = frame;
                                copy.instanceAttributes.instanceNumber = frame;
                                copy.seriesAttributes = seriesAttributes;
                                copy.seriesAttributes.instanceCount = instance.instanceAttributes.frameCount;

                                copy.studyAttributes = instance.studyAttributes;
                                return copy;
                            }),
                            seriesAttributes: seriesAttributes,
                            studyAttributes: series.studyAttributes
                        };
                    });

                    _.each(splitMultiframe, function (series) {
                        if (series.seriesAttributes.json) {
                            loadSharedMultiframeTags(series.seriesAttributes, series.instances);
                            loadPerFrameMultiframeTags(series.seriesAttributes, series.instances);
                        }
                    });

                    return _.flatten([originalSeries, splitMultiframe]);
                } else {
                    return [series];
                }
            } else {
                return [series];
            }
        } else if (settings.viewer_multiframe_split_method == 1) {
            // This method will put videos into their own series leaving the remaining instances in a single series
            // If the series is not a multiframe series, only the series itself is returned.
            if (!isMultiframe(series) && !shouldSplitInstances(series.seriesAttributes.modality)) {
                return [series];
            }

            var standAloneSeries = [];

            // Pull out each multiframe instance and create its own series
            series.instances = _.filter(series.instances, function (instance) {
                if (instance.instanceAttributes.frameCount > 1) {
                    var newSeries = {
                        uuid: Study.generateUUID(),
                        instances: _.map(_.range(0, instance.instanceAttributes.frameCount), function (frame) {
                            var copy = new Models.Instance();
                            copy.frameNumber = new Classes.FrameNumber(frame);
                            copy.id = instance.id;
                            copy.instanceAttributes = copyInstanceAttributes(instance.instanceAttributes);
                            copy.instanceAttributes.measurements = [];
                            copy.instanceAttributes.instanceIndex = frame;
                            copy.instanceAttributes.instanceNumber = frame;

                            copy.seriesAttributes = copySeriesAttributes(instance.seriesAttributes);
                            copy.seriesAttributes.instanceCount = instance.instanceAttributes.frameCount;

                            copy.studyAttributes = instance.studyAttributes;
                            return copy;
                        }),
                        seriesAttributes: series.seriesAttributes,
                        studyAttributes: series.studyAttributes
                    };

                    newSeries.seriesAttributes.instanceCount = instance.instanceAttributes.frameCount;

                    standAloneSeries.push(newSeries);

                    return false;
                } else {
                    return true;
                }
            });

            // Update number of instances to match new count in original series
            series.seriesAttributes.instanceCount = series.instances.length;

            var frameCount = 0;
            _.each(series.instances, function (instance) {
                instance.instanceAttributes.instanceIndex = frameCount;
                instance.instanceAttributes.instanceNumber = frameCount++;
            });

            // Return both the original series and the new stand-alone series
            return [series].concat(standAloneSeries);
        } else if (settings.viewer_multiframe_split_method == 2) {
            return _.map(series.instances, function (instance) {
                return {
                    uuid: Study.generateUUID(),
                    instances: _.map(_.range(0, instance.instanceAttributes.frameCount), function (frame) {
                        var copy = new Models.Instance();
                        copy.frameNumber = new Classes.FrameNumber(frame);
                        copy.id = instance.id;
                        copy.instanceAttributes = copyInstanceAttributes(instance.instanceAttributes);
                        copy.instanceAttributes.measurements = [];
                        copy.instanceAttributes.instanceIndex = frame;
                        copy.instanceAttributes.instanceNumber = frame;

                        copy.seriesAttributes = copySeriesAttributes(instance.seriesAttributes);
                        copy.seriesAttributes.instanceCount = instance.instanceAttributes.frameCount;

                        copy.studyAttributes = instance.studyAttributes;
                        return copy;
                    }),
                    seriesAttributes: series.seriesAttributes,
                    studyAttributes: series.studyAttributes,
                    loadedStatus: 0 /* None */
                };
            });
        }
    }
    Multiframe.split = split;

    function copyInstanceAttributes(attribs) {
        var copy = new Models.InstanceAttributes();
        copy.attributesLoaded = attribs.attributesLoaded;
        copy.attributesDownloadState = attribs.attributesDownloadState;
        copy.instanceIndex = attribs.instanceIndex;
        copy.rank = attribs.rank;
        copy.instanceNumber = attribs.instanceNumber;
        copy.frameCount = attribs.frameCount;
        copy.sopClass = attribs.sopClass;
        copy.version = attribs.version;
        copy.seriesDescription = attribs.seriesDescription;
        copy.pixelSpacing = attribs.pixelSpacing;
        copy.pixelSpacingMeaning = attribs.pixelSpacingMeaning;
        copy.imagePositionPatient = attribs.imagePositionPatient;
        copy.imageOrientationPatient = attribs.imageOrientationPatient;
        copy.photometricInterpretation = attribs.photometricInterpretation;
        copy.rows = attribs.rows;
        copy.columns = attribs.columns;
        copy.windowWidth = attribs.windowWidth;
        copy.windowCenter = attribs.windowCenter;
        copy.windowLevelIsCorrupt = attribs.windowLevelIsCorrupt;
        copy.bitsStored = attribs.bitsStored;
        copy.voiLutFunction = attribs.voiLutFunction;
        copy.frameTime = attribs.frameTime;
        copy.measurements = attribs.measurements;
        copy.presentationStates = attribs.presentationStates;
        copy.isKeyImage = new Subjects.ObservableValue(attribs.isKeyImage.read());
        copy.presentationLUTShape = attribs.presentationLUTShape;
        copy.keyImageId = attribs.keyImageId;
        copy.signed = attribs.signed;
        copy.containsOverlayData = attribs.containsOverlayData;
        copy.imageLaterality = attribs.imageLaterality;
        copy.rescaleSlope = attribs.rescaleSlope;
        copy.rescaleIntercept = attribs.rescaleIntercept;
        copy.json = attribs.json;
        return copy;
    }
    Multiframe.copyInstanceAttributes = copyInstanceAttributes;

    function copySeriesAttributes(attribs) {
        var copy = new Models.SeriesAttributes();
        copy.seriesUid = new Classes.SeriesUid(attribs.seriesUid.value);
        copy.instanceCount = attribs.instanceCount;
        copy.modality = attribs.modality;
        copy.imageAttributesLoaded = new Subjects.ObservableValue(0);
        copy.imageDataLoaded = new Subjects.ObservableValue(0);
        copy.documentType = attribs.documentType;
        copy.json = attribs.json;
        copy.description = attribs.description;
        return copy;
    }
    Multiframe.copySeriesAttributes = copySeriesAttributes;

    function loadSharedMultiframeTags(seriesAttributes, instances) {
        var currentTags = seriesAttributes.json.tags;
        var sharedFunctionalSeq = Dicom.findTag(currentTags, [Dicom.Multiframe.SharedFunctionalGroupSeq.toString()]);

        if (sharedFunctionalSeq && (sharedFunctionalSeq.items.length == 1)) {
            seriesAttributes.multiframe = true;

            var sharedTags = sharedFunctionalSeq.items[0].tags;

            var rescaleSlope = Dicom.findTag(sharedTags, [Dicom.Multiframe.PixelValueTransSeq.toString(), Dicom.Dictionary.RescaleSlope.toString()]);

            var rescaleIntercept = Dicom.findTag(sharedTags, [Dicom.Multiframe.PixelValueTransSeq.toString(), Dicom.Dictionary.RescaleIntercept.toString()]);

            var windowCenter = Dicom.findTag(sharedTags, [Dicom.Multiframe.FrameVOILUTSeq.toString(), Dicom.Dictionary.WindowCenter.toString()]);

            var windowWidth = Dicom.findTag(sharedTags, [Dicom.Multiframe.FrameVOILUTSeq.toString(), Dicom.Dictionary.WindowWidth.toString()]);

            var imageOrientation = Dicom.findTag(sharedTags, [Dicom.Multiframe.PlaneOrientationSeq.toString(), Dicom.Dictionary.ImageOrientationPatient.toString()]);

            var pixelSpacing = Dicom.findTag(sharedTags, [Dicom.Multiframe.PixelMeasuresSeq.toString(), Dicom.Dictionary.PixelSpacing.toString()]);

            _.each(instances, function (instance) {
                if (rescaleSlope) {
                    instance.instanceAttributes.rescaleSlope = parseFloat(rescaleSlope.value) || 1;
                }

                if (rescaleIntercept) {
                    instance.instanceAttributes.rescaleIntercept = parseFloat(rescaleIntercept.value) || 0;
                }

                if (windowCenter) {
                    instance.instanceAttributes.windowCenter = [parseFloat(windowCenter.value)];
                }

                if (windowWidth) {
                    instance.instanceAttributes.windowWidth = [parseFloat(windowWidth.value)];
                }

                if (windowCenter && windowWidth) {
                    instance.instanceAttributes.windowLevelIsCorrupt = Parsers.isCorruptDS(windowCenter.value) || Parsers.isCorruptDS(windowWidth.value);
                }

                if (imageOrientation) {
                    instance.instanceAttributes.imageOrientationPatient = Parsers.parseDS(imageOrientation.value);
                }

                if (pixelSpacing) {
                    instance.instanceAttributes.pixelSpacing = Parsers.parseDS(pixelSpacing.value);
                    instance.instanceAttributes.pixelSpacingMeaning = 0 /* PatientGeometry */;
                }
            });
        }
    }
    Multiframe.loadSharedMultiframeTags = loadSharedMultiframeTags;

    function loadPerFrameMultiframeTags(seriesAttributes, instances) {
        var currentTags = seriesAttributes.json.tags;
        var perFrameSeq = Dicom.findTag(currentTags, [Dicom.Multiframe.PerFrameFunctionalGroupSeq.toString()]);

        if (perFrameSeq && (perFrameSeq.items.length == instances.length)) {
            seriesAttributes.multiframe = true;

            _.each(perFrameSeq.items, function (item, index) {
                var imagePosition = Dicom.findTag(item.tags, [Dicom.Multiframe.PlanePositionSeq.toString(), Dicom.Dictionary.ImagePositionPatient.toString()]);
                if (imagePosition) {
                    instances[index].instanceAttributes.imagePositionPatient = Parsers.parseDS(imagePosition.value);
                }

                var imageOrientation = Dicom.findTag(item.tags, [Dicom.Multiframe.PlaneOrientationSeq.toString(), Dicom.Dictionary.ImageOrientationPatient.toString()]);
                if (imageOrientation) {
                    instances[index].instanceAttributes.imageOrientationPatient = Parsers.parseDS(imageOrientation.value);
                }

                var pixelSpacing = Dicom.findTag(item.tags, [Dicom.Multiframe.PixelMeasuresSeq.toString(), Dicom.Dictionary.PixelSpacing.toString()]);
                if (pixelSpacing) {
                    instances[index].instanceAttributes.pixelSpacing = Parsers.parseDS(pixelSpacing.value);
                    instances[index].instanceAttributes.pixelSpacingMeaning = 0 /* PatientGeometry */;
                }

                var rescaleSlope = Dicom.findTag(item.tags, [Dicom.Multiframe.PixelValueTransSeq.toString(), Dicom.Dictionary.RescaleSlope.toString()]);
                if (rescaleSlope) {
                    instances[index].instanceAttributes.rescaleSlope = parseFloat(rescaleSlope.value) || 1;
                }

                var rescaleIntercept = Dicom.findTag(item.tags, [Dicom.Multiframe.PixelValueTransSeq.toString(), Dicom.Dictionary.RescaleIntercept.toString()]);
                if (rescaleIntercept) {
                    instances[index].instanceAttributes.rescaleIntercept = parseFloat(rescaleIntercept.value) || 0;
                }

                var windowCenter = Dicom.findTag(item.tags, [Dicom.Multiframe.FrameVOILUTSeq.toString(), Dicom.Dictionary.WindowCenter.toString()]);
                if (windowCenter) {
                    instances[index].instanceAttributes.windowCenter = [parseFloat(windowCenter.value)];
                }

                var windowWidth = Dicom.findTag(item.tags, [Dicom.Multiframe.FrameVOILUTSeq.toString(), Dicom.Dictionary.WindowWidth.toString()]);
                if (windowWidth) {
                    instances[index].instanceAttributes.windowWidth = [parseFloat(windowWidth.value)];
                }

                if (windowCenter && windowWidth) {
                    instances[index].instanceAttributes.windowLevelIsCorrupt = Parsers.isCorruptDS(windowCenter.value) || Parsers.isCorruptDS(windowWidth.value);
                }
            });
        }
    }
    Multiframe.loadPerFrameMultiframeTags = loadPerFrameMultiframeTags;
})(Multiframe || (Multiframe = {}));
///<reference path="../typings/jquery/jquery.d.ts" />
///<reference path="Subject.ts" />

/**
* Implementation of the grid layout JQuery plugin.
*/
var GridLayout;
(function (GridLayout) {
    function gridLayout($this, layout, startAt, children) {
        var index = startAt;

        var $container = $('<div>').addClass('grid-layout-container');

        var elWidth = 100 / layout.columns;
        var elHeight = 100 / layout.rows;

        for (var row = 0; row < layout.rows; row++) {
            for (var column = 0; column < layout.columns; column++) {
                if (index < children.length) {
                    var $el = children[index];
                    var $cellInner = $('<div>').addClass('grid-layout-cell-inner').append($el);
                    var $cell = $('<div>').addClass('grid-layout-cell').append($cellInner);

                    $cell.css({
                        width: elWidth + '%',
                        height: elHeight + '%'
                    });

                    $container.append($cell);
                }

                index++;
            }
        }

        // Use detach() instead of remove() to keep event handlers around
        $this.children('.grid-layout-container').each(function () {
            $(this).detach();
        });
        $this.append($container);
    }

    $.prototype.grid = function (layout, startAt, children, onLayoutChanged) {
        var $this = $(this);

        var layoutHandler = function () {
            gridLayout($this, layout.read(), startAt.read(), children.read());

            if (onLayoutChanged) {
                onLayoutChanged();
            }
        };

        var layoutObserver = {
            next: function (_) {
                return layoutHandler();
            },
            done: function () {
            },
            fail: function (_) {
            }
        };

        var subscriptions = [];

        subscriptions.push(layout.updates.subscribe(layoutObserver));
        subscriptions.push(children.updates.subscribe(layoutObserver));
        subscriptions.push(startAt.updates.subscribe(layoutObserver));

        $this.unload(function (e) {
            for (var i = 0; i < subscriptions.length; i++) {
                subscriptions[i].cancel();
            }
        });

        layoutHandler();

        return $this;
    };
})(GridLayout || (GridLayout = {}));
/**
* Helper methods for working with functions
*/
var Functions;
(function (Functions) {
    /**
    * Discards method invocations occurring within 'timeout' milliseconds of the last successful invocation.
    *
    * Chooses an implementation based on the browser
    */
    function throttle(f, timeout) {
        if (Browser.isIE()) {
            return throttleImpl(f, timeout);
        } else {
            return _.throttle(f, timeout);
        }
    }
    Functions.throttle = throttle;

    /**
    * Discards method invocations occurring within 'timeout' milliseconds of the last successful invocation.
    * Similar to Underscore's throttle method, but does not use setTimeout, which causes issues in some versions of IE.
    */
    function throttleImpl(f, timeout) {
        var lastTimeCalled;

        return function (t) {
            var time = new Date().getTime();
            if (!lastTimeCalled || time - lastTimeCalled > timeout) {
                f.apply(null, arguments);
                lastTimeCalled = time;
            }
        };
    }
    Functions.throttleImpl = throttleImpl;
})(Functions || (Functions = {}));
///<reference path="../typings/jquery/jquery.d.ts" />

/**
* Helper methods for working with touch devices
*
* @see JQuery.fallback
*/
var Touch;
(function (Touch) {
    $.prototype.fallback = function (touchEvents, events, handler) {
        var lastTouchEventTime = null;

        return $(this).on(events.join(" ") + " " + touchEvents.join(" "), function (e) {
            if (lastTouchEventTime === null || (lastTouchEventTime < (new Date().getTime()) - 500)) {
                lastTouchEventTime = new Date().getTime();

                if (_.contains(touchEvents, e.type)) {
                    handler(fixEvent(e));
                } else {
                    handler(e);
                }
            }

            e.preventDefault();
        });
    };

    function fixEvent(event) {
        if (event.pageX !== undefined && event.pageY !== undefined && event.offsetX !== undefined && event.offsetY !== undefined) {
            return event;
        }

        var originalEvent = event.originalEvent;

        if (originalEvent && originalEvent.touches) {
            var touch = originalEvent.touches[0] || originalEvent.changedTouches[0];

            if (touch) {
                var offset = $(touch.target).offset();

                event.pageX = touch.pageX;
                event.pageY = touch.pageY;
                event.offsetX = touch.pageX - offset.left;
                event.offsetY = touch.pageY - offset.top;

                return event;
            }
        }

        if (originalEvent && originalEvent.pageX !== undefined && originalEvent.pageY !== undefined && originalEvent.offsetX !== undefined && originalEvent.offsetY !== undefined) {
            event.pageX = originalEvent.pageX;
            event.pageY = originalEvent.pageY;
            event.offsetX = originalEvent.offsetX;
            event.offsetY = originalEvent.offsetY;

            return event;
        }

        return null;
    }
})(Touch || (Touch = {}));
/**
* A library for working with 3D vectors and matrices.
*
* Homogeneous n-D matrices are assumed to be in standard form, i.e. consisting of a
* (n-1)-D transformation and a translation. That is, the (n, i) components are assumed
* to be zero.
*/
var Vectors;
(function (Vectors) {
    

    

    

    

    

    /**
    * Create a vector from an array of components
    */
    function fromArray(arr) {
        return {
            x: arr[0],
            y: arr[1],
            z: arr[2]
        };
    }
    Vectors.fromArray = fromArray;

    /**
    * Turn a vector into an array of components
    */
    function toArray(v) {
        return [v.x, v.y, v.z];
    }
    Vectors.toArray = toArray;

    /**
    * Compute the squared length of a vector
    */
    function len2(v) {
        return dot(v, v);
    }
    Vectors.len2 = len2;

    /**
    * Compute the length of a vector
    */
    function len(v) {
        return Math.sqrt(len2(v));
    }
    Vectors.len = len;

    /**
    * Add vectors
    */
    function add(v1, v2) {
        return {
            x: v1.x + v2.x,
            y: v1.y + v2.y,
            z: v1.z + v2.z
        };
    }
    Vectors.add = add;

    /**
    * Negate a vector
    */
    function neg(v) {
        return {
            x: -v.x,
            y: -v.y,
            z: -v.z
        };
    }
    Vectors.neg = neg;

    /**
    * Subtract vectors
    */
    function sub(v1, v2) {
        return {
            x: v1.x - v2.x,
            y: v1.y - v2.y,
            z: v1.z - v2.z
        };
    }
    Vectors.sub = sub;

    /**
    * Dot (scalar) product
    */
    function dot(v1, v2) {
        return v1.x * v2.x + v1.y * v2.y + v1.z * v2.z;
    }
    Vectors.dot = dot;

    /**
    * Multiply a vector by a scalar
    */
    function scale(v, s) {
        return {
            x: v.x * s,
            y: v.y * s,
            z: v.z * s
        };
    }
    Vectors.scale = scale;

    /**
    * Normalize a non-zero vector. That is, rescale it to unit length.
    */
    function normalize(v) {
        return scale(v, 1.0 / len(v));
    }
    Vectors.normalize = normalize;

    /**
    * Cross (vector) product of two vectors
    */
    function cross(v1, v2) {
        return {
            x: v1.y * v2.z - v2.y * v1.z,
            y: v1.z * v2.x - v2.z * v1.x,
            z: v1.x * v2.y - v2.x * v1.y
        };
    }
    Vectors.cross = cross;

    /**
    * Compute the normal to a plane spanned by two vectors
    */
    function normal(v1, v2) {
        return normalize(cross(v1, v2));
    }
    Vectors.normal = normal;

    /**
    * Projection of one vector onto another
    */
    function project(v1, v2) {
        return dot(v1, v2) / len(v2);
    }
    Vectors.project = project;

    /**
    * Project a point onto an OrientedPlane, and return the result in the coordinate system
    * determined by the plane's preferred basis.
    */
    function coords(pl, p) {
        var v = sub(p, pl.o);

        var d1 = project(v, pl.v1);
        var d2 = project(v, pl.v2);

        return [d1, d2];
    }
    Vectors.coords = coords;

    /**
    * Project a point onto a plane
    */
    function project2(pl, p) {
        var coords = Vectors.coords(pl, p);

        var v1n = normalize(pl.v1);
        var v2n = normalize(pl.v2);

        return add(pl.o, add(scale(v1n, coords[0]), scale(v2n, coords[1])));
    }
    Vectors.project2 = project2;

    /**
    * Find the time travelled along an oriented line before intersection with a plane
    *
    * Time here is interpreted as the variable t in the equation r = L.o + t * L.v for the line
    */
    function intersectionDistance(p, l) {
        var n = normalize(cross(p.v1, p.v2));
        return (dot(n, sub(p.o, l.o))) / dot(l.v, n);
    }
    Vectors.intersectionDistance = intersectionDistance;

    /**
    * Find the intersection of a line and a plane
    */
    function intersect(p, l) {
        var d = intersectionDistance(p, l);
        return add(l.o, scale(l.v, d));
    }
    Vectors.intersect = intersect;

    /**
    * Creates a line pointing from one vector to another. The line's direction is not normalized.
    */
    function between(v1, v2) {
        return { o: v1, v: sub(v2, v1) };
    }
    Vectors.between = between;

    

    

    /**
    * Generate a translation matrix
    */
    function translate(dx, dy) {
        return {
            entries: [
                1, 0, dx,
                0, 1, dy
            ]
        };
    }
    Vectors.translate = translate;

    /**
    * Generate a 3D translation matrix
    */
    function translate3(dx, dy, dz) {
        return {
            entries: [
                1, 0, 0, dx,
                0, 1, 0, dy,
                0, 0, 1, dz,
                0, 0, 0, 1
            ]
        };
    }
    Vectors.translate3 = translate3;

    /**
    * Generate a flip matrix (horizontal)
    */
    function flipH() {
        return {
            entries: [
                -1, 0, 0,
                0, 1, 0
            ]
        };
    }
    Vectors.flipH = flipH;

    /**
    * Generate a flip matrix (vertical)
    */
    function flipV() {
        return {
            entries: [
                1, 0, 0,
                0, -1, 0
            ]
        };
    }
    Vectors.flipV = flipV;

    /**
    * Generate a rotation matrix
    */
    function rotate(rads) {
        var c = Math.cos(rads);
        var s = Math.sin(rads);

        return {
            entries: [
                c, -s, 0,
                s, c, 0
            ]
        };
    }
    Vectors.rotate = rotate;

    /**
    * Generate a rotation matrix
    */
    function rotateX(rads) {
        var c = Math.cos(rads);
        var s = Math.sin(rads);

        return {
            entries: [
                1, 0, 0, 0,
                0, c, -s, 0,
                0, s, c, 0,
                0, 0, 0, 1
            ]
        };
    }
    Vectors.rotateX = rotateX;

    /**
    * Generate a rotation matrix
    */
    function rotateY(rads) {
        var c = Math.cos(rads);
        var s = Math.sin(rads);

        return {
            entries: [
                c, 0, -s, 0,
                0, 1, 0, 0,
                s, 0, c, 0,
                0, 0, 0, 1
            ]
        };
    }
    Vectors.rotateY = rotateY;

    /**
    * Generate a scale matrix
    */
    function scaleM(s) {
        return scaleNonUniformM(s, s);
    }
    Vectors.scaleM = scaleM;

    /**
    * Generate a 3D scale matrix
    */
    function scale3(sx, sy, sz) {
        return {
            entries: [
                sx, 0, 0, 0,
                0, sy, 0, 0,
                0, 0, sz, 0,
                0, 0, 0, 1
            ]
        };
    }
    Vectors.scale3 = scale3;

    /**
    * Generate a scale matrix
    */
    function scaleNonUniformM(sx, sy) {
        return {
            entries: [
                sx, 0, 0,
                0, sy, 0
            ]
        };
    }
    Vectors.scaleNonUniformM = scaleNonUniformM;

    /**
    * Identity matrix
    */
    function identity() {
        return {
            entries: [
                1, 0, 0,
                0, 1, 0
            ]
        };
    }
    Vectors.identity = identity;

    /**
    * Identity matrix
    */
    function identity3() {
        return {
            entries: [
                1, 0, 0, 0,
                0, 1, 0, 0,
                0, 0, 1, 0,
                0, 0, 0, 1
            ]
        };
    }
    Vectors.identity3 = identity3;

    /**
    * Compose two 2D (homogeneous 3D) matrices (in standard form).
    */
    function composeM(m1, m2) {
        var m1e = m1.entries;
        var m2e = m2.entries;

        return {
            entries: [
                m1e[0] * m2e[0] + m1e[1] * m2e[3],
                m1e[0] * m2e[1] + m1e[1] * m2e[4],
                m1e[0] * m2e[2] + m1e[1] * m2e[5] + m1e[2],
                m1e[3] * m2e[0] + m1e[4] * m2e[3],
                m1e[3] * m2e[1] + m1e[4] * m2e[4],
                m1e[3] * m2e[2] + m1e[4] * m2e[5] + m1e[5]
            ]
        };
    }
    Vectors.composeM = composeM;

    /**
    * Compose two 3D (homogeneous 4D) matrices (in standard form)
    */
    function compose3(m1, m2) {
        var m1e = m1.entries;
        var m2e = m2.entries;

        return {
            entries: [
                m1e[0] * m2e[0] + m1e[1] * m2e[4] + m1e[2] * m2e[8],
                m1e[0] * m2e[1] + m1e[1] * m2e[5] + m1e[2] * m2e[9],
                m1e[0] * m2e[2] + m1e[1] * m2e[6] + m1e[2] * m2e[10],
                m1e[0] * m2e[3] + m1e[1] * m2e[7] + m1e[2] * m2e[11] + m1e[3],
                m1e[4] * m2e[0] + m1e[5] * m2e[4] + m1e[6] * m2e[8],
                m1e[4] * m2e[1] + m1e[5] * m2e[5] + m1e[6] * m2e[9],
                m1e[4] * m2e[2] + m1e[5] * m2e[6] + m1e[6] * m2e[10],
                m1e[4] * m2e[3] + m1e[5] * m2e[7] + m1e[6] * m2e[11] + m1e[7],
                m1e[8] * m2e[0] + m1e[9] * m2e[4] + m1e[10] * m2e[8],
                m1e[8] * m2e[1] + m1e[9] * m2e[5] + m1e[10] * m2e[9],
                m1e[8] * m2e[2] + m1e[9] * m2e[6] + m1e[10] * m2e[10],
                m1e[8] * m2e[3] + m1e[9] * m2e[7] + m1e[10] * m2e[11] + m1e[11]
            ]
        };
    }
    Vectors.compose3 = compose3;

    /**
    * Transpose a 3D (homogeneous 4D) matrix (in standard form).
    */
    function transpose3(m) {
        var me = m.entries;

        return [
            me[0], me[4], me[8], 0,
            me[1], me[5], me[9], 0,
            me[2], me[6], me[10], 0,
            me[3], me[7], me[11], 1
        ];
    }
    Vectors.transpose3 = transpose3;

    /**
    * Determinant of a 2D matrix (in standard form).
    */
    function determinant(m) {
        var e = m.entries;
        return e[0] * e[4] - e[1] * e[3];
    }
    Vectors.determinant = determinant;

    /**
    * Multiply a 2D (homogeneous 3D) matrix (in standard form) by a point.
    */
    function multiplyM(m, p) {
        return {
            x: m.entries[0] * p.x + m.entries[1] * p.y + m.entries[2],
            y: m.entries[3] * p.x + m.entries[4] * p.y + m.entries[5]
        };
    }
    Vectors.multiplyM = multiplyM;

    /**
    * Multiply a 3D (homogeneous 4D) matrix (in standard form) by a point.
    */
    function multiply3(m, p) {
        return {
            x: m.entries[0] * p.x + m.entries[1] * p.y + m.entries[2] * p.z + m.entries[3],
            y: m.entries[4] * p.x + m.entries[5] * p.y + m.entries[6] * p.z + m.entries[7],
            z: m.entries[8] * p.x + m.entries[9] * p.y + m.entries[10] * p.z + m.entries[11]
        };
    }
    Vectors.multiply3 = multiply3;
})(Vectors || (Vectors = {}));
///<reference path="../typings/underscore/underscore.d.ts" />
///<reference path="Vector.ts" />
/**
* A library for linking geometrically related series
*/
var SeriesGeometry;
(function (SeriesGeometry) {
    

    

    /**
    * Test if instances are orthogonal. The condition used is that planes should intersect with an acute angle > 30 degrees.
    */
    function areOrthogonal(source, target) {
        var sourcePlane = normalPlaneForInstance(source.instanceAttributes);

        var n1 = sourcePlane.n;
        var n2 = target.normal;
        var cosine = n1.x * n2[0] + n1.y * n2[1] + n1.z * n2[2];

        return Math.abs(cosine) < 0.5;
    }
    SeriesGeometry.areOrthogonal = areOrthogonal;

    /**
    * Test if an instance has the DICOM tags necessary for geometric operations
    */
    function hasGeometricMetadata(instance) {
        if (instance.instanceAttributes.imagePositionPatient && instance.instanceAttributes.imageOrientationPatient && instance.instanceAttributes.pixelSpacing && instance.instanceAttributes.pixelSpacingMeaning === 0 /* PatientGeometry */) {
            return true;
        }
        return false;
    }
    SeriesGeometry.hasGeometricMetadata = hasGeometricMetadata;

    /**
    * Find the index of the instance in the collection which is geometrically closest to the specified instance
    */
    function findClosestInstance(instances, target) {
        var imagePositionPatient = Vectors.fromArray(target.imagePositionPatient);

        return findClosestInstanceToPoint(_.filter(instances, function (instance) {
            return hasGeometricMetadata(instance) && !areOrthogonal(instance, target);
        }), imagePositionPatient);
    }
    SeriesGeometry.findClosestInstance = findClosestInstance;

    /**
    * Find the index of the instance in the collection which is geometrically closest to the specified point
    */
    function findClosestInstanceToPoint(instances, patientCoords) {
        var closestIndex = null;
        var minDistance = null;

        for (var index = 0; index < instances.length; index++) {
            var instanceAtIndex = instances[index];

            if (hasGeometricMetadata(instanceAtIndex)) {
                var plane = normalPlaneForInstance(instanceAtIndex.instanceAttributes);
                var offset = Vectors.sub(patientCoords, plane.o);
                var distance = Math.abs(plane.n.x * offset.x + plane.n.y * offset.y + plane.n.z * offset.z);

                if (minDistance === null || distance < minDistance) {
                    minDistance = distance;
                    closestIndex = index;
                }
            }
        }

        return closestIndex;
    }
    SeriesGeometry.findClosestInstanceToPoint = findClosestInstanceToPoint;

    /**
    * Get the plane defined by an instance
    */
    function planeForInstanceGeometry(instance) {
        return {
            o: Vectors.fromArray(instance.imagePositionPatient),
            v1: Vectors.fromArray(instance.imageOrientationPatient.slice(0, 3)),
            v2: Vectors.fromArray(instance.imageOrientationPatient.slice(3, 6))
        };
    }
    SeriesGeometry.planeForInstanceGeometry = planeForInstanceGeometry;

    /**
    * Get the plane defined by an instance
    */
    function normalPlaneForInstance(instance) {
        if (instance.plane) {
            return instance.plane;
        }

        var origin = Vectors.fromArray(instance.imagePositionPatient);
        var v1 = Vectors.fromArray(instance.imageOrientationPatient.slice(0, 3));
        var v2 = Vectors.fromArray(instance.imageOrientationPatient.slice(3, 6));

        return instance.plane = {
            o: origin,
            n: Vectors.normalize(Vectors.cross(v1, v2))
        };
    }
    SeriesGeometry.normalPlaneForInstance = normalPlaneForInstance;

    /**
    * Get the plane defined by an instance
    */
    function planeForInstance(instance) {
        return {
            o: Vectors.fromArray(instance.instanceAttributes.imagePositionPatient),
            v1: Vectors.fromArray(instance.instanceAttributes.imageOrientationPatient.slice(0, 3)),
            v2: Vectors.fromArray(instance.instanceAttributes.imageOrientationPatient.slice(3, 6))
        };
    }
    SeriesGeometry.planeForInstance = planeForInstance;

    /**
    * Map image coords to patient coords
    */
    function mapToPatient(imageCoords, instance) {
        var plane = planeForInstance(instance);
        return Vectors.add(plane.o, Vectors.add(Vectors.scale(plane.v1, imageCoords.x * instance.instanceAttributes.pixelSpacing[0]), Vectors.scale(plane.v2, imageCoords.y * instance.instanceAttributes.pixelSpacing[0])));
    }
    SeriesGeometry.mapToPatient = mapToPatient;

    /**
    * Map patient coords to image coords
    */
    function mapFromPatient(patientCoords, instance) {
        var plane = planeForInstance(instance);
        var coords = Vectors.coords(plane, patientCoords);
        return {
            x: coords[0] / instance.instanceAttributes.pixelSpacing[0],
            y: coords[1] / instance.instanceAttributes.pixelSpacing[0]
        };
    }
    SeriesGeometry.mapFromPatient = mapFromPatient;

    /**
    * Compute the bounding vertices which bound the image geometry in 3-dimensional space
    */
    function boundingVerticesForInstance(instance) {
        var imagePositionPatient = Vectors.fromArray(instance.imagePositionPatient);

        var row = Vectors.fromArray(instance.imageOrientationPatient.slice(0, 3));
        var column = Vectors.fromArray(instance.imageOrientationPatient.slice(3, 6));

        var topEdge = Vectors.scale(Vectors.normalize(row), instance.pixelSpacing[0] * instance.columns);
        var leftEdge = Vectors.scale(Vectors.normalize(column), instance.pixelSpacing[0] * instance.rows);

        var tl = imagePositionPatient;
        var tr = Vectors.add(tl, topEdge);
        var bl = Vectors.add(tl, leftEdge);
        var br = Vectors.add(tr, leftEdge);

        return {
            tl: tl,
            tr: tr,
            bl: bl,
            br: br
        };
    }
    SeriesGeometry.boundingVerticesForInstance = boundingVerticesForInstance;

    /**
    * Compute the lines which bound the image geometry in 3-dimensional space
    */
    function boundsForInstance(instance) {
        var vs = boundingVerticesForInstance(instance);

        return [
            Vectors.between(vs.tl, vs.tr),
            Vectors.between(vs.tr, vs.br),
            Vectors.between(vs.br, vs.bl),
            Vectors.between(vs.bl, vs.tl)
        ];
    }
    SeriesGeometry.boundsForInstance = boundsForInstance;
})(SeriesGeometry || (SeriesGeometry = {}));
///<reference path='../classes/Types.ts' />
///<reference path='../models/Study.ts' />
///<reference path='Terminology.ts' />
/**
* Preset window level defaults
*/
var WindowLevelPresets;
(function (WindowLevelPresets) {
    /**
    * Default values
    */
    WindowLevelPresets.defaultCenter = 128.0;
    WindowLevelPresets.defaultWidth = 256.0;

    /**
    * Lookup default window level presets for a modality
    */
    function defaults(modality, terminology) {
        switch (modality.toUpperCase()) {
            case "CT":
                return [
                    {
                        name: terminology ? terminology.lookup(Terminology.Terms.SoftTissue) : Terminology.Terms.SoftTissue.defaultValue,
                        windowLevel: { center: 40, width: 400 },
                        opacity: 0.25
                    }, {
                        name: terminology ? terminology.lookup(Terminology.Terms.Bone) : Terminology.Terms.Bone.defaultValue,
                        windowLevel: { center: 500, width: 1500 },
                        opacity: 0.10
                    }, {
                        name: terminology ? terminology.lookup(Terminology.Terms.Head) : Terminology.Terms.Head.defaultValue,
                        windowLevel: { center: 35, width: 110 },
                        opacity: 0.25
                    }, {
                        name: terminology ? terminology.lookup(Terminology.Terms.Lung) : Terminology.Terms.Lung.defaultValue,
                        windowLevel: { center: -500, width: 1500 },
                        opacity: 1.0
                    }];
            default:
                return [];
        }
    }
    WindowLevelPresets.defaults = defaults;

    /**
    * Returns true if a valid window level is found
    * @param instance
    */
    function isValidWindowLevel(instance) {
        return instance && !instance.instanceAttributes.windowLevelIsCorrupt && instance.instanceAttributes.windowCenter && instance.instanceAttributes.windowCenter.length > 0 && instance.instanceAttributes.windowWidth && instance.instanceAttributes.windowWidth.length > 0;
    }
    WindowLevelPresets.isValidWindowLevel = isValidWindowLevel;

    /**
    * Check if an instance should be rendered using 16-bit window level
    */
    function shouldUse16BitWindowLevel(instance) {
        return instance && (instance.instanceAttributes.bitsStored > 8) && WindowLevelPresets.supports16Bit(instance) && !Browser.needsResize(instance);
    }
    WindowLevelPresets.shouldUse16BitWindowLevel = shouldUse16BitWindowLevel;

    function supports16Bit(instance) {
        return instance.instanceAttributes.photometricInterpretation.indexOf("MONO") !== -1;
    }
    WindowLevelPresets.supports16Bit = supports16Bit;

    function isFullResolutionHD(instance) {
        return (instance.instanceAttributes.bitsStored > 8 && !WindowLevelPresets.supports16Bit(instance));
    }
    WindowLevelPresets.isFullResolutionHD = isFullResolutionHD;

    /**
    * Scan image data for min/max and calculate window levels
    * @param instance
    * @param imageData
    */
    function findWindowLevels(instance, imageData) {
        var data = imageData.data;
        var fullImageSize = imageData.width * imageData.height / 2 * 4;
        var l = data.length / 2;
        var signed = instance.instanceAttributes.signed || false;
        var typeRange = (1 << instance.instanceAttributes.bitsStored);
        var typeSignedMax = (1 << instance.instanceAttributes.bitsStored - 1) - 1;
        var slope = instance.instanceAttributes.rescaleSlope || 1;
        var intercept = instance.instanceAttributes.rescaleIntercept || 0;

        var min = Number.MAX_VALUE;
        var max = -Number.MAX_VALUE;

        for (var i = 0; i < l;) {
            var raw = (data[i + fullImageSize] << 8) | data[i];

            if (signed) {
                raw = (raw & 0xffff) - 0x8000;

                if (raw > typeSignedMax) {
                    raw -= typeRange;
                }
            }

            min = Math.min(raw, min);
            max = Math.max(raw, max);

            i = i + 4;
        }

        min = (min * slope) + intercept;
        max = (max * slope) + intercept;

        var windowLevel = {
            center: (max + min) / 2.0,
            width: max - min
        };

        if (instance.instanceAttributes.presentationLUTShape === "INVERSE" || instance.instanceAttributes.photometricInterpretation === "MONOCHROME1") {
            windowLevel = {
                center: (signed ? 0 : (1 << instance.instanceAttributes.bitsStored)) * slope - windowLevel.center,
                width: windowLevel.width
            };
        }

        return windowLevel;
    }
    WindowLevelPresets.findWindowLevels = findWindowLevels;

    /**
    * True if instance needs to scan for window levels
    * @param instance
    */
    function instanceNeedsWindowLevel(instance) {
        return !WindowLevelPresets.isValidWindowLevel(instance) && !instance.instanceAttributes.windowLevelDetection;
    }
    WindowLevelPresets.instanceNeedsWindowLevel = instanceNeedsWindowLevel;
})(WindowLevelPresets || (WindowLevelPresets = {}));
///<reference path="../classes/Types.ts" />
/**
* Cine related helper methods
*/
var Cine;
(function (Cine) {
    /**
    * The default cine speed when one is not specified by the user settings
    */
    Cine.defaultCineSpeed = 12.0;

    /**
    * Rate at which storage renders frames
    */
    Cine.cineRenderRate = 30.0;

    /**
    * Check for compatibility
    */
    function isFormatSupported() {
        if (LocalViewer.isLocalViewer()) {
            return false;
        }

        if (Browser.isIE9()) {
            return false;
        }

        var testVideo = document.createElement('video');

        return testVideo.canPlayType && testVideo.canPlayType('video/mp4') !== '';
    }
    Cine.isFormatSupported = isFormatSupported;

    function isBrowserSupported() {
        return !(Browser.isIE() || Browser.isIE11());
    }
    Cine.isBrowserSupported = isBrowserSupported;

    /**
    * Get the default cine speed based on modality
    */
    function getDefaultCineSpeed(settings, modality) {
        if (settings.modalities) {
            var modalitySettings = _.find(settings.modalities, function (m) {
                return m.modality === modality;
            });

            if (modalitySettings && modalitySettings.cineSpeed) {
                return modalitySettings.cineSpeed;
            }
        }

        return Cine.defaultCineSpeed;
    }
    Cine.getDefaultCineSpeed = getDefaultCineSpeed;

    /**
    * Save the new cine speed back to the settings object
    */
    function saveCineSpeedToSettings(settings, modality, speed) {
        var copy = $.extend(true, settings, {
            modalities: []
        });

        var modalitySettings = _.find(copy.modalities, function (m) {
            return m.modality === modality;
        });

        if (!modalitySettings) {
            copy.modalities.push(modalitySettings = { modality: modality });
        }

        modalitySettings.cineSpeed = speed;

        return copy;
    }
    Cine.saveCineSpeedToSettings = saveCineSpeedToSettings;
})(Cine || (Cine = {}));
///<reference path='../typings/jquery/jquery.d.ts' />
///<reference path='../models/Study.ts' />
///<reference path="../typings/jquery/jquery.d.ts" />
///<reference path="../typings/jquery.mousewheel/jquery.mousewheel.d.ts" />
///<reference path="../typings/underscore/underscore.d.ts" />
///<reference path="../classes/Types.ts" />
///<reference path="../libs/Observable.ts" />
///<reference path="../libs/Query.ts" />
///<reference path="../libs/Services.ts" />
///<reference path="../libs/Functions.ts" />
///<reference path="../libs/V3Storage.ts" />
///<reference path="../libs/Study.ts" />
///<reference path="../libs/Subject.ts" />
///<reference path="../libs/GridLayout.ts" />
///<reference path="../libs/Touch.ts" />
///<reference path="../libs/SeriesGeometry.ts" />
///<reference path="../libs/WindowLevelPresets.ts" />
///<reference path="../libs/Cine.ts" />
///<reference path='../libs/Browser.ts' />
///<reference path="../models/StudySchema.ts" />
///<reference path="../models/Study.ts" />
///<reference path="../layer/Layer.ts" />
///<reference path="../libs/LocalViewer.ts" />

var Views;
(function (Views) {
    (function (PreloadQueueKeyType) {
        PreloadQueueKeyType[PreloadQueueKeyType["ImageAttributes"] = 0] = "ImageAttributes";
        PreloadQueueKeyType[PreloadQueueKeyType["ImageData"] = 1] = "ImageData";
        PreloadQueueKeyType[PreloadQueueKeyType["ImageMetadata"] = 2] = "ImageMetadata";
    })(Views.PreloadQueueKeyType || (Views.PreloadQueueKeyType = {}));
    var PreloadQueueKeyType = Views.PreloadQueueKeyType;

    /**
    * Priority levels
    */
    (function (Priority) {
        Priority[Priority["ImageDataBackgroundPreloading"] = -2] = "ImageDataBackgroundPreloading";
        Priority[Priority["ImageAttributeBackgroundPreloading"] = -1] = "ImageAttributeBackgroundPreloading";
        Priority[Priority["FullResolutionImage"] = 99] = "FullResolutionImage";
    })(Views.Priority || (Views.Priority = {}));
    var Priority = Views.Priority;

    var PriorityIsPriority = (function () {
        function PriorityIsPriority() {
        }
        PriorityIsPriority.prototype.compare = function (p1, p2) {
            return p1 - p2;
        };
        return PriorityIsPriority;
    })();
    Views.PriorityIsPriority = PriorityIsPriority;

    

    var PreloadQueueKeyIsKey = (function () {
        function PreloadQueueKeyIsKey() {
        }
        PreloadQueueKeyIsKey.prototype.equals = function (k1, k2) {
            if (k1.type === 0 /* ImageAttributes */ && k2.type === 0 /* ImageAttributes */) {
                return k1.seriesUid.value === k2.seriesUid.value && k1.frameNumber.value === k2.frameNumber.value && k1.instanceUid.value === k2.instanceUid.value;
            } else if (k1.type === 1 /* ImageData */ && k2.type === 1 /* ImageData */) {
                return k1.seriesUid.value === k2.seriesUid.value && k1.instanceUid.value === k2.instanceUid.value && k1.frameNumber.value === k2.frameNumber.value && k1.imageType === k2.imageType;
            } else {
                return false;
            }
        };

        PreloadQueueKeyIsKey.prototype.show = function (k) {
            switch (k.type) {
                case 0 /* ImageAttributes */:
                    return "Load Image Attributes {" + k.seriesUid.value + ", " + k.instanceUid.value + "}";
                case 1 /* ImageData */:
                    return "Load Image Data {" + k.seriesUid.value + ", " + k.instanceUid.value + ", " + k.frameNumber.value + ", " + Classes.ImageType[k.imageType] + "}";
            }
        };
        return PreloadQueueKeyIsKey;
    })();
    Views.PreloadQueueKeyIsKey = PreloadQueueKeyIsKey;

    var ByInstanceNumber = (function () {
        function ByInstanceNumber() {
            this.values = {};
        }
        /**
        * Lookup a value
        */
        ByInstanceNumber.prototype.get = function (index, def) {
            return this.values["instance" + index] || def;
        };

        /**
        * Set a value
        */
        ByInstanceNumber.prototype.put = function (index, value) {
            this.values["instance" + index] = value;
        };

        /**
        * Remove a key
        */
        ByInstanceNumber.prototype.remove = function (index) {
            this.values["instance" + index] = null;
            delete this.values["instance" + index];
        };

        /**
        * Remove all keys
        */
        ByInstanceNumber.prototype.clear = function () {
            this.values = {};
        };

        /**
        * Iterate over all values
        */
        ByInstanceNumber.prototype.each = function (f) {
            for (var p in this.values) {
                if (this.values.hasOwnProperty(p)) {
                    f(this.values[p]);
                }
            }
        };
        return ByInstanceNumber;
    })();
    Views.ByInstanceNumber = ByInstanceNumber;

    /**
    * A dictionary with keys of type ImageType
    */
    var ByImageTypeAndInstanceNumber = (function () {
        function ByImageTypeAndInstanceNumber() {
            this.values = new ByInstanceNumber();
        }
        /**
        * Lookup a value
        */
        ByImageTypeAndInstanceNumber.prototype.get = function (index, type) {
            var instance = this.values.get(index, {});
            return instance[Classes.ImageType[type]];
        };

        /**
        * Set a value
        */
        ByImageTypeAndInstanceNumber.prototype.put = function (index, type, value) {
            var instance = this.values.get(index, {});
            instance[Classes.ImageType[type]] = value;
            this.values.put(index, instance);
        };

        /**
        * Remove a key
        */
        ByImageTypeAndInstanceNumber.prototype.remove = function (index, type) {
            var instance = this.values.get(index, {});
            instance[Classes.ImageType[type]] = null;
            delete instance[Classes.ImageType[type]];
        };
        return ByImageTypeAndInstanceNumber;
    })();
    Views.ByImageTypeAndInstanceNumber = ByImageTypeAndInstanceNumber;

    

    (function (DownloadState) {
        DownloadState[DownloadState["None"] = 0] = "None";
        DownloadState[DownloadState["Queued"] = 1] = "Queued";
        DownloadState[DownloadState["Started"] = 2] = "Started";
        DownloadState[DownloadState["Success"] = 3] = "Success";
        DownloadState[DownloadState["Failed"] = 4] = "Failed";
    })(Views.DownloadState || (Views.DownloadState = {}));
    var DownloadState = Views.DownloadState;

    

    /**
    * A renderer which uses image elements
    */
    var SimpleRenderer = (function () {
        function SimpleRenderer() {
            this.imageElements = new ByInstanceNumber();
        }
        SimpleRenderer.prototype.act = function (visitor) {
            if (visitor.visitSimpleRenderer) {
                visitor.visitSimpleRenderer(this);
            }
        };

        SimpleRenderer.prototype.visit = function (visitor, def) {
            if (visitor.visitSimpleRenderer) {
                return visitor.visitSimpleRenderer(this);
            } else {
                return def;
            }
        };
        return SimpleRenderer;
    })();
    Views.SimpleRenderer = SimpleRenderer;

    /**
    * A renderer which uses a HTML5 canvas
    */
    var CanvasRenderer = (function () {
        function CanvasRenderer() {
            /**
            * Image filters in use
            */
            this.filters = [];
        }
        CanvasRenderer.prototype.act = function (visitor) {
            if (visitor.visitCanvasRenderer) {
                visitor.visitCanvasRenderer(this);
            }
        };

        CanvasRenderer.prototype.visit = function (visitor, def) {
            if (visitor.visitCanvasRenderer) {
                return visitor.visitCanvasRenderer(this);
            } else {
                return def;
            }
        };
        return CanvasRenderer;
    })();
    Views.CanvasRenderer = CanvasRenderer;

    /**
    * A renderer which uses WebGL
    */
    var WebGLRenderer = (function () {
        function WebGLRenderer() {
        }
        WebGLRenderer.prototype.act = function (visitor) {
            if (visitor.visitWebGLRenderer) {
                visitor.visitWebGLRenderer(this);
            }
        };

        WebGLRenderer.prototype.visit = function (visitor, def) {
            if (visitor.visitWebGLRenderer) {
                return visitor.visitWebGLRenderer(this);
            } else {
                return def;
            }
        };
        return WebGLRenderer;
    })();
    Views.WebGLRenderer = WebGLRenderer;

    /**
    * A rendered handle in the upper-right corner for dragging series into other viewport
    */
    var Handle = (function () {
        function Handle(view) {
            var _this = this;
            this.view = view;
            this.supportsDragImage = (typeof DataTransfer.prototype.setDragImage === "function");
            var $handle = $('<div>').addClass("handle fa fa-2x fa-bars").appendTo(view.el);
            this.$el = $handle;
            $handle.prop("draggable", true);

            this.$el.on("mousedown", function (e) {
                var dataURL = _this.view.prepareCanvasForReport(_this.supportsDragImage, 150).toDataURL("image/png");
                var img = new Image();
                img.src = dataURL;
                _this.thumbnail = img;

                if (!_this.supportsDragImage) {
                    _this.$el.css("background-image", 'url("' + dataURL + '")');
                }
            }).on('selectstart', function (e) {
                if (Browser.isIE9()) {
                    _this.$el[0].dragDrop();
                    return false;
                }
            }).on("dragstart", function (e) {
                var dataTransfer = e.originalEvent.dataTransfer;

                if (dataTransfer) {
                    dataTransfer.dropEffect = "move";
                    dataTransfer.setData("text", "swap:" + _this.view.viewKey.value + ":" + _this.view.series.uuid);
                }

                if (dataTransfer && _this.supportsDragImage) {
                    dataTransfer.setDragImage(_this.thumbnail, 0, 0);
                } else {
                    _this.$el.removeClass("fa fa-2x fa-bars");
                    _this.$el.addClass("grabbed");
                }

                _this.$el.siblings(".instance").hide();
            }).on("drag", function (e) {
                _this.$el.hide();
            }).on("mouseup", function (e) {
                _this.reset();
                _this.view.renderAll();
            }).on("dragend", function (e) {
                _this.reset();
                _this.view.renderAll();
            });

            if (this.needsHandle()) {
                this.$el.show();
            } else {
                this.$el.hide();
            }
        }
        Handle.prototype.needsHandle = function () {
            var layout = this.view.application.layout.value;
            return (layout.rows * layout.columns) > 1;
        };

        Handle.prototype.show = function () {
            if (this.needsHandle()) {
                this.$el.show();
            }
        };

        Handle.prototype.hide = function () {
            this.$el.hide();
        };

        Handle.prototype.reset = function () {
            this.$el.addClass("fa fa-2x fa-bars");
            this.$el.removeClass("grabbed");
            this.$el.css("background-image", 'none');
            this.$el.show();
            this.$el.siblings(".instance").show();
        };

        Handle.prototype.clear = function () {
            this.view = null;
            this.$el = null;
        };
        return Handle;
    })();
    Views.Handle = Handle;

    /**
    * Unique key for a Series view
    */
    var SeriesViewKey = (function () {
        function SeriesViewKey(value) {
            this.value = value;
        }
        SeriesViewKey.Null = new SeriesViewKey(null);
        return SeriesViewKey;
    })();
    Views.SeriesViewKey = SeriesViewKey;

    /**
    * A view which displays a series, consisting of a grid of instance views
    */
    var Series = (function () {
        function Series(application, sessionId, el, series, settings, accountSettings, selectedSeriesKey, selectedTool, selectedTool2, selectedToolWheel, selectedInstanceGeometry, referenceLinesActive, planeLocalizationCursor, linkSeries, imagePreloadQueue, terminology, cineSpeed, ignoreFrameTimeTag, startCinePlaybackAutomatically, noKeyboardShortcuts, recorder, playbackMode) {
            var _this = this;
            this.viewKey = new SeriesViewKey(Math.round(Math.random() * 1000000).toString());
            /**
            * The current image transformation parameters
            */
            this.transform = new Subjects.ObservableValue({ offsetX: 0, offsetY: 0, scale: 1, flipped: false, rotation: 0 });
            this.mousePressed = false;
            /**
            * True if panning is active
            */
            this.panActive = false;
            /**
            * True if press is active
            */
            this.pressActive = false;
            /**
            * True if mouse cursor moved
            */
            this.mouseMoved = false;
            /**
            * True if a mouseUp has not happened following a mouseDown
            */
            this.needsMouseUp = false;
            /**
            * True if the annotation cursor needs to be hidden (e.g., while the mouse is dragging)
            */
            this.needsHideMeasurementCursor = false;
            /**
            * Shared subject which contains the location of the probe tool
            */
            this.probeTool = new Subjects.ObservableValue(null);
            /**
            * Shared subject which contains the location of the magnifier tool
            */
            this.magnifier = new Subjects.ObservableValue(null);
            /**
            * The center of the zoom/rotate operation
            */
            this.zoomCenter = new Subjects.ObservableValue(null);
            /**
            * True if cine is active
            */
            this.cineActive = new Subjects.ObservableValue(false);
            /**
            * True if text annotations are visible
            */
            this.infoVisible = new Subjects.ObservableValue(true);
            /**
            * True if measurements are visible
            */
            this.measurementsVisible = new Subjects.ObservableValue(true);
            /**
            * True if measurement details are visible
            */
            this.measurementsDetailsVisible = new Subjects.ObservableValue(false);
            /**
            * True if annotations created by other users are visible
            */
            this.annotationsCreatedByOtherUsersVisible = new Subjects.ObservableValue(true);
            /**
            * Collection of hidden GSPS layers
            */
            this.hiddenGSPSLayers = new Subjects.ObservableValue([]);
            /**
            * True if the image is inverted
            */
            this.invertActive = new Subjects.ObservableValue(false);
            /**
            * True if XA subtraction is being used
            */
            this.subtractionActive = new Subjects.ObservableValue(false);
            /**
            *
            * @type {Subjects.ObservableValue<Classes.ColorTable>}
            */
            this.colorTable = new Subjects.ObservableValue(null);
            /**
            * True if histogram equalization is turned on
            */
            this.enhance = new Subjects.ObservableValue(false);
            /**
            * The current window level settings
            */
            this.windowLevel = new Subjects.ObservableValue({ center: 128.0, width: 255.0 });
            this.useOriginalWindowLevel = new Subjects.ObservableValue(true);
            /**
            * Taken if low resolution images should be displayed, regardless of whether the high resolution image is available.
            * This is used when mouse operations are in progress to make sure rendering is fast.
            */
            this.forceLowResolution = new Subjects.ObservableValue(Classes.Resource.unused());
            /**
            * A list of subscriptions to dispose of when unloading this series
            */
            this.subscriptions = [];
            /**
            * The layers to render when the image is loaded
            */
            this.layers = [];
            this.progressBarLayer = new Layers.ProgressBarLayer();
            /**
            * The measurement currently being drawn
            */
            this.measurementInProgress = new Subjects.ObservableValue(null);
            /**
            * The currently selected measurement
            */
            this.selectedMeasurement = new Subjects.ObservableValue(null);
            /**
            * The measurement instance currently being edited.
            * @type {Subjects.ObservableValue<Measurements.Measurement>}
            */
            this.editingMeasurement = new Subjects.ObservableValue(null);
            /**
            * Overlay data
            */
            this.overlayData = new ByInstanceNumber();
            // Avoid preloading everything twice
            this.fullPreloadStarted = false;
            /**
            * Did the init function run yet?
            */
            this.didInit = false;
            /**
            * Accelerator cache miss count
            * Increment when X-Acc-Cached response header set to MISS
            *
            * Show message only once
            */
            this.acceleratorCacheMissCount = 0;
            this.acceleratorCacheMissMessage = "This study has images missing from the accelerator cache.";
            this.acceleratorCacheMessageDidShow = false;
            this.glTextureIsLoaded = false;
            this.glCreatedPaletteTexture = false;
            this.glTexturePaletteIsLoaded = false;
            /**
            * Enable to store extra data in an annotation (e.g., stats).
            * @type {number}
            */
            this.storeExtraAnnotationData = 0;
            /**
            * Use diagnostic quality always
            * @type {boolean}
            */
            this.useDiagnosticQualityAlways = false;
            /**
            * If true, measurement stats always shown in mm
            * @type {boolean}
            */
            this.alwaysUseMillimeters = false;
            /**
            * Ultrasound regions enabled
            */
            this.ultrasoundMeasurements = false;
            /**
            * True if this series is showing a warning banner.
            */
            this.showingBanner = false;
            this.lastRenderTime = 0;
            this.updateMousePosition = function (e) {
                if (_this.cursorLayer && _this.cursorLayer.visible && _this.cursorLayer.visible.read()) {
                    _this.cursorLayer.mousePosition = { x: e.offsetX, y: e.offsetY };
                    var context = _this.cursorCanvas.getContext("2d");

                    var instance = _this.currentInstance().read();
                    var instanceIndex = _this.selectedInstanceIndex.read();
                    var instanceKey = _this.getInstanceKey(instanceIndex);

                    _this.cursorLayer.render(context, instanceKey, instance);
                }
            };
            this.hideCursor = function () {
                _this.el.css({ cursor: '' });
                _this.el.addClass('hideCursor');
            };
            this.application = application;

            this.sessionId = sessionId;

            this.el = el;
            this.series = series;
            this.settings = settings;
            this.accountSettings = accountSettings;

            this.selectedSeriesKey = selectedSeriesKey;
            this.selectedTool = selectedTool;
            this.selectedTool2 = selectedTool2;
            this.selectedToolWheel = selectedToolWheel;

            this.selectedInstanceIndex = new Subjects.ObservableValue(0);

            this.ignoreFrameTimeTag = ignoreFrameTimeTag;

            if (!this.ignoreFrameTimeTag && series && !!series.instances[0].instanceAttributes.frameTime) {
                var fps = 1000 / series.instances[0].instanceAttributes.frameTime[0];
                this.cineSpeed = new Subjects.ObservableValue(fps);
            } else {
                this.cineSpeed = cineSpeed;
            }

            this.startCinePlaybackAutomatically = startCinePlaybackAutomatically;

            this.selectedInstanceGeometry = selectedInstanceGeometry;
            this.referenceLinesActive = referenceLinesActive;
            this.linkSeries = linkSeries;
            this.planeLocalizationCursor = planeLocalizationCursor;

            this.imagePreloadQueue = imagePreloadQueue;

            this.terminology = terminology;

            this.noKeyboardShortcuts = noKeyboardShortcuts;

            this.recorder = recorder;
            this.playbackMode = playbackMode;

            this.showStandardDev = new Subjects.ObservableValue(this.accountSettings.viewer_show_std_dev == 1);
            this.rulerVisible = new Subjects.ObservableValue(this.accountSettings.viewer_hide_ruler != 1);
            this.ultrasoundRegionsVisible = new Subjects.ObservableValue(false);
            this.hideActiveMeasuremntInfo = (this.accountSettings.viewer_hide_active_measurement_info == 1);
            this.storeExtraAnnotationData = this.accountSettings.viewer_store_extra_annotation_data;

            this.measurementsDetailsVisible.write(this.settings.read().showAnnotationDetails);
            this.annotationsCreatedByOtherUsersVisible.write(this.settings.read().showAnnotationsCreatedByOthers === undefined ? true : this.settings.read().showAnnotationsCreatedByOthers);
            this.measureVolumeDefault = this.settings.read().allowVolumeMeasurements;
            this.measureVolume = new Subjects.ObservableValue(this.measureVolumeDefault);

            this.cursorLayer = new Layers.CursorLayer(new Subjects.ObservableValue(false), this, this.application.currentColor);

            // showTextOnDirected is the opposite of disableArrowMeasurementText
            var disableArrowMeasurementText = (this.settings.read().disableArrowMeasurementText == null) ? false : this.settings.read().disableArrowMeasurementText;
            this.showTextOnDirected = new Subjects.ObservableValue(!disableArrowMeasurementText);

            this.alwaysUseMillimeters = this.settings.read().alwaysUseMillimeters;
            this.ultrasoundMeasurements = this.settings.read().ultrasoundMeasurements;
            this.useDiagnosticQualityAlways = (this.accountSettings.viewer_diagnostic_quality_always == 1) || this.settings.read().diagnosticQualityAlways;

            if (this.series && !this.useDiagnosticQualityAlways) {
                var modality = this.series.seriesAttributes.modality;
                var userSettings = this.settings.read();
                if (userSettings.modalities) {
                    var modalitySettings = _.find(userSettings.modalities, function (m) {
                        return m.modality === modality;
                    });
                    if (modalitySettings && modalitySettings.diagnosticQualityAlways == true) {
                        this.useDiagnosticQualityAlways = true;
                    }
                }
            }

            if (this.series && Dicom.Subtraction.supportsSubtraction(this.series.seriesAttributes)) {
                this.subtraction = Dicom.Subtraction.buildXA(this.series.instances[0]);
            }
        }
        /**
        * Create the necessary elements, add them to the DOM, and hook up event handlers
        * @param force true to initialize even if series is not in dom
        */
        Series.prototype.init = function (force) {
            var _this = this;
            if (typeof force === "undefined") { force = false; }
            if (!this.didInit && (force || $.contains(document.documentElement, this.el[0]))) {
                this.el.empty();

                // For PDFs insert an iframe
                if (this.series && this.series.seriesAttributes && this.series.seriesAttributes.documentType && this.series.seriesAttributes.documentType.type === 0 /* PDF */) {
                    var $instance = $('<div>').addClass('instance');
                    var $iframe = $('<iframe class="instance-pdf" frameborder="0"></iframe>');

                    $iframe[0].onload = function (e) {
                        _this.renderPDF();
                    };

                    $instance.append($iframe);
                    this.el.append($instance);

                    this.didInit = true;
                    return;
                }

                // For Video use an html video player
                if (this.series && this.series.seriesAttributes && this.series.seriesAttributes.documentType && this.series.seriesAttributes.documentType.type === 1 /* Video */) {
                    var $instance = $('<div>').addClass('instance');
                    $instance.append('<div class="instance-video"></div>');
                    this.el.append($instance);

                    this.renderVideo();

                    // Prevent browser default right click
                    this.el.on('contextmenu', function (e) {
                        return false;
                    });

                    this.didInit = true;
                    return;
                }

                var $instance = $('<div>').addClass('instance');

                this.renderer = this.createRenderer($instance);

                // Initiate and attach the global image cache
                this.renderer.act({
                    visitCanvasRenderer: function (renderer) {
                        if (_this.series) {
                            if (_this.application.imagePreloadStore[_this.series.seriesAttributes.seriesUid.value] == null) {
                                _this.application.imagePreloadStore[_this.series.seriesAttributes.seriesUid.value] = new ByImageTypeAndInstanceNumber();
                            }

                            renderer.imageElements = _this.application.imagePreloadStore[_this.series.seriesAttributes.seriesUid.value];
                        }
                    },
                    visitWebGLRenderer: function (renderer) {
                        if (_this.series) {
                            if (_this.application.imagePreloadStore[_this.series.seriesAttributes.seriesUid.value] == null) {
                                _this.application.imagePreloadStore[_this.series.seriesAttributes.seriesUid.value] = new ByImageTypeAndInstanceNumber();
                            }

                            renderer.imageElements = _this.application.imagePreloadStore[_this.series.seriesAttributes.seriesUid.value];
                        }
                    }
                });

                this.createLayers();

                this.canvas = document.createElement("canvas");

                var $canvas = $(this.canvas);
                $canvas.addClass('instance-canvas');
                $canvas.appendTo($instance);

                this.progressBarCanvas = document.createElement("canvas");

                var $progressBarCanvas = $(this.progressBarCanvas);
                $progressBarCanvas.addClass('progressBar-canvas');
                $progressBarCanvas.appendTo($instance);

                this.cursorCanvas = document.createElement("canvas");
                this.cursorCanvas.width = this.el.width();
                this.cursorCanvas.height = this.el.height();

                var $cursorCanvas = $(this.cursorCanvas);
                $cursorCanvas.addClass('cursor-canvas');
                $cursorCanvas.appendTo($instance);

                if (Browser.isMobile() && this.accountSettings.viewer_enable_tap_navigation == 1) {
                    // Set up arrow overlays
                    _.each([
                        'next-series',
                        'prev-series',
                        'next-image',
                        'prev-image'
                    ], function (clazz) {
                        $('<div>').addClass('arrow-overlay').addClass(clazz).hide().appendTo($(_this.el));
                    });
                }

                $(window).resize(Functions.throttle(function (_) {
                    _this.renderAll();
                }, 10));

                this.el.append($instance);

                var $glassPane = $('<div>').addClass("glasspane").appendTo(this.el);

                $glassPane.mousewheel(Functions.throttle(function (d) {
                    return _this.handleMouseWheelEvent(d, arguments[1]);
                }, 10));

                var hammer = new Hammer.Manager($glassPane.get(0));

                hammer.add(new Hammer.Pan({
                    threshold: 0
                }));

                hammer.add(new Hammer.Pinch({
                    enable: true,
                    threshold: 0.1
                }));

                hammer.add(new Hammer.Tap({
                    event: 'doubletap',
                    taps: 2
                }));
                hammer.add(new Hammer.Tap({
                    event: 'singletap',
                    taps: 1
                }));

                hammer.add(new Hammer.Press({
                    event: 'press',
                    taps: 1,
                    time: 0
                }));

                hammer.get('doubletap').recognizeWith('singletap');
                hammer.get('singletap').requireFailure('doubletap');

                hammer.on('singletap', function (e) {
                    _this.selectedSeriesKey.write(_this.viewKey);

                    if (_this.needsMouseUp) {
                        _this.handleMouseUpEvent(e);
                    }

                    if (Browser.isMobile() && _this.accountSettings.viewer_enable_tap_navigation == 1) {
                        var $frame = $(_this.el);
                        var offset = $frame.offset();

                        var w = $frame.width();
                        var h = $frame.height();

                        var x = e.center.x - offset.left;
                        var y = e.center.y - offset.top;

                        var b1 = x * h > y * w;
                        var b2 = (w - x) * h > y * w;

                        if (b1) {
                            if (b2) {
                                var newIndex = (_this.selectedInstanceIndex.read() - 1 < 0) ? _this.series.instances.length - 1 : _this.selectedInstanceIndex.read() - 1;
                                _this.selectedInstanceIndex.write(newIndex);
                                _this.arrow('prev-image');
                            } else {
                                _this.application.loadNextSeries();
                            }
                        } else {
                            if (b2) {
                                _this.application.loadPrevSeries();
                            } else {
                                var newIndex = (_this.selectedInstanceIndex.read() + 1 >= _this.series.instances.length) ? 0 : _this.selectedInstanceIndex.read() + 1;
                                _this.selectedInstanceIndex.write(newIndex);
                                _this.arrow('next-image');
                            }
                        }
                    }
                });

                hammer.on('press', function (e) {
                    if (!_this.pressActive && ((e.button === 0) || (e.button === undefined))) {
                        _this.pressActive = true;
                        _this.handleMouseDownEvent(e);
                    }
                });

                hammer.on('pressup', function (e) {
                    if (!_this.mouseMoved && ((e.button === 0) || (e.button === undefined))) {
                        _this.handleMouseUpEvent(e);
                    }
                });

                hammer.on('panstart', function (e) {
                    if (!_this.pressActive) {
                        _this.handleMouseDownEvent(e);
                    }

                    _this.pressActive = false;
                    _this.mouseMoved = true;
                });
                hammer.on('panmove', Functions.throttle(function (e) {
                    return _this.handleMouseMoveEvent(e);
                }, 10));
                hammer.on('panmove', Functions.throttle(function (e) {
                    return _this.recordMouseMoveEvent(e);
                }, 250));
                hammer.on('panend', function (e) {
                    return _this.handleMouseUpEvent(e);
                });

                hammer.on('pinchstart', function (e) {
                    return _this.pinchZoomStart(e);
                });
                hammer.on('pinchmove', function (e) {
                    return _this.pinchZoom(e);
                });
                hammer.on('pinchend', function (e) {
                    return _this.pinchZoomEnd(e);
                });

                hammer.on('doubletap', function (e) {
                    if (_this.series) {
                        _this.application.magnifyMinify(_this);
                    }

                    if (_this.needsMouseUp) {
                        _this.handleMouseUpEvent(e);
                    }

                    _this.selectedSeriesKey.write(_this.viewKey);
                });

                // Prevent browser default right click
                $glassPane.on('contextmenu', function (e) {
                    return false;
                });

                // Prevent mobile browser overscroll
                $glassPane.on('touchmove', function (e) {
                    return false;
                });

                // Allow right mouse click to open context menu
                $glassPane.on('mouseup pointerup', function (e) {
                    var mouseOnly = (e.type == "mouseup" && !window.PointerEvent);
                    var pointerOnly = (e.type == "pointerup" && window.PointerEvent);
                    if (!_this.panActive && e.button == 2 && (mouseOnly || pointerOnly)) {
                        $('.series').contextMenu({
                            x: e.pageX,
                            y: e.pageY
                        });
                    }
                    return true;
                });

                $glassPane.on('mouseenter', function (e) {
                    if (_this.cursorLayer.cursor != 0 /* Blank */) {
                        _this.cursorLayer.visible.write(true);
                    }
                });

                $glassPane.on('mouseleave', function (e) {
                    _this.cursorLayer.visible.write(false);
                });

                if (this.settings.read().recordMouseMovements) {
                    $glassPane.on('mousemove', Functions.throttle(function (e) {
                        return _this.recordMouseMoveEvent(e);
                    }, 10));
                }

                this.subscriptions.push(Subjects.listen(this.selectedTool, function (tool) {
                    if (!_this.settings.read().disableCustomAnnotationCursor && Classes.MouseTools.isMeasurementTool(_this.selectedTool.read())) {
                        _this.enableMeasurementCursor(tool);
                    } else {
                        _this.disableMeasurementCursor();
                    }

                    if (!Classes.MouseTools.isGroupMeasurementTool(tool) && Classes.MouseTools.isGroupMeasurementTool(_this.previousTool)) {
                        _this.application.hideWarningBanner();

                        if (_this.previousTool == 33 /* ProstateTool */) {
                            Measurements.ProstateTool.reset();
                        }
                    }

                    _this.previousTool = _this.selectedTool.read();
                }));

                // Initialize the selected tool
                this.application.selectedTool.write(this.selectedTool.read());

                if (this.series) {
                    this.handle = new Handle(this);
                }

                $glassPane.on('dragover', function (e) {
                    e.preventDefault();
                }).on('drop', function (e) {
                    e.preventDefault();
                    var dataTransfer = e.originalEvent.dataTransfer;

                    if (dataTransfer) {
                        var message = dataTransfer.getData("text");
                        var seriesLikeUUID;
                        var swapViewKey;

                        if (message.indexOf("swap") != -1) {
                            var parts = message.split(":");
                            swapViewKey = parts[1];
                            seriesLikeUUID = parts[2];
                        } else {
                            seriesLikeUUID = message;
                        }

                        if (swapViewKey) {
                            var views = _this.application.seriesViews.value;
                            var swapView = _.find(views, function (view) {
                                return (view.viewKey.value == swapViewKey);
                            });

                            if (swapView) {
                                var swapViewIndex = views.indexOf(swapView);
                                var currentIndex = views.indexOf(_this);
                                views[swapViewIndex] = views.splice(currentIndex, 1, views[swapViewIndex])[0]; // swap
                                _this.application.seriesViews.raiseChangedEvent([swapView, _this]);
                                swapView.handle.reset();
                            }
                        } else {
                            var series = _.chain(_this.application.studies).map(function (s) {
                                return s.series;
                            }).flatten().filter(function (s) {
                                return s.uuid === seriesLikeUUID;
                            }).value();

                            if (!(series && series[0])) {
                                series = _.chain(_this.application.studies).map(function (s) {
                                    return s.keyImageSeries;
                                }).flatten().filter(function (s) {
                                    return s.uuid === seriesLikeUUID;
                                }).value();
                            }

                            if (series && series[0]) {
                                _this.application.selectedSeriesKey.write(_this.viewKey);
                                _this.application.replaceSelectedSeries(series[0]);
                            }
                        }
                    }
                });

                this.subscriptions.push(Subjects.listen(this.selectedSeriesKey, function (key) {
                    if (key.value === _this.viewKey.value && !_this.el.is('.selected')) {
                        _this.el.addClass('selected');
                        _this.renderAll();
                    } else if (key.value !== _this.viewKey.value && _this.el.is('.selected')) {
                        _this.el.removeClass('selected');
                        _this.renderAll();
                    }
                }));

                var sendInstanceIndexEventToRecorder = Functions.throttle(function (index) {
                    _this.recordEvent({
                        type: 6 /* SelectedImageChanged */,
                        instanceIndex: index
                    });
                }, 500);

                var accelerated = this.series && this.series.studyAttributes.accelerated;

                this.subscriptions.push(Subjects.listen(this.selectedInstanceIndex, function (index) {
                    _this.glTextureIsLoaded = false;

                    if (Browser.isChrome() && !accelerated && !_this.fullPreloadStarted) {
                        _this.fullPreloadStarted = true;
                        _this.loadAllImageData(_this.imagePreloadQueue, -2 /* ImageDataBackgroundPreloading */);
                    }

                    if (_this.application.gspsInfoVisible.read()) {
                        _this.application.gspsInfo.render();
                    }

                    _this.renderer.act({
                        visitSimpleRenderer: function (renderer) {
                            renderer.imageElements.each(function (el) {
                                return $(el.imageElement).hide();
                            });

                            var instanceKey = _this.getInstanceKey(index);

                            var imageElement = renderer.imageElements.get(instanceKey, null);

                            if (imageElement) {
                                $(imageElement.imageElement).show();
                            }
                        }
                    });

                    sendInstanceIndexEventToRecorder(index);

                    _this.renderAll();
                }));

                if (LocalViewer.isLocalViewer() || this.application.studyStorage.localAccelerator) {
                    // This line appears to cause significant issues in the local viewer. More testing is needed to see if it is required.
                    this.loadAllImageData(this.imagePreloadQueue, -2 /* ImageDataBackgroundPreloading */);
                    this.preloadAllDiagnosticImages(this.imagePreloadQueue, this.application.supportsAcceleratedPreload() ? 99 /* FullResolutionImage */ : -2 /* ImageDataBackgroundPreloading */);
                }

                if (LocalViewer.isLocalViewer() || accelerated || this.application.studyStorage.localAccelerator || this.linkSeries.read()) {
                    this.loadImageAttributes(this.imagePreloadQueue, -1 /* ImageAttributeBackgroundPreloading */);
                }

                var preloadRange = this.accountSettings.image_preload_window || this.settings.read().imagePreloadWindow || 10;

                var preloadVisibleHiResImages = _.debounce(function () {
                    return _this.preloadVisibleHiResImages(_this.imagePreloadQueue, 99 /* FullResolutionImage */);
                }, 1000);
                var preloadNearbyImages = _.debounce(function () {
                    return _this.preloadImageRange(_this.imagePreloadQueue, preloadRange, 0, preloadRange);
                }, 100);

                this.subscriptions.push(Subjects.listen(this.selectedInstanceIndex, function (index) {
                    preloadVisibleHiResImages();
                    preloadNearbyImages();
                }));

                this.subscriptions.push(Subjects.listen(this.selectedInstanceGeometry, function (data) {
                    if (data && _this.series && _this.series.studyAttributes.queryObject.studyUid.value === data.studyUid.value) {
                        if (_this.linkSeries.read() && _this.viewKey.value !== data.originator) {
                            _this.scrollToNearestInstance(data);
                        }

                        if (_this.referenceLinesActive.read() && _this.series.seriesAttributes.seriesUid.value !== data.seriesUid.value) {
                            _this.renderLayers();
                        }
                    }
                }));

                this.subscriptions.push(Subjects.listen(this.planeLocalizationCursor, function (cursor) {
                    if (cursor && _this.series && _this.series.studyAttributes.queryObject.studyUid.value === cursor.studyUid.value && _this.series.seriesAttributes.seriesUid.value !== cursor.seriesUid.value) {
                        _this.scrollToNearestInstanceToPoint(cursor.coords);
                    }

                    _this.renderAll();
                }));

                _.each([this.selectedTool, this.selectedTool2, this.selectedToolWheel], function (sub) {
                    Subjects.listen(sub, function (tool) {
                        _this.renderAll();
                    });
                });

                var renderLayers = Functions.throttle(function (_) {
                    _this.renderLayers();
                }, 25);

                var renderAll = Functions.throttle(function (_) {
                    _this.renderAll();
                }, 25);

                _.each([this.infoVisible, this.measurementsVisible, this.rulerVisible, this.ultrasoundRegionsVisible, this.measurementsDetailsVisible, this.annotationsCreatedByOtherUsersVisible, this.hiddenGSPSLayers, this.referenceLinesActive, this.probeTool, this.magnifier], function (sub) {
                    return _this.subscriptions.push(Subjects.listen(sub, function (e) {
                        renderLayers({});
                    }));
                });

                Subjects.listen(this.colorTable, function (e) {
                    _this.glTexturePaletteIsLoaded = false;
                    renderAll({});
                });

                _.each([this.invertActive, this.windowLevel, this.linkSeries, this.transform, this.enhance], function (sub) {
                    return _this.subscriptions.push(Subjects.listen(sub, function (e) {
                        renderAll({});
                    }));
                });

                this.subscriptions.push(Subjects.listen(this.subtractionActive, function (e) {
                    if (_this.subtractionActive.read()) {
                        if (!_this.subtractionWindowLevel) {
                            _this.subtractionWindowLevel = { center: 0, width: 256 };
                        }

                        var original = _this.windowLevel.read();
                        _this.windowLevel.write(_this.subtractionWindowLevel);
                        _this.preSubtractionWindowLevel = { center: original.center, width: original.width };
                    } else {
                        var original = _this.windowLevel.read();
                        _this.subtractionWindowLevel = { center: original.center, width: original.width };
                        _this.windowLevel.write(_this.preSubtractionWindowLevel);
                    }

                    renderAll({});
                }));

                this.setupMeetingEvents();

                if (!this.useDiagnosticQualityAlways) {
                    this.subscriptions.push(Subjects.listen(this.forceLowResolution, function (e) {
                        _this.renderAll();
                    }));
                }

                if (this.series) {
                    this.subscriptions.push(Subjects.listen(this.series.seriesAttributes.imageDataLoaded, function (e) {
                        _this.checkLoadedStatus();
                        _this.renderProgressBar();
                    }));
                }

                this.subscriptions.push(Subjects.listen(this.cineActive, function (active) {
                    _this.toggleCine(active);
                }));

                var saveCineSpeedToSettings = _.debounce(function () {
                    return _this.saveCineSpeedToSettings();
                }, 500);
                this.subscriptions.push(Subjects.listen(this.cineSpeed, function (speed) {
                    saveCineSpeedToSettings();
                }));

                if (this.series) {
                    if (!this.useDiagnosticQualityAlways || accelerated) {
                        var preloadRange = this.accountSettings.image_preload_window || this.settings.read().imagePreloadWindow || 10;
                        this.preloadImageRange(this.imagePreloadQueue, preloadRange, 0, preloadRange);
                    }

                    this.preloadVisibleHiResImages(this.imagePreloadQueue, 99 /* FullResolutionImage */);
                }

                this.didInit = true;

                // Initial render
                if (this.findRenderingMode() == 2 /* Simple */) {
                    this.renderAll();
                } else {
                    this.initialRenderTimer = setInterval(function () {
                        return _this.renderAll();
                    }, 200);
                }
            }
        };

        Series.prototype.arrow = function (clazz) {
            if (this.settings && !this.settings.read().disableMobileArrows) {
                var $arrow = $(this.el).find('.' + clazz);
                $arrow.show();
                $arrow.fadeOut(800);
            }
        };

        Series.prototype.enableMeasurementCursor = function (tool) {
            if (this.settings.read().disableCustomAnnotationCursor == true) {
                this.el.css({ cursor: 'crosshair' });
            } else {
                var $glassPane = this.el.find('.glasspane');

                // Show crosshair when annotation tools are selected
                this.cursorLayer.visible.write(true);

                if (Classes.MouseTools.isPaintTool(tool)) {
                    this.cursorLayer.cursor = 2 /* Paint */;
                } else {
                    this.cursorLayer.cursor = 1 /* Crosshair */;
                }

                // Enable mouse tracking for cursor layer
                $glassPane.on('mousemove', this.updateMousePosition);

                // Don't show normal operating system cursor
                $glassPane.one('mousemove', this.hideCursor);
            }
        };

        Series.prototype.disableMeasurementCursor = function () {
            if (this.settings.read().disableCustomAnnotationCursor == true) {
                this.el.css({ cursor: '' });
            } else {
                var $glassPane = this.el.find('.glasspane');

                // Don't show a custom cursor
                this.cursorLayer.visible.write(false);
                this.cursorLayer.cursor = 0 /* Blank */;

                // Stop tracking the mouse position for the cursor layer
                $glassPane.off('mousemove', this.updateMousePosition);
                $glassPane.off('mousemove', this.hideCursor);

                // Show the normal operating system cursor
                this.el.css({ cursor: '' });
                this.el.removeClass('hideCursor');
            }
        };

        /**
        * Add event listeners for meeting events
        */
        Series.prototype.setupMeetingEvents = function () {
            var _this = this;
            Subjects.listen(this.invertActive, function (inverted) {
                _this.recordEvent({
                    type: 9 /* InvertChanged */,
                    inverted: inverted
                });
            });

            Subjects.listen(this.enhance, function (enhance) {
                _this.recordEvent({
                    type: 15 /* EnhanceChanged */,
                    enhance: enhance
                });
            });

            Subjects.listen(this.infoVisible, function (visible) {
                _this.recordEvent({
                    type: 10 /* ToggleTextAnnotationsChanged */,
                    showTextAnnotations: visible
                });
            });

            Subjects.listen(this.measurementsVisible, function (visible) {
                _this.recordEvent({
                    type: 11 /* ToggleMeasurementsChanged */,
                    showMeasurements: visible
                });
            });

            // Transformation events
            var recordTransformation = _.throttle(function () {
                _this.recordTransformation();
            }, 250);

            this.subscriptions.push(Subjects.listen(this.transform, function (e) {
                recordTransformation();
                _this.application.thumbnails.render();
            }));

            // Magnifier events
            var recordMagnifier = _.throttle(function () {
                _this.recordMagnifier();
            }, 250);

            this.subscriptions.push(Subjects.listen(this.magnifier, function (e) {
                recordMagnifier();
            }));

            // Probe tool events
            var recordProbe = _.throttle(function () {
                _this.recordProbe();
            }, 250);

            this.subscriptions.push(Subjects.listen(this.probeTool, function (e) {
                recordProbe();
            }));

            // Plane localization events
            var recordPlaneLocalizationData = _.throttle(function () {
                _this.recordPlaneLocalizationData();
            }, 250);

            this.subscriptions.push(Subjects.listen(this.planeLocalizationCursor, function (e) {
                recordPlaneLocalizationData();
            }));

            if (this.startCinePlaybackAutomatically.read()) {
                this.cineActive.write(true);
            }
        };

        Series.prototype.attributesLoaded = function (instance) {
            if (this.series && instance) {
                if (this.selectedInstanceIndex.read() == instance.instanceAttributes.instanceIndex) {
                    this.preloadVisibleHiResImages(this.imagePreloadQueue, 99 /* FullResolutionImage */);
                }
            }
        };

        Series.prototype.findRenderingMode = function () {
            var mode;

            if (this.series) {
                var attribs = this.series.instances[0].instanceAttributes;
                var rows = attribs.rows;
                var cols = attribs.columns;

                mode = Rendering.getRenderingMode(Math.max(rows * 2, cols));
            } else {
                mode = Rendering.getRenderingMode();
            }

            return mode;
        };

        /**
        * Create an appropriate renderer based on capabilities
        */
        Series.prototype.createRenderer = function ($instance) {
            var mode = this.findRenderingMode();

            switch (mode) {
                case 2 /* Simple */: {
                    var simpleRenderer = new SimpleRenderer();

                    return simpleRenderer;
                }
                case 1 /* WebGL */: {
                    var webGLRenderer = new WebGLRenderer();

                    webGLRenderer.canvas3d = document.createElement("canvas");

                    var $canvas3d = $(webGLRenderer.canvas3d);
                    $canvas3d.addClass('instance-canvas-3d');
                    $canvas3d.appendTo($instance);

                    try  {
                        webGLRenderer.gl = WebGL.createContext(webGLRenderer.canvas3d);
                        webGLRenderer.program = WebGL.initShaders(this.application.shaders, webGLRenderer.gl);
                        WebGL.setupBuffersAndCreateTexture(webGLRenderer.gl, webGLRenderer.program);
                    } catch (ex) {
                        this.recordError("Unable to initialize WebGL: " + ex);
                        return new SimpleRenderer();
                    }

                    return webGLRenderer;
                }
                case 0 /* Canvas */: {
                    var canvasRenderer = new CanvasRenderer();

                    canvasRenderer.filters.push(new Filters.Subtraction(this, this.subtractionActive));
                    canvasRenderer.filters.push(new Filters.WindowLevel(this.combineWindowLevelFlags(), this.windowLevel, this.subtractionActive));
                    canvasRenderer.filters.push(new Filters.Invert(this.invertActive));
                    canvasRenderer.filters.push(new Filters.Colors(this.colorTable));

                    canvasRenderer.canvas2d = document.createElement("canvas");

                    var $canvas2d = $(canvasRenderer.canvas2d);
                    $canvas2d.addClass('instance-canvas-2d');
                    $canvas2d.appendTo($instance);

                    canvasRenderer.context = canvasRenderer.canvas2d.getContext("2d");

                    return canvasRenderer;
                }
            }

            throw "Unknown rendering mode";
        };

        /**
        * Setup the layers required for rendering
        */
        Series.prototype.createLayers = function () {
            var _this = this;
            var useMultiframeCine = Multiframe.isMultiframe(this.series) && Cine.isFormatSupported();
            var cineActive = Subjects.map(this.cineActive, function (active) {
                return active && useMultiframeCine;
            });
            var cineInactive = Subjects.map(cineActive, function (active) {
                return !active;
            });

            var settings = this.settings.read();

            if (this.series) {
                if (useMultiframeCine) {
                    this.layers.push(new Layers.ConditionalLayer(cineActive, this.cineLayer = new Layers.CineLayer(this.sessionId, cineActive, this.cineSpeed, this.transform, this.settings, this.terminology, function () {
                        return _this.renderAll();
                    }, function (frame) {
                        _this.selectedInstanceIndex.write(frame);
                        _this.renderAll();
                    })));
                }

                var textSize = Layers.DEFAULT_TEXT_SIZE;

                if (settings.annotationTextSize) {
                    var textSizeVal = parseInt(settings.annotationTextSize);
                    if (textSizeVal) {
                        textSize = textSizeVal;
                    }
                }

                this.layers.push(new Layers.OverlayLayer(this.measurementsVisible, this.overlayData, this, settings.supportROIOverlays));
                this.layers.push(new Layers.ConditionalLayer(cineInactive, this.measurementsLayer = new Layers.MeasurementLayer(this.measurementsVisible, this.measurementsDetailsVisible, this.annotationsCreatedByOtherUsersVisible, this.renderer, this.measurementInProgress, this.selectedTool, this.selectedMeasurement, this, this.playbackMode, this.showStandardDev, this.showTextOnDirected, this.editingMeasurement, this.hideActiveMeasuremntInfo, this.alwaysUseMillimeters, textSize, this.measureVolume, settings.cobbSmallAngle, this.application.user)));

                this.gspsLayer = new Layers.PresentationStateLayer(this.measurementsVisible, this.application.gspsInfoVisible, this.hiddenGSPSLayers, this.application.highlightedGraphic, this);
                this.layers.push(new Layers.ConditionalLayer(cineInactive, this.gspsLayer));

                var getImageType = function () {
                    var index = _this.selectedInstanceIndex.read();

                    return _this.getImageType(_this.series.instances[index]);
                };

                var attributesLayer = new Layers.Attributes(this.combineWindowLevelFlags(), getImageType, this.infoVisible, this.terminology, this.selectedMeasurement, this.settings, this, this);
                this.layers.push(attributesLayer);

                this.layers.push(new Layers.Orientation(this.infoVisible, this));
                this.layers.push(new Layers.ReferenceLines(this.selectedInstanceGeometry, this.referenceLinesActive, this));
                this.layers.push(new Layers.PlaneLocalizationLayer(this.planeLocalizationCursor, this.selectedTool, this));

                this.layers.push(new Layers.ProbeLayer(this.renderer, this.probeTool, this.selectedTool, this));

                this.layers.push(new Layers.MagnifierLayer(this.magnifier, this.selectedTool, this.transform, this));

                this.layers.push(new Layers.FreeRotateLayer(this.zoomCenter, this.selectedTool, this));

                this.layers.push(new Layers.RulerLayer(this, this.rulerVisible));

                if (this.ultrasoundMeasurements && Dicom.Ultrasound.isUltrasound(this.series.instances[0])) {
                    this.layers.push(new Layers.Ultrasound(this, this.ultrasoundRegionsVisible));
                }

                this.layers.push(new Layers.ErrorLayer(this.terminology));
            }

            var isSelected = Subjects.map(this.selectedSeriesKey, function (ser) {
                return ser.value === _this.viewKey.value;
            });

            var isLinked = Subjects.zip(this.linkSeries, this.selectedInstanceGeometry, function (linkSeries, selectedInstance) {
                if (_this.series && linkSeries && _this.series.instances && selectedInstance) {
                    var currentInstance = _this.currentInstance().read();
                    if (currentInstance && SeriesGeometry.hasGeometricMetadata(currentInstance)) {
                        return !SeriesGeometry.areOrthogonal(currentInstance, selectedInstance);
                    }
                }

                return false;
            });

            var borderLayer = new Layers.Border(isSelected, isLinked);
            this.layers.push(borderLayer);
        };

        /**
        * Render a single PDF
        */
        Series.prototype.renderPDF = function () {
            if (this.series && this.series.seriesAttributes && this.series.seriesAttributes.documentType && this.series.seriesAttributes.documentType.type === 0 /* PDF */) {
                var instance = this.series.instances[0];
                var uri = Routes.PDFData(this.sessionId, instance.studyAttributes.studyStorage, instance.studyAttributes.queryObject, instance.id, instance.instanceAttributes.version, this.accountSettings.cache == 1, true);

                var $iframe = $(this.el).find(".instance-pdf");
                var $pdf = $('<object style="width:100%; height:100%;" data="' + uri + '" type="application/pdf"><embed style="width:100%; height:100%;" src="' + uri + '" type="application/pdf" /></object>');

                $iframe.contents().find('html body').append($pdf);
                $iframe.contents().find('html body').css('margin', '0px');
            }
        };

        /**
        * Render the video
        */
        Series.prototype.renderVideo = function () {
            if (this.series && this.series.seriesAttributes && this.series.seriesAttributes.documentType && this.series.seriesAttributes.documentType.type === 1 /* Video */) {
                if (Cine.isFormatSupported()) {
                    var instance = this.series.instances[0];

                    var $videobody = $(this.el).find(".instance-video");

                    // var instance = this.currentInstance().read();
                    var video = new Views.Video(this.sessionId, $videobody, this.settings, instance, this.terminology, false);
                    video.render();
                } else {
                    window.alert(this.terminology.lookup(Terminology.Terms.VideoNotSupported));
                }
            }
        };

        /**
        * Render the appropriate layers into the back buffer and switch buffers
        */
        Series.prototype.renderAllForceUpdate = function () {
            this.renderAll(null, true, null, false, true);
        };
        Series.prototype.renderAll = function (renderImage, clearOverlayLayer, layers, forceSquare, forceRender) {
            var _this = this;
            if (typeof clearOverlayLayer === "undefined") { clearOverlayLayer = true; }
            if (typeof forceSquare === "undefined") { forceSquare = false; }
            if (typeof forceRender === "undefined") { forceRender = false; }
            if (!this.didInit) {
                this.init();
                return;
            }
            if (this.didInit && $.contains(document.documentElement, this.el[0])) {
                if (this.series && this.series.seriesAttributes && this.series.seriesAttributes.documentType && this.series.seriesAttributes.documentType.type === 0 /* PDF */) {
                    // Do nothing, this is a PDF
                    return;
                }

                if (this.series && this.series.seriesAttributes && this.series.seriesAttributes.documentType && this.series.seriesAttributes.documentType.type === 1 /* Video */) {
                    // Do nothing, this is a Video
                    return;
                }

                // throttle render to at most once very 10ms
                var currentRenderTime = (new Date()).getTime();
                if (!forceRender && (currentRenderTime - this.lastRenderTime < 10)) {
                    return;
                }

                this.lastRenderTime = currentRenderTime;

                var targetWidth;
                var targetHeight;

                if (forceSquare) {
                    var size = Math.min(this.el.width(), this.el.height());
                    targetWidth = targetHeight = size;
                } else {
                    targetWidth = this.el.width();
                    targetHeight = this.el.height();
                }

                this.canvas.width = targetWidth;
                this.canvas.height = targetHeight;
                this.cursorCanvas.width = targetWidth;
                this.cursorCanvas.height = targetHeight;

                this.renderer.act({
                    visitCanvasRenderer: function (renderer) {
                        renderer.canvas2d.width = targetWidth;
                        renderer.canvas2d.height = targetHeight;
                    },
                    visitWebGLRenderer: function (renderer) {
                        renderer.canvas3d.width = targetWidth;
                        renderer.canvas3d.height = targetHeight;
                    }
                });

                var instance = this.currentInstance().read();
                var instanceIndex = this.selectedInstanceIndex.read();

                this.renderer.act({
                    visitCanvasRenderer: function (renderer) {
                        renderer.context.fillStyle = "#000000";
                        renderer.context.fillRect(0, 0, renderer.context.canvas.width, renderer.context.canvas.height);
                    },
                    visitWebGLRenderer: function (renderer) {
                        renderer.gl.clearColor(0, 0, 0, 0);
                        renderer.gl.clear(renderer.gl.COLOR_BUFFER_BIT);
                    }
                });

                var context = this.canvas.getContext("2d");

                if (this.series) {
                    if (this.currentImageIsLoaded()) {
                        this.renderer.act({
                            visitSimpleRenderer: function (renderer) {
                                renderer.imageElements.each(function (imageElement) {
                                    var matrix = _this.getImageTransformation(_this.canvas.width, _this.canvas.height, instance.instanceAttributes.columns, instance.instanceAttributes.rows);

                                    var $el = $(imageElement.imageElement);

                                    $el.css("transform", "matrix({0},{1},{3},{4},{2},{5})".replace("{0}", matrix.entries[0].toString()).replace("{1}", matrix.entries[1].toString()).replace("{2}", matrix.entries[2].toString()).replace("{3}", matrix.entries[3].toString()).replace("{4}", matrix.entries[4].toString()).replace("{5}", matrix.entries[5].toString()));
                                });

                                var instanceKey = _this.getInstanceKey(instanceIndex);
                                var imageElement = renderer.imageElements.get(instanceKey, null);

                                if (imageElement) {
                                    $(imageElement.imageElement).show();
                                }
                            },
                            visitCanvasRenderer: function (renderer) {
                                var el = $(renderer.canvas2d);

                                var imageType = _this.getImageType(instance);
                                var instanceKey = _this.getInstanceKey(instanceIndex);
                                var image = renderer.imageElements.get(instanceKey, imageType);

                                if (image && image.imageElement.complete && typeof image.imageElement.naturalWidth !== "undefined" && image.imageElement.naturalWidth !== 0) {
                                    if (image.imageData == null) {
                                        renderer.imageElements.put(instanceKey, imageType, {
                                            imageElement: image.imageElement,
                                            imageData: Images.getImageData(image.imageElement),
                                            imageDownloadState: image.imageDownloadState
                                        });
                                        image = renderer.imageElements.get(instanceKey, imageType);
                                    }
                                    _this.renderUsingCanvas(el, renderer.context, image, instance);
                                }
                            },
                            visitWebGLRenderer: function (renderer) {
                                var transform = _this.getImageTransformation(context.canvas.width, context.canvas.height, instance.instanceAttributes.columns, instance.instanceAttributes.rows);

                                var imageType = _this.getImageType(instance);
                                var instanceKey = _this.getInstanceKey(instanceIndex);
                                var image = renderer.imageElements.get(instanceKey, imageType);

                                if (image && image.imageElement.complete && typeof image.imageElement.naturalWidth !== "undefined" && image.imageElement.naturalWidth !== 0) {
                                    if (image.imageData == null) {
                                        renderer.imageElements.put(instanceKey, imageType, {
                                            imageElement: image.imageElement,
                                            imageData: Images.getImageData(image.imageElement),
                                            imageDownloadState: image.imageDownloadState
                                        });
                                        image = renderer.imageElements.get(instanceKey, imageType);
                                    }
                                    _this.renderUsingWebGL(renderer.gl, renderer.program, image, instance, transform);
                                }
                            }
                        });

                        // Optional rendering step
                        if (renderImage) {
                            renderImage(context, instance);
                        }
                    } else {
                        var imageType = this.getImageType(instance);

                        this.preload(instanceIndex, imageType).subscribe({
                            done: function () {
                            },
                            next: function (_) {
                            },
                            fail: function (err) {
                                _this.recordError("Unable to preload image: " + err);
                            }
                        });

                        Rendering.writeLineCentered(context, "Image Loading", this.canvas.width / 2.0, this.canvas.height / 2.0, 12.0);
                    }
                }

                // Render layers on top
                this.renderLayers(clearOverlayLayer, layers, forceSquare);
            }
        };

        /**
        * Render just the overlay layers
        */
        Series.prototype.renderLayers = function (clearOverlayLayer, layers, hideBorder) {
            var _this = this;
            if (typeof clearOverlayLayer === "undefined") { clearOverlayLayer = true; }
            if (typeof hideBorder === "undefined") { hideBorder = false; }
            var context = this.canvas.getContext("2d");

            if (clearOverlayLayer) {
                context.clearRect(0, 0, context.canvas.width, context.canvas.height);
            }

            if (this.series && this.series.instances) {
                var instance = this.currentInstance().read();
                var instanceIndex = this.selectedInstanceIndex.read();
                var instanceKey = this.getInstanceKey(instanceIndex);

                if (layers) {
                    _.each(layers, function (layer) {
                        layer.render(context, instanceKey, instance, null, _this.series);
                    });
                } else {
                    _.each(this.layers, function (layer) {
                        if (!hideBorder || !(layer instanceof Layers.Border)) {
                            layer.render(context, instanceKey, instance, null, _this.series);
                        }
                    });
                }

                if ((instance.instanceAttributes.calibration && instance.instanceAttributes.calibration.calibrationValue) || instance.instanceAttributes.calibrationUser) {
                    if (!this.showingBanner) {
                        this.showingBanner = true;
                        this.application.showWarningBanner(this.terminology.lookup(Terminology.Terms.WarningCalibrationUsed));
                    }
                } else {
                    if (this.showingBanner) {
                        this.showingBanner = false;
                        this.application.hideWarningBanner();
                    }
                }
            }
        };

        /**
        * Render just the progress bar
        */
        Series.prototype.renderProgressBar = function (clearOverlayLayer) {
            if (typeof clearOverlayLayer === "undefined") { clearOverlayLayer = true; }
            if (this.progressBarCanvas != null) {
                var context = this.progressBarCanvas.getContext("2d");

                context.canvas.height = 15;
                context.canvas.width = this.el.width();

                var instance = this.currentInstance().read();
                var instanceIndex = this.selectedInstanceIndex.read();
                var instanceKey = this.getInstanceKey(instanceIndex);

                if (clearOverlayLayer) {
                    context.clearRect(0, 0, context.canvas.width, context.canvas.height);
                }

                this.progressBarLayer.render(context, instanceKey, instance);

                // Remove canvas and events when done loading
                var imageAttributesLoaded = instance.seriesAttributes.imageAttributesLoaded.read();
                var imageDataLoaded = instance.seriesAttributes.imageDataLoaded.read();
                if (imageAttributesLoaded == instance.seriesAttributes.instanceCount && imageDataLoaded == instance.seriesAttributes.instanceCount) {
                    // Discard the canvas
                    $(this.progressBarCanvas).remove();
                    this.progressBarCanvas = null;
                }
            }
        };

        /**
        * Get the image size, accounting for the double-height property of 16-bit images
        */
        Series.prototype.getActualImageSize = function (imageData, instance) {
            if (WindowLevelPresets.shouldUse16BitWindowLevel(instance)) {
                return { width: imageData.width, height: imageData.height / 2 };
            }

            return { width: imageData.width, height: imageData.height };
        };

        /**
        * Render the image onto a Canvas
        */
        Series.prototype.renderUsingCanvas = function (el, context, image, instance) {
            var startRender = +new Date();

            var prepared = this.prepareImageData(context, image.imageData, instance);

            var transform = this.getImageTransformation(context.canvas.width, context.canvas.height, prepared.width, prepared.height);

            context.save();
            context.transform(transform.entries[0], transform.entries[3], transform.entries[1], transform.entries[4], transform.entries[2], transform.entries[5]);
            Images.drawImageData(context, prepared.imageData);
            context.restore();

            if (this.firstImageRenderTime == null) {
                clearInterval(this.initialRenderTimer);

                var accelerated = this.series && this.series.studyAttributes.accelerated;

                if (image && image.imageElement.complete && typeof image.imageElement.naturalWidth !== "undefined" && image.imageElement.naturalWidth !== 0) {
                    var elapsed = (+new Date()) - startRender;

                    var additionalMetrics = {
                        studyUID: this.series.studyAttributes.queryObject.studyUid,
                        accelerated: accelerated,
                        imageSize: image.imageElement.naturalWidth * image.imageElement.naturalHeight,
                        imageWidth: image.imageElement.naturalWidth,
                        imageHeight: image.imageElement.naturalHeight,
                        renderTime: elapsed
                    };

                    this.firstImageRenderTime = elapsed;

                    Main.Main.recordMetric('showFirstImage', additionalMetrics);
                }

                this.application.thumbnails.render();
            }
        };

        /**
        * Prepare an image for rendering
        */
        Series.prototype.prepareImageData = function (context, imageData, instance) {
            var imageType = this.getImageType(instance);

            var imageSize = this.getActualImageSize(imageData, instance);

            var filters = this.renderer.visit({
                visitCanvasRenderer: function (renderer) {
                    return renderer.filters;
                }
            }, null);

            if (filters === null) {
                filters = [
                    new Filters.Subtraction(this, this.subtractionActive),
                    new Filters.WindowLevel(this.combineWindowLevelFlags(), this.windowLevel, this.subtractionActive),
                    new Filters.Invert(this.invertActive),
                    new Filters.Colors(this.colorTable)
                ];
            }

            for (var i = 0; i < filters.length; i++) {
                imageData = filters[i].apply(context, instance, imageData);
            }

            return {
                imageData: imageData,
                width: imageSize.width,
                height: imageSize.height
            };
        };

        Series.prototype.getAndPrepareImageData = function (context, instance) {
            var imageType = this.getImageType(instance);
            var instanceIndex = this.selectedInstanceIndex.read();
            var instanceKey = this.getInstanceKey(instanceIndex);

            var imageData = this.renderer.visit({
                visitCanvasRenderer: function (renderer) {
                    var imageElement = renderer.imageElements.get(instanceKey, imageType);

                    if (imageElement) {
                        return Images.getImageData(imageElement.imageElement);
                    }

                    return null;
                },
                visitWebGLRenderer: function (renderer) {
                    var imageElement = renderer.imageElements.get(instanceKey, imageType);

                    if (imageElement) {
                        return Images.getImageData(imageElement.imageElement);
                    }

                    return null;
                }
            }, null);

            if (imageData) {
                return this.prepareImageData(context, imageData, instance);
            }

            return null;
        };

        /**
        * Combine the window level flags to find the correct window level
        */
        Series.prototype.combineWindowLevelFlags = function () {
            var _this = this;
            return Subjects.bind(this.useOriginalWindowLevel, function (useOriginalWindowLevel) {
                if (useOriginalWindowLevel) {
                    return Subjects.map(_this.selectedInstanceIndex, function (index) {
                        var instance = _this.series.instances[index];

                        return _this.getDefaultWindowLevel(instance);
                    });
                } else {
                    return _this.windowLevel;
                }
            }, function (_, windowLevel) {
                return windowLevel;
            });
        };

        /**
        * Render the image using WebGL
        */
        Series.prototype.renderUsingWebGL = function (gl, program, image, instance, transform) {
            var startRender = +new Date();

            var toImageCoords = Vectors.scaleNonUniformM(instance.instanceAttributes.columns, instance.instanceAttributes.rows);
            var fromScreenCoords = Vectors.scaleNonUniformM(1 / gl.canvas.width, 1 / gl.canvas.height);
            var toViewport = Vectors.composeM(Vectors.translate(-1.0, 1.0), Vectors.scaleNonUniformM(2.0, -2.0));

            var imageToScreen = Vectors.composeM(toViewport, Vectors.composeM(fromScreenCoords, Vectors.composeM(transform, toImageCoords)));

            gl.viewport(0, 0, gl.canvas.width, gl.canvas.height);

            // Only send the texture to the video card if it is not already there
            if (!this.glTextureIsLoaded) {
                gl.texImage2D(gl.TEXTURE_2D, 0, gl.RGB, gl.RGB, gl.UNSIGNED_BYTE, image.imageElement);
                this.glTextureIsLoaded = true;
            }

            if (WindowLevelPresets.instanceNeedsWindowLevel(instance)) {
                instance.instanceAttributes.windowLevelDetection = true;
                var wl = WindowLevelPresets.findWindowLevels(instance, image.imageData);
                instance.instanceAttributes.windowCenter = [wl.center];
                instance.instanceAttributes.windowWidth = [wl.width];
                this.windowLevel.write(wl);
            }

            var windowLevel = this.combineWindowLevelFlags().read();
            var inverted = this.invertActive.read();
            var enhance = this.enhance.read();
            var bitDepth = WindowLevelPresets.shouldUse16BitWindowLevel(instance) ? 16 : 8;
            var rescaleIntercept = (bitDepth > 8) ? ((instance.instanceAttributes.rescaleIntercept || 0) / 65536) : 0;
            var signed = instance.instanceAttributes.signed || false;
            var slope = instance.instanceAttributes.rescaleSlope || 1;

            if (instance.instanceAttributes.presentationLUTShape === "INVERSE" || instance.instanceAttributes.photometricInterpretation === "MONOCHROME1") {
                windowLevel = {
                    center: (signed ? 0 : (1 << instance.instanceAttributes.bitsStored)) * slope - windowLevel.center,
                    width: windowLevel.width
                };
            }

            var centerLocation = gl.getUniformLocation(program, "uCenter");
            gl.uniform1f(centerLocation, windowLevel.center / 65536);

            var widthLocation = gl.getUniformLocation(program, "uWidth");
            gl.uniform1f(widthLocation, windowLevel.width / 65536);

            var signedLocation = gl.getUniformLocation(program, "uSigned");
            gl.uniform1i(signedLocation, signed ? 1 : 0);

            var sigmoidLocation = gl.getUniformLocation(program, "uSigmoid");
            gl.uniform1i(sigmoidLocation, instance.instanceAttributes.voiLutFunction === "SIGMOID" ? 1 : 0);

            var invertedLocation = gl.getUniformLocation(program, "uInverted");
            gl.uniform1i(invertedLocation, inverted ? 1 : 0);

            var enhanceLocation = gl.getUniformLocation(program, "uEnhance");
            gl.uniform1i(enhanceLocation, enhance ? 1 : 0);

            var slopeLocation = gl.getUniformLocation(program, "uSlope");
            gl.uniform1f(slopeLocation, slope);

            var interceptLocation = gl.getUniformLocation(program, "uIntercept");
            gl.uniform1f(interceptLocation, rescaleIntercept);

            var depthLocation = gl.getUniformLocation(program, "uDepth");
            gl.uniform1i(depthLocation, bitDepth);

            var transformLocation = gl.getUniformLocation(program, "uTrans");
            gl.uniformMatrix3fv(transformLocation, false, [
                imageToScreen.entries[0], imageToScreen.entries[1], 0,
                imageToScreen.entries[3], imageToScreen.entries[4], 0,
                imageToScreen.entries[2], imageToScreen.entries[5], 1]);

            var colorTableLocation = gl.getUniformLocation(program, "uColorTable");

            if (this.colorTable.read()) {
                var paletteLoc = gl.getUniformLocation(program, "uPalette");
                gl.uniform1i(paletteLoc, 1);

                if (!this.glCreatedPaletteTexture) {
                    this.glCreatedPaletteTexture = true;
                    WebGL.createPaletteTexture(gl);
                }

                gl.uniform1i(colorTableLocation, 1);

                if (!this.glTexturePaletteIsLoaded) {
                    this.glTexturePaletteIsLoaded = true;
                    var palette = new Uint8Array(ColorTablePresets.createPalette(this.colorTable.read()));
                    gl.activeTexture(gl.TEXTURE1);
                    gl.texImage2D(gl.TEXTURE_2D, 0, gl.RGBA, 256, 1, 0, gl.RGBA, gl.UNSIGNED_BYTE, palette);
                    gl.activeTexture(gl.TEXTURE0);
                }
            } else {
                gl.uniform1i(colorTableLocation, 0);
            }

            gl.drawArrays(gl.TRIANGLES, 0, 6);

            if (this.firstImageRenderTime == null) {
                clearInterval(this.initialRenderTimer);

                var accelerated = this.series && this.series.studyAttributes.accelerated;

                if (image && image.imageElement.complete && typeof image.imageElement.naturalWidth !== "undefined" && image.imageElement.naturalWidth !== 0) {
                    var elapsed = (+new Date()) - startRender;

                    var additionalMetrics = {
                        studyUID: this.series.studyAttributes.queryObject.studyUid,
                        accelerated: accelerated,
                        imageSize: image.imageElement.naturalWidth * image.imageElement.naturalHeight,
                        imageWidth: image.imageElement.naturalWidth,
                        imageHeight: image.imageElement.naturalHeight,
                        renderTime: elapsed
                    };

                    this.firstImageRenderTime = elapsed;

                    Main.Main.recordMetric('showFirstImage', additionalMetrics);
                }
            }
        };

        /**
        * Check if the current image is loaded
        */
        Series.prototype.currentImageIsLoaded = function () {
            var _this = this;
            var index = this.selectedInstanceIndex.read();
            var instance = this.series.instances[index];
            var instanceKey = this.getInstanceKey(index);

            return this.renderer.visit({
                visitSimpleRenderer: function (renderer) {
                    var imageElement = renderer.imageElements.get(instanceKey, null);

                    if (imageElement) {
                        return imageElement.imageElement.complete && instance.instanceAttributes.attributesLoaded;
                    }

                    return false;
                },
                visitCanvasRenderer: function (renderer) {
                    var imageType = _this.getImageType(instance);
                    var imageElement = renderer.imageElements.get(instanceKey, imageType);

                    if (imageElement) {
                        return imageElement.imageElement.complete && instance.instanceAttributes.attributesLoaded && typeof imageElement.imageElement.naturalWidth !== "undefined" && imageElement.imageElement.naturalWidth !== 0;
                    }
                },
                visitWebGLRenderer: function (renderer) {
                    var imageType = _this.getImageType(instance);
                    var imageElement = renderer.imageElements.get(instanceKey, imageType);

                    if (imageElement) {
                        return imageElement.imageElement.complete && instance.instanceAttributes.attributesLoaded && typeof imageElement.imageElement.naturalWidth !== "undefined" && imageElement.imageElement.naturalWidth !== 0;
                    }

                    return false;
                }
            }, false);
        };

        /**
        * Return the key used to find an instance's image data in the store
        */
        Series.prototype.getInstanceKey = function (index) {
            index = (index !== undefined) ? index : this.selectedInstanceIndex.read();
            var key = '';

            if (this.series && this.series.instances) {
                if (this.series.instances.length > index) {
                    var instance = this.series.instances[index];
                    key = instance.id.value + ':' + instance.frameNumber.value;
                }
            }

            return key;
        };

        /**
        * Check if the current image is loaded
        */
        Series.prototype.getImageState = function (index) {
            var _this = this;
            var index = index || this.selectedInstanceIndex.read();
            var instance = this.series.instances[index];
            var instanceKey = this.getInstanceKey(index);

            if (this.renderer == null) {
                return 0 /* None */;
            }

            return this.renderer.visit({
                visitSimpleRenderer: function (renderer) {
                    var imageElement = renderer.imageElements.get(instanceKey, null);

                    if (imageElement && imageElement.imageElement && imageElement.imageElement.complete) {
                        return 3 /* Success */;
                    }

                    return 0 /* None */;
                },
                visitCanvasRenderer: function (renderer) {
                    var imageType = _this.getImageType(instance);
                    var imageElement = renderer.imageElements.get(instanceKey, imageType);

                    if (imageElement) {
                        return imageElement.imageDownloadState;
                    }

                    return 0 /* None */;
                },
                visitWebGLRenderer: function (renderer) {
                    var imageType = _this.getImageType(instance);
                    var imageElement = renderer.imageElements.get(instanceKey, imageType);

                    if (imageElement) {
                        return imageElement.imageDownloadState;
                    }

                    return 0 /* None */;
                }
            }, 0 /* None */);
        };

        /**
        * Determine an appropriate image type to render
        */
        Series.prototype.getImageType = function (instance) {
            if (LocalViewer.isFileSystemViewer()) {
                return 1 /* FullResolution */;
            } else if (LocalViewer.isNodeWebkitViewer() || LocalViewer.isPersonalAccelerator()) {
                return 2 /* Diagnostic */;
            } else if (this.useDiagnosticQualityAlways) {
                return 2 /* Diagnostic */;
            } else if (this.forceLowResolution.read().used() && (!instance.instanceAttributes.mostRecentThumbnailLoadFailed || instance.instanceAttributes.mostRecentHiResLoadFailed)) {
                return 0 /* Thumbnail */;
            } else {
                var instanceIndex = this.selectedInstanceIndex.read();
                var instanceKey = this.getInstanceKey(instanceIndex);
                var useDiagnostic = this.accountSettings.viewer_diagnostic_quality == 1 || WindowLevelPresets.shouldUse16BitWindowLevel(instance);

                var hiResImageType = useDiagnostic ? 2 /* Diagnostic */ : 1 /* FullResolution */;

                if (WindowLevelPresets.isFullResolutionHD(instance)) {
                    hiResImageType = 3 /* FullResolutionHD */;
                }

                return this.renderer.visit({
                    visitCanvasRenderer: function (renderer) {
                        var imageElement = renderer.imageElements.get(instanceKey, hiResImageType);

                        if (imageElement && imageElement.imageElement.complete && typeof imageElement.imageElement.naturalWidth !== "undefined" && imageElement.imageElement.naturalWidth !== 0) {
                            return hiResImageType;
                        } else {
                            return 0 /* Thumbnail */;
                        }
                    },
                    visitWebGLRenderer: function (renderer) {
                        var imageElement = renderer.imageElements.get(instanceKey, hiResImageType);

                        if (imageElement && imageElement.imageElement.complete && typeof imageElement.imageElement.naturalWidth !== "undefined" && imageElement.imageElement.naturalWidth !== 0) {
                            return hiResImageType;
                        } else {
                            return 0 /* Thumbnail */;
                        }
                    }
                }, 1 /* FullResolution */);
            }
        };

        Series.prototype.hasSelectedMeasurement = function () {
            var instanceIndex = this.selectedInstanceIndex.read();
            var instance = this.series.instances[instanceIndex];
            var measurements = instance.instanceAttributes.measurements;
            var index = measurements.indexWhere(function (measurement) {
                return measurement.selected;
            });
            return (index >= 0);
        };

        /**
        * Delete the selected measurement
        */
        Series.prototype.deleteSelectedMeasurement = function () {
            var _this = this;
            var instanceIndex = this.selectedInstanceIndex.read();
            var instance = this.series.instances[instanceIndex];
            var measurements = instance.instanceAttributes.measurements;
            var index = measurements.indexWhere(function (measurement) {
                return measurement.selected;
            });

            if (index >= 0) {
                var measurement = measurements[index];
                if (measurement.editable) {
                    if (measurement == instance.instanceAttributes.calibration) {
                        instance.instanceAttributes.calibration = null;
                    }

                    if (measurement.volume) {
                        _.each(measurement.volume.slices, function (s) {
                            s.measurement.volume = null;
                        });
                    }

                    if (measurement.propagation) {
                        for (var ctr = 0; ctr < this.series.instances.length; ctr += 1) {
                            var currentMeasurements = this.series.instances[ctr].instanceAttributes.measurements;
                            var currentIndex = _.indexOf(currentMeasurements, measurement);
                            if (currentIndex >= 0) {
                                currentMeasurements.splice(currentIndex, 1);
                            }
                        }
                    } else {
                        measurements.splice(index, 1);
                    }

                    this.selectedMeasurement.write(null);

                    if (measurement.id) {
                        Services.deleteImageAnnotation(this.sessionId, this.series.studyAttributes.queryObject, measurement.id).subscribe({
                            done: function () {
                            },
                            next: function (_) {
                            },
                            fail: function (err) {
                                _this.recordError("Unable to delete annotation: " + err);
                            }
                        });
                    }

                    instance.instanceAttributes.calibrationUser = !!_.find(measurements, function (m) {
                        return !!m.pixelSpacingUser || !!m.sliceSpacingUser;
                    });

                    this.renderAll();
                }
            }
        };

        /**
        * Deselect all measurements
        */
        Series.prototype.clearSelection = function () {
            this.selectedEndpoint = null;
            var instanceIndex = this.selectedInstanceIndex.read();
            var measurements = this.series.instances[instanceIndex].instanceAttributes.measurements;
            _.each(measurements, function (measurement) {
                measurement.selected = false;
                if (measurement.volume) {
                    measurement.volume.selected = false;
                }
            });
            this.selectedMeasurement.write(null);
        };

        /**
        * Save the new cine speed back to the settings object
        */
        Series.prototype.saveCineSpeedToSettings = function () {
            var _this = this;
            if (this.series) {
                Subjects.modify(this.settings, function (settings) {
                    return Cine.saveCineSpeedToSettings(settings, _this.series.seriesAttributes.modality, _this.cineSpeed.read());
                });
            }
        };

        /**
        * Cancel all existing subscriptions
        */
        Series.prototype.unload = function () {
            _.each(this.subscriptions, function (s) {
                return s.cancel();
            });

            if (this.handle) {
                this.handle.clear();
                this.handle = null;
            }
        };

        /**
        * Get the default window level for this series
        */
        Series.prototype.getDefaultWindowLevel = function (instance) {
            if (!instance && this.series) {
                instance = this.series.instances[0];
            }

            if (instance) {
                if (this.subtractionActive.read()) {
                    return this.subtraction.windowLevel || this.subtractionWindowLevel;
                } else if (WindowLevelPresets.shouldUse16BitWindowLevel(instance)) {
                    var windowWidth = instance.instanceAttributes.windowWidth;
                    var windowCenter = instance.instanceAttributes.windowCenter;

                    if (windowWidth && windowWidth.length && windowCenter && windowCenter.length) {
                        return { center: windowCenter[0], width: windowWidth[0] };
                    }
                }
            }

            return { center: WindowLevelPresets.defaultCenter, width: WindowLevelPresets.defaultWidth };
        };

        /**
        * Get the current display options
        */
        Series.prototype.getDisplayOptions = function () {
            var transform = this.transform.read();
            var windowLevel = this.windowLevel.read();

            return {
                flipped: transform.flipped,
                rotation: transform.rotation,
                invert: this.invertActive.read(),
                windowLevel: this.useOriginalWindowLevel.read() ? Maybe.Nothing() : Maybe.Just({
                    windowCenter: windowLevel.center,
                    windowWidth: windowLevel.width
                }),
                showMeasurements: this.measurementsVisible.read(),
                showTextAnnotations: this.infoVisible.read()
            };
        };

        Series.prototype.getExtendedDisplayOptions = function () {
            var transform = this.transform.read();
            var displayOptions = this.getDisplayOptions();
            displayOptions.scale = transform.scale;
            displayOptions.offsetX = transform.offsetX;
            displayOptions.offsetY = transform.offsetY;
            displayOptions.hideThumbnails = !this.application.thumbnailsVisible.read();
            return displayOptions;
        };

        /**
        * Apply display options from a hanging protocol
        */
        Series.prototype.applyDisplayOptions = function (displayOptions) {
            if (displayOptions) {
                var extendedOptions = displayOptions;

                var transform = this.transform.read();
                transform.flipped = displayOptions.flipped;
                transform.rotation = displayOptions.rotation;

                if (extendedOptions.scale) {
                    transform.scale = extendedOptions.scale;
                }

                if (extendedOptions.offsetX) {
                    transform.offsetX = extendedOptions.offsetX;
                }

                if (extendedOptions.offsetY) {
                    transform.offsetY = extendedOptions.offsetY;
                }

                this.transform.write(transform);

                this.transformHP = new Subjects.ObservableValue({ offsetX: 0, offsetY: 0, scale: 1, flipped: false, rotation: 0 });
                this.transformHP.write(transform);

                if (displayOptions.invert) {
                    this.invertActive.write(true);
                }

                if (!displayOptions.showTextAnnotations) {
                    this.infoVisible.write(false);
                }

                if (!displayOptions.showMeasurements) {
                    this.measurementsVisible.write(false);
                }

                if (extendedOptions.hideThumbnails) {
                    this.application.thumbnailsVisible.write(false);
                }

                if (displayOptions.windowLevel) {
                    var defaultWindowLevel = this.getDefaultWindowLevel();

                    var windowLevel = Maybe.maybe(displayOptions.windowLevel, defaultWindowLevel, function (wl) {
                        return {
                            center: wl.windowCenter,
                            width: wl.windowWidth
                        };
                    });

                    if (Maybe.hasValue(displayOptions.windowLevel)) {
                        this.useOriginalWindowLevel.write(false);
                    }

                    this.windowLevel.write(windowLevel);
                }
            } else {
                var defaultWindowLevel = this.getDefaultWindowLevel();

                this.windowLevel.write(defaultWindowLevel);
            }

            if (this.series) {
                var modality = this.series.seriesAttributes.modality;

                var settings = this.settings.read();

                if (settings.modalities) {
                    var modalitySettings = _.find(settings.modalities, function (m) {
                        return m.modality === modality;
                    });

                    if (modalitySettings && modalitySettings.displayMiddleSlice === true) {
                        this.selectedInstanceIndex.write(Math.floor(this.series.instances.length / 2));
                    }
                }
            }
        };

        /**
        * Reload all frames
        */
        Series.prototype.reloadAllFrames = function () {
            var preloadRange = this.accountSettings.image_preload_window || this.settings.read().imagePreloadWindow || 10;

            this.preloadImageRange(this.imagePreloadQueue, preloadRange, 0, preloadRange);
            this.preloadVisibleHiResImages(this.imagePreloadQueue, 99 /* FullResolutionImage */);

            this.renderAll();
        };

        /**
        * Load all image attributes with low priority
        * Necessary for series linking and text annotations to show up
        */
        Series.prototype.loadImageAttributes = function (queue, priority) {
            var _this = this;
            if (this.series) {
                var onLoad = Multiframe.shouldSplitInstances(this.series.seriesAttributes.modality) ? function () {
                    _this.application.thumbnails.render();
                } : function () {
                };

                _.each(this.series.instances, function (instance, index, list) {
                    var key = {
                        type: 0 /* ImageAttributes */,
                        seriesUid: _this.series.seriesAttributes.seriesUid,
                        instanceUid: instance.id,
                        frameNumber: instance.frameNumber
                    };

                    var loadAttr = Observable._finally(Study.loadImageAttributes(_this.sessionId, instance, _this), onLoad);
                    queue.enqueue(key, loadAttr, priority, new PreloadQueueKeyIsKey(), new PriorityIsPriority());

                    var isMultiframe = _.any(_this.series.instances, function (instance) {
                        return instance.instanceAttributes.frameCount > 1;
                    });
                    var settings = _this.settings.read();

                    if (!isMultiframe && (settings && settings.ultrasoundMeasurements && Dicom.Ultrasound.isUltrasoundSeries(_this.series.seriesAttributes))) {
                        var keyMetadata = {
                            type: 2 /* ImageMetadata */,
                            seriesUid: _this.series.seriesAttributes.seriesUid,
                            instanceUid: instance.id,
                            frameNumber: instance.frameNumber
                        };

                        queue.enqueue(keyMetadata, _this.loadImageJSON(instance), priority, new PreloadQueueKeyIsKey(), new PriorityIsPriority());
                    }
                });
            }
        };

        /**
        * Load all image thumbnail data with low priority
        */
        Series.prototype.loadAllImageData = function (queue, priority) {
            if (this.series) {
                this.preloadImageRange(queue, this.series.instances.length, priority, priority);
            }
        };

        /**
        * Preload image attributes and image data for instances or frames in the specified range.
        *
        * @param {queue} The queue to use for preloading
        * @param {window} The number of instances to load either side of the current instance
        * @param {minPriority} The priority to use for the most distant instance
        * @param {maxPriority} The priority to use for the current instance
        */
        Series.prototype.preloadImageRange = function (queue, window, minPriority, maxPriority) {
            var _this = this;
            var accelerated = this.series && this.series.studyAttributes.accelerated;

            var imageType = (accelerated || this.useDiagnosticQualityAlways) ? 2 /* Diagnostic */ : 0 /* Thumbnail */;

            if (this.series && this.renderer.visit({
                visitSimpleRenderer: function (renderer) {
                    return false;
                }
            }, true)) {
                var selectedInstanceIndex = this.selectedInstanceIndex.read();

                // Make sure each image is only loaded once.
                // Sort indices by their distance from the loaded slice, and keep only unique indices.
                var indices = _.uniq(_.sortBy(_.map(_.range(-window, window), function (offset) {
                    var index = (offset + selectedInstanceIndex) % _this.series.instances.length;

                    if (index < 0) {
                        index += _this.series.instances.length;
                    }

                    return [offset, index];
                }), function (offset) {
                    return Math.abs(offset[0]);
                }), false, function (offset) {
                    return offset[1];
                });

                _.each(indices, function (offset) {
                    var instance = _this.series.instances[offset[1]];
                    var priority = Math.round(maxPriority - (maxPriority - minPriority) * Math.abs(offset[0]) / window);
                    var key = {
                        type: 1 /* ImageData */,
                        seriesUid: _this.series.seriesAttributes.seriesUid,
                        instanceUid: instance.id,
                        frameNumber: instance.frameNumber,
                        imageType: 0 /* Thumbnail */
                    };
                    queue.enqueue(key, _this.preload(offset[1], imageType), priority, new PreloadQueueKeyIsKey(), new PriorityIsPriority());
                });
            }
        };

        /**
        * For the local viewer
        */
        Series.prototype.preloadAllDiagnosticImages = function (queue, priority) {
            var _this = this;
            if (this.series && this.renderer.visit({
                visitSimpleRenderer: function (renderer) {
                    return false;
                }
            }, true)) {
                _.each(this.series.instances, function (instance, index) {
                    var key = {
                        type: 1 /* ImageData */,
                        seriesUid: _this.series.seriesAttributes.seriesUid,
                        instanceUid: instance.id,
                        frameNumber: instance.frameNumber,
                        imageType: 2 /* Diagnostic */
                    };

                    // order series loading, first to last
                    var priorityMod = 0;
                    if (_this.application.supportsAcceleratedPreload()) {
                        var study = _.find(_this.application.studies, function (study) {
                            return study.studyAttributes.queryObject.toString() === _this.series.studyAttributes.queryObject.toString();
                        });
                        if (study) {
                            var seriesIndex = _.indexOf(study.series, _this.series);
                            if (seriesIndex > 0) {
                                priorityMod = seriesIndex;
                            }
                        }
                    }

                    queue.enqueue(key, _this.preload(index, 2 /* Diagnostic */), priority - priorityMod, new PreloadQueueKeyIsKey(), new PriorityIsPriority());
                });
            }
        };

        /**
        * Load the high resolution image for the visible frames of the visible instances
        */
        Series.prototype.preloadVisibleHiResImages = function (queue, priority) {
            if (this.series && !LocalViewer.isLocalViewer() && this.renderer && this.renderer.visit({
                visitSimpleRenderer: function (renderer) {
                    return false;
                }
            }, true)) {
                var selectedInstanceIndex = this.selectedInstanceIndex.read();
                var instance = this.series.instances[selectedInstanceIndex];

                var useDiagnostic = this.useDiagnosticQualityAlways || WindowLevelPresets.shouldUse16BitWindowLevel(instance);

                var hiResImageType = useDiagnostic ? 2 /* Diagnostic */ : 1 /* FullResolution */;

                if (WindowLevelPresets.isFullResolutionHD(instance)) {
                    hiResImageType = 3 /* FullResolutionHD */;
                }

                var key = {
                    type: 1 /* ImageData */,
                    seriesUid: this.series.seriesAttributes.seriesUid,
                    instanceUid: instance.id,
                    frameNumber: instance.frameNumber,
                    imageType: hiResImageType
                };

                queue.enqueue(key, this.preload(selectedInstanceIndex, hiResImageType), priority, new PreloadQueueKeyIsKey(), new PriorityIsPriority());
            }
        };

        /**
        * Preload the image attributes and image data for the specified image type
        */
        Series.prototype.preload = function (instanceIndex, imageType) {
            var _this = this;
            if (this.series) {
                var instance = this.series.instances[instanceIndex];

                var loadMetadata = Study.loadImageAttributes(this.sessionId, instance);

                return Observable.bind2(loadMetadata, function (_) {
                    var loadImageData = Observable.on(Observable.ifThenElse(function () {
                        if (imageType == 0 /* Thumbnail */) {
                            return !instance.instanceAttributes.mostRecentThumbnailLoadFailed;
                        } else {
                            return !instance.instanceAttributes.mostRecentHiResLoadFailed;
                        }
                    }, _this.preloadImage(instanceIndex, imageType), Observable.ret(false)), function (success) {
                        if (success) {
                            if (_this.useDiagnosticQualityAlways) {
                                if (imageType !== 0 /* Thumbnail */) {
                                    Subjects.modify(instance.seriesAttributes.imageDataLoaded, function (n) {
                                        return n + 1;
                                    });
                                }
                            } else {
                                if (imageType === 0 /* Thumbnail */) {
                                    Subjects.modify(instance.seriesAttributes.imageDataLoaded, function (n) {
                                        return n + 1;
                                    });
                                }
                            }

                            if (imageType === 0 /* Thumbnail */) {
                                instance.instanceAttributes.mostRecentThumbnailLoadFailed = false;
                            } else {
                                instance.instanceAttributes.mostRecentHiResLoadFailed = false;
                            }

                            // Don't render if this is a preload
                            if (instanceIndex == _this.selectedInstanceIndex.read()) {
                                _this.renderAll();
                            }
                        }
                    }, function (err) {
                        if (imageType === 0 /* Thumbnail */) {
                            instance.instanceAttributes.mostRecentThumbnailLoadFailed = true;
                        } else {
                            instance.instanceAttributes.mostRecentHiResLoadFailed = true;
                        }

                        if (!err) {
                            err = "Image GET failed (no error): " + instance.id.value;
                        }

                        _this.recordError(err);
                        _this.renderAll();
                    }, function () {
                    });

                    var loadOverlay = Observable.forget(_this.loadOverlays(instanceIndex));

                    if (!Multiframe.isMultiframe(_this.series) && _this.settings.read().ultrasoundMeasurements && Dicom.Ultrasound.isUltrasoundSeries(_this.series.seriesAttributes) && !instance.instanceAttributes.json) {
                        _this.loadImageJSON(instance).subscribe({
                            done: function () {
                            },
                            next: function (_) {
                            },
                            fail: function (err) {
                                _this.recordError("Unable to load image metadata: " + err);
                            } });
                    }

                    return Observable.zip(loadImageData, loadOverlay, function (_1, _2) {
                        return _1;
                    });
                }, function (_1, _2) {
                    return _2;
                });
            }

            return Observable.ret({});
        };

        Series.prototype.loadImageJSON = function (instance) {
            return Observable.invoke(Study.loadImageJSON(this.sessionId, instance, this.series, false), function (json) {
                instance.instanceAttributes.json = json;
                _.each(instance.instanceAttributes.measurements, function (measurement) {
                    var result = Dicom.Ultrasound.findSpacing(measurement, instance);
                    if (result && result.valid) {
                        measurement.ultrasoundPixelSpacing = [result.spacingX, result.spacingY];
                    }
                });
            });
        };

        /**
        * Load overlay data
        */
        Series.prototype.loadOverlays = function (instanceIndex) {
            var futures = [];
            var instance = this.series.instances[instanceIndex];

            if (instance.instanceAttributes.attributesLoaded) {
                for (var ctr = 0; ctr < instance.instanceAttributes.containsOverlayDataPlane.length; ctr += 1) {
                    if (instance.instanceAttributes.containsOverlayDataPlane[ctr]) {
                        var loadOverlay = this.loadOverlayPlane(instanceIndex, ctr);

                        if (loadOverlay) {
                            futures.push(loadOverlay);
                        }
                    }
                }
            }

            return Observable.sequenceA(futures);
        };

        Series.prototype.loadOverlayPlane = function (instanceIndex, overlayPlane) {
            var _this = this;
            var overlayPlaneKey = (this.getInstanceKey(instanceIndex) + '-' + overlayPlane);
            var roiOverlay = this.settings.read().supportROIOverlays;

            return Observable.ifThenElse(function () {
                return _this.overlayData.get(overlayPlaneKey, null) === null;
            }, Observable.forget(Observable.invoke(Observable.catchError(this.preloadOverlayImage(instanceIndex, overlayPlane), function (err) {
                _this.recordError("Cannot load overlay: " + err);
                return null;
            }), function (data) {
                if (data) {
                    if (roiOverlay) {
                        var rgb = Annotations.fromHex(Annotations.OVERLY_COLORS[Math.min(1, overlayPlane)]);

                        for (var i = 0; i < data.data.length; i += 4) {
                            if (data.data[i]) {
                                data.data[i] = rgb.r;
                                data.data[i + 1] = rgb.g;
                                data.data[i + 2] = rgb.b;
                                data.data[i + 3] = overlayPlane ? 255 : 128;
                            } else {
                                data.data[i + 3] = data.data[i];
                            }
                        }
                    } else {
                        for (var i = 0; i < data.data.length; i += 4) {
                            data.data[i + 3] = data.data[i];
                        }
                    }
                }

                _this.overlayData.put(overlayPlaneKey, data);

                if (instanceIndex == _this.selectedInstanceIndex.read()) {
                    if (_this.firstImageRenderTime != null) {
                        _this.renderAll();
                    }
                }
            })), Observable.ret({}));
        };

        Series.prototype.preloadOverlayImage = function (instanceIndex, overlayPlane) {
            var _this = this;
            var instance = this.series.instances[instanceIndex];
            var overlayPlaneKey = (this.getInstanceKey(instanceIndex) + '-' + overlayPlane);

            return this.renderer.visit({
                visitSimpleRenderer: function (_) {
                    var uri = Routes.ImageData(_this.sessionId, instance.studyAttributes.studyStorage, _this.series.studyAttributes.queryObject, instance.id, instance.instanceAttributes.version, new Classes.FrameNumber(parseInt((0x6000 + (overlayPlane * 2)).toString(16) + "3000", 16)), 2 /* Diagnostic */, 8, _this.accountSettings.cache == 1, true);

                    return Images.loadImage(uri);
                },
                visitCanvasRenderer: function (renderer) {
                    var imageData = renderer.imageElements.get(overlayPlaneKey, 2 /* Diagnostic */);

                    if (!imageData) {
                        var image = new Image();
                        var state = 1 /* Queued */;

                        var cachedImage = {
                            imageElement: image,
                            imageData: null,
                            imageDownloadState: state
                        };

                        renderer.imageElements.put(overlayPlaneKey, 2 /* Diagnostic */, cachedImage);

                        var uri = Routes.ImageData(_this.sessionId, instance.studyAttributes.studyStorage, _this.series.studyAttributes.queryObject, instance.id, instance.instanceAttributes.version, new Classes.FrameNumber(parseInt((0x6000 + (overlayPlane * 2)).toString(16) + "3000", 16)), 2 /* Diagnostic */, 8, _this.accountSettings.cache == 1, true);

                        return Observable.map(Images.loadImageOnly(image, uri, cachedImage), function (_) {
                            cachedImage.imageData = Images.getImageData(image);
                            return cachedImage.imageData;
                        });
                    }

                    return Observable.ret(imageData.imageData);
                },
                visitWebGLRenderer: function (renderer) {
                    var imageData = renderer.imageElements.get(overlayPlaneKey, 2 /* Diagnostic */);

                    if (!imageData) {
                        var image = new Image();
                        var state = 1 /* Queued */;

                        var cachedImage = {
                            imageElement: image,
                            imageData: null,
                            imageDownloadState: state
                        };

                        renderer.imageElements.put(overlayPlaneKey, 2 /* Diagnostic */, cachedImage);

                        var uri = Routes.ImageData(_this.sessionId, instance.studyAttributes.studyStorage, _this.series.studyAttributes.queryObject, instance.id, instance.instanceAttributes.version, new Classes.FrameNumber(parseInt((0x6000 + (overlayPlane * 2)).toString(16) + "3000", 16)), 2 /* Diagnostic */, 8, _this.accountSettings.cache == 1, true);

                        return Observable.map(Images.loadImageOnly(image, uri, cachedImage), function (_) {
                            cachedImage.imageData = Images.getImageData(image);
                            return cachedImage.imageData;
                        });
                    }

                    return Observable.ret(imageData.imageData);
                }
            }, Observable.ret(null));
        };

        /**
        * Load image data with the specified image index
        */
        Series.prototype.preloadImage = function (instanceIndex, imageType) {
            var _this = this;
            var $instance = this.el.find('.instance');
            var instance = this.series.instances[instanceIndex];
            var instanceKey = this.getInstanceKey(instanceIndex);

            // Fail immediately if the image attributes are not available
            if (instance.instanceAttributes.attributesDownloadState != 3 /* Success */) {
                return Observable.ret(false);
            }

            return this.renderer.visit({
                visitSimpleRenderer: function (renderer) {
                    var imageData = renderer.imageElements.get(instanceKey, null);

                    if (!imageData) {
                        var image = new Image();

                        renderer.imageElements.put(instanceKey, {
                            imageElement: image
                        });

                        var $image = $(image);

                        $image.addClass('instance-image');
                        $image.appendTo($instance);

                        $image.hide();

                        var uri = Routes.ImageData(_this.sessionId, instance.studyAttributes.studyStorage, instance.studyAttributes.queryObject, instance.id, instance.instanceAttributes.version, instance.frameNumber, 1 /* FullResolution */, 8, _this.accountSettings.cache == 1, true);

                        return Observable.map(Images.loadImageOnly(image, uri), function (res) {
                            _this.handleImageLoadResponse(res);
                            return true;
                        });
                    }

                    return Observable.ret(false);
                },
                visitCanvasRenderer: function (renderer) {
                    var instance = _this.series.instances[instanceIndex];

                    var imageData = renderer.imageElements.get(instanceKey, imageType);

                    if (!imageData) {
                        var image = new Image();
                        var state = 1 /* Queued */;

                        var cachedImage = {
                            imageElement: image,
                            imageData: null,
                            imageDownloadState: state
                        };

                        renderer.imageElements.put(instanceKey, imageType, cachedImage);

                        var use16Bit = WindowLevelPresets.shouldUse16BitWindowLevel(instance);

                        var uri = Routes.ImageData(_this.sessionId, instance.studyAttributes.studyStorage, instance.studyAttributes.queryObject, instance.id, instance.instanceAttributes.version, instance.frameNumber, imageType, use16Bit ? 16 : 8, _this.accountSettings.cache == 1, true, (Browser.needsResize(instance) ? Browser.resize(instance) : 0));

                        var accelerated = (_this.series && _this.series.studyAttributes && _this.series.studyAttributes.accelerated) ? true : false;

                        return Observable.map(Images.loadImageOnly(image, uri, cachedImage, accelerated), function (res) {
                            _this.handleImageLoadResponse(res);
                            return true;
                        });
                    }

                    return Observable.ret(false);
                },
                visitWebGLRenderer: function (renderer) {
                    var imageData = renderer.imageElements.get(instanceKey, imageType);

                    if (!imageData) {
                        var image = new Image();
                        var state = 1 /* Queued */;

                        var cachedImage = {
                            imageElement: image,
                            imageData: null,
                            imageDownloadState: state
                        };

                        renderer.imageElements.put(instanceKey, imageType, cachedImage);

                        var use16Bit = WindowLevelPresets.shouldUse16BitWindowLevel(instance);

                        var uri = Routes.ImageData(_this.sessionId, instance.studyAttributes.studyStorage, instance.studyAttributes.queryObject, instance.id, instance.instanceAttributes.version, instance.frameNumber, imageType, use16Bit ? 16 : 8, _this.accountSettings.cache == 1, true);

                        var accelerated = (_this.series && _this.series.studyAttributes && _this.series.studyAttributes.accelerated) ? true : false;

                        return Observable.map(Images.loadImageOnly(image, uri, cachedImage, accelerated), function (res) {
                            _this.handleImageLoadResponse(res);
                            return true;
                        });
                    }

                    return Observable.ret(false);
                }
            }, Observable.ret(false));
        };

        Series.prototype.checkLoadedStatus = function () {
            var preloading = this.application.supportsAcceleratedPreload();
            var seriesLoaded = this.application.seriesLoaded(this.series);

            var loadedStatusStyle = preloading ? "" : "hide-icon";
            if (seriesLoaded == 3 /* Diagnostic */) {
                loadedStatusStyle = "loaded-hd-icon";
            } else if (seriesLoaded == 2 /* Thumbnail */) {
                loadedStatusStyle = "loaded-sd-icon";
            }

            if (preloading) {
                $("#series-" + this.series.uuid).find(".accelerated-icon").removeClass("hide-icon loaded-hd-icon loaded-sd-icon").addClass(loadedStatusStyle);
            } else {
                $("#series-" + this.series.uuid).find(".load-icon").removeClass("hide-icon loaded-hd-icon loaded-sd-icon").addClass(loadedStatusStyle);
            }
        };

        Series.prototype.handleImageLoadResponse = function (response) {
            if (response.acceleratorCached != null && response.acceleratorCached != '' && response.acceleratorCached != 'HIT') {
                this.acceleratorCacheMissCount = this.acceleratorCacheMissCount + 1;

                if (this.acceleratorCacheMissCount == 5 && !this.acceleratorCacheMessageDidShow) {
                    // Indicate problem on thumbnails area
                    $('.fa-flash').css('color', '#F33').attr('title', this.acceleratorCacheMissMessage);
                }
            }

            this.checkLoadedStatus();
        };

        Series.prototype.handleMouseDownEvent = function (e) {
            var tool = (e.button === 0 || typeof e.button == 'undefined' || e.button == null) ? this.selectedTool.read() : this.selectedTool2.read();
            var foundSelectedMeasurement = false;

            this.panActive = true;
            this.needsMouseUp = true;
            this.measurementsLayer.hideMeasurementInfo = this.hideActiveMeasuremntInfo;

            if (this.series) {
                this.mousePressed = true;
                this.measureVolume.write(false);

                var instance = this.currentInstance().read();

                var inverseTransform = this.getInverseImageTransformation(this.canvas.width, this.canvas.height, instance.instanceAttributes.columns, instance.instanceAttributes.rows);

                var mousePosition = Mouse.getOffset(e.pointers[0]);
                var mousePositionImage = Vectors.multiplyM(inverseTransform, mousePosition);

                this.recordEvent({
                    type: 8 /* MouseDown */,
                    mousePositionImage: mousePositionImage
                });

                // select annotation, even if measurement tool is not active
                var transform = this.getImageTransformation(this.canvas.width, this.canvas.height, instance.instanceAttributes.columns, instance.instanceAttributes.rows);
                var instanceIndex = this.selectedInstanceIndex.read();
                var measurements = this.series.instances[instanceIndex].instanceAttributes.measurements;
                this.clearSelection();

                var selection = Measurements.select(tool, measurements, this.application.currentColor.read(), mousePositionImage, 20 / Math.sqrt(Math.abs(Vectors.determinant(transform))), Classes.MouseTools.isDropMeasurementTool(tool));

                if (selection.measurement) {
                    foundSelectedMeasurement = true;

                    if (selection.endpoint) {
                        this.selectedEndpoint = selection.endpoint;
                    }

                    selection.measurement.saveOriginalEndpoints(mousePositionImage);
                    this.selectedMeasurement.write(selection.measurement);
                    this.editingMeasurement.write(selection.measurement);
                }

                if (Classes.MouseTools.isDropMeasurementTool(tool) && !foundSelectedMeasurement) {
                    this.originalMousePosition = mousePositionImage;

                    var hideMeasurement = Classes.MouseTools.isAnnotationTool(tool, this.showTextOnDirected.read());
                    var newMeasurement = Measurements.createMeasurement(tool, this.originalMousePosition, this.terminology, hideMeasurement, false, instance, this.application.currentColor.read(), this.accountSettings.viewer_default_drop_shape_width);

                    this.addMeasurementToSelectedInstance(newMeasurement);
                    this.recordMeasurementAdded(newMeasurement, mousePositionImage);
                    this.clearSelection();
                } else if (Classes.MouseTools.isPaintTool(tool) && foundSelectedMeasurement) {
                    this.originalMousePosition = mousePositionImage;
                    this.measurementInProgress.write(selection.measurement);
                    selection.measurement.startDrawing(mousePositionImage);
                } else if (Classes.MouseTools.isMeasurementTool(tool) && !this.isPropagating()) {
                    this.originalMousePosition = mousePositionImage;
                    this.needsHideMeasurementCursor = true;

                    if (!foundSelectedMeasurement && (tool != 4 /* Select */)) {
                        if (tool != this.selectedTool.read()) {
                            this.enableMeasurementCursor(tool);
                        }

                        if (!this.measurementInProgress.read()) {
                            var hideMeasurement = Classes.MouseTools.isAnnotationTool(tool, this.showTextOnDirected.read());
                            var calibration = Classes.MouseTools.isCalibrationTool(tool);
                            var groupTool = Classes.MouseTools.isGroupMeasurementTool(tool);
                            var newMeasurement = Measurements.createMeasurement(tool, this.originalMousePosition, this.terminology, hideMeasurement, calibration, instance, this.application.currentColor.read());

                            if (newMeasurement) {
                                this.measurementInProgress.write(newMeasurement);

                                if (groupTool) {
                                    var groupMeasurement = newMeasurement;
                                    if (groupMeasurement.getCurrentStep() == 0) {
                                        this.application.showWarningBanner(this.terminology.lookup(groupMeasurement.getCurrentTermToken()));
                                    }
                                }
                            }
                        } else {
                            this.measurementInProgress.read().startDrawing(this.originalMousePosition);
                        }
                    }

                    this.renderLayers();
                } else {
                    this.forceLowResolution.write(this.forceLowResolution.read().take());
                    this.originalMousePosition = mousePosition;

                    this.zoomCenter.write(mousePosition);

                    this.originalTransform = this.transform.read();

                    var selectedInstanceIndex = this.selectedInstanceIndex.read();

                    this.originalIndex = selectedInstanceIndex;

                    // Recalculate window level if default value is stored
                    if (this.windowLevel.read() != null) {
                        if (this.windowLevel.read().center != WindowLevelPresets.defaultCenter && this.windowLevel.read().width != WindowLevelPresets.defaultWidth) {
                            this.originalWindowLevel = this.windowLevel.read();
                        } else {
                            this.originalWindowLevel = this.getDefaultWindowLevel();
                        }
                    }
                }
            }

            if (tool === 9 /* Localization */ || tool === 11 /* Probe */ || tool === 15 /* Magnify */) {
                this.el.css({ cursor: 'none' });
            } else {
                if (!this.cursorLayer.visible.read()) {
                    this.el.css({ cursor: 'crosshair' });
                }
            }

            this.selectedSeriesKey.write(this.viewKey);

            this.handleMouseMoveEvent(e);
        };

        Series.prototype.handleMouseUpEvent = function (e) {
            var _this = this;
            var tool = (e.button === 0 || typeof e.button == 'undefined' || e.button == null) ? this.selectedTool.read() : this.selectedTool2.read();

            this.panActive = false;
            this.needsMouseUp = false;
            this.editingMeasurement.write(null);

            if (this.mousePressed) {
                this.mousePressed = false;
                this.measureVolume.write(this.measureVolumeDefault);

                var instance = this.currentInstance().read();

                var newMousePosition = Mouse.getOffset(e.pointers.length ? e.pointers[0] : e.changedPointers[0]);
                var newMousePositionImage = this.mapToImage(newMousePosition, this.canvas.width, this.canvas.height, instance.instanceAttributes.columns, instance.instanceAttributes.rows);

                var measurementInProgress = this.measurementInProgress.read();
                var selectedMeasurement = this.selectedMeasurement.read();

                if (selectedMeasurement) {
                    selectedMeasurement.active = false;
                }

                if (Classes.MouseTools.isMeasurementTool(tool)) {
                    if (this.mouseMoved || Classes.MouseTools.isMultiClickMeasurementTool(tool) || Classes.MouseTools.isPaintTool(tool)) {
                        this.enableMeasurementCursor(tool);

                        if (this.selectedEndpoint) {
                            this.selectedEndpoint = null;
                        } else if (measurementInProgress) {
                            if (measurementInProgress.isNonEmpty() || Classes.MouseTools.isPaintTool(tool)) {
                                var transform = this.getImageTransformation(this.canvas.width, this.canvas.height, instance.instanceAttributes.columns, instance.instanceAttributes.rows);

                                if (measurementInProgress.stopDrawing(newMousePositionImage, transform)) {
                                    var validMeasurement = true;
                                    if (tool == 10 /* Text */) {
                                        var text = window.prompt(this.terminology.lookup(Terminology.Terms.EnterText), "");
                                        if (text) {
                                            var measurement = measurementInProgress;
                                            measurement.text = text;
                                        } else {
                                            validMeasurement = false;
                                        }
                                    } else if (tool == 31 /* Stamp */) {
                                        var measurement = measurementInProgress;
                                        measurement.text = "\uD83D\uDD12 " + (this.application.user.name || this.application.user.email);
                                    }

                                    if (validMeasurement && this.ultrasoundMeasurements && Dicom.Ultrasound.isUltrasound(instance)) {
                                        var result = Dicom.Ultrasound.findSpacing(measurementInProgress, instance);

                                        if (result.valid) {
                                            measurementInProgress.ultrasoundPixelSpacing = [result.spacingX, result.spacingY];
                                        } else {
                                            measurementInProgress.ultrasoundPixelSpacing = null;
                                            this.ultrasoundRegionsVisible.write(true);
                                            this.renderAll();
                                            window.alert(this.terminology.lookup(Terminology.Terms.InvalidUltrasoundMeasurement));
                                        }
                                    }

                                    if (measurementInProgress.calibration) {
                                        validMeasurement = this.updateCalibration(measurementInProgress, instance);
                                    }

                                    if (validMeasurement && selectedMeasurement && Classes.MouseTools.isPaintTool(tool)) {
                                        if (measurementInProgress.isNonEmpty()) {
                                            this.editMeasurement(selectedMeasurement, instance);
                                            this.selectedMeasurement.write(measurementInProgress);
                                            this.measurementInProgress.write(measurementInProgress = null);
                                        } else {
                                            this.selectedMeasurement.write(measurementInProgress);
                                            this.deleteSelectedMeasurement();
                                            this.clearSelection();
                                            this.measurementInProgress.write(measurementInProgress = null);
                                        }
                                    } else if (validMeasurement) {
                                        var groupTool = Classes.MouseTools.isGroupMeasurementTool(tool);

                                        this.addMeasurementToSelectedInstance(measurementInProgress);
                                        this.recordMeasurementAdded(measurementInProgress, newMousePositionImage);
                                        this.clearSelection();
                                        this.selectedMeasurement.write(measurementInProgress);
                                        measurementInProgress.selected = true;

                                        if (groupTool) {
                                            var groupMeasurement = measurementInProgress;

                                            if (groupMeasurement.step()) {
                                                groupMeasurement.reset();
                                                this.application.hideWarningBanner();
                                            } else {
                                                this.application.updateWarningBanner(this.terminology.lookup(groupMeasurement.getCurrentTermToken()));
                                            }
                                        }
                                    } else {
                                        this.measurementInProgress.write(measurementInProgress = null);
                                    }
                                }
                            } else {
                                this.measurementInProgress.write(measurementInProgress = null);
                            }
                        }

                        if (!measurementInProgress && selectedMeasurement) {
                            if (selectedMeasurement.hasMoved() && selectedMeasurement.id && selectedMeasurement.editable) {
                                if (this.ultrasoundMeasurements && Dicom.Ultrasound.isUltrasound(instance)) {
                                    var result = Dicom.Ultrasound.findSpacing(selectedMeasurement, instance);

                                    if (result && result.valid) {
                                        selectedMeasurement.ultrasoundPixelSpacing = [result.spacingX, result.spacingY];
                                        this.editMeasurement(selectedMeasurement, instance);
                                    } else {
                                        selectedMeasurement.ultrasoundPixelSpacing = null;
                                        this.ultrasoundRegionsVisible.write(true);
                                        this.renderAll();
                                        window.alert(this.terminology.lookup(Terminology.Terms.InvalidUltrasoundMeasurement));
                                    }
                                } else {
                                    this.editMeasurement(selectedMeasurement, instance);
                                }
                            }
                        }

                        // End custom cursor for alternate mouse button
                        if (!Classes.MouseTools.isMeasurementTool(this.selectedTool.read()) && Classes.MouseTools.isMeasurementTool(tool)) {
                            var delayedRemoveCursor = function (e) {
                                var dist = Geometry.distanceBetween(newMousePosition, { x: e.offsetX, y: e.offsetY });

                                if (dist > 40) {
                                    $glassPane.off('mousemove', delayedRemoveCursor);
                                    _this.disableMeasurementCursor();
                                }
                            };

                            var $glassPane = this.el.find('.glasspane');
                            $glassPane.on('mousemove', delayedRemoveCursor);
                        }
                    } else if (tool == 10 /* Text */) {
                        if (measurementInProgress) {
                            var text = window.prompt(this.terminology.lookup(Terminology.Terms.EnterText), "");
                            if (text) {
                                var measurement = measurementInProgress;

                                if (measurementInProgress.isNonEmpty()) {
                                    measurement.text = text;
                                } else {
                                    var transform = this.getImageTransformation(this.canvas.width, this.canvas.height, instance.instanceAttributes.columns, instance.instanceAttributes.rows);
                                    measurement.createAtPoint(text, transform);
                                }

                                this.addMeasurementToSelectedInstance(measurementInProgress);
                                this.recordMeasurementAdded(measurementInProgress, newMousePositionImage);
                                this.clearSelection();
                                this.selectedMeasurement.write(measurementInProgress);
                                measurementInProgress.selected = true;
                            } else {
                                this.measurementInProgress.write(measurementInProgress = null);
                            }
                        } else if (selectedMeasurement) {
                            var measurement = selectedMeasurement;
                            var text = window.prompt(this.terminology.lookup(Terminology.Terms.EnterText), measurement.text);
                            if (text) {
                                measurement.text = text;
                                this.editMeasurement(selectedMeasurement, instance);
                            } else {
                                this.measurementInProgress.write(measurementInProgress = null);
                            }
                        }
                    } else {
                        this.measurementInProgress.write(measurementInProgress = null);
                    }
                } else {
                    this.forceLowResolution.write(this.forceLowResolution.read().release());
                }

                if (this.propagatingInstance) {
                    this.finishPropagation();
                }

                this.recordMouseUpEvent(newMousePositionImage);

                this.zoomCenter.write(null);

                this.measurementsLayer.hideMeasurementInfo = false;
                this.renderAllForceUpdate();

                if (!this.cursorLayer.visible.read()) {
                    this.el.css({ cursor: '' });
                }
            }

            this.mouseMoved = false;
            this.pressActive = false;
            this.needsHideMeasurementCursor = false;
        };

        Series.prototype.isPropagating = function () {
            return this.propagatingInstance != null;
        };

        Series.prototype.startPropagation = function () {
            this.propagatingInstance = this.currentInstance().read();
            this.propagatingMeasurement = this.selectedMeasurement.read();
        };

        Series.prototype.finishPropagation = function () {
            var instance = this.currentInstance().read();
            var currentIndex = _.indexOf(this.series.instances, instance);
            var originalIndex = this.propagatingMeasurement.propagation ? this.propagatingMeasurement.propagation.originalIndex : _.indexOf(this.series.instances, this.propagatingInstance);

            var startIndex, endIndex;

            if (currentIndex < originalIndex) {
                startIndex = currentIndex;
                endIndex = this.propagatingMeasurement.propagation ? this.propagatingMeasurement.propagation.endIndex : currentIndex;
            } else {
                endIndex = currentIndex;
                startIndex = this.propagatingMeasurement.propagation ? this.propagatingMeasurement.propagation.startIndex : currentIndex;
            }

            Annotations.propagateAnnotation(this.propagatingMeasurement, this.series, originalIndex, startIndex, endIndex, true);
            this.editMeasurement(this.propagatingMeasurement, this.propagatingInstance);
            this.application.propagateMode.write(false);
            this.propagatingMeasurement = null;
            this.propagatingInstance = null;
            this.clearSelection();
        };

        Series.prototype.propapateToSeries = function () {
            var instance = this.currentInstance().read();
            var measurement = this.selectedMeasurement.read();
            var originalIndex = measurement.propagation ? measurement.propagation.originalIndex : _.indexOf(this.series.instances, instance);
            var startIndex = 0;
            var endIndex = this.series.instances.length - 1;

            Annotations.propagateAnnotation(measurement, this.series, originalIndex, startIndex, endIndex, true);
            this.editMeasurement(measurement, this.series.instances[originalIndex]);
            this.propagatingMeasurement = null;
            this.propagatingInstance = null;

            window.alert(this.terminology.lookup(Terminology.Terms.PropagateAllComplete));
        };

        Series.prototype.thresholdToArea = function (shrinkwrap, range) {
            var _this = this;
            if (typeof shrinkwrap === "undefined") { shrinkwrap = false; }
            if (typeof range === "undefined") { range = false; }
            var valMin;

            if (range) {
                valMin = window.prompt(this.terminology.lookup(Terminology.Terms.ThresholdMinPrompt));
            } else {
                valMin = window.prompt(this.terminology.lookup(Terminology.Terms.ThresholdPrompt));
            }

            var instance = this.currentInstance().read();
            var instanceIndex = this.selectedInstanceIndex.read();
            var instanceKey = this.getInstanceKey(instanceIndex);
            var selected;

            if (valMin) {
                var thresholdMin = parseFloat(valMin);
                var thresholdMax = Number.MAX_VALUE;

                if (range) {
                    var valMax = window.prompt(this.terminology.lookup(Terminology.Terms.ThresholdMaxPrompt));
                    thresholdMax = parseFloat(valMax);
                }

                var bounds;
                if (this.selectedMeasurement.read()) {
                    selected = this.selectedMeasurement.read();
                    if (selected instanceof Measurements.Rectangle) {
                        bounds = Geometry.findBoundingBox(selected.points);
                    } else {
                        selected = null;
                    }
                }

                if (!bounds) {
                    bounds = [{ x: 0, y: 0 }, { x: instance.instanceAttributes.columns - 1, y: instance.instanceAttributes.rows - 1 }];
                }

                if (WindowLevelPresets.shouldUse16BitWindowLevel(instance)) {
                    var width = instance.instanceAttributes.columns;
                    var height = instance.instanceAttributes.rows;

                    var imageData = this.renderer.visit({
                        visitCanvasRenderer: function (renderer) {
                            var imageElement = renderer.imageElements.get(instanceKey, 2 /* Diagnostic */);

                            if (imageElement) {
                                return Images.getImageData(imageElement.imageElement);
                            }

                            return null;
                        },
                        visitWebGLRenderer: function (renderer) {
                            var imageElement = renderer.imageElements.get(instanceKey, 2 /* Diagnostic */);

                            if (imageElement) {
                                return Images.getImageData(imageElement.imageElement);
                            }

                            return null;
                        }
                    }, null);

                    if (imageData) {
                        var minX = Math.floor(Math.min(bounds[0].x, bounds[1].x));
                        var minY = Math.floor(Math.min(bounds[0].y, bounds[1].y));
                        var maxX = Math.ceil(Math.max(bounds[0].x, bounds[1].x));
                        var maxY = Math.ceil(Math.max(bounds[0].y, bounds[1].y));

                        var area;

                        var currentColor = this.application.currentColor.read();
                        var found = _.find(instance.instanceAttributes.measurements, function (m) {
                            return (m instanceof Measurements.Area) && m.color === currentColor;
                        });
                        if (found) {
                            area = found.area;
                        } else {
                            area = new Uint8Array(width * height);
                        }

                        var terminology = this.terminology;
                        var processResult = function (area, foundData) {
                            if (foundData) {
                                if (selected) {
                                    _this.deleteSelectedMeasurement();
                                    _this.clearSelection();
                                }

                                if (found) {
                                    var foundArea = found;
                                    foundArea.updateArea(area);
                                    _this.editMeasurement(foundArea, instance);
                                    foundArea.selected = true;
                                    _this.selectedMeasurement.write(foundArea);
                                } else {
                                    var length = Math.max(width, height);
                                    var cursorSize = Math.max(Measurements.Area.ELEMENT_RADIUS_MIN, Math.round(length * Measurements.Area.ELEMENT_RADIUS_RATIO));
                                    var areaMeasurement = new Measurements.Area([{ x: 0, y: 0 }, { x: width - 1, y: height - 1 }], width, height, cursorSize, currentColor, null, area);
                                    areaMeasurement.editable = true;
                                    _this.addMeasurementToSelectedInstance(areaMeasurement);
                                    areaMeasurement.selected = true;
                                    _this.selectedMeasurement.write(areaMeasurement);
                                }
                            } else {
                                window.alert(terminology.lookup(Terminology.Terms.OperationNoResult));
                            }

                            $('.overlay').removeClass('application-loading');
                            _this.renderLayers();
                        };

                        if (Worker && (!Browser.isIE() || Browser.isIE11())) {
                            $('.overlay').addClass('application-loading');

                            var worker = new Worker("./resources/workers/ThresholdWorker.js");
                            worker.onmessage = function (e) {
                                if (e.data.action === "success") {
                                    processResult(new Uint8Array(e.data.area), e.data.foundData);
                                } else {
                                    window.alert(terminology.lookup(Terminology.Terms.OperationExceededLimits));
                                    $('.overlay').removeClass('application-loading');
                                }
                            };

                            worker.postMessage({
                                action: shrinkwrap ? "shrinkwrap" : "threshold",
                                thresholdMin: thresholdMin,
                                thresholdMax: thresholdMax,
                                imageData: imageData.data.buffer,
                                width: width,
                                height: height,
                                slope: instance.instanceAttributes.rescaleSlope,
                                intercept: instance.instanceAttributes.rescaleIntercept,
                                signed: instance.instanceAttributes.signed,
                                minX: minX,
                                minY: minY,
                                maxX: maxX,
                                maxY: maxY,
                                area: area.buffer
                            }, [imageData.data.buffer, area.buffer]);
                        } else {
                            window.alert(terminology.lookup(Terminology.Terms.OperationNotSupported));
                        }
                    }
                }
            }
        };

        Series.prototype.updateCalibration = function (measurement, instance) {
            var pixelSpacing = instance.instanceAttributes.pixelSpacing;
            var validMeasurement = true;

            if (pixelSpacing && pixelSpacing[0]) {
                measurement.calibration = false;
            } else {
                var val = window.prompt(this.terminology.lookup(Terminology.Terms.LineCalibratePrompt));

                if (val && (measurement instanceof Measurements.LineMeasurement)) {
                    var length = parseFloat(val);
                    if (length > 0) {
                        measurement.calibrationValue = length;
                    } else {
                        validMeasurement = false;
                    }
                } else {
                    validMeasurement = false;
                }
            }

            return validMeasurement;
        };

        Series.prototype.pinchZoomStart = function (e) {
            this.originalTransform = this.transform.read();

            this.zoomCenter.write(e.center);
            this.originalMousePosition = e.center;
            this.forceLowResolution.write(this.forceLowResolution.read().take());

            this.selectedSeriesKey.write(this.viewKey);
        };

        Series.prototype.pinchZoomEnd = function (e) {
            this.zoomCenter.write(null);
            this.forceLowResolution.write(this.forceLowResolution.read().release());
        };

        Series.prototype.pinchZoom = function (e) {
            this.applyZoomRotate(this.originalTransform.scale * e.scale, this.originalTransform.rotation);
        };

        /**
        * Send a mouse up event to the recorder
        */
        Series.prototype.recordMouseUpEvent = function (newMousePositionImage) {
            switch (this.selectedTool.read()) {
                case 3 /* Window */:
                    this.recordWindowLevel(newMousePositionImage);
                    break;
            }
        };

        /**
        * Send an event to the recorder
        */
        Series.prototype.recordEvent = function (event) {
            if (this.series) {
                event.seriesInstanceUid = this.series.seriesAttributes.seriesUid;

                var index = this.selectedInstanceIndex.read();

                event.instanceUid = this.series.instances[index].id;

                this.recorder.read().append(event);
            }
        };

        /**
        * Send the current image transformation to the recorder
        */
        Series.prototype.recordTransformation = function () {
            this.recordEvent({
                type: 0 /* ImageTransformationChanged */,
                newTransformation: this.transform.read()
            });
        };

        /**
        * Send the current magnifier position to the recorder
        */
        Series.prototype.recordMagnifier = function () {
            this.recordEvent({
                type: 12 /* MagnifierPositionChanged */,
                magnifierPosition: this.magnifier.read()
            });
        };

        /**
        * Send the current probe tool position to the recorder
        */
        Series.prototype.recordProbe = function () {
            this.recordEvent({
                type: 13 /* ProbeToolChanged */,
                probeTool: this.probeTool.read()
            });
        };

        /**
        * Send the current plane localization data to the recorder
        */
        Series.prototype.recordPlaneLocalizationData = function () {
            this.recordEvent({
                type: 14 /* PlaneLocalizationChanged */,
                planeLocalizationData: this.planeLocalizationCursor.read()
            });
        };

        /**
        * Send the current window level to the recorder
        */
        Series.prototype.recordWindowLevel = function (newMousePositionImage) {
            this.recordEvent({
                type: 1 /* WindowLevelChanged */,
                mousePositionImage: newMousePositionImage,
                newWindowLevel: this.useOriginalWindowLevel.read() ? null : this.windowLevel.read()
            });
        };

        /**
        * Add a measurement
        */
        Series.prototype.addMeasurementToSelectedInstance = function (measurement) {
            var instance = this.currentInstance().read();
            this.addMeasurementToInstance(measurement, instance);
        };

        /**
        * Returns a measurement's annotation data.
        * @param {Measurements.Measurement} measurement
        * @param {Models.Instance} instance
        * @returns {Models.AnnotationData}
        */
        Series.prototype.getAnnotationData = function (measurement, instance) {
            var annotationData = measurement.toAnnotationData();

            if (measurement.label) {
                annotationData.label = measurement.label;
            }

            if (measurement.colocationId) {
                annotationData.colocationId = measurement.colocationId;
            }

            if (measurement.propagation) {
                annotationData.propagation = {
                    startIndex: measurement.propagation.startIndex,
                    originalIndex: measurement.propagation.originalIndex,
                    endIndex: measurement.propagation.endIndex
                };
            }

            if (measurement.pixelSpacingUser) {
                annotationData.pixelSpacing = measurement.pixelSpacingUser;
            }

            if (measurement.sliceSpacingUser) {
                annotationData.sliceSpacing = measurement.sliceSpacingUser;
            }

            if (this.storeExtraAnnotationData && instance) {
                var instanceIndex = this.selectedInstanceIndex.read();
                var instanceKey = this.getInstanceKey(instanceIndex);
                var statistics = measurement.statistics(instance, instanceKey, this.renderer, null);
                if (statistics) {
                    annotationData.stats = statistics;
                    annotationData.stats.pixelSpacing = instance.instanceAttributes.pixelSpacing ? instance.instanceAttributes.pixelSpacing[0] : 0;

                    if (this.storeExtraAnnotationData == 2) {
                        var pixelData = measurement.getAllPixelValues(instance, instanceKey, this.renderer);
                        if (pixelData) {
                            annotationData.stats.pixelData = pixelData;
                        }
                    }
                }

                annotationData.description = JSON.stringify([instance.instanceAttributes.seriesDescription, instance.instanceAttributes.imageType]);
                annotationData.instanceIndex = instance.instanceAttributes.instanceIndex;
            }

            return annotationData;
        };

        /**
        * Add a measurement
        */
        Series.prototype.addMeasurementToInstance = function (measurement, instance) {
            var _this = this;
            this.measurementInProgress.write(null);

            var stamped = false;
            var stamp = _.find(instance.instanceAttributes.measurements, function (m) {
                return m instanceof Measurements.Stamp;
            });
            if (stamp) {
                var stampCreatorId = stamp.creatorId;
                stamped = (stampCreatorId != this.application.user.uuid);
            }

            if (stamped) {
                var message = this.terminology.lookup(Terminology.Terms.WarningInstanceStamped);
                window.alert(message);
            } else {
                measurement.creatorId = this.application.user.uuid;
                var measurements = instance.instanceAttributes.measurements;
                measurements.push(measurement);

                if (!(this.application.user.is_anonymous == true && this.accountSettings.viewer_anon_annotations_clear == 1)) {
                    Services.addImageAnnotation(this.sessionId, this.series.studyAttributes.queryObject, instance.seriesAttributes.seriesUid, instance.id, instance.frameNumber, this.getAnnotationData(measurement, instance)).subscribe({
                        next: function (response) {
                            measurement.id = new Classes.AnnotationId(response.uuid);
                            measurement.creator = _this.application.user.name;

                            _this.renderLayers();
                        },
                        done: function () {
                        },
                        fail: function (err) {
                            _this.recordError("Unable to add annotation: " + err);
                        }
                    });
                }
            }
        };

        /**
        * Send a measurement-added event to the recorder
        */
        Series.prototype.recordMeasurementAdded = function (measurement, mousePositionImage) {
            this.recordEvent({
                type: 2 /* AnnotationAdded */,
                mousePositionImage: mousePositionImage,
                annotation: measurement.toAnnotationData(),
                instanceIndex: this.selectedInstanceIndex.read()
            });
        };

        /**
        * Save a measurement to services
        */
        Series.prototype.editMeasurement = function (measurement, instance) {
            var _this = this;
            Services.editImageAnnotation(this.application.sessionId, this.series.studyAttributes.queryObject, measurement.id, this.getAnnotationData(measurement, instance)).subscribe({
                done: function () {
                },
                next: function (_) {
                },
                fail: function (err) {
                    _this.recordError("Unable to edit image annotation: " + err);
                }
            });
        };

        Series.prototype.recordMouseMoveEvent = function (e) {
            if (this.series) {
                var instance = this.currentInstance().read();

                var newMousePosition;

                if (e.pointers && e.pointers.length) {
                    newMousePosition = Mouse.getOffset(e.pointers[0]);
                } else {
                    newMousePosition = Mouse.getOffset(e);
                }

                var newMousePositionImage = this.mapToImage(newMousePosition, this.canvas.width, this.canvas.height, instance.instanceAttributes.columns, instance.instanceAttributes.rows);

                this.recordEvent({
                    type: 7 /* MouseMove */,
                    mousePositionImage: newMousePositionImage
                });
            }
        };

        Series.prototype.handleMouseMoveEvent = function (e) {
            if (this.series) {
                var tool = (e.button === 0 || typeof e.button == 'undefined' || e.button == null) ? this.selectedTool.read() : this.selectedTool2.read();

                var instance = this.currentInstance().read();

                var newMousePosition = Mouse.getOffset(e.pointers[0]);
                var newMousePositionImage = this.mapToImage(newMousePosition, this.canvas.width, this.canvas.height, instance.instanceAttributes.columns, instance.instanceAttributes.rows);

                if (Classes.MouseTools.isMeasurementTool(tool)) {
                    if (this.mousePressed) {
                        if (this.mouseMoved && this.needsHideMeasurementCursor) {
                            this.needsHideMeasurementCursor = false;

                            // hide special annotation cursor while dragging
                            if (!this.settings.read().disableCustomAnnotationCursor && !Classes.MouseTools.isPaintTool(tool)) {
                                this.disableMeasurementCursor();
                                this.el.css({ cursor: '' });
                                this.el.addClass('hideCursor');
                            }
                        }

                        var selectedMeasurement = this.selectedMeasurement.read();
                        var measurementInProgress = this.measurementInProgress.read();

                        var transform = this.getImageTransformation(this.canvas.width, this.canvas.height, instance.instanceAttributes.columns, instance.instanceAttributes.rows);

                        if (measurementInProgress && (measurementInProgress instanceof Measurements.Trace)) {
                            if (measurementInProgress.stopDrawing(newMousePositionImage, transform)) {
                                this.addMeasurementToSelectedInstance(measurementInProgress);
                                this.recordMeasurementAdded(measurementInProgress, newMousePositionImage);
                                this.clearSelection();
                                measurementInProgress.selected = true;
                            } else {
                                measurementInProgress.startDrawing(newMousePositionImage, transform);
                            }
                        } else if (this.selectedEndpoint || measurementInProgress || selectedMeasurement) {
                            var dx = newMousePositionImage.x - this.originalMousePosition.x;
                            var dy = newMousePositionImage.y - this.originalMousePosition.y;

                            if (selectedMeasurement) {
                                selectedMeasurement.active = true;
                            }

                            if (this.selectedEndpoint) {
                                if (Math.max(Math.abs(dx), Math.abs(dy)) > 5 / Math.sqrt(Math.abs(Vectors.determinant(transform)))) {
                                    this.selectedEndpoint.move(newMousePositionImage);
                                } else {
                                    this.selectedEndpoint.reset();
                                }
                            } else if (measurementInProgress !== null) {
                                measurementInProgress.continueDrawing(newMousePositionImage);
                            } else if (selectedMeasurement && selectedMeasurement.editable) {
                                if (Math.max(Math.abs(dx), Math.abs(dy)) > 5 / Math.sqrt(Math.abs(Vectors.determinant(transform)))) {
                                    selectedMeasurement.continueMoving(dx, dy);
                                } else {
                                    selectedMeasurement.continueMoving(0, 0);
                                }
                            }
                        }
                    }

                    this.renderLayers();
                } else {
                    if (this.mousePressed) {
                        switch (tool) {
                            case 0 /* Move */:
                                this.updateOffset(newMousePosition);
                                break;
                            case 2 /* Zoom */:
                                this.updateZoom(newMousePosition, false);
                                break;
                            case 14 /* FreeRotate */:
                                this.updateZoom(newMousePosition, true);
                                break;
                            case 1 /* Scroll */:
                                this.updateScroll(newMousePosition);
                                break;
                            case 3 /* Window */:
                                this.updateWindowLevel(newMousePosition);
                                break;
                            case 9 /* Localization */:
                                this.updateLocalizationCursor(newMousePosition);
                                break;
                            case 11 /* Probe */:
                                this.probeTool.write(newMousePositionImage);
                                break;
                            case 15 /* Magnify */:
                                this.magnifier.write(newMousePositionImage);
                                break;
                        }
                    }
                }
            }
        };

        /**
        * Apply a window level preset based on settings or defaults
        */
        Series.prototype.applyWindowLevelPreset = function (index) {
            var modalitySettings = this.settings.read().modalities;

            var presets = this.getWindowLevelPresets(modalitySettings);

            var windowLevel = presets[index];

            if (windowLevel) {
                this.useOriginalWindowLevel.write(false);
                this.windowLevel.write(windowLevel.windowLevel);
            }
        };

        /**
        * Apply color table preset
        * @param {number} index
        */
        Series.prototype.applyColorTablePreset = function (index) {
            var modalitySettings = this.settings.read().modalities;

            var presets = this.getColorTablePresets(modalitySettings);

            var colorTable = presets[index];

            if (colorTable) {
                this.colorTable.write(colorTable.colorTable);
            }
        };

        /**
        * Detect the window level based on the selected ROI
        */
        Series.prototype.detectWindowLevel = function () {
            var instanceIndex = this.selectedInstanceIndex.read();
            var instance = this.series.instances[instanceIndex];
            var instanceKey = this.getInstanceKey(instanceIndex);

            // Prefer the selected measurement, if there is one.
            var selectedMeasurement = this.selectedMeasurement.read();

            // If there is no selected measurement, try to return the _only_ measurement.
            if (!selectedMeasurement) {
                selectedMeasurement = instance.instanceAttributes.measurements.only();
            }

            if (selectedMeasurement) {
                var statistics = selectedMeasurement.statistics(instance, instanceKey, this.renderer, null);

                if (!statistics) {
                    window.alert(this.terminology.lookup(Terminology.Terms.CannotDetectWindowLevel));
                } else {
                    this.useOriginalWindowLevel.write(false);
                    this.windowLevel.write({
                        center: (statistics.max + statistics.min) / 2.0,
                        width: statistics.max - statistics.min
                    });
                }
            } else {
                window.alert(this.terminology.lookup(Terminology.Terms.PleaseSelectAnAnnotation));
            }
        };

        /**
        * Get window level presets based on settings or defaults
        */
        Series.prototype.getWindowLevelPresets = function (modalitySettings) {
            var modality = this.series.seriesAttributes.modality;

            if (modalitySettings) {
                var settings = _.find(modalitySettings, function (m) {
                    return m.modality === modality;
                });

                if (settings && settings.presets !== undefined && settings.presets.length > 0) {
                    _.each(settings.presets, function (preset) {
                        if (preset.windowLevel) {
                            preset.windowLevel.center = Number(preset.windowLevel.center);
                            preset.windowLevel.width = Number(preset.windowLevel.width);
                        }
                    });

                    return settings.presets;
                }
            }

            var index = this.selectedInstanceIndex.read();

            return WindowLevelPresets.defaults(modality, this.terminology);
        };

        /**
        * Get a color table preset
        * @param {Classes.ModalitySettings[]} modalitySettings
        * @returns {Classes.ColorTablePreset[]}
        */
        Series.prototype.getColorTablePresets = function (modalitySettings) {
            var modality = this.series.seriesAttributes.modality;

            if (modalitySettings) {
                var settings = _.find(modalitySettings, function (m) {
                    return m.modality === modality;
                });

                if (settings && settings.colorTablePresets !== undefined && settings.colorTablePresets.length > 0) {
                    return settings.colorTablePresets;
                }
            }

            return ColorTablePresets.defaults(modality, this.terminology);
        };

        /**
        * Update the offset portion of the current image transformation based on mouse movement
        */
        Series.prototype.updateOffset = function (newMousePosition) {
            var dx = newMousePosition.x - this.originalMousePosition.x;
            var dy = newMousePosition.y - this.originalMousePosition.y;

            this.transform.write({
                offsetX: this.originalTransform.offsetX + dx,
                offsetY: this.originalTransform.offsetY + dy,
                scale: this.originalTransform.scale,
                flipped: this.originalTransform.flipped,
                rotation: this.originalTransform.rotation
            });
        };

        /**
        * Update the zoom portion of the current image transformation based on mouse movement
        */
        Series.prototype.updateZoom = function (newMousePosition, rotate) {
            var dx = this.originalMousePosition.x - newMousePosition.x;
            var dy = this.originalMousePosition.y - newMousePosition.y;

            var newScale = this.originalTransform.scale;
            var newRotation = this.originalTransform.rotation;

            if (rotate) {
                var dx2 = dx;
                var dy2 = dy;

                if (Math.abs(dx2) < 5) {
                    dx2 = 0;
                }

                if (Math.abs(dy2) < 5) {
                    dy2 = 0;
                }

                if (dx2 * dx2 + dy2 * dy2 > 25) {
                    newRotation += Math.atan2(dy2, dx2) / Math.PI * 2;
                }
            } else {
                newScale *= Math.exp(dy / 75.0);
            }

            this.applyZoomRotate(newScale, newRotation);
        };

        Series.prototype.applyZoomRotate = function (newScale, newRotation) {
            var instance = this.currentInstance().read();

            var newTransform = Transform.zoomRotateAround(instance, this.originalMousePosition, this.originalTransform, newScale, newRotation, this.canvas.width, this.canvas.height);

            this.transform.write(newTransform);
        };

        /**
        * Update the currently selected instance or frame based on mouse movement
        */
        Series.prototype.updateScroll = function (newMousePosition) {
            var dy = this.originalMousePosition.y - newMousePosition.y;

            // Scroll one frame after a mouse drag is longer than pixelScrollingThreshold
            var pixelScrollingThreshold = 40;

            // On series larger than 20 images use a percentage of the screen to determine when to scroll,
            // Otherwise use the pixelScrollingThreshold
            var usePercentScrollingMethod = (this.series.instances.length > 20) ? true : false;

            var newIndex = 0;

            if (usePercentScrollingMethod) {
                newIndex = Math.round(this.originalIndex + dy * this.series.instances.length / this.el.height()) % this.series.instances.length;
            } else {
                newIndex = (this.originalIndex + Math.floor(dy / pixelScrollingThreshold)) % this.series.instances.length;
            }

            if (newIndex < 0) {
                newIndex += this.series.instances.length;
            }

            if (this.selectedInstanceIndex.read() !== newIndex) {
                this.selectedInstanceIndex.write(newIndex);
            }
        };

        /**
        * Update the window level settings based on mouse movement
        */
        Series.prototype.updateWindowLevel = function (newMousePosition) {
            var dx = this.originalMousePosition.x - newMousePosition.x;
            var dy = this.originalMousePosition.y - newMousePosition.y;

            var settings = this.settings.read();
            dx = (settings.reverseWLHorizontal === true) ? -dx : dx;
            dy = (settings.reverseWLVertical === true) ? -dy : dy;

            if (Math.abs(dx) > 5 || Math.abs(dy) > 5) {
                if (this.useOriginalWindowLevel.read()) {
                    this.originalWindowLevel = this.getDefaultWindowLevel(this.currentInstance().read());
                    this.useOriginalWindowLevel.write(false);
                }

                var sensitivity = Math.min(Math.max(1.0, this.originalWindowLevel.center / 200.0), 10.0);

                var newWindowLevel = {
                    center: Math.round(this.originalWindowLevel.center + sensitivity * dy),
                    width: Math.round(Math.max(1.0, this.originalWindowLevel.width / Math.exp(dx / 100.0)))
                };

                this.windowLevel.write(newWindowLevel);
            }
        };

        Series.prototype.handleMouseWheelEvent = function (e, delta) {
            if (this.selectedToolWheel.read() == 1 /* Scroll */) {
                this.wheelScroll(e, delta);
            } else if (this.selectedToolWheel.read() == 2 /* Zoom */) {
                this.wheelZoom(delta);
            } else {
                switch (this.selectedTool.read()) {
                    case 2 /* Zoom */:
                    case 14 /* FreeRotate */:
                    case 0 /* Move */:
                    case 15 /* Magnify */:
                        this.wheelZoom(delta);
                        break;
                    default:
                        this.wheelScroll(e, delta);
                        break;
                }
            }

            e.stopPropagation();
            e.preventDefault();
        };

        /**
        * Update the zoom of the current image transformation based on mouse wheel movement
        */
        Series.prototype.wheelZoom = function (delta) {
            var transform = this.transform.read();

            this.transform.write({
                offsetX: transform.offsetX,
                offsetY: transform.offsetY,
                scale: delta < 0 ? transform.scale / 1.2 : transform.scale * 1.2,
                flipped: transform.flipped,
                rotation: transform.rotation
            });
        };

        /**
        * Update the currently selected instance or frame based on mouse wheel movement
        */
        Series.prototype.wheelScroll = function (e, delta) {
            var d;

            if (delta > 0) {
                d = -1;
            } else {
                d = 1;
            }

            this.scrollBy(d);
        };

        /**
        * Move the instance number by the specified amount
        */
        Series.prototype.scrollBy = function (n) {
            var instance = this.selectedInstanceIndex.read();

            if (this.series) {
                var newInstance = (instance + n) % this.series.instances.length;

                if (newInstance < 0) {
                    newInstance += this.series.instances.length;
                }

                this.selectedInstanceIndex.write(newInstance);
            }
        };

        /**
        * Reset the image transformation settings
        */
        Series.prototype.fit = function () {
            this.transform.write({
                offsetX: 0,
                offsetY: 0,
                scale: 1,
                flipped: false,
                rotation: 0
            });
        };

        /**
        * Reset all settings for all frames
        */
        Series.prototype.reset = function () {
            this.deleteAllEditableMeasurements();

            if (this.subtractionActive.read()) {
                this.subtractionActive.write(false);
            }

            this.resetWindowLevel();
            this.invertActive.write(false);
            this.infoVisible.write(true);
            this.measurementsVisible.write(true);
            this.hiddenGSPSLayers.write([]);
            this.colorTable.write(null);

            this.fit();

            if (this.transformHP) {
                this.transform.write(this.transformHP.read());
            }

            this.reloadAllFrames();
        };

        /**
        * Delete all editable measurements
        */
        Series.prototype.deleteAllEditableMeasurements = function (skipArea) {
            var _this = this;
            if (typeof skipArea === "undefined") { skipArea = false; }
            var deleted = [];

            _.each(this.series.instances, function (instance) {
                var calibrationUser = false;

                _.each(instance.instanceAttributes.measurements, function (measurement) {
                    if (measurement.editable && !measurement.temporary && (!skipArea || !(measurement instanceof Measurements.Area))) {
                        if (measurement.id && !_.contains(deleted, measurement.id)) {
                            deleted.push(measurement.id);

                            Services.deleteImageAnnotation(_this.sessionId, _this.series.studyAttributes.queryObject, measurement.id).subscribe({
                                done: function () {
                                },
                                next: function (_) {
                                },
                                fail: function (err) {
                                    _this.recordError("Unable to delete annotation:" + err);
                                }
                            });

                            if (measurement == instance.instanceAttributes.calibration) {
                                instance.instanceAttributes.calibration = null;
                            }
                        }
                    } else {
                        if (measurement.pixelSpacingUser || measurement.sliceSpacingUser) {
                            calibrationUser = true;
                        }
                    }
                });
                instance.instanceAttributes.measurements = _.filter(instance.instanceAttributes.measurements, function (m) {
                    return !m.editable || (skipArea && (m instanceof Measurements.Area));
                });
                instance.instanceAttributes.calibrationUser = calibrationUser;
            });
        };

        Series.prototype.updateAreaColorSelection = function (color) {
            var _this = this;
            var instanceIndex = this.selectedInstanceIndex.read();
            var instance = this.series.instances[instanceIndex];

            this.clearSelection();

            _.each(instance.instanceAttributes.measurements, function (m) {
                if (m instanceof Measurements.Area) {
                    var area = m;
                    if (area.color === color) {
                        _this.selectedMeasurement.write(m);
                        m.selected = true;
                    }
                }
            });
        };

        /**
        * Reset only the window level
        */
        Series.prototype.resetWindowLevel = function () {
            this.useOriginalWindowLevel.write(true);
            this.windowLevel.write(this.getDefaultWindowLevel());
            this.subtractionWindowLevel = null;
            var original = this.windowLevel.read();
            this.preSubtractionWindowLevel = { center: original.center, width: original.width };
            this.recordWindowLevel();
        };

        /**
        * Export the current instance to PNG format
        */
        Series.prototype.exportCurrentImages = function () {
            var uri = this.renderImageToDataURI();
            this.renderAll();

            if (LocalViewer.isStandardLocalViewer()) {
                window.open(uri);
            } else {
                var html = $('<img>').attr('src', uri)[0].outerHTML;
                $(window.open().document.body).html(html);
            }
        };

        /**
        * Store current instance capture
        */
        Series.prototype.storeCurrentImage = function () {
            var dataURL = this.prepareCanvasForReport().toDataURL("image/png");
            this.renderAll(); // clear the prepared canvas
            SecondaryCapture.postCurrentImage(this.sessionId, this, dataURL);
        };

        /**
        * Export the current series to video
        */
        Series.prototype.exportVideo = function (format) {
            var uri = Routes.VideoDownload(this.sessionId, this.series.studyAttributes.studyStorage, this.series.studyAttributes.queryObject, this.series.seriesAttributes.seriesUid, format);

            window.location.href = uri;
        };

        /**
        * Export the current series to zip
        */
        Series.prototype.exportSeries = function () {
            window.location.href = Routes.SeriesDownload(this.sessionId, this.series.studyAttributes.studyStorage, this.series.studyAttributes.queryObject, this.series.seriesAttributes.seriesUid);
            ;
        };

        /**
        * Export this series' study to zip
        */
        Series.prototype.exportStudy = function () {
            window.location.href = Routes.StudyDownload(this.sessionId, this.series.studyAttributes.studyStorage, this.series.studyAttributes.queryObject);
            ;
        };

        /**
        * Export this series' study as ISO
        */
        Series.prototype.exportISO = function () {
            var uri = window.location.href = Routes.ISODownload(this.sessionId, this.series.studyAttributes.studyStorage, this.series.studyAttributes.queryObject);
            ;
        };

        /**
        * Export this series' study as a local viewer
        */
        Series.prototype.exportLocalViewer = function () {
            window.location.href = Routes.LocalViewerDownload(this.sessionId, this.series.studyAttributes.studyStorage, this.series.studyAttributes.queryObject);
            ;
        };

        /**
        * Prepare the canvas for export to PNG
        */
        Series.prototype.prepareCanvasForExport = function (layers, forceSquare) {
            var _this = this;
            if (typeof forceSquare === "undefined") { forceSquare = false; }
            var canvas = this.canvas;

            this.renderAll(function (context, instance) {
                context.fillStyle = "#000000";
                context.fillRect(0, 0, canvas.width, canvas.height);

                _this.renderer.act({
                    visitSimpleRenderer: function (renderer) {
                        var transform = _this.getImageTransformation(canvas.width, canvas.height, instance.instanceAttributes.columns, instance.instanceAttributes.rows);

                        context.save();
                        context.transform(transform.entries[0], transform.entries[3], transform.entries[1], transform.entries[4], transform.entries[2], transform.entries[5]);

                        var instanceIndex = _this.selectedInstanceIndex.read();
                        var instanceKey = _this.getInstanceKey(instanceIndex);

                        var $image = renderer.imageElements.get(instanceKey, null);

                        if ($image) {
                            context.drawImage($image.imageElement, 0, 0);
                        }

                        context.restore();
                    },
                    visitCanvasRenderer: function (renderer) {
                        context.drawImage(renderer.canvas2d, 0, 0);
                    },
                    visitWebGLRenderer: function (renderer) {
                        context.drawImage(renderer.canvas3d, 0, 0);
                    }
                });
            }, false, layers, forceSquare);

            return canvas;
        };

        /**
        * Draw canvas in square, thumbnail-sized dimensions, hiding all small textual info, for use in reports
        * @returns {HTMLCanvasElement}
        */
        Series.prototype.prepareCanvasForReport = function (selected, targetSize) {
            if (typeof selected === "undefined") { selected = false; }
            if (typeof targetSize === "undefined") { targetSize = 256; }
            var offScreenCanvas = document.createElement('canvas');
            offScreenCanvas.width = targetSize;
            offScreenCanvas.height = targetSize;
            var offScreenContext = offScreenCanvas.getContext("2d");

            // fit image to thumbnail frame
            var instanceIndex = this.selectedInstanceIndex.read();
            var instance = this.series.instances[instanceIndex];
            var imageCols = instance.instanceAttributes.columns;
            var imageRows = instance.instanceAttributes.rows;
            var transform = this.getImageTransformation(this.canvas.width, this.canvas.height, imageCols, imageRows);
            var longDim;

            if (imageCols > imageRows) {
                longDim = this.canvas.width - (transform.entries[2] * 2.0);
            } else {
                longDim = this.canvas.height - (transform.entries[5] * 2.0);
            }

            var layers = [];

            var measurementLayer = new Layers.MeasurementLayer(this.measurementsVisible, this.measurementsDetailsVisible, this.annotationsCreatedByOtherUsersVisible, this.renderer, this.measurementInProgress, this.selectedTool, this.selectedMeasurement, this, this.playbackMode, this.showStandardDev, this.showTextOnDirected, this.editingMeasurement, this.hideActiveMeasuremntInfo, this.alwaysUseMillimeters, Layers.DEFAULT_TEXT_SIZE, this.measureVolume, false, this.application.user);
            measurementLayer.alwaysHideInfo = true; // textual info is too small
            measurementLayer.lineWidth = (measurementLayer.lineWidth * (longDim / targetSize)); // maintain relative line thickness

            layers.push(new Layers.OverlayLayer(this.measurementsVisible, this.overlayData, this, false));
            layers.push(measurementLayer);
            var canvas = this.prepareCanvasForExport(layers);

            offScreenContext.fillStyle = "#000000";
            offScreenContext.fillRect(0, 0, offScreenCanvas.width, offScreenCanvas.height);
            offScreenContext.drawImage(canvas, (canvas.width - longDim) / 2.0, (canvas.height - longDim) / 2.0, longDim, longDim, 0, 0, offScreenCanvas.width, offScreenCanvas.height);

            if (selected) {
                Rendering.drawRectangleBorder(offScreenContext, 1.5, 1.5, offScreenContext.canvas.width - 3, offScreenContext.canvas.height - 3, '#00ffff');
            }

            return offScreenCanvas;
        };

        Series.prototype.renderImageToDataURI = function () {
            return this.prepareCanvasForExport(null, this.settings.read().exportSquare).toDataURL("image/png");
        };

        /**
        * Export the current instance as a secondary capture image
        */
        Series.prototype.secondaryCapture = function () {
            var _this = this;
            var uri = this.renderImageToDataURI();
            this.renderAll();

            var storeAndLoadImage = SecondaryCapture.secondaryCapture(this.sessionId, this.series, uri);

            storeAndLoadImage.subscribe({
                next: function (newSeries) {
                    _this.application.addSeries(newSeries);
                },
                done: function () {
                },
                fail: function (err) {
                    _this.recordError("Unable to save secondary capture image: " + err);
                    window.alert(_this.terminology.lookup(Terminology.Terms.SecondaryCaptureFailed));
                }
            });
        };

        /**
        * Export the current annotations as a GSPS object
        */
        Series.prototype.saveGSPS = function () {
            var _this = this;
            var editableAnnotations = _.flatten(_.map(this.series.instances, function (instance) {
                return _.filter(instance.instanceAttributes.measurements, function (m) {
                    return m.editable && !(m instanceof Measurements.Area);
                });
            }));

            if (_.any(editableAnnotations)) {
                $('.overlay').addClass('application-loading');

                var instanceIndex = this.selectedInstanceIndex.read();
                var instance = this.series.instances[instanceIndex];
                var instanceKey = this.getInstanceKey(instanceIndex);

                var gsps = GSPS.createGSPS(editableAnnotations, this.renderer, instanceKey, instance, this.showStandardDev.read());

                var storeAndLoadObject = Observable._finally(V3Storage.postGSPS(this.sessionId, this.series.studyAttributes.studyStorage, this.series.studyAttributes.queryObject, gsps), function () {
                    return $('.overlay').removeClass('application-loading');
                });

                storeAndLoadObject.subscribe({
                    next: function (imageResult) {
                        var study = _.find(_this.application.studies, function (study) {
                            return study.studyAttributes.queryObject.toString() === _this.series.studyAttributes.queryObject.toString();
                        });

                        _this.deleteAllEditableMeasurements(true);
                        GSPS.apply([gsps], study);

                        _this.renderAll();
                    },
                    done: function () {
                    },
                    fail: function (err) {
                        _this.recordError("Unable to save GSPS: " + err);
                        window.alert(_this.terminology.lookup(Terminology.Terms.SaveGSPSFailed));
                    }
                });
            }
        };

        /**
        * Toggle cine mode
        */
        Series.prototype.toggleCine = function (active) {
            var _this = this;
            var imagePositionSubject = this.selectedInstanceIndex;

            if (this.series && !Multiframe.isMultiframe(this.series) || !Cine.isFormatSupported()) {
                if (active) {
                    if (!LocalViewer.isLocalViewer()) {
                        this.loadAllImageData(this.imagePreloadQueue, -2 /* ImageDataBackgroundPreloading */);
                    }

                    this.resetCineStartPosition();
                    this.cineStartFrame = imagePositionSubject.read();
                    this.forceLowResolution.write(this.forceLowResolution.read().take());
                    this.cineTimer = window.setInterval(function () {
                        var time = new Date().getTime();
                        var elapsed = time - _this.cineStartTime;
                        var cineSpeed = _this.cineSpeed.read();
                        var frame = (_this.cineStartFrame + elapsed * cineSpeed / 1000) % _this.series.instances.length;
                        imagePositionSubject.write(Math.floor(frame));
                    }, 10);
                } else {
                    this.imagePreloadQueue.clear();
                    this.forceLowResolution.write(this.forceLowResolution.read().release());
                    window.clearInterval(this.cineTimer);
                }
            }

            this.renderAll();
        };

        /**
        * Update the cineStartFrame and cineStartTime in response to a change in frame rate, or start/stop event
        */
        Series.prototype.resetCineStartPosition = function () {
            this.cineStartFrame = this.selectedInstanceIndex.read();
            this.cineStartTime = new Date().getTime();
        };

        /**
        * Increase the cine playback frame rate
        */
        Series.prototype.increaseCineSpeed = function () {
            var cineSpeed = this.cineSpeed.read();
            if (++cineSpeed > 60) {
                cineSpeed = 60;
            }
            this.cineSpeed.write(cineSpeed);

            this.resetCineStartPosition();
        };

        /**
        * Decrease the cine playback frame rate
        */
        Series.prototype.decreaseCineSpeed = function () {
            var cineSpeed = this.cineSpeed.read();
            if (--cineSpeed < 1) {
                cineSpeed = 1;
            }
            this.cineSpeed.write(cineSpeed);

            this.resetCineStartPosition();
        };

        /**
        * Delete the selected image
        */
        Series.prototype.deleteSelectedImage = function () {
            var _this = this;
            if (this.series) {
                var selectedInstanceIndex = this.selectedInstanceIndex.read();
                var instance = this.series.instances[selectedInstanceIndex];

                if (this.application.singleSeriesEnabled.read()) {
                    selectedInstanceIndex = _.indexOf(this.application.seriesViews.read(), this) + this.application.singleSeriesIndex;
                    instance = this.application.singleSeries.instances[selectedInstanceIndex];
                }

                var message = this.terminology.lookup(Terminology.Terms.DeleteImageWarning) + "\n\n" + instance.instanceAttributes.seriesDescription + ", " + this.terminology.lookup(Terminology.Terms.Instance) + " #" + (instance.instanceAttributes.instanceNumber + 1);

                if (window.confirm(message)) {
                    var deleteImage = V3Storage.deleteImage(this.sessionId, instance.studyAttributes.studyStorage, instance.studyAttributes.queryObject, instance.id, instance.instanceAttributes.version);

                    $('.overlay').addClass('application-loading');

                    Observable._finally(deleteImage, function () {
                        $('.overlay').removeClass('application-loading');
                    }).subscribe({
                        done: function () {
                            if (_this.application.singleSeriesEnabled.read()) {
                                _this.application.singleSeries.instances.splice(selectedInstanceIndex, 1);

                                if (_this.application.singleSeries.instances.length) {
                                    _this.application.resetSeriesViews();
                                    _this.application.render(_this.application.singleSeries);
                                } else {
                                    _this.application.removeEmptySeries(_this.series);
                                }
                            } else {
                                _this.series.instances.splice(selectedInstanceIndex, 1);

                                if (_this.series.instances.length) {
                                    _this.selectedInstanceIndex.write(0);
                                } else {
                                    _this.application.removeEmptySeries(_this.series);
                                }
                            }
                        },
                        next: function (_) {
                        },
                        fail: function (err) {
                            _this.recordError("Unable to delete image: " + err);
                            window.alert(_this.terminology.lookup(Terminology.Terms.ErrorDeletingImage));
                        }
                    });
                }
            }
        };

        /**
        * Find the instance which is geometrically closest to the specified instance and show it.
        */
        Series.prototype.scrollToNearestInstance = function (instance) {
            if (this.allAttributesAreLoaded()) {
                var index = SeriesGeometry.findClosestInstance(this.series.instances, instance);

                if (index !== null) {
                    this.selectedInstanceIndex.write(index);
                }
            }
        };

        /**
        * Find the instance which is geometrically closest to the specified point and show it.
        */
        Series.prototype.scrollToNearestInstanceToPoint = function (patientCoords) {
            if (this.allAttributesAreLoaded()) {
                var index = SeriesGeometry.findClosestInstanceToPoint(this.series.instances, patientCoords);

                if (index !== null) {
                    this.selectedInstanceIndex.write(index);
                }
            }
        };

        /**
        * Check whether all image attributes got loaded
        */
        Series.prototype.allAttributesAreLoaded = function () {
            return _.all(this.series.instances, function (instance) {
                return instance.instanceAttributes.attributesLoaded;
            });
        };

        /**
        * Gets the currently selected instance
        */
        Series.prototype.currentInstance = function () {
            var _this = this;
            return Subjects.map(this.selectedInstanceIndex, function (index) {
                return _this.series ? _this.series.instances[index] : null;
            });
        };

        /**
        * Update the global plane localization cursor
        */
        Series.prototype.updateLocalizationCursor = function (screenCoords) {
            var instance = this.currentInstance().read();

            if (SeriesGeometry.hasGeometricMetadata(instance)) {
                var el = this.canvas;

                var rows = instance.instanceAttributes.rows;
                var cols = instance.instanceAttributes.columns;

                var imageCoords = this.mapToImage(screenCoords, el.width, el.height, cols, rows);

                var patientCoords = SeriesGeometry.mapToPatient(imageCoords, instance);

                this.planeLocalizationCursor.write({
                    studyUid: this.series.studyAttributes.queryObject.studyUid,
                    seriesUid: this.series.seriesAttributes.seriesUid,
                    coords: patientCoords
                });
            }
        };

        /**
        * Cine opens in a new tab on iOS devices
        */
        Series.prototype.openMultiframeCineInNewTab = function () {
            if (Cine.isFormatSupported()) {
                var $popupBody = Popup.createPopup();

                var instance = this.currentInstance().read();

                var video = new Views.Video(this.sessionId, $popupBody, this.settings, instance, this.terminology);
                video.render();
            } else {
                window.alert(this.terminology.lookup(Terminology.Terms.VideoNotSupported));
            }
        };

        /**
        * Mark the selected image as a key image or not
        */
        Series.prototype.updateKeyImageStatusInServices = function () {
            var _this = this;
            var instance = this.currentInstance().read();

            if (instance) {
                if (instance.instanceAttributes.isKeyImage.read()) {
                    Services.addKeyImage(this.sessionId, instance.studyAttributes.queryObject, instance.seriesAttributes.seriesUid, instance.id, instance.frameNumber, instance.instanceAttributes.version).subscribe({
                        done: function () {
                        },
                        next: function (response) {
                            instance.instanceAttributes.keyImageId = new Classes.KeyImageId(response.uuid);
                        },
                        fail: function (err) {
                            _this.recordError("Unable to add key image: " + err);
                        }
                    });
                } else if (instance.instanceAttributes.keyImageId) {
                    Services.deleteKeyImage(this.sessionId, instance.instanceAttributes.keyImageId).subscribe({
                        done: function () {
                        },
                        next: function (_) {
                            instance.instanceAttributes.keyImageId = null;
                        },
                        fail: function (err) {
                            _this.recordError("Unable to delete key image: " + err);
                        }
                    });
                }
            }
        };

        /**
        * Get the image rendering parameters based on the current transformation settings
        */
        Series.prototype.getImageTransformation = function (dw, dh, sw, sh) {
            var transform = this.transform.read();

            return Transform.transformToMatrix(transform, dw, dh, sw, sh);
        };

        /**
        * Get the inverse image rendering parameters based on the current transformation settings
        */
        Series.prototype.getInverseImageTransformation = function (dw, dh, sw, sh) {
            var transform = this.transform.read();

            return Transform.transformToInverseMatrix(transform, dw, dh, sw, sh);
        };

        /**
        * Map canvas coordinates to image coordinates
        */
        Series.prototype.mapToImage = function (p, dw, dh, sw, sh) {
            var transform = this.getInverseImageTransformation(dw, dh, sw, sh);

            return Vectors.multiplyM(transform, p);
        };

        /**
        * Map image coordinates to canvas coordinates
        */
        Series.prototype.mapFromImage = function (p, dw, dh, sw, sh) {
            var transform = this.getImageTransformation(dw, dh, sw, sh);

            return Vectors.multiplyM(transform, p);
        };

        Series.prototype.recordError = function (message) {
            this.application.recordSeriesError(this.series.seriesAttributes.seriesUid, message, "Series");
        };

        /**
        * Returns true if this series is currently being displayed.
        * @returns {boolean}
        */
        Series.prototype.isActive = function () {
            return (this.canvas && ($(this.canvas).hasClass('instance-canvas')) && document.body.contains(this.canvas));
        };

        Series.prototype.supportsSubtraction = function () {
            return this.series && this.subtraction && Dicom.Subtraction.supportsSubtraction(this.series.seriesAttributes) && (this.findRenderingMode() === 0 /* Canvas */);
        };
        return Series;
    })();
    Views.Series = Series;
})(Views || (Views = {}));
///<reference path='../typings/jquery/jquery.d.ts' />
///<reference path='../typings/underscore/underscore.d.ts' />
///<reference path='../classes/Types.ts' />
///<reference path='../libs/Observable.ts' />
///<reference path='../libs/Query.ts' />
///<reference path='../libs/Services.ts' />
///<reference path='../libs/V3Storage.ts' />
///<reference path='../libs/Study.ts' />
///<reference path='../models/StudySchema.ts' />
///<reference path='../libs/Subject.ts' />
///<reference path='../libs/Touch.ts' />
///<reference path='../libs/WindowLevelPresets.ts' />
///<reference path='../models/Study.ts' />
///<reference path='Series.ts' />
var Views;
(function (Views) {
    

    

    /**
    * A collection of helper methods for creating toolbars
    */
    var MenuBar = (function () {
        function MenuBar($toolbar) {
            this.$toolbar = $toolbar;
            this.listeners = [];
        }
        /**
        * Add a text label to a toolbar
        */
        MenuBar.prototype.addLabel = function (options) {
            var label = $('<div>').addClass('toolbarLabel').text(options.text.read());

            var li = $('<div>').append(label);

            this.listeners.push(Subjects.listen(options.text, function (text) {
                label.text(text);
            }));

            if (options.visible) {
                this.listeners.push(Subjects.listen(options.visible, function (value) {
                    if (!value) {
                        li.hide();
                    } else {
                        li.show();
                    }
                }));

                if (!options.visible.read()) {
                    li.hide();
                }
            }

            this.$toolbar.append(li);

            return li;
        };

        /**
        * Add a button to a toolbar
        *
        * @param {$toolbar} The toolbar
        * @param {options.title} The text to display on the button
        * @param {options.tooltip} An optional tooltip
        * @param {options.icon} The name of the icon to display
        * @param {options.click} An optional action to perform on clicking
        * @param {options.enabled} A subject which determines whether the button is active or not
        * @param {options.closePopupsOnClick} True if clicking the button should close all open dropdown menus
        *
        * @returns The button jQuery DOM wrapper object
        */
        MenuBar.prototype.addToolbarButton = function (options) {
            if (!options.tooltip && options.title) {
                options.tooltip = options.title;
            }

            var icon = $('<div>').addClass("fa").addClass('imageControlIcon');

            var label = $('<div>').addClass('imageControlLabel');

            var li = $('<div>').addClass('imageControl').append(icon).append(label).addClass('glow');

            if (options.dataDynamic) {
                var data = options.dataDynamic.read();

                label.text(data.title);
                li.attr({ title: data.tooltip });
                icon.removeClass(icon.data("current-icon")).addClass(ToolbarButtons.iconFor(data.icon)).data("current-icon", ToolbarButtons.iconFor(data.icon));

                this.listeners.push(Subjects.listen(options.dataDynamic, function (data) {
                    label.text(data.title);
                    li.attr({ title: data.tooltip });

                    icon.removeClass(icon.data("current-icon")).addClass(ToolbarButtons.iconFor(data.icon)).data("current-icon", ToolbarButtons.iconFor(data.icon));
                }));
            } else {
                if (options.title) {
                    label.text(options.title);
                } else {
                    label.html('&nbsp;');
                    li.addClass('blank');
                }
                li.attr({ title: options.tooltip });

                icon.addClass(ToolbarButtons.iconFor(options.icon)).data("current-icon", ToolbarButtons.iconFor(options.icon));
            }

            if (options.click) {
                li.fallback(["touchstart", "pointerdown"], ["click"], function (e) {
                    if (!options.enabled || options.enabled.read()) {
                        options.click(e);
                    }

                    if (options.closePopupsOnClick === true) {
                        $('.dropdown-icon').removeClass('dropdown-visible');
                    }

                    e.stopPropagation();
                });
            }

            li.on("contextmenu", function (e) {
                e.preventDefault();
            });

            li.dblclick(function (e) {
                e.stopPropagation();
                e.preventDefault();
            });

            if (options.enabled) {
                this.listeners.push(Subjects.listen(options.enabled, function (value) {
                    if (!value) {
                        li.addClass('disabled');
                    } else {
                        li.removeClass('disabled');
                    }
                }));

                if (!options.enabled.read()) {
                    li.addClass('disabled');
                }
            }

            this.$toolbar.append(li);

            return li;
        };

        /**
        * Add a toggle button to a toolbar
        *
        * @param {$toolbar} The toolbar
        * @param {options.title} The text to display on the button
        * @param {options.tooltip} An optional tooltip
        * @param {options.icon} The name of the icon to display
        * @param {options.tool} The mouse tool to activate on clicking
        * @param {options.selected} A writable subject which determines whether the button is selected
        * @param {options.closePopupsOnClick} True if clicking the button should close all open dropdown menus
        *
        * @returns The button jQuery DOM wrapper object
        */
        MenuBar.prototype.addToggleButton = function (options) {
            var li = this.addToolbarButton({
                title: options.title,
                tooltip: options.tooltip,
                enabled: options.enabled,
                visible: options.visible,
                icon: options.icon,
                dataDynamic: options.toggledData ? Subjects.map(options.selected, function (s) {
                    return s ? options.toggledData : {
                        title: options.title,
                        tooltip: options.tooltip,
                        icon: options.icon
                    };
                }) : null,
                click: function (e) {
                    var newValue = !options.selected.read();
                    options.selected.write(newValue);

                    if (newValue) {
                        if (options.toggledData && options.toggledData.baseClass) {
                            li.addClass(options.toggledData.baseClass + '-selected');
                            li.removeClass(options.toggledData.baseClass + '-unselected');
                        } else {
                            li.addClass('selected');

                            if (options.highlightParent) {
                                li.parents('.dropdown-icon').addClass('selected');
                            }
                        }
                    } else {
                        if (options.toggledData && options.toggledData.baseClass) {
                            li.removeClass(options.toggledData.baseClass + '-selected');
                            li.addClass(options.toggledData.baseClass + '-unselected');
                        } else {
                            li.removeClass('selected');

                            // Only remove selected if no other children are selected
                            if (options.highlightParent && li.parents('.dropdown-icon').children('.dropdown-content').children('.selected').length < 1) {
                                li.parents('.dropdown-icon').removeClass('selected');
                            }
                        }
                    }
                    if (options.click) {
                        options.click(e);
                    }
                },
                closePopupsOnClick: options.closePopupsOnClick
            });

            if (options.highlightParent) {
                li.addClass("highlight-parent");
            }

            this.listeners.push(Subjects.listen(options.selected, function (value) {
                if (value) {
                    if (options.toggledData && options.toggledData.baseClass) {
                        li.addClass(options.toggledData.baseClass + '-selected');
                        li.removeClass(options.toggledData.baseClass + '-unselected');
                    } else {
                        li.addClass('selected');

                        if (options.highlightParent) {
                            li.parents('.dropdown-icon').addClass('selected');
                        }
                    }
                } else {
                    if (options.toggledData && options.toggledData.baseClass) {
                        li.removeClass(options.toggledData.baseClass + '-selected');
                        li.addClass(options.toggledData.baseClass + '-unselected');
                    } else {
                        li.removeClass('selected');

                        // Only remove selected if no other children are selected
                        if (options.highlightParent && li.parents('.dropdown-icon').children('.dropdown-content').children('.selected').length < 1) {
                            li.parents('.dropdown-icon').removeClass('selected');
                        }
                    }
                }
            }));

            if (options.selected.read()) {
                if (options.toggledData && options.toggledData.baseClass) {
                    li.addClass(options.toggledData.baseClass + '-selected');
                } else {
                    li.addClass('selected');
                }

                li.find('.selectedTool').addClass('disp');
            }

            if (options.selected2) {
                this.listeners.push(Subjects.listen(options.selected2, function (value) {
                    if (value) {
                        li.find('.selectedTool2').addClass('disp');
                    } else {
                        li.find('.selectedTool2').removeClass('disp');
                    }
                }));

                if (options.selected2.read()) {
                    li.find('.selectedTool2').addClass('disp');
                }
            }

            return li;
        };

        /**
        * Add a drop down button to a toolbar
        *
        * @param {$toolbar} The toolbar
        * @param {options.title} The text to display on the button
        * @param {options.tooltip} An optional tooltip
        * @param {options.icon} The name of the icon to display
        *
        * @returns The dropdown menu jQuery DOM wrapper object.
        */
        MenuBar.prototype.addDropDownButton = function (options, buttons) {
            var li = this.addToolbarButton({
                title: options.title,
                tooltip: options.tooltip,
                visible: options.visible,
                icon: options.icon,
                click: function () {
                    if (li.is('.dropdown-visible')) {
                        li.removeClass('dropdown-visible');
                    } else {
                        $('.dropdown-icon').removeClass('dropdown-visible');
                        li.addClass('dropdown-visible');
                    }
                }
            });

            li.addClass('dropdown-icon');

            if (!options.title) {
                li.addClass('no-title');
            }

            var $submenu = $('<div>').addClass('dropdown-content');
            var submenu = new MenuBar($submenu);

            this.listeners.concat(submenu.listeners);

            _.each(buttons, function (button) {
                return button.add(submenu);
            });

            $submenu.appendTo(li);

            if ($submenu.children('.highlight-parent.selected').length > 0) {
                li.addClass('selected');
            }

            return li;
        };

        /**
        * Add layout buttons to a toolbar for the standard layout modes (1x1, 1x2, 2x2 and 2x3)
        *
        * @param {$toolbar} The toolbar
        * @param {layout} A subject which determines the current layout setting
        */
        MenuBar.prototype.addLayoutButtons = function (layout) {
            var $layoutControl = $('<div>').addClass('layoutControl');

            $layoutControl.fallback(["touchstart", "pointerdown"], ["mouseup"], function (e) {
                var d = $(e.target).data();

                if (d != null && d.rows != null && d.cols != null) {
                    layout.write({ rows: d.rows, columns: d.cols });
                }
            });

            _.each(_.range(1, 5), function (rows) {
                var $row = $('<div>').addClass('layoutRow');

                _.each(_.range(1, 5), function (cols) {
                    var $cell = $('<span>').addClass('layoutCell').data({
                        rows: rows,
                        cols: cols
                    });

                    $cell.hover(function () {
                        var $cells = $layoutControl.find('.layoutCell');
                        $cells.each(function () {
                            var r = $(this).data('rows');
                            var c = $(this).data('cols');
                            $(this).toggleClass('layoutGlow', r <= rows && c <= cols);
                        });
                    }, function () {
                        var $cells = $layoutControl.find('.layoutCell');
                        $cells.removeClass('layoutGlow');
                    });

                    var $square = $('<span>').addClass('fa fa-square').css('display', 'inline-block').attr('title', cols + "x" + rows).data({
                        rows: rows,
                        cols: cols
                    }).appendTo($cell);

                    $row.append($cell);
                });

                $layoutControl.append($row);
            });

            this.$toolbar.append($layoutControl);

            return [$layoutControl];
        };

        MenuBar.prototype.addColorButtons = function (currentColor) {
            var $colorControl = $('<div>').addClass('colorControl');
            $colorControl.fallback(["touchstart", "pointerdown"], ["mouseup"], function (e) {
                var d = $(e.target).data();

                if (d != null) {
                    currentColor.write(d.color);
                    $colorControl.parents('.dropdown-icon').css("color", Annotations.COLORS[d.color]);
                }
            });

            var $row = $('<div>').addClass('layoutRow');

            _.each(Annotations.COLORS, function (color, index) {
                var $cell = $('<span>').addClass('layoutCell').data({ color: index });

                $('<span>').addClass('fa fa-square').css('display', 'inline-block').css("color", color).attr('title', "Color " + (index + 1)).data({ color: index }).appendTo($cell);

                $row.append($cell);
            });

            $colorControl.append($row);
            this.$toolbar.append($colorControl);

            return [$colorControl];
        };

        MenuBar.prototype.addMouseToolSettings = function (application) {
            var settings = application.settings.read();
            var selectedToolOriginalValue = settings.toolSelection;

            var selectedToolDidChange = false;

            var $overlay = $('.overlay');

            $overlay.html('');
            $overlay.addClass('overlay-standard');
            $overlay.show();

            var mousePanel = $('<div id="mouseSettingsPanel">');
            mousePanel.on("contextmenu", function (e) {
                e.preventDefault();
            });

            var header = $('<div id="mouseSettingsHeader">');

            var txtHeader = application.terminology.lookup(Terminology.Terms.MouseTooltip);
            var h1 = $("<h1>").text(txtHeader);

            var txtCancel = application.terminology.lookup(Terminology.Terms.Cancel);
            var btnCancel = $('<button type="button" class="btn">' + txtCancel + '</button>');

            var txtSave = application.terminology.lookup(Terminology.Terms.Save);
            var btnSave = $('<button type="button" class="btn btn-primary">' + txtSave + '</button>');

            var hidePanel = function () {
                $overlay.hide();
                mousePanel.remove();
            };

            btnCancel.on("click", function (e) {
                hidePanel();
            });

            btnSave.on("click", function (e) {
                // Only update the left mouse button tool if the user changes the dropdown
                if (selectedToolDidChange) {
                    application.selectedTool.write(parseInt(toolSelection.val()));

                    settings.toolSelection = parseInt(toolSelection.val());
                }

                // Always update the right mouse button and wheel
                application.selectedTool2.write(parseInt(toolSelection2.val()));
                application.selectedToolWheel.write(parseInt(toolSelectionWheel.val()));

                settings.toolSelection2 = parseInt(toolSelection2.val());
                settings.toolSelectionWheel = parseInt(toolSelectionWheel.val());

                // Writing to settings automatically fires the AJAX request to save the settings (See Application.ts)
                application.settings.write(settings);

                hidePanel();
            });

            var list = [
                { type: 0 /* Move */, name: application.terminology.lookup(Terminology.Terms.Move) },
                { type: 1 /* Scroll */, name: application.terminology.lookup(Terminology.Terms.Scroll) },
                { type: 2 /* Zoom */, name: application.terminology.lookup(Terminology.Terms.Zoom) },
                { type: 14 /* FreeRotate */, name: application.terminology.lookup(Terminology.Terms.FreeRotate) },
                { type: 3 /* Window */, name: application.terminology.lookup(Terminology.Terms.WindowLevelTooltip) },
                { type: 4 /* Select */, name: application.terminology.lookup(Terminology.Terms.SelectAnnotationTooltip) },
                { type: 5 /* Measure */, name: application.terminology.lookup(Terminology.Terms.Line) },
                { type: 12 /* Arrow */, name: application.terminology.lookup(Terminology.Terms.Arrow) },
                { type: 13 /* Angle */, name: application.terminology.lookup(Terminology.Terms.Angle) },
                { type: 7 /* CobbAngle */, name: application.terminology.lookup(Terminology.Terms.Cobb) },
                { type: 6 /* Rectangle */, name: application.terminology.lookup(Terminology.Terms.Rectangle) },
                { type: 8 /* Ellipse */, name: application.terminology.lookup(Terminology.Terms.Ellipse) },
                { type: 10 /* Text */, name: application.terminology.lookup(Terminology.Terms.Text) },
                { type: 11 /* Probe */, name: application.terminology.lookup(Terminology.Terms.Probe) },
                { type: 9 /* Localization */, name: application.terminology.lookup(Terminology.Terms.PlaneLocalization) }
            ];

            var listWheel = [
                { type: 34 /* None */, name: application.terminology.lookup(Terminology.Terms.ToolDependent) },
                { type: 2 /* Zoom */, name: application.terminology.lookup(Terminology.Terms.Zoom) },
                { type: 1 /* Scroll */, name: application.terminology.lookup(Terminology.Terms.Scroll) }
            ];

            // Left Mouse
            var toolSelection = $('<select id="toolSelection" name="toolSelection" />');
            for (var i = 0; i < list.length; i++) {
                $("<option />", { value: list[i].type, text: list[i].name }).appendTo(toolSelection);
            }

            toolSelection.val(settings.toolSelection.toString());

            toolSelection.on("change", function () {
                selectedToolDidChange = parseInt(toolSelection.val()) != selectedToolOriginalValue;
            });

            // Right Mouse
            var toolSelection2 = $('<select id="toolSelection2" name="toolSelection2" />');
            for (var i = 0; i < list.length; i++) {
                $("<option />", { value: list[i].type, text: list[i].name }).appendTo(toolSelection2);
            }

            toolSelection2.val(settings.toolSelection2.toString());

            // Wheel
            var toolSelectionWheel = $('<select id="toolSelectionWheel" name="toolSelectionWheel" />');
            for (var i = 0; i < listWheel.length; i++) {
                $("<option />", { value: listWheel[i].type, text: listWheel[i].name }).appendTo(toolSelectionWheel);
            }

            toolSelectionWheel.val(settings.toolSelectionWheel.toString());

            // Build Panel
            header.append(btnSave);
            header.append(btnCancel);
            header.append($("<h2/>").text(application.terminology.lookup(Terminology.Terms.Mouse)));

            mousePanel.append(header);

            mousePanel.append(toolSelection2);
            mousePanel.append($("<h4/>").text(application.terminology.lookup(Terminology.Terms.RightButton)));

            mousePanel.append(toolSelection);
            mousePanel.append($("<h4/>").text(application.terminology.lookup(Terminology.Terms.LeftButton)));

            mousePanel.append(toolSelectionWheel);
            mousePanel.append($("<h4/>").text(application.terminology.lookup(Terminology.Terms.WheelScroll)));

            $overlay.after(mousePanel);

            return [];
        };
        return MenuBar;
    })();
    Views.MenuBar = MenuBar;

    /**
    * A collection of helper methods for creating toolbars
    */
    var ContextMenu = (function () {
        function ContextMenu() {
        }
        ContextMenu.prototype.addLabel = function (options) {
            var item = {
                name: options.text.read(),
                type: "html"
            };

            if (options.visible !== undefined && !options.visible.read()) {
                item.disabled = true;
            }

            return item;
        };

        ContextMenu.prototype.addToolbarButton = function (options) {
            var item = {
                name: options.tooltip
            };

            if (options.click !== undefined) {
                item.callback = function (key, opts) {
                    options.click();
                    return true;
                };
            }

            if (options.enabled !== undefined && !options.enabled.read()) {
                item.disabled = true;
            }

            return item;
        };

        ContextMenu.prototype.addToggleButton = function (options) {
            return this.addToolbarButton({
                title: (options.selected.read() ? "\u2713 " : "") + options.title,
                tooltip: (options.selected.read() ? "\u2713 " : "") + options.tooltip,
                click: function () {
                    return Subjects.modify(options.selected, function (b) {
                        return !b;
                    });
                },
                enabled: options.enabled
            });
        };

        /**
        * Add a drop down button to a toolbar
        *
        * @param {$toolbar} The toolbar
        * @param {options.title} The text to display on the button
        * @param {options.tooltip} An optional tooltip
        * @param {options.icon} The name of the icon to display
        *
        * @returns The dropdown menu jQuery DOM wrapper object.
        */
        ContextMenu.prototype.addDropDownButton = function (options, buttons) {
            return {
                name: options.title,
                items: _.flatten(_.map(buttons, function (button) {
                    return button.add(new ContextMenu());
                }))
            };
        };

        /**
        * Context menus do not support layout buttons
        */
        ContextMenu.prototype.addLayoutButtons = function (layout) {
            return [];
        };

        ContextMenu.prototype.addColorButtons = function (currentColor) {
            return [];
        };

        ContextMenu.prototype.addMouseToolSettings = function (application) {
            return [];
        };

        /**
        * Apply a toolbar configuration, by attaching its items to that element's
        * context menu
        */
        ContextMenu.apply = function (selector, toolbarConfiguration, application) {
            $.contextMenu({
                selector: selector,
                build: function () {
                    var items = AbstractToolbars.createToolbarFromConfiguration(new ContextMenu(), toolbarConfiguration, application);

                    return {
                        items: new ContextMenuItemRenamer().rename(items)
                    };
                }
            });
        };
        return ContextMenu;
    })();
    Views.ContextMenu = ContextMenu;

    /**
    * This class is used to assign unique keys to submenu items to work around
    * the way in which jQuery.contextMenu handles arrays.
    */
    var ContextMenuItemRenamer = (function () {
        function ContextMenuItemRenamer() {
            this.index = 0;
        }
        ContextMenuItemRenamer.prototype.rename = function (items) {
            var _this = this;
            var result = {};

            _.each(items, function (item) {
                var key = "item" + _this.index;
                result[key] = item;

                if (item.items) {
                    item.items = _this.rename(item.items);
                }

                _this.index++;
            });

            return result;
        };
        return ContextMenuItemRenamer;
    })();

    var AbstractToolbars = (function () {
        function AbstractToolbars() {
        }
        /**
        * Add a dropdown button to an abstract toolbar.
        *
        * Note: this function is only needed here to avoid type inference issues in TypeScript.
        */
        AbstractToolbars.addDropDownButton = function (toolbar, options, buttons) {
            return toolbar.addDropDownButton(options, _.map(buttons, function (button) {
                return {
                    add: function (toolbar) {
                        return Either.either(button.action, function (onClick) {
                            return toolbar.addToolbarButton({
                                title: button.title,
                                tooltip: button.tooltip,
                                icon: button.icon,
                                click: onClick
                            });
                        }, function (tool) {
                            return AbstractToolbars.addMouseToolButton(toolbar, {
                                title: button.title,
                                tooltip: button.tooltip,
                                icon: button.icon,
                                tool: tool.tool,
                                selectedTool: tool.selectedTool,
                                selectedTool2: tool.selectedTool2
                            });
                        });
                    }
                };
            }));
        };

        /**
        * Add a button to a toolbar which allows the user to activate a mouse tool
        *
        * The button will appear selected when the mouse tool is active
        *
        * @param {$toolbar} The toolbar
        * @param {options.title} The text to display on the button
        * @param {options.tooltip} An optional tooltip
        * @param {options.icon} The name of the icon to display
        * @param {options.tool} The mouse tool to activate on clicking
        * @param {options.selectedTool} A writable subject which determines the currently selected tool
        *
        * @returns The button jQuery DOM wrapper object
        */
        AbstractToolbars.addMouseToolButton = function (toolbar, options) {
            return toolbar.addToggleButton({
                title: options.title,
                tooltip: options.tooltip,
                icon: options.icon,
                enabled: options.enabled,
                visible: options.visible,
                closePopupsOnClick: true,
                selected: Subjects.dimap(options.selectedTool, function (t) {
                    return t === options.tool;
                }, function (b) {
                    return b ? options.tool : 0 /* Move */;
                }),
                selected2: Subjects.dimap(options.selectedTool2, function (t) {
                    return t === options.tool;
                }, function (b) {
                    return b ? options.tool : 0 /* Move */;
                }),
                highlightParent: true
            });
        };

        /**
        * Create a toolbar from a configuration object
        */
        AbstractToolbars.createToolbarFromConfiguration = function (toolbar, config, application) {
            return _.flatten(_.map(config.items, function (item) {
                switch (item.itemType) {
                    case 0 /* Button */:
                        return AbstractToolbars.addStandardToolbarButton(toolbar, item.button, false, application);
                    case 1 /* Group */:
                        return [toolbar.addDropDownButton({
                                title: item.text_i18n && item.text ? application.terminology.lookup(new Classes.Term(item.text_i18n, item.text)) : item.text,
                                tooltip: item.tooltip_i18n && item.tooltip ? application.terminology.lookup(new Classes.Term(item.tooltip_i18n, item.tooltip)) : item.tooltip,
                                icon: item.icon
                            }, _.map(item.buttons, function (button) {
                                return {
                                    add: function (toolbar) {
                                        return AbstractToolbars.addStandardToolbarButton(toolbar, button, true, application);
                                    }
                                };
                            }))];
                }
            }));
        };

        /**
        * Add a standard toolbar button to the toolbar, or to a group
        */
        AbstractToolbars.addStandardToolbarButton = function (toolbar, buttonType, group, application) {
            var container = application.selectedSeriesContainer.read();
            var settings = application.settings.read();

            switch (buttonType) {
                case 9 /* Fit */:
                    return [toolbar.addToolbarButton({
                            title: application.terminology.lookup(Terminology.Terms.Fit),
                            tooltip: application.terminology.lookup(Terminology.Terms.FitTooltip),
                            icon: "w_fit",
                            click: function () {
                                var series = application.selectedSeriesContainer.read();

                                if (series != null) {
                                    series.fit();
                                }
                            },
                            enabled: Subjects.map(application.selectedSeriesContainer, function (s) {
                                return s != null;
                            })
                        })];
                case 1 /* Zoom */:
                    return [AbstractToolbars.addMouseToolButton(toolbar, {
                            title: application.terminology.lookup(Terminology.Terms.Zoom),
                            tooltip: application.terminology.lookup(Terminology.Terms.ZoomTooltip),
                            icon: "w_zoom",
                            tool: 2 /* Zoom */,
                            selectedTool: application.selectedTool,
                            selectedTool2: application.selectedTool2
                        })];
                case 54 /* FreeRotate */:
                    return [AbstractToolbars.addMouseToolButton(toolbar, {
                            title: application.terminology.lookup(Terminology.Terms.FreeRotate),
                            tooltip: application.terminology.lookup(Terminology.Terms.FreeRotateTooltip),
                            icon: "w_rotate",
                            tool: 14 /* FreeRotate */,
                            selectedTool: application.selectedTool,
                            selectedTool2: application.selectedTool2
                        })];
                case 2 /* Move */:
                    return [AbstractToolbars.addMouseToolButton(toolbar, {
                            title: application.terminology.lookup(Terminology.Terms.Move),
                            tooltip: application.terminology.lookup(Terminology.Terms.MoveTooltip),
                            icon: "w_pan",
                            tool: 0 /* Move */,
                            selectedTool: application.selectedTool,
                            selectedTool2: application.selectedTool2
                        })];
                case 27 /* FlipV */:
                    return [toolbar.addToolbarButton({
                            title: application.terminology.lookup(Terminology.Terms.FlipV),
                            tooltip: application.terminology.lookup(Terminology.Terms.FlipVTooltip),
                            icon: "w_flip_v",
                            click: function () {
                                var series = application.selectedSeriesContainer.read();

                                if (series !== null) {
                                    var transform = series.transform;

                                    Subjects.modify(transform, function (t) {
                                        return {
                                            offsetX: t.offsetX,
                                            offsetY: t.offsetY,
                                            scale: t.scale,
                                            flipped: !t.flipped,
                                            rotation: (t.rotation + 2) % 4
                                        };
                                    });
                                }
                            }
                        })];
                case 3 /* Flip */:
                    return [toolbar.addToolbarButton({
                            title: application.terminology.lookup(Terminology.Terms.Flip),
                            tooltip: application.terminology.lookup(Terminology.Terms.FlipTooltip),
                            icon: "w_flip",
                            click: function () {
                                var series = application.selectedSeriesContainer.read();

                                if (series !== null) {
                                    var transform = series.transform;

                                    Subjects.modify(transform, function (t) {
                                        return {
                                            offsetX: t.offsetX,
                                            offsetY: t.offsetY,
                                            scale: t.scale,
                                            flipped: !t.flipped,
                                            rotation: t.rotation
                                        };
                                    });
                                }
                            }
                        })];
                case 4 /* Rotate */:
                    return [toolbar.addToolbarButton({
                            title: application.terminology.lookup(Terminology.Terms.Rotate),
                            tooltip: application.terminology.lookup(Terminology.Terms.RotateTooltip),
                            icon: "w_rotate",
                            click: function () {
                                var series = application.selectedSeriesContainer.read();

                                if (series != null) {
                                    Subjects.modify(series.transform, function (transform) {
                                        return {
                                            offsetX: transform.offsetX,
                                            offsetY: transform.offsetX,
                                            scale: transform.scale,
                                            flipped: transform.flipped,
                                            rotation: transform.rotation + 1
                                        };
                                    });
                                }
                            },
                            enabled: Subjects.map(application.selectedSeriesContainer, function (s) {
                                return s != null;
                            })
                        })];
                case 5 /* Scroll */:
                    return [AbstractToolbars.addMouseToolButton(toolbar, {
                            title: application.terminology.lookup(Terminology.Terms.Scroll),
                            tooltip: application.terminology.lookup(Terminology.Terms.ScrollTooltip),
                            icon: "w_scroll",
                            tool: 1 /* Scroll */,
                            selectedTool: application.selectedTool,
                            selectedTool2: application.selectedTool2
                        })];
                case 6 /* WindowLevel */:
                    return [AbstractToolbars.addMouseToolButton(toolbar, {
                            title: application.terminology.lookup(Terminology.Terms.SelectWindowLevel),
                            tooltip: application.terminology.lookup(Terminology.Terms.SelectWindowLevelTooltip),
                            icon: "w_pan",
                            tool: 3 /* Window */,
                            selectedTool: application.selectedTool,
                            selectedTool2: application.selectedTool2,
                            enabled: Subjects.bind(application.selectedSeriesContainer, function (series) {
                                return series === null ? Subjects.ret(null) : series.windowLevel;
                            }, function (_, windowLevel) {
                                return windowLevel !== null;
                            })
                        })];
                case 76 /* WindowLevelDirect */:
                    return [AbstractToolbars.addMouseToolButton(toolbar, {
                            title: application.terminology.lookup(Terminology.Terms.WindowLevel),
                            tooltip: application.terminology.lookup(Terminology.Terms.SelectWindowLevelTooltip),
                            icon: "w_window",
                            tool: 3 /* Window */,
                            selectedTool: application.selectedTool,
                            selectedTool2: application.selectedTool2,
                            enabled: Subjects.bind(application.selectedSeriesContainer, function (series) {
                                return series === null ? Subjects.ret(null) : series.windowLevel;
                            }, function (_, windowLevel) {
                                return windowLevel !== null;
                            })
                        })];
                case 28 /* ResetWindowLevel */:
                    return [toolbar.addToolbarButton({
                            title: application.terminology.lookup(Terminology.Terms.Reset),
                            tooltip: application.terminology.lookup(Terminology.Terms.ResetTooltip),
                            icon: "w_reload",
                            click: function () {
                                var series = application.selectedSeriesContainer.read();

                                if (series != null) {
                                    series.resetWindowLevel();
                                }
                            },
                            enabled: Subjects.map(application.selectedSeriesContainer, function (s) {
                                return s != null;
                            })
                        })];
                case 24 /* SavePreset */:
                    if (!LocalViewer.isLocalViewer()) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.SavePreset),
                                tooltip: application.terminology.lookup(Terminology.Terms.SavePresetTooltip),
                                icon: "w_save",
                                click: function () {
                                    application.saveWindowLevelPreset();
                                },
                                enabled: Subjects.zip(Subjects.map(application.selectedSeriesContainer, function (series) {
                                    if (series && series.series && series.series.instances.length > 0) {
                                        return WindowLevelPresets.shouldUse16BitWindowLevel(series.series.instances[0]);
                                    }

                                    return false;
                                }), application.isRecording, function (b1, b2) {
                                    return b1 && !b2;
                                })
                            })];
                    }
                    return [];
                case 29 /* WindowLevelPresets */:
                    if (container) {
                        var series = container.series;

                        if (series) {
                            var instance = series.instances[0];

                            if (WindowLevelPresets.shouldUse16BitWindowLevel(instance)) {
                                var modality = series.seriesAttributes.modality;
                                var modalitySettings;

                                if (settings.modalities) {
                                    modalitySettings = _.find(settings.modalities, function (m) {
                                        return m.modality === modality;
                                    });
                                }

                                var presets;

                                if (modalitySettings && modalitySettings.presets !== undefined && modalitySettings.presets.length > 0) {
                                    presets = modalitySettings.presets;
                                } else {
                                    var index = container.selectedInstanceIndex.read();
                                    if (container.series) {
                                        var instance = container.series.instances[index];
                                        presets = WindowLevelPresets.defaults(modality, application.terminology);
                                    } else {
                                        presets = [];
                                    }
                                }

                                return _.map(_.take(presets, 10), function (preset, index) {
                                    return toolbar.addToolbarButton({
                                        title: preset.name.substr(0, 8),
                                        tooltip: preset.name,
                                        icon: "w_window",
                                        closePopupsOnClick: true,
                                        click: function () {
                                            container.applyWindowLevelPreset(index);
                                        }
                                    });
                                });
                            }
                        }
                    }
                    return [];
                case 103 /* ColorTablePresets */:
                    if (container) {
                        var series = container.series;

                        if (series) {
                            var modality = series.seriesAttributes.modality;
                            var modalitySettings;

                            if (settings.modalities) {
                                modalitySettings = _.find(settings.modalities, function (m) {
                                    return m.modality === modality;
                                });
                            }

                            var colorTablePresets;

                            if (modalitySettings && modalitySettings.colorTablePresets !== undefined && modalitySettings.colorTablePresets.length > 0) {
                                colorTablePresets = modalitySettings.colorTablePresets;
                            } else {
                                if (container.series) {
                                    colorTablePresets = ColorTablePresets.defaults(modality, application.terminology);
                                } else {
                                    colorTablePresets = [];
                                }
                            }

                            return _.map(_.take(colorTablePresets, 10), function (preset, index) {
                                return toolbar.addToolbarButton({
                                    title: preset.name.substr(0, 8),
                                    tooltip: preset.name,
                                    icon: "w_colors",
                                    closePopupsOnClick: true,
                                    click: function () {
                                        container.applyColorTablePreset(index);
                                    }
                                });
                            });
                        }
                    }
                    return [];
                case 73 /* DetectWindowLevel */:
                    if (application.permissions.annotation_edit !== 0) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.DetectWindowLevel),
                                tooltip: application.terminology.lookup(Terminology.Terms.DetectWindowLevelTooltip),
                                icon: "w_window",
                                closePopupsOnClick: true,
                                click: function () {
                                    container.detectWindowLevel();
                                }
                            })];
                    }
                    return [];
                case 30 /* SelectAnnotation */:
                    if (application.permissions.annotation_edit !== 0) {
                        return [AbstractToolbars.addMouseToolButton(toolbar, {
                                title: application.terminology.lookup(Terminology.Terms.SelectAnnotation),
                                tooltip: application.terminology.lookup(Terminology.Terms.SelectAnnotationTooltip),
                                icon: "w_select",
                                tool: 4 /* Select */,
                                selectedTool: application.selectedTool,
                                selectedTool2: application.selectedTool2,
                                enabled: Subjects.map(application.isRecording, function (b) {
                                    return !b;
                                })
                            })];
                    }
                    return [];
                case 31 /* DeleteAnnotation */:
                    if (application.permissions.annotation_edit !== 0) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.DeleteAnnotation),
                                tooltip: application.terminology.lookup(Terminology.Terms.DeleteAnnotationTooltip),
                                icon: "w_delete",
                                click: function () {
                                    var series = application.selectedSeriesContainer.read();
                                    if (series != null) {
                                        series.deleteSelectedMeasurement();
                                    }
                                },
                                enabled: Subjects.zip(Subjects.bind(application.selectedSeriesContainer, function (series) {
                                    return series === null ? Subjects.ret(null) : series.selectedMeasurement;
                                }, function (_, measurement) {
                                    return measurement !== null && measurement.editable;
                                }), application.isRecording, function (b1, b2) {
                                    return b1 && !b2;
                                })
                            })];
                    }
                    return [];
                case 32 /* FillAnnotation */:
                    if (application.permissions.annotation_edit !== 0) {
                        return [toolbar.addToggleButton({
                                title: application.terminology.lookup(Terminology.Terms.Filled),
                                tooltip: application.terminology.lookup(Terminology.Terms.FilledTooltip),
                                icon: "w_fill",
                                enabled: Subjects.zip(Subjects.bind(application.selectedSeriesContainer, function (series) {
                                    return series === null ? Subjects.ret(null) : series.selectedMeasurement;
                                }, function (_, measurement) {
                                    return measurement != null && measurement.editable;
                                }), application.isRecording, function (b1, b2) {
                                    return b1 && !b2;
                                }),
                                selected: Subjects.lens(Subjects.bindW(application.selectedSeriesContainer, function (series) {
                                    return series === null ? Subjects.retW(null) : series.selectedMeasurement;
                                }), function (measurement) {
                                    return measurement != null && measurement.filled;
                                }, function (measurement, value) {
                                    if (measurement) {
                                        measurement.filled = value;

                                        var series = application.selectedSeriesContainer.read();
                                        series.editMeasurement(measurement);
                                        series.clearSelection();
                                    }
                                    application.renderAllFrames();
                                })
                            })];
                    }
                    return [];
                case 33 /* ExportGSPS */:
                    if (application.permissions.annotation_edit !== 0 && !LocalViewer.isLocalViewer()) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.ExportGSPS),
                                tooltip: application.terminology.lookup(Terminology.Terms.ExportGSPSTooltip),
                                icon: "w_save",
                                click: function () {
                                    var series = application.selectedSeriesContainer.read();
                                    if (series != null) {
                                        series.saveGSPS();
                                    }
                                },
                                enabled: Subjects.map(application.isRecording, function (b) {
                                    return !b;
                                })
                            })];
                    }
                    return [];
                case 34 /* Line */:
                    if (application.permissions.annotation_edit !== 0) {
                        return [AbstractToolbars.addMouseToolButton(toolbar, {
                                title: application.terminology.lookup(Terminology.Terms.Line),
                                tooltip: application.terminology.lookup(Terminology.Terms.LineTooltip),
                                icon: "w_line",
                                tool: 5 /* Measure */,
                                selectedTool: application.selectedTool,
                                selectedTool2: application.selectedTool2
                            })];
                    }
                    return [];
                case 97 /* LineAnnotate */:
                    if (application.permissions.annotation_edit !== 0) {
                        return [AbstractToolbars.addMouseToolButton(toolbar, {
                                title: application.terminology.lookup(Terminology.Terms.LineAnnotate),
                                tooltip: application.terminology.lookup(Terminology.Terms.LineAnnotateTooltip),
                                icon: "w_line",
                                tool: 25 /* LineAnnotate */,
                                selectedTool: application.selectedTool,
                                selectedTool2: application.selectedTool2
                            })];
                    }
                    return [];
                case 107 /* CalibrateLine */:
                    if (application.permissions.annotation_edit !== 0) {
                        return [AbstractToolbars.addMouseToolButton(toolbar, {
                                title: application.terminology.lookup(Terminology.Terms.LineCalibrate),
                                tooltip: application.terminology.lookup(Terminology.Terms.LineCalibrateTooltip),
                                icon: "w_line",
                                tool: 30 /* CalibrateLine */,
                                selectedTool: application.selectedTool,
                                selectedTool2: application.selectedTool2
                            })];
                    }
                    return [];
                case 35 /* Arrow */:
                    if (application.permissions.annotation_edit !== 0) {
                        return [AbstractToolbars.addMouseToolButton(toolbar, {
                                title: application.terminology.lookup(Terminology.Terms.Arrow),
                                tooltip: application.terminology.lookup(Terminology.Terms.ArrowTooltip),
                                icon: "w_arrow",
                                tool: 12 /* Arrow */,
                                selectedTool: application.selectedTool,
                                selectedTool2: application.selectedTool2
                            })];
                    }
                    return [];
                case 94 /* ArrowAnnotate */:
                    if (application.permissions.annotation_edit !== 0) {
                        return [AbstractToolbars.addMouseToolButton(toolbar, {
                                title: application.terminology.lookup(Terminology.Terms.ArrowAnnotate),
                                tooltip: application.terminology.lookup(Terminology.Terms.ArrowAnnotateTooltip),
                                icon: "w_arrow",
                                tool: 22 /* ArrowAnnotate */,
                                selectedTool: application.selectedTool,
                                selectedTool2: application.selectedTool2
                            })];
                    }
                    return [];
                case 53 /* Angle */:
                    if (application.permissions.annotation_edit !== 0) {
                        return [AbstractToolbars.addMouseToolButton(toolbar, {
                                title: application.terminology.lookup(Terminology.Terms.Angle),
                                tooltip: application.terminology.lookup(Terminology.Terms.AngleTooltip),
                                icon: "w_angle",
                                tool: 13 /* Angle */,
                                selectedTool: application.selectedTool,
                                selectedTool2: application.selectedTool2
                            })];
                    }
                    return [];
                case 36 /* Cobb */:
                    if (application.permissions.annotation_edit !== 0) {
                        return [AbstractToolbars.addMouseToolButton(toolbar, {
                                title: application.terminology.lookup(Terminology.Terms.Cobb),
                                tooltip: application.terminology.lookup(Terminology.Terms.CobbTooltip),
                                icon: "w_cobb",
                                tool: 7 /* CobbAngle */,
                                selectedTool: application.selectedTool,
                                selectedTool2: application.selectedTool2
                            })];
                    }
                    return [];
                case 37 /* Rectangle */:
                    if (application.permissions.annotation_edit !== 0) {
                        return [AbstractToolbars.addMouseToolButton(toolbar, {
                                title: application.terminology.lookup(Terminology.Terms.Rectangle),
                                tooltip: application.terminology.lookup(Terminology.Terms.RectangleTooltip),
                                icon: "w_rect",
                                tool: 6 /* Rectangle */,
                                selectedTool: application.selectedTool,
                                selectedTool2: application.selectedTool2
                            })];
                    }
                    return [];
                case 98 /* RectangleAnnotate */:
                    if (application.permissions.annotation_edit !== 0) {
                        return [AbstractToolbars.addMouseToolButton(toolbar, {
                                title: application.terminology.lookup(Terminology.Terms.RectangleAnnotate),
                                tooltip: application.terminology.lookup(Terminology.Terms.RectangleAnnotateTooltip),
                                icon: "w_rect",
                                tool: 26 /* RectangleAnnotate */,
                                selectedTool: application.selectedTool,
                                selectedTool2: application.selectedTool2
                            })];
                    }
                    return [];
                case 38 /* Ellipse */:
                    if (application.permissions.annotation_edit !== 0) {
                        return [AbstractToolbars.addMouseToolButton(toolbar, {
                                title: application.terminology.lookup(Terminology.Terms.Ellipse),
                                tooltip: application.terminology.lookup(Terminology.Terms.EllipseTooltip),
                                icon: "w_ellipse",
                                tool: 8 /* Ellipse */,
                                selectedTool: application.selectedTool,
                                selectedTool2: application.selectedTool2
                            })];
                    }
                    return [];
                case 96 /* EllipseAnnotate */:
                    if (application.permissions.annotation_edit !== 0) {
                        return [AbstractToolbars.addMouseToolButton(toolbar, {
                                title: application.terminology.lookup(Terminology.Terms.EllipseAnnotate),
                                tooltip: application.terminology.lookup(Terminology.Terms.EllipseAnnotateTooltip),
                                icon: "w_ellipse",
                                tool: 24 /* EllipseAnnotate */,
                                selectedTool: application.selectedTool,
                                selectedTool2: application.selectedTool2
                            })];
                    }
                    return [];
                case 58 /* Radius */:
                    if (application.permissions.annotation_edit !== 0) {
                        return [AbstractToolbars.addMouseToolButton(toolbar, {
                                title: application.terminology.lookup(Terminology.Terms.Radius),
                                tooltip: application.terminology.lookup(Terminology.Terms.RadiusTooltip),
                                icon: "w_ellipse",
                                tool: 16 /* Circle */,
                                selectedTool: application.selectedTool,
                                selectedTool2: application.selectedTool2
                            })];
                    }
                    return [];
                case 85 /* Circle */:
                    if (application.permissions.annotation_edit !== 0) {
                        return [AbstractToolbars.addMouseToolButton(toolbar, {
                                title: application.terminology.lookup(Terminology.Terms.Circle),
                                tooltip: application.terminology.lookup(Terminology.Terms.CircleTooltip),
                                icon: "w_ellipse",
                                tool: 18 /* DropCircle */,
                                selectedTool: application.selectedTool,
                                selectedTool2: application.selectedTool2
                            })];
                    }
                    return [];
                    return [];
                case 95 /* CircleAnnotate */:
                    if (application.permissions.annotation_edit !== 0) {
                        return [AbstractToolbars.addMouseToolButton(toolbar, {
                                title: application.terminology.lookup(Terminology.Terms.CircleAnnotate),
                                tooltip: application.terminology.lookup(Terminology.Terms.CircleAnnotateTooltip),
                                icon: "w_ellipse",
                                tool: 23 /* CircleAnnotate */,
                                selectedTool: application.selectedTool,
                                selectedTool2: application.selectedTool2
                            })];
                    }
                    return [];
                case 86 /* Square */:
                    if (application.permissions.annotation_edit !== 0) {
                        return [AbstractToolbars.addMouseToolButton(toolbar, {
                                title: application.terminology.lookup(Terminology.Terms.Square),
                                tooltip: application.terminology.lookup(Terminology.Terms.SquareTooltip),
                                icon: "w_rect",
                                tool: 19 /* DropSquare */,
                                selectedTool: application.selectedTool,
                                selectedTool2: application.selectedTool2
                            })];
                    }
                    return [];
                case 99 /* SquareAnnotate */:
                    if (application.permissions.annotation_edit !== 0) {
                        return [AbstractToolbars.addMouseToolButton(toolbar, {
                                title: application.terminology.lookup(Terminology.Terms.SquareAnnotate),
                                tooltip: application.terminology.lookup(Terminology.Terms.SquareAnnotateTooltip),
                                icon: "w_rect",
                                tool: 27 /* SquareAnnotate */,
                                selectedTool: application.selectedTool,
                                selectedTool2: application.selectedTool2
                            })];
                    }
                    return [];
                case 88 /* OrthoAxes */:
                    if (application.permissions.annotation_edit !== 0) {
                        return [AbstractToolbars.addMouseToolButton(toolbar, {
                                title: application.terminology.lookup(Terminology.Terms.OrthoAxes),
                                tooltip: application.terminology.lookup(Terminology.Terms.OrthoAxesTooltip),
                                icon: "w_axes",
                                tool: 20 /* OrthoAxes */,
                                selectedTool: application.selectedTool,
                                selectedTool2: application.selectedTool2
                            })];
                    }
                    return [];
                case 93 /* FemoralHead */:
                    if (application.permissions.annotation_edit !== 0) {
                        return [AbstractToolbars.addMouseToolButton(toolbar, {
                                title: application.terminology.lookup(Terminology.Terms.FemoralHead),
                                tooltip: application.terminology.lookup(Terminology.Terms.FemoralHeadTooltip),
                                icon: "w_ellipse",
                                tool: 21 /* FemoralHead */,
                                selectedTool: application.selectedTool,
                                selectedTool2: application.selectedTool2
                            })];
                    }
                    return [];
                case 105 /* Polygon */:
                    if (application.permissions.annotation_edit !== 0) {
                        return [AbstractToolbars.addMouseToolButton(toolbar, {
                                title: application.terminology.lookup(Terminology.Terms.Polygon),
                                tooltip: application.terminology.lookup(Terminology.Terms.PolygonTooltip),
                                icon: "w_trace",
                                tool: 28 /* Polygon */,
                                selectedTool: application.selectedTool,
                                selectedTool2: application.selectedTool2
                            })];
                    }
                    return [];
                case 106 /* Trace */:
                    if (application.permissions.annotation_edit !== 0) {
                        return [AbstractToolbars.addMouseToolButton(toolbar, {
                                title: application.terminology.lookup(Terminology.Terms.Trace),
                                tooltip: application.terminology.lookup(Terminology.Terms.TraceTooltip),
                                icon: "w_trace",
                                tool: 29 /* Trace */,
                                selectedTool: application.selectedTool,
                                selectedTool2: application.selectedTool2
                            })];
                    }
                    return [];
                case 140 /* ProstateTool */:
                    if (application.permissions.annotation_edit !== 0) {
                        return [AbstractToolbars.addMouseToolButton(toolbar, {
                                title: application.terminology.lookup(Terminology.Terms.ProstateTool),
                                tooltip: application.terminology.lookup(Terminology.Terms.ProstateToolTooltip),
                                icon: "w_line",
                                tool: 33 /* ProstateTool */,
                                selectedTool: application.selectedTool,
                                selectedTool2: application.selectedTool2
                            })];
                    }
                    return [];
                case 127 /* Area */:
                    if (application.permissions.annotation_edit !== 0) {
                        return [AbstractToolbars.addMouseToolButton(toolbar, {
                                title: application.terminology.lookup(Terminology.Terms.Area),
                                tooltip: application.terminology.lookup(Terminology.Terms.PaintTooltip),
                                icon: "w_paint",
                                tool: 32 /* Area */,
                                selectedTool: application.selectedTool,
                                selectedTool2: application.selectedTool2
                            })];
                    }
                    return [];
                case 39 /* Text */:
                    if (application.permissions.annotation_edit !== 0) {
                        return [AbstractToolbars.addMouseToolButton(toolbar, {
                                title: application.terminology.lookup(Terminology.Terms.Text),
                                tooltip: application.terminology.lookup(Terminology.Terms.TextTooltip),
                                icon: "w_text",
                                tool: 10 /* Text */,
                                selectedTool: application.selectedTool,
                                selectedTool2: application.selectedTool2
                            })];
                    }
                    return [];
                case 121 /* Stamp */:
                    if (application.permissions.annotation_edit !== 0) {
                        return [AbstractToolbars.addMouseToolButton(toolbar, {
                                title: application.terminology.lookup(Terminology.Terms.Stamp),
                                tooltip: application.terminology.lookup(Terminology.Terms.StampTooltip),
                                icon: "w_lock",
                                tool: 31 /* Stamp */,
                                selectedTool: application.selectedTool,
                                selectedTool2: application.selectedTool2
                            })];
                    }
                    return [];
                case 119 /* Propagate */:
                    return [toolbar.addToggleButton({
                            title: application.terminology.lookup(Terminology.Terms.Propagate),
                            tooltip: application.terminology.lookup(Terminology.Terms.PropagateTooltip),
                            icon: "w_copy",
                            enabled: Subjects.zip(Subjects.bind(application.selectedSeriesContainer, function (series) {
                                return series === null ? Subjects.ret(null) : series.selectedMeasurement;
                            }, function (_, measurement) {
                                return measurement != null && measurement.editable && (measurement instanceof Measurements.Text);
                            }), application.isRecording, function (b1, b2) {
                                return b1 && !b2;
                            }),
                            selected: application.propagateMode
                        })];
                case 120 /* PropagateAll */:
                    return [toolbar.addToolbarButton({
                            title: application.terminology.lookup(Terminology.Terms.Propagate),
                            tooltip: application.terminology.lookup(Terminology.Terms.PropagateAllTooltip),
                            icon: "w_copy",
                            click: function () {
                                var series = application.selectedSeriesContainer.read();

                                if (series != null) {
                                    series.propapateToSeries();
                                }
                            },
                            enabled: Subjects.zip(Subjects.bind(application.selectedSeriesContainer, function (series) {
                                return series === null ? Subjects.ret(null) : series.selectedMeasurement;
                            }, function (_, measurement) {
                                return measurement != null && measurement.editable && (measurement instanceof Measurements.Text);
                            }), application.isRecording, function (b1, b2) {
                                return b1 && !b2;
                            })
                        })];
                case 128 /* ThresholdToArea */:
                    return [toolbar.addToolbarButton({
                            title: application.terminology.lookup(Terminology.Terms.Threshold),
                            tooltip: application.terminology.lookup(Terminology.Terms.ThresholdTooltip),
                            icon: "w_threshold",
                            click: function () {
                                var series = application.selectedSeriesContainer.read();

                                if (series != null) {
                                    series.thresholdToArea(false, false);
                                }
                            }
                        })];
                case 132 /* ThresholdRangeToArea */:
                    return [toolbar.addToolbarButton({
                            title: application.terminology.lookup(Terminology.Terms.Range),
                            tooltip: application.terminology.lookup(Terminology.Terms.RangeTooltip),
                            icon: "w_threshold",
                            click: function () {
                                var series = application.selectedSeriesContainer.read();

                                if (series != null) {
                                    series.thresholdToArea(false, true);
                                }
                            }
                        })];
                case 129 /* ShrinkWrapToArea */:
                    return [toolbar.addToolbarButton({
                            title: application.terminology.lookup(Terminology.Terms.ShrinkWrap),
                            tooltip: application.terminology.lookup(Terminology.Terms.ShrinkWrapTooltip),
                            icon: "w_shrinkwrap",
                            click: function () {
                                var series = application.selectedSeriesContainer.read();

                                if (series != null) {
                                    series.thresholdToArea(true, false);
                                }
                            }
                        })];
                case 8 /* Probe */:
                    if (Rendering.getRenderingMode() !== 2 /* Simple */) {
                        return [AbstractToolbars.addMouseToolButton(toolbar, {
                                title: application.terminology.lookup(Terminology.Terms.Probe),
                                tooltip: application.terminology.lookup(Terminology.Terms.ProbeTooltip),
                                icon: "w_select",
                                tool: 11 /* Probe */,
                                selectedTool: application.selectedTool,
                                selectedTool2: application.selectedTool2,
                                enabled: Subjects.zip(Subjects.map(application.selectedSeriesContainer, function (series) {
                                    if (series && series.series && series.series.instances.length > 0) {
                                        return WindowLevelPresets.shouldUse16BitWindowLevel(series.series.instances[0]);
                                    }

                                    return false;
                                }), application.recordingMode, function (b1, b2) {
                                    return b1 && !b2;
                                })
                            })];
                    }
                    return [];
                case 10 /* Reset */:
                    return [toolbar.addToolbarButton({
                            title: application.terminology.lookup(Terminology.Terms.Reset),
                            tooltip: application.terminology.lookup(Terminology.Terms.ResetTooltip),
                            icon: "w_reload",
                            click: function () {
                                var series = application.selectedSeriesContainer.read();

                                if (series != null) {
                                    series.reset();
                                }
                            },
                            enabled: Subjects.map(application.selectedSeriesContainer, function (s) {
                                return s != null;
                            })
                        })];
                case 40 /* ReferenceLines */:
                    return [toolbar.addToggleButton({
                            title: application.terminology.lookup(Terminology.Terms.ReferenceLines),
                            tooltip: application.terminology.lookup(Terminology.Terms.ReferenceLinesTooltip),
                            icon: "w_reflines",
                            selected: application.referenceLinesActive
                        })];
                case 41 /* LinkedSeries */:
                    return [toolbar.addToggleButton({
                            title: application.terminology.lookup(Terminology.Terms.LinkedSeries),
                            tooltip: application.terminology.lookup(Terminology.Terms.LinkedSeriesTooltip),
                            icon: "w_link",
                            selected: application.linkSeries
                        })];
                case 42 /* PlaneLocalization */:
                    return [AbstractToolbars.addMouseToolButton(toolbar, {
                            title: application.terminology.lookup(Terminology.Terms.PlaneLocalization),
                            tooltip: application.terminology.lookup(Terminology.Terms.PlaneLocalizationTooltip),
                            icon: "w_3d",
                            selectedTool: application.selectedTool,
                            selectedTool2: application.selectedTool2,
                            tool: 9 /* Localization */,
                            enabled: Subjects.map(application.selectedInstance(), function (instance) {
                                return instance !== null && SeriesGeometry.hasGeometricMetadata(instance.instance);
                            })
                        })];
                case 70 /* MPR */:
                    var enableMPR = application.accountSettings.viewer_enable_mpr;
                    if (enableMPR !== undefined && enableMPR > 0) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.MPR),
                                tooltip: application.terminology.lookup(Terminology.Terms.MPRTooltip),
                                icon: "w_mpr",
                                click: function () {
                                    application.mpr();
                                },
                                enabled: Subjects.map(application.isRecording, function (b) {
                                    return !b;
                                })
                            })];
                    }
                    return [];
                case 43 /* Maximize */:
                    return [toolbar.addToolbarButton({
                            title: application.terminology.lookup(Terminology.Terms.Maximize),
                            tooltip: application.terminology.lookup(Terminology.Terms.MaximizeTooltip),
                            icon: "w_max",
                            enabled: Subjects.zip(Subjects.map(application.layout, function (layout) {
                                return layout.rows > 1 || layout.columns > 1;
                            }), application.isRecording, function (b1, b2) {
                                return b1 && !b2;
                            }),
                            click: function () {
                                var selectedSeries = application.selectedSeriesContainer.read();

                                if (selectedSeries) {
                                    application.magnifyMinify(selectedSeries);
                                }
                            }
                        })];
                case 44 /* NewWindow */:
                    if (!LocalViewer.isLocalViewer()) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.NewWindow),
                                tooltip: application.terminology.lookup(Terminology.Terms.NewWindowTooltip),
                                icon: "w_tab",
                                click: function () {
                                    application.openNewWindow(0);
                                }
                            })];
                    }
                    return [];
                case 45 /* LayoutButtons */:
                    return toolbar.addLayoutButtons(application.layout);
                case 130 /* ColorButtons */:
                    return toolbar.addColorButtons(application.currentColor);
                case 13 /* Print */:
                    var enablePrinting = application.accountSettings.enable_viewer_print;

                    if (!LocalViewer.isStandardLocalViewer() && (enablePrinting === undefined || enablePrinting > 0)) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.Print),
                                tooltip: application.terminology.lookup(Terminology.Terms.PrintTooltip),
                                icon: "w_print",
                                click: function () {
                                    application.print();
                                },
                                enabled: Subjects.map(application.isRecording, function (b) {
                                    return !b;
                                })
                            })];
                    }
                    return [];
                case 15 /* TextAnnotations */:
                    var selected;

                    if (application.globalTextToggle) {
                        selected = Subjects.bindW(application.selectedSeriesContainer, function (_) {
                            return application.globalTextVisible;
                        });
                    } else {
                        selected = Subjects.bindW(application.selectedSeriesContainer, function (series) {
                            return series === null ? Subjects.retW(false) : series.infoVisible;
                        });
                    }

                    return [toolbar.addToggleButton({
                            title: application.terminology.lookup(Terminology.Terms.Show),
                            tooltip: application.terminology.lookup(Terminology.Terms.InfoTooltip),
                            icon: "w_info",
                            selected: selected,
                            toggledData: {
                                title: application.terminology.lookup(Terminology.Terms.Hide),
                                tooltip: application.terminology.lookup(Terminology.Terms.InfoTooltip),
                                icon: "w_info",
                                baseClass: 'showHide'
                            }
                        })];
                case 16 /* Measurements */:
                    var enableToggleAnnotations = application.accountSettings.enable_viewer_toggle_annotations;

                    if (enableToggleAnnotations === undefined || enableToggleAnnotations > 0) {
                        var selected;

                        if (application.globalMeasurementsToggle) {
                            selected = Subjects.bindW(application.selectedSeriesContainer, function (_) {
                                return application.globalMeasurementsVisible;
                            });
                        } else {
                            selected = Subjects.bindW(application.selectedSeriesContainer, function (series) {
                                return series === null ? Subjects.retW(false) : series.measurementsVisible;
                            });
                        }

                        return [toolbar.addToggleButton({
                                title: application.terminology.lookup(Terminology.Terms.Show),
                                tooltip: application.terminology.lookup(Terminology.Terms.ShowAnnotationsTooltip),
                                icon: "w_measurements",
                                selected: selected,
                                toggledData: {
                                    title: application.terminology.lookup(Terminology.Terms.Hide),
                                    tooltip: application.terminology.lookup(Terminology.Terms.ShowAnnotationsTooltip),
                                    icon: "w_measurements",
                                    baseClass: 'showHide'
                                }
                            })];
                    }
                    return [];
                case 104 /* UltrasoundRegions */:
                    var selected = Subjects.bindW(application.selectedSeriesContainer, function (series) {
                        return series === null ? Subjects.retW(false) : series.ultrasoundRegionsVisible;
                    });

                    return [toolbar.addToggleButton({
                            title: application.terminology.lookup(Terminology.Terms.ShowUltrasoundRegion),
                            tooltip: application.terminology.lookup(Terminology.Terms.UltrasoundRegionTooltip),
                            icon: "w_rect",
                            selected: selected,
                            toggledData: {
                                title: application.terminology.lookup(Terminology.Terms.HideUltrasoundRegion),
                                tooltip: application.terminology.lookup(Terminology.Terms.UltrasoundRegionTooltip),
                                icon: "w_rect",
                                baseClass: 'showHide'
                            }
                        })];
                case 84 /* Ruler */:
                    var enableToggleAnnotations = application.accountSettings.enable_viewer_toggle_annotations;

                    if (enableToggleAnnotations === undefined || enableToggleAnnotations > 0) {
                        var selected;

                        if (application.globalRulerToggle) {
                            selected = Subjects.bindW(application.selectedSeriesContainer, function (_) {
                                return application.globalRulerVisible;
                            });
                        } else {
                            selected = Subjects.bindW(application.selectedSeriesContainer, function (series) {
                                return series === null ? Subjects.retW(false) : series.rulerVisible;
                            });
                        }

                        return [toolbar.addToggleButton({
                                title: application.terminology.lookup(Terminology.Terms.Show),
                                tooltip: application.terminology.lookup(Terminology.Terms.ShowRulerTooltip),
                                icon: "w_ruler",
                                selected: selected,
                                toggledData: {
                                    title: application.terminology.lookup(Terminology.Terms.Hide),
                                    tooltip: application.terminology.lookup(Terminology.Terms.ShowRulerTooltip),
                                    icon: "w_ruler",
                                    baseClass: 'showHide'
                                }
                            })];
                    }
                    return [];
                case 74 /* AnnotationsDetailToggle */:
                    var enableToggleAnnotations = application.accountSettings.enable_viewer_toggle_annotations;

                    if (enableToggleAnnotations === undefined || enableToggleAnnotations > 0) {
                        return [toolbar.addToggleButton({
                                title: application.terminology.lookup(Terminology.Terms.Show),
                                tooltip: application.terminology.lookup(Terminology.Terms.ShowAnnotationsDetailTooltip),
                                icon: "w_ann_detail_toggle",
                                selected: Subjects.bindW(application.selectedSeriesContainer, function (series) {
                                    return series === null ? Subjects.retW(false) : series.measurementsDetailsVisible;
                                }),
                                click: function () {
                                    var selectedSeries = application.selectedSeriesContainer.read();

                                    if (selectedSeries) {
                                        var measurementsVisible = selectedSeries.measurementsVisible.read();
                                        var measurementsDetailsVisible = selectedSeries.measurementsDetailsVisible.read();
                                        if (!measurementsVisible && measurementsDetailsVisible) {
                                            selectedSeries.measurementsVisible.write(true);
                                        }
                                    }
                                },
                                toggledData: {
                                    title: application.terminology.lookup(Terminology.Terms.Hide),
                                    tooltip: application.terminology.lookup(Terminology.Terms.ShowAnnotationsDetailTooltip),
                                    icon: "w_ann_detail_toggle",
                                    baseClass: 'showHide'
                                }
                            })];
                    }
                    return [];
                case 139 /* AnnotationsCreatedByOthersToggle */:
                    var enableToggleAnnotations = application.accountSettings.enable_viewer_toggle_annotations;

                    if (enableToggleAnnotations === undefined || enableToggleAnnotations > 0) {
                        return [toolbar.addToggleButton({
                                title: application.terminology.lookup(Terminology.Terms.Show),
                                tooltip: application.terminology.lookup(Terminology.Terms.ToggleCreatedByOthersAnnotationsTooltip),
                                icon: "w_ann_detail_toggle",
                                selected: Subjects.bindW(application.selectedSeriesContainer, function (series) {
                                    return series === null ? Subjects.retW(false) : series.annotationsCreatedByOtherUsersVisible;
                                }),
                                click: function () {
                                    var selectedSeries = application.selectedSeriesContainer.read();

                                    if (selectedSeries) {
                                        var measurementsVisible = selectedSeries.measurementsVisible.read();
                                        var measurementsCreatedByOthersVisible = selectedSeries.annotationsCreatedByOtherUsersVisible.read();
                                        if (!measurementsVisible && measurementsCreatedByOthersVisible) {
                                            selectedSeries.measurementsVisible.write(true);
                                        }
                                    }
                                },
                                toggledData: {
                                    title: application.terminology.lookup(Terminology.Terms.Hide),
                                    tooltip: application.terminology.lookup(Terminology.Terms.ToggleCreatedByOthersAnnotationsTooltip),
                                    icon: "w_ann_detail_toggle",
                                    baseClass: 'showHide'
                                }
                            })];
                    }
                    return [];
                case 17 /* Invert */:
                    if (Rendering.getRenderingMode() !== 2 /* Simple */) {
                        return [toolbar.addToggleButton({
                                title: application.terminology.lookup(Terminology.Terms.Invert),
                                tooltip: application.terminology.lookup(Terminology.Terms.InvertTooltip),
                                icon: "w_invert",
                                selected: Subjects.bindW(application.selectedSeriesContainer, function (series) {
                                    return series === null ? Subjects.retW(false) : series.invertActive;
                                })
                            })];
                    }
                    return [];
                case 118 /* Subtraction */:
                    if (Rendering.getRenderingMode() !== 2 /* Simple */) {
                        return [toolbar.addToggleButton({
                                title: application.terminology.lookup(Terminology.Terms.Subtract),
                                tooltip: application.terminology.lookup(Terminology.Terms.SubtractTooltip),
                                icon: "w_calc",
                                enabled: Subjects.bindW(application.selectedSeriesContainer, function (series) {
                                    return series === null ? Subjects.retW(false) : Subjects.retW(series.supportsSubtraction());
                                }),
                                selected: Subjects.bindW(application.selectedSeriesContainer, function (series) {
                                    return series === null ? Subjects.retW(false) : series.subtractionActive;
                                })
                            })];
                    }
                    return [];
                case 71 /* Enhance */:
                    if (Rendering.getRenderingMode() === 1 /* WebGL */) {
                        return [toolbar.addToggleButton({
                                title: application.terminology.lookup(Terminology.Terms.Enhance),
                                tooltip: application.terminology.lookup(Terminology.Terms.EnhanceTooltip),
                                icon: "w_enhance",
                                selected: Subjects.bindW(application.selectedSeriesContainer, function (series) {
                                    return series === null ? Subjects.retW(false) : series.enhance;
                                })
                            })];
                    }
                    return [];
                case 46 /* ExportPNG */:
                    var enableExport = application.accountSettings.enable_viewer_export;

                    if (LocalViewer.isLocalViewer() || (enableExport === undefined || enableExport > 0)) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.ExportPNG),
                                tooltip: application.terminology.lookup(Terminology.Terms.ExportPNGTooltip),
                                icon: "w_png",
                                click: function () {
                                    var selectedSeries = application.selectedSeriesContainer.read();

                                    if (selectedSeries) {
                                        selectedSeries.exportCurrentImages();
                                    }
                                },
                                enabled: Subjects.map(application.isRecording, function (b) {
                                    return !b;
                                })
                            })];
                    }
                    return [];
                case 92 /* StorePNG */:
                    var enableExport = application.accountSettings.enable_viewer_export;

                    if (!LocalViewer.isLocalViewer() && ((enableExport === undefined || enableExport > 0))) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.StorePNG),
                                tooltip: application.terminology.lookup(Terminology.Terms.StorePNGTooltip),
                                icon: "w_png",
                                click: function () {
                                    var selectedSeries = application.selectedSeriesContainer.read();

                                    if (selectedSeries) {
                                        selectedSeries.storeCurrentImage();
                                    }
                                },
                                enabled: Subjects.map(application.isRecording, function (b) {
                                    return !b;
                                })
                            })];
                    }
                    return [];
                case 72 /* ExportAllPNG */:
                    var enableExport = application.accountSettings.enable_viewer_export;

                    if (!LocalViewer.isLocalViewer() && (enableExport === undefined || enableExport > 0)) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.ExportAllPNG),
                                tooltip: application.terminology.lookup(Terminology.Terms.ExportAllPNGTooltip),
                                icon: "w_png",
                                click: function () {
                                    application.exportCurrentImages();
                                },
                                enabled: Subjects.map(application.isRecording, function (b) {
                                    return !b;
                                })
                            })];
                    }
                    return [];
                case 47 /* SecondaryCapture */:
                    var enableExport = application.accountSettings.enable_viewer_export;
                    var uploadPermssion = application.permissions.study_upload !== 0;

                    if (!LocalViewer.isStandardLocalViewer() && (enableExport === undefined || enableExport > 0) && uploadPermssion) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.SecondaryCapture),
                                tooltip: application.terminology.lookup(Terminology.Terms.SecondaryCaptureTooltip),
                                icon: "w_dcm",
                                click: function () {
                                    var selectedSeries = application.selectedSeriesContainer.read();

                                    if (selectedSeries) {
                                        selectedSeries.secondaryCapture();
                                    }
                                },
                                enabled: Subjects.map(application.isRecording, function (b) {
                                    return !b;
                                })
                            })];
                    }
                    return [];
                case 48 /* Metadata */:
                    var enableExport = application.accountSettings.enable_viewer_export;

                    if (!LocalViewer.isLocalViewer() && (enableExport === undefined || enableExport > 0)) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.EditMetadata),
                                tooltip: application.terminology.lookup(Terminology.Terms.EditMetadataTooltip),
                                icon: "w_dump",
                                click: function () {
                                    return application.editMetadata();
                                },
                                enabled: Subjects.map(application.isRecording, function (b) {
                                    return !b;
                                })
                            })];
                    }
                    return [];
                case 21 /* KeyImage */:
                    if (!LocalViewer.isStandardLocalViewer() && application.permissions.keyimage_edit !== 0) {
                        return [toolbar.addToggleButton({
                                title: application.terminology.lookup(Terminology.Terms.KeyImage),
                                tooltip: application.terminology.lookup(Terminology.Terms.KeyImageTooltip),
                                icon: "w_key",
                                selected: Subjects.bindW(application.selectedSeriesContainer, function (series) {
                                    return series === null ? Subjects.retW(false) : Subjects.onWrite(Subjects.bindW(series.selectedInstanceIndex, function (index) {
                                        if (series.series) {
                                            var instance = series.series.instances[index];

                                            return instance.instanceAttributes.isKeyImage;
                                        }
                                        return Subjects.retW(false);
                                    }), function (_) {
                                        return series.updateKeyImageStatusInServices();
                                    });
                                }),
                                enabled: Subjects.map(application.isRecording, function (b) {
                                    return !b;
                                })
                            })];
                    }
                    return [];
                case 14 /* Thumbs */:
                    return [toolbar.addToggleButton({
                            title: application.terminology.lookup(Terminology.Terms.Show),
                            tooltip: application.terminology.lookup(Terminology.Terms.ThumbnailsTooltip),
                            icon: "w_thumb",
                            selected: application.thumbnailsVisible,
                            toggledData: {
                                title: application.terminology.lookup(Terminology.Terms.Hide),
                                tooltip: application.terminology.lookup(Terminology.Terms.ThumbnailsTooltip),
                                icon: "w_thumb",
                                baseClass: 'showHide'
                            }
                        })];
                case 23 /* Actions */:
                    if (!LocalViewer.isStandardLocalViewer() && application.studies[0].actions.length > 0) {
                        return _.map(application.studies[0].actions, function (action) {
                            return toolbar.addToolbarButton({
                                title: action.name.truncate(10),
                                tooltip: action.name,
                                icon: "w_action",
                                click: function () {
                                    application.invokeStudyAction(application.studies[0], action);
                                }
                            });
                        });
                    }
                    return [];
                case 25 /* DeleteImage */:
                    if (!LocalViewer.isStandardLocalViewer() && application.permissions.study_delete_image !== 0) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.DeleteImage),
                                icon: "w_delete",
                                tooltip: application.terminology.lookup(Terminology.Terms.DeleteImageTooltip),
                                click: function () {
                                    var selectedSeries = application.selectedSeriesContainer.read();

                                    if (selectedSeries) {
                                        selectedSeries.deleteSelectedImage();
                                    }
                                }
                            })];
                    }
                    return [];
                case 26 /* DeleteSeries */:
                    if (!LocalViewer.isStandardLocalViewer() && application.permissions.study_delete_image !== 0) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.DeleteSeries),
                                icon: "w_delete",
                                tooltip: application.terminology.lookup(Terminology.Terms.DeleteSeriesTooltip),
                                click: function () {
                                    var selectedSeries = application.selectedSeriesContainer.read();

                                    if (selectedSeries && selectedSeries.series) {
                                        application.deleteSeries(selectedSeries.series);
                                    }
                                }
                            })];
                    }
                    return [];
                case 22 /* Settings */:
                    if (!LocalViewer.isStandardLocalViewer()) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.Settings),
                                tooltip: application.terminology.lookup(Terminology.Terms.SettingsTooltip),
                                icon: "w_settings",
                                click: function () {
                                    application.editSettings();
                                }
                            })];
                    }
                    return [];
                case 19 /* Cine */:
                    if (container && container.series && !LocalViewer.isLocalViewer() && !Cine.isBrowserSupported() && Multiframe.isMultiframe(container.series)) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.Cine),
                                tooltip: application.terminology.lookup(Terminology.Terms.CineTooltip),
                                icon: "w_cine",
                                click: function () {
                                    var selectedSeries = application.selectedSeriesContainer.read();

                                    if (selectedSeries) {
                                        selectedSeries.openMultiframeCineInNewTab();
                                    }
                                },
                                enabled: Subjects.map(application.isRecording, function (b) {
                                    return !b;
                                })
                            })];
                    }
                    return [];
                case 49 /* Play */:
                    if (container && container.series && (LocalViewer.isLocalViewer() || Cine.isBrowserSupported() || !Multiframe.isMultiframe(container.series))) {
                        return [toolbar.addToggleButton({
                                title: application.terminology.lookup(Terminology.Terms.Play),
                                tooltip: application.terminology.lookup(Terminology.Terms.PlayTooltip),
                                icon: "w_play",
                                toggledData: {
                                    title: application.terminology.lookup(Terminology.Terms.Pause),
                                    tooltip: application.terminology.lookup(Terminology.Terms.PauseTooltip),
                                    icon: "w_pause",
                                    baseClass: null
                                },
                                selected: Subjects.bindW(application.selectedSeriesContainer, function (series) {
                                    return series === null ? Subjects.retW(false) : series.cineActive;
                                })
                            })];
                    }
                    return [];
                case 50 /* Faster */:
                    if (container && container.series && (LocalViewer.isLocalViewer() || Cine.isBrowserSupported() || !Multiframe.isMultiframe(container.series))) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.Faster),
                                tooltip: application.terminology.lookup(Terminology.Terms.FasterTooltip),
                                icon: "w_cine_faster",
                                click: function () {
                                    var selectedSeries = application.selectedSeriesContainer.read();

                                    if (selectedSeries) {
                                        selectedSeries.increaseCineSpeed();
                                    }
                                },
                                enabled: Subjects.bindW(application.selectedSeriesContainer, function (series) {
                                    return series === null ? Subjects.retW(false) : series.cineActive;
                                })
                            })];
                    }
                    return [];
                case 51 /* Slower */:
                    if (container && container.series && (LocalViewer.isLocalViewer() || Cine.isBrowserSupported() || !Multiframe.isMultiframe(container.series))) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.Slower),
                                tooltip: application.terminology.lookup(Terminology.Terms.SlowerTooltip),
                                icon: "w_cine_slower",
                                click: function () {
                                    var selectedSeries = application.selectedSeriesContainer.read();

                                    if (selectedSeries) {
                                        selectedSeries.decreaseCineSpeed();
                                    }
                                },
                                enabled: Subjects.bindW(application.selectedSeriesContainer, function (series) {
                                    return series === null ? Subjects.retW(false) : series.cineActive;
                                })
                            })];
                    }
                    return [];
                case 52 /* FPSLabel */:
                    if (container && container.series && (LocalViewer.isLocalViewer() || Cine.isBrowserSupported() || !Multiframe.isMultiframe(container.series))) {
                        return [toolbar.addLabel({
                                text: Subjects.bind(application.selectedSeriesContainer, function (series) {
                                    return series === null ? Subjects.ret("") : Subjects.map(series.cineSpeed, function (fps) {
                                        return "{0} {fps}".replace("{fps}", application.terminology.lookup(Terminology.Terms.FPS)).replace("{0}", fps.toFixed(2));
                                    });
                                }, function (_, text) {
                                    return text;
                                }),
                                visible: Subjects.bindW(application.selectedSeriesContainer, function (series) {
                                    return series === null ? Subjects.retW(false) : series.cineActive;
                                })
                            })];
                    }
                    return [];
                case 20 /* Record */:
                    if (!LocalViewer.isLocalViewer() && application.permissions.study_audio_record) {
                        return [toolbar.addToggleButton({
                                title: application.terminology.lookup(Terminology.Terms.Record),
                                tooltip: application.terminology.lookup(Terminology.Terms.RecordTooltip),
                                icon: "w_record",
                                toggledData: {
                                    title: application.terminology.lookup(Terminology.Terms.Stop),
                                    tooltip: application.terminology.lookup(Terminology.Terms.StopTooltip),
                                    icon: "w_stop",
                                    baseClass: null
                                },
                                selected: application.recordingMode,
                                enabled: Subjects.map(application.meetingHost, function (b) {
                                    return !b;
                                })
                            })];
                    }
                    return [];
                case 116 /* ShowRecordings */:
                    if (container && container.series && !LocalViewer.isLocalViewer()) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.Recordings),
                                tooltip: application.terminology.lookup(Terminology.Terms.RecordingsTooltip),
                                icon: "w_recordings",
                                click: function () {
                                    container.application.toggleRecordingsInfo();
                                }
                            })];
                    }
                    return [];
                case 131 /* ShowAttachments */:
                    if (container && container.series && !LocalViewer.isLocalViewer()) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.Reports),
                                tooltip: application.terminology.lookup(Terminology.Terms.ReportsTooltip),
                                icon: "w_attachment",
                                click: function () {
                                    container.application.toggleAttachmentInfo();
                                }
                            })];
                    }
                    return [];
                case 141 /* ShowAllGSPS */:
                    if (container && container.series && !LocalViewer.isLocalViewer()) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.GSPS),
                                tooltip: application.terminology.lookup(Terminology.Terms.GSPSTooltip),
                                icon: "w_annotations",
                                click: function () {
                                    container.application.toggleGSPSInfo();
                                }
                            })];
                    }
                    return [];
                case 109 /* PlayRecording */:
                    if (container && container.series && !LocalViewer.isLocalViewer()) {
                        return [toolbar.addToggleButton({
                                title: application.terminology.lookup(Terminology.Terms.Play),
                                tooltip: application.terminology.lookup(Terminology.Terms.PlayTooltip),
                                icon: "w_play",
                                toggledData: {
                                    title: application.terminology.lookup(Terminology.Terms.Pause),
                                    tooltip: application.terminology.lookup(Terminology.Terms.PauseTooltip),
                                    icon: "w_pause",
                                    baseClass: null
                                },
                                selected: application.playbackMode,
                                enabled: Subjects.retW(true)
                            })];
                    }
                    return [];
                case 110 /* StopPlayback */:
                    if (container && container.series && !LocalViewer.isLocalViewer()) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.Stop),
                                tooltip: application.terminology.lookup(Terminology.Terms.StopTooltip),
                                icon: "w_stop",
                                click: function () {
                                    container.application.playbackStopped();
                                }
                            })];
                    }
                    return [];
                case 111 /* RewindPlayback */:
                    if (container && container.series && !LocalViewer.isLocalViewer()) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.Rewind),
                                tooltip: application.terminology.lookup(Terminology.Terms.RewindTooltip),
                                icon: "w_rewind",
                                click: function () {
                                    container.application.rewindPlayback();
                                }
                            })];
                    }
                    return [];
                case 112 /* FastForwardPlayback */:
                    if (container && container.series && !LocalViewer.isLocalViewer()) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.Forward),
                                tooltip: application.terminology.lookup(Terminology.Terms.ForwardTooltip),
                                icon: "w_forward",
                                click: function () {
                                    container.application.fastForwardPlayback();
                                }
                            })];
                    }
                    return [];
                case 55 /* GSPSLayers */:
                    var series = container.series;

                    if (series && series.studyAttributes && series.studyAttributes.presentationStateIODs) {
                        return _.map(series.studyAttributes.presentationStateIODs, function (gsps) {
                            var title = gsps.identificationModule.presentationCreationDate;
                            var tooltip = title + " (" + gsps.graphicAnnotationModule.graphicAnnotationSequence.length + ")";

                            return toolbar.addToggleButton({
                                title: title,
                                tooltip: tooltip,
                                icon: "w_measurements",
                                selected: Subjects.lens(container.hiddenGSPSLayers, function (layers) {
                                    return !_.contains(layers, gsps);
                                }, function (layers, shown) {
                                    if (shown) {
                                        return _.without(layers, gsps);
                                    } else {
                                        return [gsps].concat(layers);
                                    }
                                })
                            });
                        });
                    }

                    return [];
                case 56 /* Magnify */:
                    return [AbstractToolbars.addMouseToolButton(toolbar, {
                            title: application.terminology.lookup(Terminology.Terms.Magnify),
                            tooltip: application.terminology.lookup(Terminology.Terms.MagnifyTooltip),
                            icon: "w_zoom",
                            tool: 15 /* Magnify */,
                            selectedTool: application.selectedTool,
                            selectedTool2: application.selectedTool2
                        })];
                case 57 /* Anonymize */:
                    if (container && application.permissions.annotation_edit !== 0 && !LocalViewer.isLocalViewer()) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.Anonymize),
                                tooltip: application.terminology.lookup(Terminology.Terms.AnonymizeTooltip),
                                icon: "w_dcm",
                                click: function () {
                                    container.application.anonymize(1 /* Series */);
                                }
                            })];
                    }
                    return [];
                case 59 /* AnonymizeStudy */:
                    if (container && application.permissions.annotation_edit !== 0 && !LocalViewer.isLocalViewer()) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.AnonymizeStudy),
                                tooltip: application.terminology.lookup(Terminology.Terms.AnonymizeStudyTooltip),
                                icon: "w_dcm",
                                click: function () {
                                    container.application.anonymize(0 /* Study */);
                                }
                            })];
                    }
                    return [];
                case 60 /* AnonymizeImage */:
                    if (container && application.permissions.annotation_edit !== 0 && !LocalViewer.isLocalViewer()) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.AnonymizeImage),
                                tooltip: application.terminology.lookup(Terminology.Terms.AnonymizeImageTooltip),
                                icon: "w_dcm",
                                click: function () {
                                    container.application.anonymize(2 /* Image */);
                                }
                            })];
                    }
                    return [];
                case 125 /* CropSeries */:
                    if (container && application.permissions.annotation_edit !== 0 && !LocalViewer.isLocalViewer()) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.CropSeries),
                                tooltip: application.terminology.lookup(Terminology.Terms.CropSeriesTooltip),
                                icon: "w_crop",
                                click: function () {
                                    container.application.cropStudy();
                                }
                            })];
                    }
                    return [];
                case 108 /* SplitStudy */:
                    if (container && application.permissions.study_split !== 0 && !LocalViewer.isLocalViewer()) {
                        return [toolbar.addToggleButton({
                                title: application.terminology.lookup(Terminology.Terms.SplitStudy),
                                tooltip: application.terminology.lookup(Terminology.Terms.SplitStudyTooltip),
                                icon: "w_cut",
                                selected: application.splitStudyEnabled
                            })];
                    }
                    return [];
                case 133 /* ReverseSeries */:
                    if (container && application.permissions.study_split !== 0 && !LocalViewer.isLocalViewer()) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.ReverseSeries),
                                tooltip: application.terminology.lookup(Terminology.Terms.ReverseSeriesTooltip),
                                icon: "w_random",
                                enabled: Subjects.bindW(application.selectedSeriesContainer, function (series) {
                                    return series === null ? Subjects.retW(false) : Subjects.retW(!Multiframe.isMultiframe(series.series));
                                }),
                                click: function () {
                                    container.application.reverseSeries();
                                }
                            })];
                    }
                    return [];
                case 134 /* UnweaveSeries */:
                    if (container && application.permissions.study_split !== 0 && !LocalViewer.isLocalViewer()) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.UnweaveSeries),
                                tooltip: application.terminology.lookup(Terminology.Terms.UnweaveSeriesTooltip),
                                icon: "w_random",
                                enabled: Subjects.bindW(application.selectedSeriesContainer, function (series) {
                                    return series === null ? Subjects.retW(false) : Subjects.retW(!Multiframe.isMultiframe(series.series));
                                }),
                                click: function () {
                                    container.application.unweaveSeries();
                                }
                            })];
                    }
                    return [];
                case 135 /* RearrangeSeries */:
                    if (container && application.permissions.study_split !== 0 && !LocalViewer.isLocalViewer()) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.RearrangeSeries),
                                tooltip: application.terminology.lookup(Terminology.Terms.RearrangeSeriesTooltip),
                                icon: "w_random",
                                enabled: Subjects.bindW(application.selectedSeriesContainer, function (series) {
                                    return series === null ? Subjects.retW(false) : Subjects.retW(!Multiframe.isMultiframe(series.series));
                                }),
                                click: function () {
                                    container.application.rearrangeSeries();
                                }
                            })];
                    }
                    return [];
                case 136 /* PartSeries */:
                    if (container && application.permissions.study_split !== 0 && !LocalViewer.isLocalViewer()) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.PartSeries),
                                tooltip: application.terminology.lookup(Terminology.Terms.PartSeriesTooltip),
                                icon: "w_random",
                                enabled: Subjects.bindW(application.selectedSeriesContainer, function (series) {
                                    return series === null ? Subjects.retW(false) : Subjects.retW(!Multiframe.isMultiframe(series.series));
                                }),
                                click: function () {
                                    container.application.partSeries();
                                }
                            })];
                    }
                    return [];
                case 138 /* MergeSeries */:
                    if (container && application.permissions.study_split !== 0 && !LocalViewer.isLocalViewer()) {
                        return [toolbar.addToggleButton({
                                title: application.terminology.lookup(Terminology.Terms.MergeSeries),
                                tooltip: application.terminology.lookup(Terminology.Terms.MergeSeriesTooltip),
                                icon: "w_random",
                                enabled: Subjects.bindW(application.selectedSeriesContainer, function (series) {
                                    return series === null ? Subjects.retW(false) : Subjects.retW(!Multiframe.isMultiframe(series.series));
                                }),
                                selected: application.mergeSeriesEnabled
                            })];
                    }
                    return [];
                case 137 /* ResetStudy */:
                    if (container && application.permissions.study_split !== 0 && !LocalViewer.isLocalViewer()) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.ResetStudyArrangement),
                                tooltip: application.terminology.lookup(Terminology.Terms.ResetStudyArrangementTooltip),
                                icon: "w_undo",
                                enabled: Subjects.bindW(application.selectedSeriesContainer, function (series) {
                                    return series === null ? Subjects.retW(false) : Subjects.retW(!Multiframe.isMultiframe(series.series));
                                }),
                                click: function () {
                                    container.application.resetArrangement();
                                }
                            })];
                    }
                    return [];
                case 61 /* PreviousSeriesSet */:
                    if (container) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.PreviousSeriesSet),
                                tooltip: application.terminology.lookup(Terminology.Terms.PreviousSeriesSetTooltip),
                                icon: "w_action",
                                click: function () {
                                    container.application.loadSeriesSet(false);
                                }
                            })];
                    }
                    return [];
                case 62 /* NextSeriesSet */:
                    if (container) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.NextSeriesSet),
                                tooltip: application.terminology.lookup(Terminology.Terms.NextSeriesSetTooltip),
                                icon: "w_action",
                                click: function () {
                                    container.application.loadSeriesSet(true);
                                }
                            })];
                    }
                    return [];
                case 63 /* CineAll */:
                    if (container) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.PreviousSeriesSet),
                                tooltip: application.terminology.lookup(Terminology.Terms.PreviousSeriesSetTooltip),
                                icon: "w_action",
                                click: function () {
                                    container.application.toggleCineAll();
                                }
                            })];
                    }
                    return [];
                case 64 /* DuplicateAnnotation */:
                    if (container) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.DuplicateAnnotation),
                                tooltip: application.terminology.lookup(Terminology.Terms.DuplicateAnnotationTooltip),
                                icon: "w_action",
                                click: function () {
                                    container.application.duplicateShape();
                                }
                            })];
                    }
                    return [];
                case 65 /* CopyAnnotation */:
                    if (container) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.CopyAnnotation),
                                tooltip: application.terminology.lookup(Terminology.Terms.CopyAnnotationTooltip),
                                icon: "w_action",
                                click: function () {
                                    container.application.copyShape();
                                }
                            })];
                    }
                    return [];
                case 66 /* PasteAnnotation */:
                    if (container) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.PasteAnnotation),
                                tooltip: application.terminology.lookup(Terminology.Terms.PasteAnnotationTooltip),
                                icon: "w_action",
                                click: function () {
                                    container.application.pasteShape();
                                }
                            })];
                    }
                    return [];
                case 87 /* CoLocalization */:
                    if (application.permissions.annotation_edit !== 0 && !LocalViewer.isLocalViewer()) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.CoLocalization),
                                tooltip: application.terminology.lookup(Terminology.Terms.CoLocalizationTooltip),
                                icon: "w_colocate",
                                enabled: Subjects.zip(Subjects.bind(application.selectedSeriesContainer, function (series) {
                                    return series === null ? Subjects.ret(null) : series.selectedMeasurement;
                                }, function (_, measurement) {
                                    return measurement != null && measurement.editable;
                                }), application.isRecording, function (b1, b2) {
                                    return b1 && !b2;
                                }),
                                click: function () {
                                    container.application.copyShapeToAllSeries();
                                }
                            })];
                    }
                    return [];
                case 89 /* Label */:
                    if (application.permissions.annotation_edit !== 0 && !LocalViewer.isLocalViewer()) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.Label),
                                tooltip: application.terminology.lookup(Terminology.Terms.LabelTooltip),
                                icon: "w_header",
                                enabled: Subjects.zip(Subjects.bind(application.selectedSeriesContainer, function (series) {
                                    return series === null ? Subjects.ret(null) : series.selectedMeasurement;
                                }, function (_, measurement) {
                                    return measurement != null && measurement.editable && !(measurement instanceof Measurements.Text);
                                }), application.isRecording, function (b1, b2) {
                                    return b1 && !b2;
                                }),
                                click: function () {
                                    container.application.labelAnnotation();
                                }
                            })];
                    }
                    return [];
                case 122 /* PixelSpacingUser */:
                    if (application.permissions.annotation_edit !== 0 && !LocalViewer.isLocalViewer()) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.AssignPixelSpacing),
                                tooltip: application.terminology.lookup(Terminology.Terms.AssignPixelSpacingTooltip),
                                icon: "w_edit",
                                enabled: Subjects.zip(Subjects.bind(application.selectedSeriesContainer, function (series) {
                                    return series === null ? Subjects.ret(null) : series.selectedMeasurement;
                                }, function (_, measurement) {
                                    return measurement != null && measurement.editable && !(measurement instanceof Measurements.Text);
                                }), application.isRecording, function (b1, b2) {
                                    return b1 && !b2;
                                }),
                                click: function () {
                                    container.application.assignPixelSpacing();
                                }
                            })];
                    }
                    return [];
                case 123 /* SliceSpacingUser */:
                    if (application.permissions.annotation_edit !== 0 && !LocalViewer.isLocalViewer()) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.AssignSliceSpacing),
                                tooltip: application.terminology.lookup(Terminology.Terms.AssignSliceSpacingTooltip),
                                icon: "w_edit",
                                enabled: Subjects.zip(Subjects.bind(application.selectedSeriesContainer, function (series) {
                                    return series === null ? Subjects.ret(null) : series.selectedMeasurement;
                                }, function (_, measurement) {
                                    return measurement != null && measurement.editable && !(measurement instanceof Measurements.Text);
                                }), application.isRecording, function (b1, b2) {
                                    return b1 && !b2;
                                }),
                                click: function () {
                                    container.application.assignSliceSpacing();
                                }
                            })];
                    }
                    return [];
                case 67 /* StartMeeting */:
                    if (container && application.permissions.meeting_edit !== 0 && !LocalViewer.isLocalViewer()) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.StartMeeting),
                                tooltip: application.terminology.lookup(Terminology.Terms.StartMeetingTooltip),
                                icon: "w_meeting",
                                click: function () {
                                    application.startMeeting();
                                },
                                enabled: Subjects.map(application.recordingMode, function (b) {
                                    return !b;
                                })
                            })];
                    }
                    return [];
                case 68 /* UseDiagnosticQuality */:
                    if (!LocalViewer.isLocalViewer() && application.accountSettings.viewer_diagnostic_quality != 1 && application.accountSettings.viewer_diagnostic_quality_always != 1) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.UseDiagnosticQuality),
                                tooltip: application.terminology.lookup(Terminology.Terms.UseDiagnosticQualityTooltip),
                                icon: "w_dcm",
                                click: function () {
                                    application.useDiagosticQualityViewer();
                                },
                                enabled: Subjects.map(application.recordingMode, function (b) {
                                    return !b;
                                })
                            })];
                    }
                    return [];
                case 69 /* RecordAudio */:
                    if (!LocalViewer.isLocalViewer() && application.accountSettings.viewer_diagnostic_quality != 1 && application.accountSettings.viewer_diagnostic_quality_always != 1) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.RecordAudio),
                                tooltip: application.terminology.lookup(Terminology.Terms.RecordAudio),
                                icon: "w_audio",
                                click: function () {
                                    application.recordAudio();
                                },
                                enabled: Subjects.map(application.recordingMode, function (b) {
                                    return !b;
                                })
                            })];
                    }
                    return [];
                case 75 /* Blank */:
                    return [toolbar.addToolbarButton({ icon: 'w_blank' })];
                case 77 /* MouseToolSettings */:
                    return [toolbar.addToolbarButton({
                            title: application.terminology.lookup(Terminology.Terms.Mouse),
                            tooltip: application.terminology.lookup(Terminology.Terms.MouseTooltip),
                            icon: "w_pointer",
                            click: function () {
                                toolbar.addMouseToolSettings(application);
                            }
                        })];
                case 78 /* ShowOnlyKeyImages */:
                    if (application.permissions.keyimage_view !== 0) {
                        return [toolbar.addToggleButton({
                                title: application.terminology.lookup(Terminology.Terms.ShowOnlyKeyImages),
                                tooltip: application.terminology.lookup(Terminology.Terms.ShowOnlyKeyImagesTooltip),
                                icon: "w_key",
                                selected: application.keyImageSeriesEnabled
                            })];
                    }
                    return [];
                case 126 /* SaveKeyImageLayout */:
                    if (application.permissions.keyimage_view !== 0) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.SaveKeyImageLayout),
                                tooltip: application.terminology.lookup(Terminology.Terms.SaveKeyImageLayoutTooltip),
                                icon: "w_key",
                                click: function () {
                                    application.saveKeyImageLayoutState();
                                }
                            })];
                    }
                    return [];
                case 79 /* ExportAVI */:
                    var enableExport = application.accountSettings.enable_viewer_export;

                    if (LocalViewer.isLocalViewer() || (enableExport === undefined || enableExport > 0)) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.ExportAVI),
                                tooltip: application.terminology.lookup(Terminology.Terms.ExportAVITooltip),
                                icon: "w_export_video",
                                click: function () {
                                    var selectedSeries = application.selectedSeriesContainer.read();

                                    if (selectedSeries) {
                                        selectedSeries.exportVideo('avi');
                                    }
                                },
                                enabled: Subjects.map(application.isRecording, function (b) {
                                    return !b;
                                })
                            })];
                    }
                    return [];
                case 80 /* ExportMP4 */:
                    var enableExport = application.accountSettings.enable_viewer_export;

                    if (LocalViewer.isLocalViewer() || (enableExport === undefined || enableExport > 0)) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.ExportMP4),
                                tooltip: application.terminology.lookup(Terminology.Terms.ExportMP4Tooltip),
                                icon: "w_export_video",
                                click: function () {
                                    var selectedSeries = application.selectedSeriesContainer.read();

                                    if (selectedSeries) {
                                        selectedSeries.exportVideo('mp4');
                                    }
                                },
                                enabled: Subjects.map(application.isRecording, function (b) {
                                    return !b;
                                })
                            })];
                    }
                    return [];
                case 81 /* ExportSeries */:
                    var enableExport = application.accountSettings.enable_viewer_export;

                    if (LocalViewer.isLocalViewer() || (enableExport === undefined || enableExport > 0)) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.ExportSeries),
                                tooltip: application.terminology.lookup(Terminology.Terms.ExportSeriesTooltip),
                                icon: "w_download",
                                click: function () {
                                    var selectedSeries = application.selectedSeriesContainer.read();

                                    if (selectedSeries) {
                                        selectedSeries.exportSeries();
                                    }
                                },
                                enabled: Subjects.map(application.isRecording, function (b) {
                                    return !b;
                                })
                            })];
                    }
                    return [];
                case 113 /* ExportStudy */:
                    var enableExport = application.accountSettings.enable_viewer_export;

                    if (LocalViewer.isLocalViewer() || (enableExport === undefined || enableExport > 0)) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.ExportStudy),
                                tooltip: application.terminology.lookup(Terminology.Terms.ExportStudyTooltip),
                                icon: "w_download",
                                click: function () {
                                    var selectedSeries = application.selectedSeriesContainer.read();

                                    if (selectedSeries) {
                                        selectedSeries.exportStudy();
                                    }
                                },
                                enabled: Subjects.map(application.isRecording, function (b) {
                                    return !b;
                                })
                            })];
                    }
                    return [];
                case 114 /* ExportISO */:
                    var enableExport = application.accountSettings.enable_viewer_export;

                    if (LocalViewer.isLocalViewer() || (enableExport === undefined || enableExport > 0)) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.ExportISO),
                                tooltip: application.terminology.lookup(Terminology.Terms.ExportISOTooltip),
                                icon: "w_download",
                                click: function () {
                                    var selectedSeries = application.selectedSeriesContainer.read();

                                    if (selectedSeries) {
                                        selectedSeries.exportISO();
                                    }
                                },
                                enabled: Subjects.map(application.isRecording, function (b) {
                                    return !b;
                                })
                            })];
                    }
                    return [];
                case 115 /* ExportLocalViewer */:
                    var enableExport = application.accountSettings.enable_viewer_export;

                    if (LocalViewer.isLocalViewer() || (enableExport === undefined || enableExport > 0)) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.ExportViewer),
                                tooltip: application.terminology.lookup(Terminology.Terms.ExportViewerTooltip),
                                icon: "w_download",
                                click: function () {
                                    var selectedSeries = application.selectedSeriesContainer.read();

                                    if (selectedSeries) {
                                        selectedSeries.exportLocalViewer();
                                    }
                                },
                                enabled: Subjects.map(application.isRecording, function (b) {
                                    return !b;
                                })
                            })];
                    }
                    return [];
                case 82 /* NextStudy */:
                    if (!LocalViewer.isStandardLocalViewer()) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.NextStudy),
                                tooltip: application.terminology.lookup(Terminology.Terms.NextStudyTooltip),
                                icon: "w_chevron_right",
                                click: function () {
                                    application.nextWorklistStudy(false);
                                },
                                enabled: Subjects.ret(application.hasNextWorklistStudy(false))
                            })];
                    }
                    return [];
                case 83 /* PreviousStudy */:
                    if (!LocalViewer.isStandardLocalViewer()) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.PreviousStudy),
                                tooltip: application.terminology.lookup(Terminology.Terms.PreviousStudyTooltip),
                                icon: "w_chevron_left",
                                click: function () {
                                    application.previousWorklistStudy(false);
                                },
                                enabled: Subjects.ret(application.hasPreviousWorklistStudy(false))
                            })];
                    }
                    return [];
                case 90 /* NextStudyByMRN */:
                    if (!LocalViewer.isStandardLocalViewer()) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.NextStudyByMRN),
                                tooltip: application.terminology.lookup(Terminology.Terms.NextStudyByMRNTooltip),
                                icon: "w_chevron_right",
                                click: function () {
                                    application.nextWorklistStudy(true);
                                },
                                enabled: Subjects.ret(application.hasNextWorklistStudy(true))
                            })];
                    }
                    return [];
                case 91 /* PreviousStudyByMRN */:
                    if (!LocalViewer.isStandardLocalViewer()) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.PreviousStudyByMRN),
                                tooltip: application.terminology.lookup(Terminology.Terms.PreviousStudyByMRNTooltip),
                                icon: "w_chevron_left",
                                click: function () {
                                    application.previousWorklistStudy(true);
                                },
                                enabled: Subjects.ret(application.hasPreviousWorklistStudy(true))
                            })];
                    }
                    return [];
                case 101 /* NextImage */:
                    if (!LocalViewer.isStandardLocalViewer()) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.NextImage),
                                tooltip: application.terminology.lookup(Terminology.Terms.NextImageTooltip),
                                icon: "w_chevron_right",
                                click: function () {
                                    application.nextSingleSeriesImage();
                                },
                                enabled: Subjects.ret(application.hasNextSingleSeriesImage())
                            })];
                    }
                    return [];
                case 100 /* PreviousImage */:
                    if (!LocalViewer.isStandardLocalViewer()) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.PreviousImage),
                                tooltip: application.terminology.lookup(Terminology.Terms.PreviousImageTooltip),
                                icon: "w_chevron_left",
                                click: function () {
                                    application.previousSingleSeriesImage();
                                },
                                enabled: Subjects.ret(application.hasPreviousSingleSeriesImage())
                            })];
                    }
                    return [];
                case 102 /* LayoutSingleSeries */:
                    return [toolbar.addToggleButton({
                            title: application.terminology.lookup(Terminology.Terms.LayoutSingleSeries),
                            tooltip: application.terminology.lookup(Terminology.Terms.LayoutSingleSeriesTooltip),
                            icon: "w_series_view",
                            selected: application.singleSeriesEnabled,
                            enabled: Subjects.bindW(application.layout, function (layout) {
                                return Subjects.retW(application.canDisplaySingleSeries(layout));
                            })
                        })];
                case 117 /* LoadReport */:
                    if (!LocalViewer.isLocalViewer()) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.NewReport),
                                tooltip: application.terminology.lookup(Terminology.Terms.NewReportTooltip),
                                icon: "w_dump",
                                click: function () {
                                    application.loadReport(container.series.studyAttributes.uuid);
                                },
                                enabled: Subjects.ret(true)
                            })];
                    }
                    return [];
                case 124 /* RemoveImages */:
                    if (!LocalViewer.isStandardLocalViewer() && application.permissions.study_delete_image !== 0) {
                        return [toolbar.addToolbarButton({
                                title: application.terminology.lookup(Terminology.Terms.RemoveImages),
                                icon: "w_delete",
                                tooltip: application.terminology.lookup(Terminology.Terms.RemoveImagesTooltip),
                                click: function () {
                                    application.removeImages();
                                }
                            })];
                    }
                    return [];
            }
        };
        return AbstractToolbars;
    })();
    Views.AbstractToolbars = AbstractToolbars;

    /**
    * The main application toolbar
    */
    var Toolbar = (function () {
        function Toolbar(el, application) {
            this.el = el;
            this.application = application;
        }
        /**
        * Get an appropriate toolbar configuration for a modality
        */
        Toolbar.prototype.readToolbarConfiguration = function (settings, modality) {
            var modalitySettings;

            if (settings.modalities) {
                modalitySettings = _.find(settings.modalities, function (m) {
                    return m.modality === modality;
                });
            }

            if (LocalViewer.isStandardLocalViewer()) {
                return ToolbarButtons.localViewerToolbar(modality);
            } else if (modalitySettings && modalitySettings.toolbar !== undefined) {
                return modalitySettings.toolbar;
            } else if (settings.toolbar !== undefined) {
                return settings.toolbar;
            } else {
                return ToolbarButtons.createDefaultToolbar(modality);
            }
        };

        /**
        * Create the toolbar based on the user settings, for the specified modality
        */
        Toolbar.prototype.configure = function (settings, modality) {
            var toolbarConfiguration = this.readToolbarConfiguration(settings, modality);
            var $toolbar = $('<div>').addClass('toolbar-scroll');

            if (this.menuBar) {
                _.each(this.menuBar.listeners, function (l) {
                    l.cancel();
                });

                this.menuBar.listeners = [];
                this.menuBar = null;
            }

            this.menuBar = new MenuBar($toolbar);

            AbstractToolbars.createToolbarFromConfiguration(this.menuBar, toolbarConfiguration, this.application);

            this.el.empty();
            this.el.append($toolbar);

            // special handling of color control icon
            $(".colorControl").parents('.dropdown-icon').css("color", Annotations.COLORS[this.application.currentColor.read()]);
        };

        /**
        * Create necessary elements, add them to the DOM and hook up event handlers
        */
        Toolbar.prototype.render = function () {
            var _this = this;
            var container = this.application.selectedSeriesContainer.read();
            var settings = this.application.settings.read();

            if (container && container.series && settings) {
                if (container.playbackMode.read()) {
                    this.configure({ toolbar: ToolbarButtons.createPlaybackToolbar() }, null);
                } else {
                    var modality = container.series.seriesAttributes.modality;
                    this.configure(settings, modality);
                }
            }

            if (this.subscription) {
                this.subscription.cancel();
                this.subscription = null;
            }

            this.subscription = this.application.selectedSeriesContainer.updates.subscribe({
                next: function (container) {
                    var settings = _this.application.settings.read();
                    if (container.series && container.series.seriesAttributes.modality) {
                        if (container.playbackMode.read()) {
                            _this.configure({ toolbar: ToolbarButtons.createPlaybackToolbar() }, null);
                            _this.application.showPlaybackBanner(_this.application.terminology.lookup(Terminology.Terms.PlaybackInProgress));
                        } else {
                            _this.configure(settings, container.series.seriesAttributes.modality);
                        }
                    }
                },
                done: function () {
                },
                fail: function (_) {
                }
            });
        };
        return Toolbar;
    })();
    Views.Toolbar = Toolbar;
})(Views || (Views = {}));
///<reference path='../typings/jquery/jquery.d.ts' />
///<reference path='../typings/underscore/underscore.d.ts' />
///<reference path='../typings/virtual-dom/virtual-dom.d.ts' />
///<reference path='../classes/Types.ts' />
///<reference path='../libs/Observable.ts' />
///<reference path='../libs/Query.ts' />
///<reference path='../libs/Services.ts' />
///<reference path='../libs/V3Storage.ts' />
///<reference path='../libs/Multiframe.ts' />
///<reference path='../libs/Study.ts' />
///<reference path='../libs/Date.ts' />
///<reference path='../libs/Number.ts' />
///<reference path='../models/StudySchema.ts' />
///<reference path='../libs/Subject.ts' />
///<reference path='../libs/Touch.ts' />
///<reference path='../models/Study.ts' />
///<reference path='Series.ts' />
var Views;
(function (Views) {
    var FoldInHook = (function () {
        function FoldInHook() {
        }
        FoldInHook.prototype.hook = function (node, propertyName, previousValue) {
            if (!previousValue) {
                // Use animate instead of fadeIn so that the display property is
                // still inline-block when the animation completes.
                $(node).css({ opacity: 0.0 }).animate({ opacity: 1.0 }, 500);
            }
        };
        return FoldInHook;
    })();
    Views.FoldInHook = FoldInHook;

    /**
    * View which displays study information, priors, and thumbnails for all available series
    */
    var Thumbnails = (function () {
        function Thumbnails(el, application) {
            this.el = el;
            this.application = application;
        }
        Thumbnails.prototype.toVNode = function (studies, priors) {
            var vdom = virtualDom;

            var dateFormat = this.application.settings.read().dateFormat;

            return vdom.h('div', {}, [
                this.studyInfo(studies[0], dateFormat),
                this.thumbnailIcons(studies, priors, dateFormat),
                this.placementIcon()
            ]);
        };

        Thumbnails.prototype.studyInfo = function (study, dateFormat) {
            var _this = this;
            var vdom = virtualDom;
            var term = function (t) {
                return _this.application.terminology.lookup(t);
            };
            var hideDateTime = this.application.settings.read().hideThumbnailsDateTime;
            var elements = [];

            elements.push(vdom.h('div', {}, [study.studyAttributes.patientName || term(Terminology.Terms.Patient)]));
            elements.push(vdom.h('div', {}, [this.patientText(study, dateFormat)]));
            elements.push(vdom.h('div', {}, (study.studyAttributes.accelerated || this.application.studyStorage.localAccelerator ? [vdom.h('span', { className: 'fa fa-flash', title: 'This study is accelerated.' }, []), " "] : []).concat([study.studyAttributes.studyDescription || term(Terminology.Terms.Study)])));

            if (!hideDateTime) {
                elements.push(vdom.h('div', {}, [this.studyText(study, dateFormat)]));
            }

            elements.push(vdom.h('div', {}, [study.series.length + ' ' + term(Terminology.Terms.SeriesPlural)]));
            elements.push(this.studyPageLink(study));

            return vdom.h('div.study-thumbnails-container-info', {}, elements, 'study-info');
        };

        Thumbnails.prototype.studyPageLink = function (study) {
            var vdom = virtualDom;

            if (LocalViewer.isLocalViewer()) {
                return vdom.h('div', {}, []);
            } else if (this.application.accountSettings.viewer_study_page_link_visible == null || this.application.accountSettings.viewer_study_page_link_visible == 1) {
                var studyUri = null;
                if (this.application.accountSettings.viewer_study_page_link_url != null && this.application.accountSettings.viewer_study_page_link_url.length > 0) {
                    studyUri = this.application.accountSettings.viewer_study_page_link_url;
                } else {
                    var query = study.studyAttributes.queryObject;
                    studyUri = '/?route=view_study_meta_with_sid' + '&namespace_id=' + encodeURIComponent(query.storageNamespace.value) + '&study_uid=' + encodeURIComponent(query.studyUid.value) + '&phi_namespace=' + encodeURIComponent(query.phiNamespace.value) + '&sid=' + encodeURIComponent(this.application.sessionId.value);
                }

                return vdom.h('p.study-page-link', {}, [
                    vdom.h('a', {
                        href: studyUri,
                        target: '_blank',
                        style: {
                            'text-shadow': 'none'
                        }
                    }, [
                        this.application.terminology.lookup(Terminology.Terms.StudyPage)
                    ])
                ]);
            } else {
                return vdom.h('div', {}, []);
            }
        };

        Thumbnails.prototype.patientText = function (study, dateFormat) {
            var patientText = "(" + (study.studyAttributes.patientSex || "-") + ") ";

            if (study.studyAttributes.patientBirthDate) {
                patientText += study.studyAttributes.patientBirthDate.toShortDateString(dateFormat);
            }

            return patientText;
        };

        Thumbnails.prototype.studyText = function (study, dateFormat) {
            var hideThumbnailsDateTime = this.application.settings.read().hideThumbnailsDateTime || (this.application.accountSettings.viewer_hide_thumbnails_datetime != 0);

            if (study.studyAttributes.studyCreateDate && !hideThumbnailsDateTime) {
                return study.studyAttributes.studyCreateDate.toShortDateString(dateFormat);
            }

            return "-";
        };

        Thumbnails.prototype.thumbnailIcons = function (studies, priors, dateFormat) {
            var _this = this;
            var vdom = virtualDom;

            var term = function (t) {
                return _this.application.terminology.lookup(t);
            };

            var keyImageSeriesVisible = false;
            var stackSeriesVisible = false;
            var stackSeriesPosition = 0 /* None */;

            var settings = this.application.settings.read();

            // Check if we should display key image series
            if (this.application.keyImageSeriesEnabled.read()) {
                keyImageSeriesVisible = true;
            }

            // Look for modality setting based on first series in first study
            if (settings.modalities && studies[0] != null && studies[0].series[0] != null && studies[0].series[0].seriesAttributes != null) {
                var modalitySettings = _.find(settings.modalities, function (m) {
                    return m.modality === studies[0].series[0].seriesAttributes.modality;
                });

                if (modalitySettings && modalitySettings.stackSeriesVisible) {
                    stackSeriesVisible = modalitySettings.stackSeriesVisible;
                }

                if (modalitySettings && modalitySettings.stackSeriesPosition) {
                    stackSeriesPosition = modalitySettings.stackSeriesPosition;
                }
            }

            // Default legacy setting, overrides modality specific settings, should only be set at a modality level going forward
            if (settings.showStackSeries) {
                stackSeriesVisible = true;
                stackSeriesPosition = 2 /* Last */;
            }

            var priorsNotLoaded = _.filter(priors, function (prior) {
                return !_.any(_this.application.studies, function (study) {
                    return study.studyAttributes.queryObject.studyUid.value === prior.study_uid || study.studyAttributes.uuid.value === prior.uuid;
                });
            });

            var allStudies = _.map(studies, function (s) {
                return Either.Left(s);
            }).concat(_.map(priorsNotLoaded, function (s) {
                return Either.Right(s);
            }));

            var sortedStudies = _.sortBy(allStudies, function (e) {
                var maybeDate = Either.either(e, function (study) {
                    return study === _this.application.studies[0] ? Maybe.Nothing() : Maybe.fromNull(study.studyAttributes.studyCreateDate);
                }, function (prior) {
                    return Maybe.fromNull(prior.study_date_value);
                });

                return Maybe.fromMaybe(Maybe.fmap(maybeDate, function (d) {
                    return -d.getTime();
                }), -Infinity);
            });

            // Display report series before other series
            if (settings.displayReportsFirst) {
                sortedStudies = _.map(sortedStudies, function (e) {
                    return Either.either(e, function (study) {
                        study.series = _.sortBy(study.series, function (series) {
                            // For sorting purposes, return a zero for report series and a 1 for other series
                            return (Study.isPDFSeries(series) || Study.isImagedORUSeries(series)) ? 0 : 1;
                        });
                        return Either.Left(study);
                    }, function (prior) {
                        return Either.Right(prior);
                    });
                });
            }

            var icons = [];

            var keyImageOrder = settings.keyImageOrder;
            if (keyImageSeriesVisible && keyImageOrder) {
                var allKeyImageSeries = _.toArray(_.flatten(_.map(this.application.studies, function (study) {
                    return study.keyImageSeries;
                })));

                if (allKeyImageSeries && allKeyImageSeries.length && allKeyImageSeries[0]) {
                    var allSeries = Study.orderSeries(allKeyImageSeries, keyImageOrder == "desc");
                    var keyimageSeries = _.map(allSeries, function (series) {
                        return _this.keyImageSeriesIcon(series);
                    });
                    icons = _.flatten([keyimageSeries]);
                }
            }

            if (icons.length == 0) {
                icons = _.chain(sortedStudies).map(function (e, studyIndex, list) {
                    return Either.either(e, function (study) {
                        var studyIcons = _.map(study.series, function (series) {
                            return _this.thumbnailIcon(study, studyIndex, series, dateFormat);
                        });

                        var spacer = vdom.h('li.series-thumbnail-spacer', {}, []);

                        if (keyImageSeriesVisible && study.keyImageSeries && study.keyImageSeries.length > 0) {
                            var keyimageSeries = _.map(study.keyImageSeries, function (series) {
                                return _this.keyImageSeriesIcon(series);
                            });
                            studyIcons = _.flatten([keyimageSeries]);
                        }

                        if (stackSeriesVisible) {
                            var stackSeries = _this.stackSeriesIcon(study);

                            if (stackSeriesPosition == 1 /* First */) {
                                studyIcons = _.flatten([stackSeries, spacer, studyIcons]);
                            } else if (stackSeriesPosition == 2 /* Last */) {
                                studyIcons = _.flatten([studyIcons, stackSeries, spacer]);
                            }
                        }

                        return studyIcons;
                    }, function (prior) {
                        return _this.relatedStudy(prior, studies[0]);
                    });
                }).flatten().value();
            }

            return vdom.h('ul.thumbnail-icons', {}, icons, 'thumbnail-icons');
        };

        Thumbnails.prototype.thumbnailIcon = function (study, studyIndex, series, dateFormat) {
            var _this = this;
            var vdom = virtualDom;

            var instance = series.instances[0];
            var first = instance;

            var modality = series.seriesAttributes.modality;

            var settings = this.application.settings.read();

            if (settings.modalities) {
                var modalitySettings = _.find(settings.modalities, function (m) {
                    return m.modality === modality;
                });

                if (modalitySettings && modalitySettings.displayMiddleSlice) {
                    instance = series.instances[Math.floor(series.instances.length / 2)];
                }
            }

            var uri;

            // Cache the thumbnail URI to avoid unnecessary patching when using
            // multiple image hosts.
            if (series.seriesAttributes.documentType) {
                uri = '';
            } else if (instance.instanceAttributes.thumbnailUri) {
                uri = instance.instanceAttributes.thumbnailUri;
            } else {
                instance.instanceAttributes.thumbnailUri = uri = Routes.ImageData(this.application.sessionId, instance.studyAttributes.studyStorage, instance.studyAttributes.queryObject, instance.id, instance.instanceAttributes.version, new Classes.FrameNumber(0), 0 /* Thumbnail */, 8, this.application.accountSettings.cache == 1, true);
            }

            var selectedSeries = this.application.selectedSeriesContainer.read();

            var layout = this.application.layout.read();
            var displayedSeries = _.take(this.application.seriesViews.read(), layout.rows * layout.columns);

            var selected = selectedSeries && selectedSeries.series && selectedSeries.series.seriesAttributes.seriesUid.value === series.seriesAttributes.seriesUid.value && selectedSeries.series.instances[0].id.value === series.instances[0].id.value;

            if (this.application.singleSeriesEnabled.read() && !selected && selectedSeries && selectedSeries.series && selectedSeries.series.parent) {
                selected = selectedSeries.series.parent.seriesAttributes.seriesUid.value === series.seriesAttributes.seriesUid.value;
            }

            var displayed = _.any(displayedSeries, function (container) {
                return container && container.series && container.series.seriesAttributes.seriesUid.value === series.seriesAttributes.seriesUid.value && container.series.instances[0].id.value === series.instances[0].id.value;
            });

            function formatDateTime(date, time, hide) {
                // Use MST timezone for imaged ORU series
                var timeZone = Study.isImagedORUSeries(series) ? "America/Phoenix" : null;

                if (!hide) {
                    // Combine fields and return formatted datetime
                    if (date && time) {
                        var datetime = new Date(date.getFullYear(), date.getMonth(), date.getDate(), time.getHours(), time.getMinutes(), time.getSeconds());

                        return datetime.toShortDateTimeString(dateFormat, timeZone);
                    }

                    if (date && !time) {
                        return date.toShortDateString(dateFormat);
                    }

                    if (!date && time) {
                        return time.toShortTimeString();
                    }
                }

                return '';
            }

            var dynamicThumbnail = false;
            var backgroungImageStyleDynamic;

            // make thumbnail appear with same orientation as series
            var seriesView = this.application.findSeriesView(series);
            if (!Browser.isIE9()) {
                var transform;
                if (seriesView && seriesView.canvas) {
                    transform = _.clone(seriesView.transform.read());
                } else if (series.displayOptions) {
                    transform = {
                        offsetX: 0, offsetY: 0, scale: 1,
                        flipped: series.displayOptions.flipped,
                        rotation: series.displayOptions.rotation };
                }

                if (transform) {
                    dynamicThumbnail = true;

                    backgroungImageStyleDynamic = {
                        'background-image': 'url(' + uri + ')'
                    };

                    transform.scale = 1.0; // ignore zoom
                    var DEFAULT_SIZE = 100;
                    var matrix = Transform.transformToInverseMatrix(transform, DEFAULT_SIZE, DEFAULT_SIZE, DEFAULT_SIZE, DEFAULT_SIZE);
                    backgroungImageStyleDynamic['transform'] = "matrix({0},{1},{2},{3},0,0)".replace("{0}", matrix.entries[0].toString()).replace("{1}", matrix.entries[1].toString()).replace("{2}", matrix.entries[3].toString()).replace("{3}", matrix.entries[4].toString());
                }
            }

            var seriesLoaded = this.application.seriesLoaded(series);
            var preloadAccelerated = this.application.supportsAcceleratedPreload();

            var loadedStatusStyle = "hide-icon";
            if (seriesLoaded == 3 /* Diagnostic */) {
                loadedStatusStyle = "loaded-hd-icon";
            } else if (seriesLoaded == 2 /* Thumbnail */) {
                loadedStatusStyle = "loaded-sd-icon";
            }

            var loadedStatusStyleAccelerated = preloadAccelerated && (seriesLoaded == 3 /* Diagnostic */) ? "loaded-hd-icon" : "";

            var hideThumbnailsDateTime = this.application.settings.read().hideThumbnailsDateTime || (this.application.accountSettings.viewer_hide_thumbnails_datetime != 0);

            return vdom.h('li.box', {
                onclick: function (e) {
                    var $thumbnail = $("#series-" + series.uuid);

                    if (_this.application.splitStudyEnabled.read()) {
                        // only allow splitting primary study
                        if (series.studyAttributes.uuid.value == _this.application.studies[0].studyAttributes.uuid.value) {
                            $thumbnail.toggleClass("splitting");
                            series.splitSelected = $thumbnail.hasClass("splitting");
                        }
                    } else if (_this.application.mergeSeriesEnabled.read()) {
                        // only allow merging primary study
                        if (series.studyAttributes.uuid.value == _this.application.studies[0].studyAttributes.uuid.value) {
                            $thumbnail.toggleClass("merging");
                            series.mergeSelected = $thumbnail.hasClass("merging");
                        }
                    } else {
                        _this.application.replaceSelectedSeries(series);
                    }

                    e.preventDefault();
                },
                ondragstart: function (e) {
                    if (e.dataTransfer) {
                        e.dataTransfer.setData("text", series.uuid);
                    }
                },
                foldIn: series.studyAttributes.queryObject.studyUid.value !== this.application.queryObject.studyUid.value ? new FoldInHook() : null
            }, [
                vdom.h('div', {
                    className: ['series-thumbnail-container'].concat(selected ? ['selected'] : []).concat(displayed ? ['displayed'] : []).join(' '),
                    draggable: true,
                    id: "series-" + series.uuid,
                    style: {
                        'background-image': dynamicThumbnail ? 'none' : 'url(' + uri + ')'
                    }
                }, [
                    dynamicThumbnail ? vdom.h('div', {
                        className: 'series-thumbnail-container-image',
                        style: backgroungImageStyleDynamic
                    }, []) : null,
                    vdom.h('span', { className: 'fa fa-3x thumb-icon ' + ((series.seriesAttributes.documentType) ? series.seriesAttributes.documentType.thumbnailIcon : '') }, []),
                    vdom.h('div.series-thumbnail-container-info', {}, (studyIndex > 0 && study.studyAttributes.studyCreateDate ? [
                        vdom.h('div', {}, [
                            study.studyAttributes.priorNumber > 0 ? this.application.terminology.lookup(Terminology.Terms.Prior) + " #" + study.studyAttributes.priorNumber : this.application.terminology.lookup(Terminology.Terms.Current)
                        ]),
                        vdom.h('div', {}, [hideThumbnailsDateTime ? "" : "(" + study.studyAttributes.studyCreateDate.toShortDateString(dateFormat) + ")"])
                    ] : []).concat([
                        vdom.h('div', {}, [first.instanceAttributes.seriesDescription]),
                        vdom.h('div', { className: 'series-info-normal' }, [formatDateTime(first.instanceAttributes.seriesCreateDate, first.instanceAttributes.seriesCreateTime, hideThumbnailsDateTime)]),
                        vdom.h('div', { className: 'series-info-small' }, [formatDateTime(first.instanceAttributes.seriesCreateDate, null, hideThumbnailsDateTime)]),
                        vdom.h('div', { className: 'series-info-small' }, [formatDateTime(null, first.instanceAttributes.seriesCreateTime, hideThumbnailsDateTime)]),
                        vdom.h('div', {}, [(!series.seriesAttributes.documentType) ? series.instances.length + ' ' + this.application.terminology.lookup(Terminology.Terms.Instances) : '']),
                        vdom.h('div.series-thumbnail-document-type', {}, [(series.seriesAttributes.documentType) ? series.seriesAttributes.documentType.thumbnailText : ''])
                    ])),
                    vdom.h('div', {}, (study.studyAttributes.accelerated || study.studyAttributes.studyStorage.localAccelerator ? [vdom.h('span', { className: 'fa fa-flash accelerated-icon ' + loadedStatusStyleAccelerated, title: 'This series is accelerated.' }, []), " "] : [vdom.h('span', { className: 'fa fa-check accelerated-icon load-icon ' + loadedStatusStyle, title: 'This series is loaded.' }, []), " "]))
                ])
            ], 'series-' + series.seriesAttributes.seriesUid.value);
        };

        Thumbnails.prototype.keyImageSeriesIcon = function (series) {
            var _this = this;
            var vdom = virtualDom;
            var term = function (t) {
                return _this.application.terminology.lookup(t);
            };
            var selectedSeries = this.application.selectedSeriesContainer.read();
            var selected = selectedSeries && selectedSeries.series && selectedSeries.series.seriesAttributes.seriesUid.value === series.seriesAttributes.seriesUid.value;

            if (this.application.singleSeriesEnabled.read() && !selected && selectedSeries && selectedSeries.series && selectedSeries.series.parent) {
                selected = selectedSeries.series.parent.seriesAttributes.seriesUid.value === series.seriesAttributes.seriesUid.value;
            }

            var instance = series.instances[0];
            var uri = '';
            if (instance.instanceAttributes.thumbnailUri) {
                uri = instance.instanceAttributes.thumbnailUri;
            } else {
                instance.instanceAttributes.thumbnailUri = uri = Routes.ImageData(this.application.sessionId, instance.studyAttributes.studyStorage, instance.studyAttributes.queryObject, instance.id, instance.instanceAttributes.version, new Classes.FrameNumber(0), 0 /* Thumbnail */, 8, this.application.accountSettings.cache == 1, true);
            }

            var icon = vdom.h('li.box', {
                onclick: function (e) {
                    _this.application.replaceSelectedSeries(series);
                    e.preventDefault();
                },
                ondragstart: function (e) {
                    if (e.dataTransfer) {
                        e.dataTransfer.setData("text", series.uuid);
                    }
                }
            }, [
                vdom.h('div', {
                    className: ['series-thumbnail-container'].concat(selected ? ['selected'] : []).join(' '),
                    draggable: true,
                    style: {
                        'background-image': 'url(' + uri + ')'
                    }
                }, [
                    vdom.h('span', { className: 'fa fa-3x fa-star thumb-icon' }, []),
                    vdom.h('div.series-thumbnail-container-info', {}, [
                        vdom.h('div', {}, [term(Terminology.Terms.KeyImageSeries)]),
                        vdom.h('div', {}, [series.instances.length + ' ' + term(Terminology.Terms.Instances)])
                    ])
                ])
            ], 'stack-series');

            return icon;
        };

        Thumbnails.prototype.stackSeriesIcon = function (study) {
            var _this = this;
            var vdom = virtualDom;
            var term = function (t) {
                return _this.application.terminology.lookup(t);
            };
            var selectedSeries = this.application.selectedSeriesContainer.read();
            var selected = selectedSeries && selectedSeries.series && study.stackSeries && selectedSeries.series.seriesAttributes.seriesUid.value === study.stackSeries.seriesAttributes.seriesUid.value;

            if (this.application.singleSeriesEnabled.read() && !selected && selectedSeries && selectedSeries.series && selectedSeries.series.parent) {
                selected = selectedSeries.series.parent.seriesAttributes.seriesUid.value === study.stackSeries.seriesAttributes.seriesUid.value;
            }

            var icon = vdom.h('li.box', {
                onclick: function (e) {
                    _this.application.replaceSelectedSeries(study.stackSeries);
                    e.preventDefault();
                }
            }, [
                vdom.h('div', {
                    className: ['series-thumbnail-container'].concat(selected ? ['selected'] : []).join(' ')
                }, [
                    vdom.h('span', { className: 'fa fa-3x fa-bars thumb-icon' }, []),
                    vdom.h('div.series-thumbnail-container-info', {}, [
                        vdom.h('div', {}, [study.studyAttributes.studyDescription]),
                        vdom.h('div', {}, [term(Terminology.Terms.StackSeries)])
                    ])
                ])
            ], 'stack-series');

            return icon;
        };

        Thumbnails.prototype.relatedStudy = function (prior, primary) {
            var _this = this;
            var enablePriors = this.application.settings.read().showPriors;
            var thin = prior.thin;

            if (LocalViewer.isLocalViewer() || (enablePriors !== undefined && enablePriors === false)) {
                return [];
            } else {
                if (!!this.application.settings.read().showOnlyPriors) {
                    var primaryStudyDate = primary.studyAttributes.studyCreateDate;
                    if (primaryStudyDate && prior.study_date_value) {
                        if (prior.study_date_value.getTime() > primaryStudyDate.getTime()) {
                            return [];
                        }
                    }
                }

                var vdom = virtualDom;
                var term = function (t) {
                    return _this.application.terminology.lookup(t);
                };
                var hideThumbnailsDateTime = this.application.settings.read().hideThumbnailsDateTime || (this.application.accountSettings.viewer_hide_thumbnails_datetime != 0);

                return [
                    vdom.h('li.box', {}, [
                        vdom.h('div.series-thumbnail-container', {
                            onclick: function (e) {
                                if (thin) {
                                    var app = _this.application;
                                    var sid = app.sessionId;

                                    var studyPoll = function () {
                                        Services.getStudyInfo(sid, new Classes.StudyUUID(prior.uuid)).subscribe({
                                            done: function () {
                                            },
                                            next: function (result) {
                                                if (result.thin == 1) {
                                                    setTimeout(studyPoll, 10000);
                                                } else {
                                                    app.loadPrior({
                                                        storage_namespace: result.storage_namespace,
                                                        phi_namespace: result.phi_namespace,
                                                        study_uid: result.study_uid,
                                                        prior_number: prior.prior_number
                                                    });
                                                }
                                            },
                                            fail: function (err) {
                                                Services.AuditLog("study poll call failed: " + err, "Thumbnails", sid);
                                            }
                                        });
                                    };

                                    Services.retrieveStudy(sid, new Classes.StudyUUID(prior.uuid)).subscribe({
                                        done: function () {
                                        },
                                        next: function (_) {
                                            $(e.target).find("span").removeClass("fa-arrow-circle-o-down").addClass("fa-gear fa-spin");
                                            setTimeout(studyPoll, 10000);
                                        },
                                        fail: function (err) {
                                            Services.AuditLog("study/retrieve call of prior failed: " + err, "Thumbnails", sid);
                                            window.alert(_this.application.terminology.lookup(Terminology.Terms.CannotRetrieveThinStudy));
                                        }
                                    });
                                } else {
                                    _this.application.loadPrior(prior);
                                }

                                e.preventDefault();
                            }
                        }, [
                            thin ? vdom.h('span', { className: 'fa fa-3x fa-arrow-circle-o-down thumb-icon' }, []) : vdom.h('span', { className: 'fa fa-3x fa-clock-o thumb-icon' }, []),
                            vdom.h('div.series-thumbnail-container-info', {}, [
                                vdom.h('div', {}, [prior.study_description]),
                                vdom.h('div', {}, [term(Terminology.Terms.AccessionNumberShort) + ": " + prior.accession_number]),
                                vdom.h('div', {}, [hideThumbnailsDateTime ? "" : term(Terminology.Terms.Date) + ": " + prior.study_date])
                            ])
                        ])
                    ], 'prior-' + prior.study_uid)
                ];
            }
        };

        Thumbnails.prototype.placementIcon = function () {
            var _this = this;
            var vdom = virtualDom;
            var $body = $('body');
            var left = $body.is(".thumbs-on-left");
            var top = $body.is(".thumbs-on-top");
            var bottom = !left && !top;
            var placementIcon;

            if (top) {
                placementIcon = "fa-caret-square-o-down";
            } else if (bottom) {
                placementIcon = "fa-caret-square-o-left";
            } else {
                placementIcon = "fa-caret-square-o-up";
            }

            return vdom.h('div.placement-button.glow', {}, [
                vdom.h('i', {
                    className: 'fa ' + placementIcon,
                    onclick: function (e) {
                        var small = _this.application.settings.read().smallThumbnails;
                        var $placementButton = $(".placement-button > i");

                        $body.removeClass("thumbs-small");
                        $placementButton.removeClass("fa-caret-square-o-down fa-caret-square-o-left fa-caret-square-o-up");

                        if ($body.is(".thumbs-on-left")) {
                            $body.toggleClass("thumbs-on-left thumbs-on-top"); // top

                            if (small) {
                                $body.addClass("thumbs-small");
                            }

                            $placementButton.addClass("fa-caret-square-o-down");
                        } else if ($body.is(".thumbs-on-top")) {
                            $body.toggleClass("thumbs-on-top"); // bottom

                            if (small) {
                                $body.addClass("thumbs-small");
                            }

                            $placementButton.addClass("fa-caret-square-o-left");
                        } else {
                            $body.toggleClass("thumbs-on-left"); // left
                            $placementButton.addClass("fa-caret-square-o-up");
                        }

                        _this.application.renderAllFrames();
                        e.preventDefault();
                    }
                }, [])
            ], 'placement-button');
        };

        /**
        * Create necessary elements, add them to the DOM and hook up event handlers.
        *
        * If the component has already been rendered, we use virtual-dom to apply the differences
        * to the existing node.
        */
        Thumbnails.prototype.render = function () {
            var priors = this.application.priors;
            var vnode = this.toVNode(this.application.studies, priors);

            if (this.rendered) {
                var patches = virtualDom.diff(this.rendered.vnode, vnode);

                virtualDom.patch(this.rendered.el, patches);

                this.rendered.vnode = vnode;
            } else {
                var initial = virtualDom.create(vnode);

                this.el.empty().append($(initial));

                this.rendered = {
                    el: initial,
                    vnode: vnode
                };
            }
        };
        return Thumbnails;
    })();
    Views.Thumbnails = Thumbnails;
})(Views || (Views = {}));
///<reference path='../typings/jquery/jquery.d.ts' />
///<reference path='../classes/Types.ts' />
///<reference path='../libs/Observable.ts' />
///<reference path="../libs/Queue.ts" />
///<reference path='../libs/Query.ts' />
///<reference path='../libs/Services.ts' />
///<reference path='../libs/Browser.ts' />
///<reference path='../libs/V3Storage.ts' />
///<reference path='../libs/Study.ts' />
///<reference path='../libs/Date.ts' />
///<reference path='../libs/Number.ts' />
///<reference path='../libs/Multiframe.ts' />
///<reference path='../models/StudySchema.ts' />
///<reference path='../libs/Subject.ts' />
///<reference path='../libs/GridLayout.ts' />
///<reference path='../models/Study.ts' />
///<reference path='Series.ts' />
///<reference path='Toolbar.ts' />
///<reference path='Thumbnails.ts' />
var Views;
(function (Views) {
    

    

    

    /**
    * The main application view
    *
    * Contains a grid of several series views
    *
    * Coordination between views takes place via subjects defined here
    */
    var Application = (function () {
        function Application(el, studyStorage, queryObject, sessionId, user, settings, accountSettings, permissions, terminology, keyImagesOnly, owner, shaders, hideStudyUUID, showStudyUUID) {
            var _this = this;
            this.studies = [];
            this.priors = [];
            this.stackedStudies = [];
            this.singleSeriesIndex = 0;
            /**
            * The series which is current magnified
            */
            this.magnifiedSeries = new Subjects.ObservableValue(null);
            /**
            * Cine frame rate
            */
            this.cineSpeed = new Subjects.ObservableValue(0);
            /**
            * True if playback should start automatically
            */
            this.startCinePlaybackAutomatically = new Subjects.ObservableValue(false);
            /**
            * True if the hanging protocol should override the frame time tag
            */
            this.ignoreFrameTimeTag = false;
            /**
            * Taken if keyboard shortcuts should be ignored
            */
            this.noKeyboardShortcuts = new Subjects.ObservableValue(Classes.Resource.unused());
            /**
            * Records viewer events
            */
            this.recorder = new Subjects.ObservableValue(new Recording.NullRecorder());
            /**
            * Indicates whether or not a script is being recorded
            */
            this.recordingMode = new Subjects.ObservableValue(false);
            /**
            * Indicates whether or not we are hosting a meeting
            */
            this.meetingHost = new Subjects.ObservableValue(false);
            /**
            * Indicates whether or not we are viewing a meeting
            */
            this.inMeeting = new Subjects.ObservableValue(false);
            /**
            * Subscriptions registered while render the UI
            * @type {any[]}
            */
            this.renderSubscriptions = [];
            /**
            * True if the recorder is recording
            */
            this.isRecording = Subjects.zip(this.recordingMode, this.meetingHost, function (b1, b2) {
                return b1 || b2;
            });
            /**
            * Indicates whether or not a script is being replayed
            */
            this.playbackMode = new Subjects.ObservableValue(false);
            // This object holds the status of the arrow keys (pressed or not)
            this.keyStatus = {};
            /**
            * True if the UI has been rendered
            * @type {boolean}
            */
            this.initializedLayout = false;
            /**
            * Toggle application keyboard listener, which is not focus-based.
            * @type {Subjects.ObservableValue<boolean>}
            */
            this.listenForKeyboardInput = new Subjects.ObservableValue(true);
            this.localAccelerator = false;
            this.splitStudyMode = false;
            this.mergeSeriesMode = false;
            this.keyImageLayout = [];
            /**
            * True if the GSPSInfo panel is visible
            */
            this.gspsInfoVisible = new Subjects.ObservableValue(false);
            this.highlightedGraphic = new Subjects.ObservableValue(null);
            this.el = el;
            this.studyStorage = studyStorage;
            this.queryObject = queryObject;
            this.sessionId = sessionId;
            this.user = user;
            this.settings = new Subjects.ObservableValue(settings);
            this.accountSettings = accountSettings;
            this.permissions = permissions;
            this.terminology = terminology;
            this.keyImagesOnly = keyImagesOnly;
            this.owner = owner;
            this.shaders = shaders;
            this.hideStudyUUID = hideStudyUUID;
            this.showStudyUUID = showStudyUUID;

            var viewer_local_accelerator = Query.findParameter(window.location, "viewer_local_accelerator");
            if (viewer_local_accelerator) {
                var local_acceleartor = parseInt(viewer_local_accelerator);
                if (local_acceleartor == 1) {
                    this.localAccelerator = true;
                    studyStorage.localAccelerator = true;
                    accountSettings.viewer_diagnostic_quality = 1;
                    accountSettings.viewer_diagnostic_quality_always = 1;
                }
            }

            this.layout = new Subjects.ObservableValue({ rows: 1, columns: 1 });
            this.currentColor = new Subjects.ObservableValue(0);
            this.selectedTool = new MultiMonitor.SharedSubject("selectedTool", owner, new Subjects.ObservableValue(this.getDefaultMouseTool(settings, accountSettings)));

            this.selectedTool2 = new MultiMonitor.SharedSubject("selectedTool2", owner, new Subjects.ObservableValue(this.getDefaultMouseTool2(settings, accountSettings)));

            this.selectedToolWheel = new MultiMonitor.SharedSubject("selectedToolWheel", owner, new Subjects.ObservableValue(this.getDefaultMouseToolWheel(settings, accountSettings)));

            this.globalMeasurementsToggle = settings.toggleAllSeriesMeasurements;
            this.globalMeasurementsVisible = new MultiMonitor.SharedSubject("globalMeasurementsVisible", owner, new Subjects.ObservableValue(true));

            this.globalTextToggle = settings.toggleAllSeriesText;
            this.globalTextVisible = new MultiMonitor.SharedSubject("globalTextVisible", owner, new Subjects.ObservableValue(true));

            this.globalRulerToggle = settings.toggleAllSeriesRuler;
            this.globalRulerVisible = new MultiMonitor.SharedSubject("globalRulerVisible", owner, new Subjects.ObservableValue(true));

            Subjects.listen(this.currentColor, function (color) {
                _.each(_this.seriesViews.value, function (series) {
                    if (series.series) {
                        series.updateAreaColorSelection(color);
                    }
                });

                _this.renderAllFrames();
            });

            Subjects.listen(this.layout, function (layout) {
                _this.recorder.read().append({
                    type: 4 /* LayoutChanged */,
                    newLayout: layout
                });

                if (_this.singleSeriesEnabled.read() && _this.seriesViews && _this.seriesViews.value.length) {
                    if (_this.canDisplaySingleSeries(layout)) {
                        if (_this.singleSeriesSingleLayout) {
                            _this.singleSeries = _this.seriesViews.value[0].series;
                            _this.seriesViews.value[0].series = _this.singleSeriesSingleLayout;
                            _this.seriesViews.value[0].selectedInstanceIndex.write(_this.singleSeriesIndex);
                            _this.selectedSeriesKey.write(_this.seriesViews.value[0].viewKey);
                            _this.seriesViews.value[0].renderAll();
                        }
                    } else {
                        if (_this.singleSeries) {
                            _this.singleSeriesSingleLayout = _this.seriesViews.value[0].series;
                            _this.seriesViews.value[0].series = _this.singleSeries;
                            _this.seriesViews.value[0].selectedInstanceIndex.write(0);
                            _this.selectedSeriesKey.write(_this.seriesViews.value[0].viewKey);
                            _this.seriesViews.value[0].renderAll();
                        }
                    }

                    _this.seriesViews.raiseChangedEvent(_this.seriesViews.value);
                }

                _this.updateSeriesHandles((layout.rows * layout.columns) > 1);
                _this.thumbnails.render();
            });

            Subjects.listen(this.globalMeasurementsVisible, function (show) {
                _.each(_this.seriesViews.value, function (series) {
                    series.measurementsVisible.write(show);
                });
            });

            Subjects.listen(this.globalRulerVisible, function (show) {
                _.each(_this.seriesViews.value, function (series) {
                    series.rulerVisible.write(show);
                });
            });

            Subjects.listen(this.globalTextVisible, function (show) {
                _.each(_this.seriesViews.value, function (series) {
                    series.infoVisible.write(show);
                });
            });

            Subjects.listen(this.selectedTool, function (tool) {
                _this.recorder.read().append({
                    type: 3 /* MouseToolChanged */,
                    newMouseTool: tool
                });
            });

            Subjects.listen(this.settings, function (settings) {
                Services.putSettings(_this.sessionId, settings).subscribe({
                    done: function () {
                    },
                    next: function (_) {
                    },
                    fail: function (err) {
                        _this.recordError("Unable to save settings to services: " + err);
                    }
                });
            });

            Subjects.listen(this.playbackMode, function (playing) {
                _this.playbackChanged(playing);
            });

            this.seriesViews = new Subjects.ObservableValue([]);

            this.seriesViewsSubscription = this.seriesViews.updates.subscribe({
                done: function () {
                },
                fail: function (_) {
                },
                next: function (_) {
                    _this.thumbnails.render();
                }
            });

            /* Shared subjects */
            this.selectedSeriesKey = new MultiMonitor.SharedSubject("selectedSeries", owner, new Subjects.ObservableValue(Views.SeriesViewKey.Null));
            this.selectedSeriesContainer = Subjects.map(this.selectedSeriesKey, function (key) {
                return _.find(_this.seriesViews.value, function (sc) {
                    return sc.viewKey.value === key.value;
                }) || null;
            });

            this.selectedSeriesContainer.updates.subscribe({
                done: function () {
                },
                fail: function (_) {
                },
                next: function (_) {
                    _this.thumbnails.render();
                }
            });

            this.selectedInstanceGeometry = new MultiMonitor.SharedSubject("selectedInstanceGeometry", owner, new Subjects.ObservableValue(null));

            this.clipboard = new MultiMonitor.SharedSubject("clipboard", owner, new Subjects.ObservableValue(null));

            this.thumbnailsVisible = new MultiMonitor.SharedSubject("thumbnailsVisible", owner, new Subjects.ObservableValue(true));

            this.keyImageSeriesEnabled = new MultiMonitor.SharedSubject("keyImageSeriesEnabled", owner, new Subjects.ObservableValue(false));

            this.splitStudyEnabled = new Subjects.ObservableValue(false);
            this.mergeSeriesEnabled = new Subjects.ObservableValue(false);

            this.singleSeriesEnabled = new Subjects.ObservableValue(false);

            this.planeLocalizationCursor = new MultiMonitor.SharedSubject("planeLocalizationCursor", owner, new Subjects.ObservableValue(null));

            this.referenceLinesActive = new MultiMonitor.SharedSubject("referenceLinesActive", owner, new Subjects.ObservableValue(true));

            this.linkSeries = new MultiMonitor.SharedSubject("linkSeries", owner, new Subjects.ObservableValue(accountSettings.viewer_link_series == 1));

            this.propagateMode = new MultiMonitor.SharedSubject("propagateMode", owner, new Subjects.ObservableValue(false));

            Subjects.listen(this.selectedInstance(), function (value) {
                if (value && SeriesGeometry.hasGeometricMetadata(value.instance)) {
                    var instance = value.instance;

                    var instancePlane = SeriesGeometry.normalPlaneForInstance(instance.instanceAttributes);

                    var instanceGeometry = {
                        originator: value.container.viewKey.value,
                        seriesUid: instance.seriesAttributes.seriesUid,
                        studyUid: instance.studyAttributes.queryObject.studyUid,
                        imagePositionPatient: instance.instanceAttributes.imagePositionPatient,
                        imageOrientationPatient: instance.instanceAttributes.imageOrientationPatient,
                        pixelSpacing: instance.instanceAttributes.pixelSpacing,
                        rows: instance.instanceAttributes.rows,
                        columns: instance.instanceAttributes.columns,
                        normal: Vectors.toArray(instancePlane.n)
                    };

                    _this.selectedInstanceGeometry.write(instanceGeometry);
                } else {
                    _this.selectedInstanceGeometry.write(null);
                }

                if (_this.gspsInfoVisible.read()) {
                    _this.gspsInfo.render();
                }
            });

            this.imagePreloadQueue = new Queue.ObservablePriorityQueue();

            Observable.queue(this.imagePreloadQueue, 16, new Views.PreloadQueueKeyIsKey()).subscribe({
                next: function (_) {
                },
                done: function () {
                    throw new Error("The impossible happened in Observable.queue");
                },
                fail: function (_) {
                    throw new Error("The impossible happened in Observable.queue");
                }
            });

            this.imagePreloadStore = {};

            $(document).on("keydown", function (e) {
                if (!_this.noKeyboardShortcuts.read().used()) {
                    _this.handleKeyboardShortcut(e);
                }
            }).on("keyup", function (e) {
                if (!_this.noKeyboardShortcuts.read().used()) {
                    _this.handleKeyupEvent(e);
                }
            });

            if (Browser.isMobile()) {
                var message = this.terminology.lookup(Terminology.Terms.NotForDiagnosticUse);
                $('<div>').addClass('banner').append(message).appendTo($(document.body));
                $('.applicationFrame').css({ top: '6px' });
            }
        }
        /**
        * Load a study, and subscribe for updates to annotations and key images using websockets
        */
        Application.prototype.loadStudyAndSubscribeForUpdates = function (studyStorage, query) {
            var _this = this;
            return Observable.bind2(Study.loadStudy(this.sessionId, studyStorage, query, this.keyImagesOnly, this.permissions, this.settings.read(), this.accountSettings), function (study) {
                return Observable.take(1, Observable.catchError(Observable.timeout(_this.subscribeToStudyUpdates(study), 1000), function () {
                    return {};
                }));
            }, function (study, _) {
                return study;
            });
        };

        Application.prototype.loadPriorAndSubscribeForUpdates = function (studyStorage, query) {
            var _this = this;
            return Observable.bind2(Study.loadStudy(this.sessionId, studyStorage, query, this.keyImagesOnly, this.permissions, this.settings.read(), null), function (study) {
                return Observable.take(1, Observable.catchError(Observable.timeout(_this.subscribeToStudyUpdates(study), 1000), function () {
                    return {};
                }));
            }, function (study, _) {
                return study;
            });
        };

        /**
        * Modify a study-loading computation to also load annotations
        */
        Application.prototype.withAnnotations = function (query, loadStudy) {
            var _this = this;
            var loadAnnotations = this.permissions.annotation_view !== 0 ? Observable.catchError(Observable.timeout(Services.listImageAnnotations(this.sessionId, query), 2000), function () {
                return { annotations: [] };
            }) : Observable.ret({ annotations: [] });

            return Observable.map(Observable.invoke(Observable.zip(loadStudy, loadAnnotations, function (study, annotations) {
                return { study: study, annotations: annotations };
            }), function (output) {
                Annotations.applyAll(output.annotations, output.study, _this.user, _this.settings.read());
            }), function (output) {
                return output.study;
            });
        };

        /**
        * Modify a study-loading computation to also load GSPS data
        */
        Application.prototype.withGSPSData = function (loadStudy, f) {
            var _this = this;
            return Observable.bind(loadStudy, function (t) {
                var study = f(t);
                return Observable.map(Study.loadAndApplyAllGSPSData(_this.sessionId, study, _this.settings.read().supportsSRAnnotations), function (_) {
                    return study;
                });
            });
        };

        /**
        * Initialize the view
        *
        * Loads study and metadata from storage, and calls the render method
        */
        Application.prototype.load = function () {
            var _this = this;
            var loadStudy = Observable.bind2(this.loadStudyAndSubscribeForUpdates(this.studyStorage, this.queryObject), function (study) {
                if (LocalViewer.isLocalViewer()) {
                    // Local viewer returns a mock of the getStudyList call
                    return Observable.ret([LocalViewer.makePrior(study)]);
                } else if (_this.showStudyUUID.length > 0) {
                    return Services.getStudyListByUuid(_this.sessionId, _this.showStudyUUID);
                } else if (study.studyAttributes.patientId == null || (study.studyAttributes.patientId && study.studyAttributes.patientId.value != null && study.studyAttributes.patientId.value.length == 0)) {
                    // If we don't have an MRN / Patient Id we cannot find priors so just return null
                    return Observable.ret(null);
                } else {
                    var worklistFilter;

                    if (_this.accountSettings.viewer_show_priors_worklist_only == 1) {
                        worklistFilter = _this.getWorklistFilter();
                    }

                    // Attempt to query services for a list of related studies (priors), filtering out this study from that list
                    if (worklistFilter) {
                        return Services.getStudyListFiltered(_this.sessionId, _this.queryObject.phiNamespace, study.studyAttributes.patientId, null, null, _this.hideStudyUUID, worklistFilter);
                    } else {
                        return Services.getStudyList(_this.sessionId, _this.queryObject.phiNamespace, study.studyAttributes.patientId, null, null, _this.hideStudyUUID);
                    }
                }
            }, function (study, priors) {
                return {
                    study: study,
                    priors: priors
                };
            });

            return Observable.invoke(this.withAnnotations(this.queryObject, this.withGSPSData(Observable.invoke(loadStudy, function (output) {
                output.study.originalSeries = output.study.series;
                output.study.series = _.flatten(_.map(output.study.series, function (series) {
                    return Multiframe.split(series, _this.accountSettings);
                }));

                _this.setPriorNumbers(output.study, _.filter(output.priors, function (prior) {
                    return prior.study_uid !== output.study.studyAttributes.queryObject.studyUid.value;
                }));
                _this.studies.push(output.study);
                _this.priors = output.priors;

                var settings = _this.settings.read();

                // The stack series is kept separate to make it easy to find and work with later (ie in thumbnails)
                if (_this.isStackSeriesEnabled()) {
                    output.study.stackSeries = Study.createStackSeries(output.study);
                }

                // Key Images
                var priorsStack;

                _this.keyImageSeriesEnabled.write(settings.hangKeyImageSeriesFirst);
                _this.singleSeriesEnabled.write(settings.layoutSeries);
                _this.showingKeyImagesOnly = _this.keyImageSeriesEnabled.read();

                Subjects.listen(_this.singleSeriesEnabled, function (enabled) {
                    var seriesView = _this.selectedSeriesContainer.read();
                    var layout = _this.layout.read();
                    var layoutCount = layout.columns * layout.rows;
                    _this.resetSeriesViews();

                    if (enabled && seriesView) {
                        if (seriesView.series.instances.length <= layoutCount) {
                            _this.singleSeriesIndex = 0;
                        } else {
                            _this.singleSeriesIndex = seriesView.selectedInstanceIndex.read();
                        }

                        _this.render(seriesView.series);
                        _this.selectedSeriesKey.write(_this.seriesViews.value[0].viewKey);
                    } else {
                        var series;

                        if (_this.singleSeries) {
                            series = _this.singleSeries;
                        }

                        _this.singleSeries = null;
                        _this.render();

                        for (var ctr = 0; ctr < _this.seriesViews.value.length; ctr++) {
                            if (_this.seriesViews.value[ctr].series == series) {
                                _this.selectedSeriesKey.write(_this.seriesViews.value[ctr].viewKey);
                                _this.seriesViews.value[ctr].selectedInstanceIndex.write(_this.singleSeriesIndex);
                                break;
                            }
                        }
                    }

                    _this.seriesViews.raiseChangedEvent(_this.seriesViews.value);
                });

                if (!_this.keyImagesOnly && _this.keyImageSeriesEnabled.read()) {
                    if (_this.settings.read().splitKeyImageSeries) {
                        var keyImageSeries = Study.createKeyImageSeries(output.study, true);
                        if (keyImageSeries.length && keyImageSeries[0].instances.length) {
                            output.study.keyImageSeries = Study.splitSeries(keyImageSeries[0]);
                        } else {
                            output.study.keyImageSeries = [];
                        }
                    } else {
                        var stackKeyImageSeries = _this.settings.read().stackKeyImageSeries;
                        output.study.keyImageSeries = Study.createKeyImageSeries(output.study, stackKeyImageSeries);

                        if (stackKeyImageSeries && _this.settings.read().stackKeyImageSeriesPriors) {
                            priorsStack = output.study.keyImageSeries[0];
                        }
                    }
                }

                if (settings.expandPriors) {
                    var showOnlyPriors = !!settings.showOnlyPriors;
                    var thins = [];
                    _.each(_this.priors, function (prior) {
                        if (_this.studies[0].studyAttributes.queryObject.studyUid.value != prior.study_uid) {
                            var usePrior = true;

                            if (showOnlyPriors) {
                                var primaryStudyDate = _this.studies[0].studyAttributes.studyCreateDate;
                                if (primaryStudyDate && prior.study_date_value) {
                                    if (prior.study_date_value.getTime() > primaryStudyDate.getTime()) {
                                        usePrior = false;
                                    }
                                }
                            }

                            if (usePrior) {
                                if (prior.thin) {
                                    thins.push(prior);
                                } else {
                                    _this.loadPrior(prior, priorsStack);
                                }
                            }
                        }
                    });

                    _this.priors = thins;
                }

                var isReloadable = parseInt(Query.findParameter(window.location, "reloadable") || "0");
                var needsReport = parseInt(Query.findParameter(window.location, "openreport") || "0");
                if (isReloadable) {
                    if (needsReport) {
                        isReloadable = 2;
                    }

                    var app = _this;
                    function reloadListener(evt) {
                        var data = JSON.parse(evt.data);
                        window.location.href = "/viewer/#study/" + data.storageNs + "/" + data.studyUID + "/" + data.phiNs + "?sid=" + app.sessionId.value + "&reloadable=" + isReloadable;
                        window.location.reload();
                    }
                    window.addEventListener("message", reloadListener, false);
                    if (isReloadable === 2) {
                        window.localStorage.setItem("reportEvent", JSON.stringify({
                            studyId: app.studies[0].studyAttributes.uuid.value
                        }));
                    }
                }

                if (needsReport) {
                    _this.loadReport(_this.studies[0].studyAttributes.uuid, true);
                }
            }), function (output) {
                return output.study;
            })), function (_) {
                _this.render();
            });
        };

        Application.prototype.localAcceleratorCached = function (storage_namespace, study_uid, study) {
            if (this.isWorklistAccelerated()) {
                return {
                    subscribe: function (ob) {
                        var img = new Image();

                        img.onload = function () {
                            study.studyAttributes.studyStorage.localAccelerator = (this.width == Routes.LocalAccelerator.LOCAL_ACCELERATOR_TRUE);
                            ob.next(study);
                            ob.done();
                        };

                        img.onerror = function () {
                            study.studyAttributes.studyStorage.localAccelerator = false;
                            ob.next(study);
                            ob.done();
                        };

                        var data = {
                            namespace: storage_namespace,
                            studyUID: study_uid
                        };

                        var params = $.param({ data: JSON.stringify(data) });

                        img.src = Routes.LocalAccelerator.LOCAL_ACCELERATOR_PROTOCOL + Routes.LocalAccelerator.LOCAL_ACCELERATOR_HOSTNAME + ":" + Routes.LocalAccelerator.LOCAL_ACCELERATOR_PORT + "/cached" + "?" + params + "&a=" + +(new Date());

                        return Observable.memptySubscription;
                    }
                };
            } else {
                return Observable.ret(study);
            }
        };

        Application.prototype.checkAcceleration = function (storage_namespace, study_uid, withGSPSData, f) {
            var _this = this;
            return Observable.bind(withGSPSData, function (t) {
                var study = f(t);
                return Observable.invoke(_this.localAcceleratorCached(storage_namespace, study_uid, study), function (_) {
                    return study;
                });
            });
        };

        /**
        * Load a prior and make its series available
        */
        Application.prototype.loadPrior = function (prior, stack) {
            var _this = this;
            var priorQuery = new Classes.QueryObject(new Classes.StorageNamespace(prior.storage_namespace), new Classes.PhiNamespace(prior.phi_namespace), new Classes.StudyUid(prior.study_uid));

            var loadPrior = Observable.bind(Services.getStudyStorageInfo(this.sessionId, priorQuery), function (studyStorage) {
                return _this.loadPriorAndSubscribeForUpdates(studyStorage, priorQuery);
            });

            $('.overlay').addClass('application-loading');

            var loadAll = Observable._finally(this.withAnnotations(priorQuery, Observable.invoke(this.checkAcceleration(prior.storage_namespace, prior.study_uid, this.withGSPSData(loadPrior, function (study) {
                return study;
            }), function (study) {
                return study;
            }), function (study) {
                study.studyAttributes.isPrimary = false;
                study.studyAttributes.priorNumber = prior.prior_number;
                study.originalSeries = study.series;
                study.series = _.flatten(_.map(study.series, function (series) {
                    return Multiframe.split(series, _this.accountSettings);
                }));

                // The stack series is kept separate to make it easy to find and work with later (ie in thumbnails)
                if (_this.isStackSeriesEnabled()) {
                    study.stackSeries = Study.createStackSeries(study);
                }

                if (!_this.keyImagesOnly && _this.keyImageSeriesEnabled.read()) {
                    if (_this.settings.read().splitKeyImageSeries) {
                        var keyImageSeries = Study.createKeyImageSeries(study, true);
                        if (keyImageSeries.length && keyImageSeries[0].instances.length) {
                            study.keyImageSeries = Study.splitSeries(keyImageSeries[0]);
                        } else {
                            study.keyImageSeries = [];
                        }
                    } else {
                        var stackkeyimageSeries = _this.settings.read().stackKeyImageSeries;
                        study.keyImageSeries = Study.createKeyImageSeries(study, stackkeyimageSeries);
                    }
                }

                if (stack) {
                    stack.instances = _.uniq(_.union(study.keyImageSeries[0].instances, stack.instances), false, function (instance) {
                        return instance.id.value;
                    });
                    _this.stackedStudies.push(study);
                } else {
                    _this.studies.push(study);
                }

                _this.thumbnails.render();
            })), function () {
                $('.overlay').removeClass('application-loading');
            });

            loadAll.subscribe({
                next: function (_) {
                    if (_this.settings.read().expandPriors) {
                        _this.resetSeriesViews();
                        _this.render();
                        _this.selectedSeriesKey.write(_this.seriesViews.value[0].viewKey);
                    }
                },
                done: function () {
                },
                fail: function (err) {
                    _this.recordError("Unable to load prior study: " + err);
                }
            });
        };

        /**
        * Should the stack series be available
        */
        Application.prototype.isStackSeriesEnabled = function () {
            var _this = this;
            var settings = this.settings.read();

            var stackSeriesVisible = false;

            // Look for modality setting based on first series in first study
            if (settings.modalities && this.studies != null && this.studies[0] != null && this.studies[0].series[0] != null && this.studies[0].series[0].seriesAttributes != null) {
                var modalitySettings = _.find(settings.modalities, function (m) {
                    return m.modality === _this.studies[0].series[0].seriesAttributes.modality;
                });

                if (modalitySettings && modalitySettings.stackSeriesVisible) {
                    stackSeriesVisible = modalitySettings.stackSeriesVisible;
                }
            }

            // Default legacy setting, overrides modality specific settings, should only be set at a modality level going forward
            if (settings.showStackSeries) {
                stackSeriesVisible = true;
            }

            return stackSeriesVisible;
        };

        /**
        * Should the stack series be shown when the page loads
        */
        Application.prototype.isStackSeriesDisplayedFirst = function () {
            var _this = this;
            var settings = this.settings.read();

            var stackSeriesStartSelected = false;

            // Look for modality setting based on first series in first study
            if (settings.modalities && this.studies != null && this.studies[0] != null && this.studies[0].series[0] != null && this.studies[0].series[0].seriesAttributes != null) {
                var modalitySettings = _.find(settings.modalities, function (m) {
                    return m.modality === _this.studies[0].series[0].seriesAttributes.modality;
                });

                if (modalitySettings && modalitySettings.stackSeriesStartSelected) {
                    stackSeriesStartSelected = modalitySettings.stackSeriesStartSelected;
                }
            }

            return stackSeriesStartSelected;
        };

        /**
        * Get the default mouse tool based on settings
        */
        Application.prototype.getDefaultMouseTool = function (settings, accountSettings) {
            if (settings.toolSelection != null) {
                return settings.toolSelection;
            }

            if (accountSettings.viewer_default_mouse_tool != null) {
                return Classes.MouseTool[accountSettings.viewer_default_mouse_tool];
            }

            return 1 /* Scroll */;
        };

        /**
        * Get the default mouse tool based on settings
        */
        Application.prototype.getDefaultMouseTool2 = function (settings, accountSettings) {
            if (settings.toolSelection2 != null) {
                return settings.toolSelection2;
            }

            if (accountSettings.viewer_default_mouse_tool2 != null) {
                return Classes.MouseTool[accountSettings.viewer_default_mouse_tool2];
            }

            return 0 /* Move */;
        };

        /**
        * Get the default mouse tool based on settings
        */
        Application.prototype.getDefaultMouseToolWheel = function (settings, accountSettings) {
            if (settings.toolSelectionWheel != null) {
                return settings.toolSelectionWheel;
            }

            if (accountSettings.viewer_default_mouse_tool_wheel != null) {
                return Classes.MouseTool[accountSettings.viewer_default_mouse_tool_wheel];
            }

            return 0 /* Move */;
        };

        /**
        * Subscribe to study updates for the specified study
        */
        Application.prototype.subscribeToStudyUpdates = function (study) {
            var _this = this;
            var request = Messaging.studyChannel(this.sessionId, study.studyAttributes.queryObject);

            return Observable.forget(Observable.invoke(Messaging.channel(request), function (channel) {
                channel.subscribe({
                    done: function () {
                    },
                    next: function (message) {
                        _this.handleStudyChannelMessage(study, message);
                    },
                    fail: function (err) {
                        _this.recordError("Unable to subscribe for study updates: " + err);
                    }
                });
            }));
        };

        /**
        * Handle an asynchronous study message
        */
        Application.prototype.handleStudyChannelMessage = function (study, message) {
            if (message.sid_md5 !== this.user.sid_md5) {
                if (/^study\.annotation/.test(message.channel) || /^study\.annotation/.test(message.primary_channel)) {
                    this.handleAnnotationEvent(study, message);
                } else if (/^study\.keyimage/.test(message.channel) || /^study\.keyimage/.test(message.primary_channel)) {
                    this.handleKeyImageEvent(study, message);
                } else if (/^study\./.test(message.channel)) {
                    switch (message.event) {
                        case "NEW_MEETING":
                            this.addMeeting(study, message.data);
                            break;
                    }
                }
            }
        };

        /**
        * Handle an annotation event
        */
        Application.prototype.handleAnnotationEvent = function (study, message) {
            var _this = this;
            if (message.event === "DELETE") {
                Annotations.merge(study, message.event, { uuid: message.id }, this.user);
                this.renderAllFrames();
            } else {
                var loadAnnotation = Services.getImageAnnotation(this.sessionId, this.queryObject, new Classes.AnnotationId(message.id));

                loadAnnotation.subscribe({
                    done: function () {
                    },
                    next: function (ann) {
                        Annotations.merge(study, message.event, ann, _this.user);
                        _this.renderAllFrames();
                    },
                    fail: function (err) {
                        _this.recordError("Unable to retrieve annotation data: " + err);
                    }
                });
            }
        };

        /**
        * Handle a key image event
        */
        Application.prototype.handleKeyImageEvent = function (study, message) {
            var _this = this;
            if (message.event === "DELETE") {
                _.each(study.series, function (series) {
                    _.each(series.instances, function (instance) {
                        if (instance.instanceAttributes.keyImageId && instance.instanceAttributes.isKeyImage.read() && instance.instanceAttributes.keyImageId.value === message.id) {
                            instance.instanceAttributes.isKeyImage.write(false);
                            instance.instanceAttributes.keyImageId = null;
                        }
                    });
                });
            } else {
                var loadKeyImage = Services.getKeyImage(this.sessionId, new Classes.KeyImageId(message.id));

                loadKeyImage.subscribe({
                    done: function () {
                    },
                    next: function (keyImage) {
                        var series = _.find(study.series, function (s) {
                            return s.seriesAttributes.seriesUid.value === keyImage.series_uid;
                        });

                        if (series) {
                            var instance = _.find(series.instances, function (i) {
                                return i.id.value === keyImage.instance_uid;
                            });

                            if (instance) {
                                instance.instanceAttributes.isKeyImage.write(true);
                                instance.instanceAttributes.keyImageId = new Classes.KeyImageId(keyImage.uuid);
                            }
                        }
                    },
                    fail: function (err) {
                        _this.recordError("Unable to load key image data: " + err);
                    }
                });
            }
        };

        /**
        * Add a meeting to the thumbnails area
        */
        Application.prototype.addMeeting = function (study, meeting) {
            var newMeeting = new Classes.StudyMeeting();

            newMeeting.id = new Classes.MeetingId(meeting.uuid);
            newMeeting.name = meeting.name;
            newMeeting.host = meeting.user_name;

            study.meetings.push(newMeeting);

            this.thumbnails.render();
        };

        /**
        * Set the priorNumber field on prior study records related to their relative age
        */
        Application.prototype.setPriorNumbers = function (study, priors) {
            var _this = this;
            _.each(priors, function (prior) {
                prior.study_date_value = Parsers.parseDicomDate(prior.study_date);
                if (prior.study_date_value) {
                    var dateFormat = _this.settings.read().dateFormat;
                    prior.study_date = prior.study_date_value.toShortDateString(dateFormat);
                }
            });

            var currentStudyTime = study.studyAttributes.studyCreateDate ? study.studyAttributes.studyCreateDate.getTime() : 0;

            var sortedPriors = _.sortBy(priors, function (prior) {
                return prior.study_date_value ? -prior.study_date_value.getTime() : 0;
            });

            _.each(sortedPriors, function (prior, index, list) {
                var priorDate = prior.study_date_value;
                prior.prior_number = !priorDate || currentStudyTime > priorDate.getTime() ? index + 1 : index;
            });

            study.studyAttributes.isPrimary = true;
            study.studyAttributes.priorNumber = _.filter(priors, function (prior) {
                var priorDate = prior.study_date_value;
                return priorDate ? priorDate.getTime() > currentStudyTime : false;
            }).length;
        };

        Application.prototype.resetSeriesViews = function () {
            var _this = this;
            _.each(this.seriesViews.read(), function (s) {
                s.unload();
            });

            if (this.seriesViewsSubscription) {
                this.seriesViewsSubscription.cancel();
                this.seriesViewsSubscription = null;
            }

            this.seriesViews = new Subjects.ObservableValue([]);

            this.seriesViewsSubscription = this.seriesViews.updates.subscribe({
                done: function () {
                },
                fail: function (_) {
                },
                next: function (_) {
                    _this.thumbnails.render();
                }
            });
        };

        /**
        * Layout the main application grid. This method is only called when the collection of available series changes.
        */
        Application.prototype.render = function (seriesToLoad) {
            var _this = this;
            // clear listeners
            _.each(this.renderSubscriptions, function (sub) {
                return sub.cancel();
            });
            this.renderSubscriptions = [];

            // build toolbar UI
            var $toolbar;
            if (this.toolbar) {
                $toolbar = this.toolbar.el;
            } else {
                $toolbar = $('<div>').addClass('toolbar');
                this.toolbar = new Views.Toolbar($toolbar, this);
            }

            this.toolbar.render();

            // build application UI
            this.$applicationFrame = $('<div>').addClass('applicationFrame');
            this.$applicationFrame.append($toolbar);
            this.$applicationFrameInner = $('<div>').addClass('applicationFrame-inner').appendTo(this.$applicationFrame);

            // build thumbnails UI
            if (!this.thumbnails) {
                var $thumbnails = $('<div>').addClass('thumbs');
                this.thumbnails = new Views.Thumbnails($thumbnails, this);
                this.$applicationFrameInner.append($thumbnails);
            } else {
                this.$applicationFrameInner.append(this.thumbnails.el);
            }

            this.thumbnails.render();

            var $applicationFrameInner2 = $('<div>').addClass('applicationFrame-inner-2').appendTo(this.$applicationFrameInner);

            // build side panel UI
            var $meeting = $('<div>').addClass('meeting');
            this.meetingInfo = new Views.MeetingInfo($meeting, this.sessionId, this.permissions, this.terminology, this.meetingHost, function (meetingId) {
                return _this.endMeeting(meetingId);
            }, function (meetingId) {
                return _this.leaveMeeting(meetingId);
            }, function (meetingId, userUuid) {
                return _this.changeMeetingPresenter(meetingId, userUuid);
            }, this.listenForKeyboardInput);
            this.recordingsInfo = new Views.RecordingsInfo($meeting, this.sessionId, this.permissions, this.terminology, this.listenForKeyboardInput);
            this.attachmentInfo = new Views.AttachmentInfo($meeting, this.sessionId, this.permissions, this.terminology, this.listenForKeyboardInput);
            this.gspsInfo = new Views.GSPSInfo($meeting, this.sessionId, this.terminology, this);
            $applicationFrameInner2.append($meeting);

            // configure application instance
            var viewerSettings = this.settings.read();
            _.each(this.studies, function (study) {
                study.series = Study.filterSeries(study.series, viewerSettings);
            });

            this.renderSubscriptions.push(Subjects.listen(Subjects.zip(this.thumbnailsVisible, this.inMeeting, function (b1, b2) {
                return b1 && !b2;
            }), function (visible) {
                _this.updateThumbnailsVisibility(visible);
                _this.renderAllFrames();
            }));

            this.renderSubscriptions.push(Subjects.listen(Subjects.zip(this.inMeeting, this.meetingHost, function (b1, b2) {
                return b1 || b2;
            }), function (visible) {
                _this.updateMeetingInfoVisibility(visible);
                _this.renderAllFrames();
            }));

            this.renderSubscriptions.push(Subjects.listen(this.inMeeting, function (inMeeting) {
                if (inMeeting) {
                    _this.showMeetingGlassPane();
                } else {
                    _this.hideMeetingGlassPane();
                }
            }));

            // Hide thumbnails if necessary
            var hideThumbnails = (this.accountSettings.viewer_hide_thumbnails == 1) || this.settings.read().hideThumbnails;
            this.thumbnailsVisible.write(!hideThumbnails);

            this.updateThumbnailsVisibility(this.thumbnailsVisible.read());

            this.renderSubscriptions.push(Subjects.listen(this.splitStudyEnabled, function (enabled) {
                if (enabled) {
                    _this.splitStudyMode = true;
                    _this.showWarningBanner(_this.terminology.lookup(Terminology.Terms.SplitInstructions));
                } else {
                    if (_this.splitStudyMode) {
                        _this.hideWarningBanner();
                        _this.splitStudyMode = false;
                        var splitting = _.map(_.filter(_this.studies[0].series, function (series) {
                            return series.splitSelected;
                        }), function (series) {
                            return series.seriesAttributes.seriesUid.value;
                        });
                        _.each(_this.studies[0].series, function (series) {
                            series.splitSelected = false;
                        });
                        $(".series-thumbnail-container").removeClass("splitting");
                        _this.splitStudy(_this.studies[0], splitting);
                    }
                }
            }));

            this.renderSubscriptions.push(Subjects.listen(this.mergeSeriesEnabled, function (enabled) {
                if (enabled) {
                    _this.mergeSeriesMode = true;
                    _this.showWarningBanner(_this.terminology.lookup(Terminology.Terms.MergeInstructions));
                } else {
                    if (_this.mergeSeriesMode) {
                        _this.hideWarningBanner();
                        _this.mergeSeriesMode = false;
                        var merging = _.filter(_this.studies[0].series, function (series) {
                            return series.mergeSelected;
                        });
                        _.each(_this.studies[0].series, function (series) {
                            series.mergeSelected = false;
                        });
                        $(".series-thumbnail-container").removeClass("merging");
                        _this.mergeSeries(_this.studies[0], merging);
                    }
                }
            }));

            this.renderSubscriptions.push(Subjects.listen(this.keyImageSeriesEnabled, function (enabled) {
                var studies = _this.studies;
                var stackedStudies = _this.stackedStudies;

                // it's possible for touch/click events to trigger this multiple times very rapidly leading to a bad state
                if (_this.showingKeyImagesOnly != enabled) {
                    if (enabled) {
                        var stackKeyImageSeries = _this.settings.read().stackKeyImageSeries;
                        var splitKeyImageSeries = _this.settings.read().splitKeyImageSeries;
                        var stackKeyImageSeriesPriors = _this.settings.read().stackKeyImageSeriesPriors;

                        _.each(_this.studies, function (study) {
                            if (splitKeyImageSeries) {
                                var keyImageSeries = Study.createKeyImageSeries(study, true);
                                if (keyImageSeries.length && keyImageSeries[0].instances.length) {
                                    study.keyImageSeries = Study.splitSeries(keyImageSeries[0]);
                                } else {
                                    study.keyImageSeries = [];
                                }
                            } else {
                                study.keyImageSeries = Study.createKeyImageSeries(study, stackKeyImageSeries);
                            }
                        });

                        if (stackKeyImageSeriesPriors && stackKeyImageSeries && !splitKeyImageSeries) {
                            var stack = _this.studies[0].keyImageSeries[0];

                            _.each(_this.studies, function (study, index) {
                                if (index > 0) {
                                    study.keyImageSeries = Study.createKeyImageSeries(study, stackKeyImageSeries);
                                    stack.instances = _.uniq(_.union(study.keyImageSeries[0].instances, stack.instances), false, function (instance) {
                                        return instance.id.value;
                                    });
                                    stackedStudies.push(study);
                                }
                            });

                            _.each(_this.stackedStudies, function (study) {
                                studies = _.without(studies, study);
                            });

                            _this.studies = studies;
                            _this.priors = [];
                        }
                    } else {
                        _.each(_this.stackedStudies, function (study) {
                            studies.push(study);
                        });

                        _this.stackedStudies = [];
                    }

                    _this.resetSeriesViews();
                    _this.render();
                    _this.selectedSeriesKey.write(_this.seriesViews.value[0].viewKey);
                    _this.thumbnails.render();
                }

                _this.showingKeyImagesOnly = enabled;
            }));

            var allSeries = _.toArray(_.flatten(_.map(this.studies, function (study) {
                return study.series;
            })));

            // If the key images series is displayed do not display the regular series
            if (this.keyImageSeriesEnabled.read()) {
                var allKeyImageSeries = _.toArray(_.flatten(_.map(this.studies, function (study) {
                    return study.keyImageSeries;
                })));

                if (allKeyImageSeries && allKeyImageSeries.length && allKeyImageSeries[0]) {
                    allSeries = allKeyImageSeries;

                    var keyImageOrder = viewerSettings.keyImageOrder;

                    if (keyImageOrder) {
                        allSeries = Study.orderSeries(allSeries, keyImageOrder == "desc");
                    }
                }
            }

            // Hang the stack series in the first available screen if requested to in modality settings
            if (this.isStackSeriesDisplayedFirst() && this.studies[0].stackSeries) {
                allSeries = [this.studies[0].stackSeries].concat(allSeries);
            }

            var $allSeries = $('<div>').addClass('study-series');
            var firstLayout = false;

            if (_.any(allSeries)) {
                var screenIndex = parseInt(Query.findParameter(window.location, "screen") || "0");
                var hangingProtocol;

                if (this.singleSeriesEnabled.read()) {
                    this.singleSeries = seriesToLoad || allSeries[0];
                    allSeries = Study.splitSeries(this.singleSeries).slice(this.singleSeriesIndex);

                    if (this.initializedLayout) {
                        var layout = this.layout.read();
                        hangingProtocol = HangingProtocols.selectDefault(this.studies[0], this.settings.read(), { rows: layout.rows, columns: layout.columns });
                    } else {
                        firstLayout = true;
                        hangingProtocol = HangingProtocols.selectDefault(this.studies[0], this.settings.read());
                    }

                    this.hang(allSeries, $allSeries, hangingProtocol, screenIndex);
                } else {
                    hangingProtocol = HangingProtocols.select(this.studies[0], this.settings.read());
                    this.hang(allSeries, $allSeries, hangingProtocol, screenIndex);

                    if (screenIndex == 0) {
                        var isChildWindow = parseInt(Query.findParameter(window.location, "isChildWindow") || "0");

                        if (isChildWindow == 0) {
                            if (window.self == window.top) {
                                if (this.accountSettings.fullscreen) {
                                    this.fullscreenOnDisplay(0, hangingProtocol.screens.length);
                                } else if (!Browser.uiFound()) {
                                    this.needsDirectFullscreen(function (fullscreen) {
                                        if (fullscreen) {
                                            _this.fullscreenOnDisplay(0, hangingProtocol.screens.length, null, true);
                                        }
                                    });
                                }

                                window.addEventListener("storage", function (e) {
                                    try  {
                                        if (e.key === "viewerReloadEvent") {
                                            var data = JSON.parse(e.newValue);
                                            window.location.href = data.url;
                                        }
                                    } catch (ex) {
                                    }
                                });
                            }

                            for (var screen = 1; screen < hangingProtocol.screens.length; screen++) {
                                this.openNewWindow(screen, hangingProtocol.screens.length);
                            }
                        }
                    }
                }
            } else {
                $allSeries.append(this.terminology.lookup(Terminology.Terms.NoImageSeries));
            }

            if (firstLayout && this.singleSeriesEnabled.read() && !this.canDisplaySingleSeries(this.layout.read())) {
                this.layout.write({ columns: 1, rows: 1 });
            }

            $applicationFrameInner2.append($allSeries);

            this.renderSubscriptions.push(Subjects.listen(this.linkSeries, function (link) {
                if (link) {
                    _this.loadVisibleImageAttributes(_this.imagePreloadQueue, -1 /* ImageAttributeBackgroundPreloading */);
                }
            }));

            this.renderSubscriptions.push(Subjects.listen(this.propagateMode, function (enabled) {
                if (enabled) {
                    var series = _this.selectedSeriesContainer.read();
                    if (series) {
                        series.startPropagation();
                        _this.showWarningBanner(_this.terminology.lookup(Terminology.Terms.PropagateInstructions));
                    }
                } else {
                    _this.hideWarningBanner();
                }
            }));

            this.renderSubscriptions.push(Subjects.listen(Subjects.zip(this.selectedTool, this.layout, function (tool, _) {
                return tool;
            }), function (tool) {
                if (tool === 9 /* Localization */) {
                    _this.loadVisibleImageAttributes(_this.imagePreloadQueue, -1 /* ImageAttributeBackgroundPreloading */);
                    if (Browser.isChrome()) {
                        _this.loadVisibleThumbnails(_this.imagePreloadQueue, -2 /* ImageDataBackgroundPreloading */);
                    }
                }
            }));

            if (this.selectedTool.read() === 9 /* Localization */ || LocalViewer.isLocalViewer()) {
                this.loadVisibleImageAttributes(this.imagePreloadQueue, -1 /* ImageAttributeBackgroundPreloading */);
            }

            this.renderSubscriptions.push(Subjects.listen(this.recordingMode, function (recording) {
                if (recording) {
                    _this.startRecording();
                }
            }));

            // build context menu UI
            var contextMenu = (function () {
                var settings = _this.settings.read();

                if (settings.contextMenu !== undefined) {
                    return settings.contextMenu;
                } else {
                    return ContextMenus.createDefaultContextMenu();
                }
            })();

            Views.ContextMenu.apply(".series", contextMenu, this);

            // draw application
            this.el.empty();
            this.el.append(this.$applicationFrame);

            if (!!this.studyStorage.is_frozen) {
                this.showWarningBanner(this.terminology.lookup(Terminology.Terms.FrozenStudyWarning));
            }

            if (this.keyImageSeriesEnabled.read()) {
                this.layoutKeyImages();
            }
        };

        Application.prototype.layoutKeyImages = function () {
            var app = this;

            if (this.hangKeyImageHandlerId) {
                clearTimeout(this.hangKeyImageHandlerId);
            }

            this.hangKeyImageHandlerId = setTimeout(function () {
                Services.getKeyImageSettings(app.sessionId).subscribe({
                    done: function () {
                    },
                    next: function (res) {
                        if (res) {
                            app.keyImageLayout = res;
                            var keyImageLayout = _.find(res, function (k) {
                                return app.studies[0].studyAttributes.patientId.value == k.patientId;
                            });
                            if (keyImageLayout) {
                                app.applyKeyImageLayoutState(keyImageLayout);
                            }
                        }
                    },
                    fail: function (err) {
                        console.error("Unable to get key image settings: " + err);
                    }
                });
            }, 500);
        };

        /**
        * Hang series as described by a single screen of a hanging protocol object
        */
        Application.prototype.hang = function (allSeries, $allSeries, hangingProtocol, screenIndex) {
            var _this = this;
            var screen = hangingProtocol.screens[screenIndex];
            this.layout.write({ rows: screen.rows, columns: screen.cols });

            if (hangingProtocol.cineOptions) {
                this.cineSpeed.write(hangingProtocol.cineOptions.defaultCineFrameRate);
                this.ignoreFrameTimeTag = hangingProtocol.cineOptions.ignoreFrameTimeTag || false;
                this.startCinePlaybackAutomatically.write(hangingProtocol.cineOptions.startCinePlaybackAutomatically);
            }

            var protocolSeries = HangingProtocols.apply(allSeries, hangingProtocol);
            var screenSeries = protocolSeries.matches[screenIndex];
            var layoutOriginal = this.layout.read();
            var maxViews = Math.max(layoutOriginal.rows * layoutOriginal.columns, 16);

            for (var i = 0; i < maxViews; i++) {
                (function (i) {
                    var series = null;
                    var displayOptions = null;

                    if (i < screenSeries.length) {
                        var match = screenSeries[i];
                        if (match) {
                            series = match.series;
                            displayOptions = match.displayOptions;
                            series.displayOptions = match.displayOptions;
                        }
                    } else if (i - screenSeries.length < protocolSeries.rest.length) {
                        series = protocolSeries.rest[i - screenSeries.length];
                    }

                    var $series = $('<div>').addClass('series');

                    var seriesView = new Views.Series(_this, _this.sessionId, $series, series, _this.settings, _this.accountSettings, _this.selectedSeriesKey, _this.selectedTool, _this.selectedTool2, _this.selectedToolWheel, _this.selectedInstanceGeometry, _this.referenceLinesActive, _this.planeLocalizationCursor, _this.linkSeries, _this.imagePreloadQueue, _this.terminology, _this.cineSpeed, _this.ignoreFrameTimeTag, _this.startCinePlaybackAutomatically, _this.noKeyboardShortcuts, Subjects.map(_this.recorder, function (r) {
                        return _this.modifyRecorder(r, i);
                    }), _this.playbackMode);

                    seriesView.applyDisplayOptions(displayOptions);

                    // if locally accelerated but not currently displayed, preload the series (queues storage calls)
                    if (!displayOptions && _this.supportsAcceleratedPreload()) {
                        if (series != null && series.seriesAttributes != null && series.seriesAttributes.seriesUid != null && series.seriesAttributes.seriesUid.value != null) {
                            seriesView.init(true);
                        }
                    }

                    _this.seriesViews.value.push(seriesView);

                    if (series != null && series.seriesAttributes != null && series.seriesAttributes.seriesUid != null && series.seriesAttributes.seriesUid.value != null) {
                        if (_this.imagePreloadStore[series.seriesAttributes.seriesUid.value] == null) {
                            _this.imagePreloadStore[series.seriesAttributes.seriesUid.value] = new Views.ByImageTypeAndInstanceNumber();
                        }
                    }

                    if (series != null && series.seriesAttributes != null && Multiframe.shouldSplitInstances(series.seriesAttributes.modality)) {
                        seriesView.loadImageAttributes(_this.imagePreloadQueue, -1 /* ImageAttributeBackgroundPreloading */);
                    }
                })(i);
            }

            this.thumbnails.render();

            var seriesElements = Subjects.zip(this.seriesViews, this.magnifiedSeries, function (ss, mag) {
                return mag === null ? _.map(ss, function (s) {
                    return s.el;
                }) : [mag.el];
            });

            var layout = Subjects.zip(this.layout, this.magnifiedSeries, function (layout, series) {
                return series === null ? layout : { rows: 1, columns: 1 };
            });

            var startingSeriesUid = Query.findParameter(window.location, "showSeriesUID");

            // Initial series specified
            if (this.selectedSeriesKey.read() === Views.SeriesViewKey.Null && startingSeriesUid != null) {
                var series = this.seriesViews.read();
                for (var i = 0; i < allSeries.length; i++) {
                    if (series[i].series && series[i].series.seriesAttributes && series[i].series.seriesAttributes.seriesUid && series[i].series.seriesAttributes.seriesUid.value == startingSeriesUid) {
                        this.loadSeriesAt(series[i].series, 0);
                        this.seriesViews.raiseChangedEvent(series);
                        this.selectedSeriesKey.write(series[i].viewKey);
                        break;
                    }
                }
            }

            // Either no starting series selected or starting series not found
            if (this.selectedSeriesKey.read() === Views.SeriesViewKey.Null) {
                var firstSeries = _.find(this.seriesViews.read(), function (s) {
                    return s.series != null;
                });
                if (firstSeries) {
                    this.selectedSeriesKey.write(firstSeries.viewKey);
                } else {
                    this.selectedSeriesKey.write(_.first(this.seriesViews.read()).viewKey);
                }
            }

            $allSeries.grid(layout, Subjects.ret(0), seriesElements, function () {
                _.delay(function () {
                    return _this.renderAllFrames();
                }, 10);
            });

            this.initializedLayout = true;
        };

        Application.prototype.supportsAcceleratedPreload = function () {
            return this.settings.read().preloadAccelerated && this.studyStorage.localAccelerator && this.isDiagnosticQualityAlways();
        };

        /**
        * Create a recorder which inserts the correct value for seriesContainerIndex
        */
        Application.prototype.modifyRecorder = function (recorder, index) {
            return {
                append: function (e) {
                    e.seriesContainerIndex = index;
                    recorder.append(e);
                },
                getEvents: recorder.getEvents,
                setZeroTimestamp: recorder.setZeroTimestamp
            };
        };

        /**
        * Load image attributes for all visible series to enable linking
        */
        Application.prototype.loadVisibleImageAttributes = function (queue, priority) {
            var _this = this;
            var layout = this.layout.read();

            _.each(_.range(0, Math.min(this.seriesViews.value.length, layout.rows * layout.columns)), function (index) {
                _this.seriesViews.value[index].loadImageAttributes(queue, priority);
            });
        };

        /**
        * Start loading thumbnails for all visible series
        */
        Application.prototype.loadVisibleThumbnails = function (queue, priority) {
            var _this = this;
            var layout = this.layout.read();

            _.each(_.range(0, Math.min(this.seriesViews.value.length, layout.rows * layout.columns)), function (index) {
                _this.seriesViews.value[index].loadAllImageData(queue, priority);
            });
        };

        Application.prototype.handleKeyupEvent = function (e) {
            switch (e.keyCode) {
                case 38:
                    if (this.keyStatus.up !== undefined) {
                        window.clearInterval(this.keyStatus.up);
                        delete this.keyStatus.up;
                    }
                    if (this.keyStatus.upWait !== undefined) {
                        window.clearTimeout(this.keyStatus.upWait);
                        delete this.keyStatus.upWait;
                    }
                    break;
                case 40:
                    if (this.keyStatus.down !== undefined) {
                        window.clearInterval(this.keyStatus.down);
                        delete this.keyStatus.down;
                    }
                    if (this.keyStatus.downWait !== undefined) {
                        window.clearTimeout(this.keyStatus.downWait);
                        delete this.keyStatus.downWait;
                    }
                    break;
            }
        };

        /**
        * Handle a key-press event
        */
        Application.prototype.handleKeyboardShortcut = function (e) {
            var _this = this;
            if (!this.listenForKeyboardInput.read()) {
                return;
            }

            var series = this.selectedSeriesContainer.read();

            if (series != null) {
                var handled = true;

                switch (e.keyCode) {
                    case 8:
                    case 46:
                        if ((this.permissions.annotation_edit !== 0) && series.hasSelectedMeasurement()) {
                            series.deleteSelectedMeasurement();
                        } else if (this.keyImageSeriesEnabled.read()) {
                            this.clearKeyImageLayoutState();
                        }
                        break;
                    case 33:
                        series.scrollBy(-10);
                        break;
                    case 34:
                        series.scrollBy(10);
                        break;
                    case 38:
                        series.scrollBy(-1);
                        if (this.keyStatus.upWait === undefined) {
                            this.keyStatus.upWait = window.setTimeout(function () {
                                delete _this.keyStatus.upWait;
                                if (_this.keyStatus.up === undefined) {
                                    _this.keyStatus.up = window.setInterval(function () {
                                        series.scrollBy(-1);
                                    }, 10);
                                }
                            }, 250);
                        }
                        break;
                    case 40:
                        series.scrollBy(1);
                        if (this.keyStatus.downWait === undefined) {
                            this.keyStatus.downWait = window.setTimeout(function () {
                                delete _this.keyStatus.downWait;
                                if (_this.keyStatus.down === undefined) {
                                    _this.keyStatus.down = window.setInterval(function () {
                                        series.scrollBy(1);
                                    }, 10);
                                }
                            }, 250);
                        }
                        break;
                    case 49:
                    case 50:
                    case 51:
                    case 52:
                    case 53:
                    case 54:
                    case 55:
                    case 56:
                    case 57:
                        series.applyWindowLevelPreset(e.keyCode - 49);
                        break;
                    case 97:
                    case 98:
                    case 99:
                    case 100:
                    case 101:
                    case 102:
                    case 103:
                    case 104:
                    case 105:
                        series.applyWindowLevelPreset(e.keyCode - 97);
                        break;
                    default:
                        var settings = this.settings.read();

                        handled = (function () {
                            var bindings = settings.keyBindings || Keyboard.createDefaultKeyConfiguration();

                            if (bindings && bindings.items) {
                                var key = _.find(bindings.items, function (item) {
                                    return item.keyCode === e.keyCode && !!item.ctrl == e.ctrlKey && !!item.alt == e.altKey && !!item.meta == e.metaKey && !!item.shift == e.shiftKey;
                                });

                                if (key) {
                                    _this.invokeToolbarAction(key.action);
                                    return true;
                                }

                                switch (e.keyCode) {
                                    case 37:
                                        if (_this.gspsInfoVisible.read()) {
                                            _this.gspsInfo.previousGSPS();
                                        }
                                        break;
                                    case 39:
                                        if (_this.gspsInfoVisible.read()) {
                                            _this.gspsInfo.nextGSPS();
                                        }
                                        break;
                                }
                            }

                            return false;
                        })();
                }

                if (handled) {
                    e.preventDefault();
                }
            }
        };

        /**
        * Invoke a toolbar action manually
        */
        Application.prototype.invokeToolbarAction = function (action) {
            Views.AbstractToolbars.addStandardToolbarButton({
                addLabel: function (_) {
                    return {};
                },
                addToolbarButton: function (opts) {
                    opts.click();
                    return {};
                },
                addToggleButton: function (opts) {
                    Subjects.modify(opts.selected, function (b) {
                        return !b;
                    });
                    return {};
                },
                addDropDownButton: function (_1, _2) {
                    return {};
                },
                addLayoutButtons: function (_) {
                    return [];
                },
                addColorButtons: function (_) {
                    return [];
                },
                addMouseToolSettings: function (_) {
                    return [];
                }
            }, action, false, this);
        };

        /**
        * Synchronize the visibility of the thumbnails with the current setting
        */
        Application.prototype.updateThumbnailsVisibility = function (visible) {
            if (visible) {
                this.$applicationFrameInner.addClass('withthumbs');
            } else {
                this.$applicationFrameInner.removeClass('withthumbs');
            }
        };

        /**
        * Synchronize the visibility of the meetings panel with the current setting
        */
        Application.prototype.updateMeetingInfoVisibility = function (visible) {
            if (visible) {
                this.$applicationFrameInner.addClass('withmeeting');
            } else {
                this.$applicationFrameInner.removeClass('withmeeting');
            }
        };

        Application.prototype.updateRecordingsInfoVisibility = function (visible) {
            if (visible) {
                this.$applicationFrameInner.addClass('withmeeting withrecording');
            } else {
                this.$applicationFrameInner.removeClass('withmeeting withrecording');
            }
        };

        Application.prototype.updateAttachmentInfoVisibility = function (visible) {
            if (visible) {
                this.$applicationFrameInner.addClass('withmeeting withattachments');
            } else {
                this.$applicationFrameInner.removeClass('withmeeting withattachments');
            }
        };

        Application.prototype.updateGSPSInfoVisibility = function (visible) {
            this.gspsInfoVisible.write(visible);

            if (visible) {
                this.$applicationFrameInner.addClass('withmeeting withgsps');
            } else {
                this.$applicationFrameInner.removeClass('withmeeting withgsps');
            }
        };

        /**
        * Render all visible series
        */
        Application.prototype.renderAllFrames = function () {
            for (var i = 0; i < this.seriesViews.value.length; i++) {
                var series = this.seriesViews.value[i];
                series.renderAll();
            }
        };

        /**
        * A subject which returns the model of the first visible instance in the selected series
        */
        Application.prototype.selectedInstance = function () {
            return Subjects.bind(this.selectedSeriesContainer, function (series) {
                return series === null ? Subjects.ret(null) : series.selectedInstanceIndex;
            }, function (series, index) {
                if (!series || !series.series) {
                    return null;
                } else {
                    var instance = series.series.instances[index % series.series.instances.length];
                    return { instance: instance, container: series };
                }
            });
        };

        /**
        *  Replace the selected series with the previous series in study.series
        */
        Application.prototype.loadPrevSeries = function (startAtLastInstance) {
            if (typeof startAtLastInstance === "undefined") { startAtLastInstance = false; }
            var selectedSeriesContainer = this.selectedSeriesContainer.read();

            if (selectedSeriesContainer) {
                var currentUID = selectedSeriesContainer.series.seriesAttributes.seriesUid;
                var currentIndex = null;
                var nextSeries = null;

                var currentStudy = _.filter(this.studies, function (study) {
                    return study.studyAttributes.uuid.value == selectedSeriesContainer.series.studyAttributes.uuid.value;
                });
                var allSeries = _.toArray(_.flatten(_.map(currentStudy, function (study) {
                    return study.series;
                })));

                for (var i = 0; i < allSeries.length; i++) {
                    if (allSeries[i].seriesAttributes.seriesUid.value == currentUID.value) {
                        currentIndex = i;
                        break;
                    }
                }

                if (currentIndex != null) {
                    // Loop to the last series if we are viewing the first series, otherwise load the previous series
                    nextSeries = (currentIndex == 0) ? allSeries[allSeries.length - 1] : allSeries[currentIndex - 1];
                }

                if (nextSeries != null) {
                    this.replaceSelectedSeries(nextSeries);

                    var loadedSeries = this.selectedSeriesContainer.read();
                    loadedSeries.init();
                    loadedSeries.arrow('prev-series');

                    if (startAtLastInstance) {
                        loadedSeries.scrollBy(loadedSeries.series.instances.length - 1);
                    }
                }
            }
        };

        /**
        *  Replace the selected series with the next series in study.series
        */
        Application.prototype.loadNextSeries = function () {
            var selectedSeriesContainer = this.selectedSeriesContainer.read();

            if (selectedSeriesContainer) {
                var currentUID = selectedSeriesContainer.series.seriesAttributes.seriesUid;
                var currentIndex = null;
                var nextSeries = null;

                var currentStudy = _.filter(this.studies, function (study) {
                    return study.studyAttributes.uuid.value == selectedSeriesContainer.series.studyAttributes.uuid.value;
                });
                var allSeries = _.toArray(_.flatten(_.map(currentStudy, function (study) {
                    return study.series;
                })));

                for (var i = 0; i < allSeries.length; i++) {
                    if (allSeries[i].seriesAttributes.seriesUid.value == currentUID.value) {
                        currentIndex = i;
                        break;
                    }
                }

                if (currentIndex != null) {
                    // Loop to the first series if we are viewing the last series, otherwise load the next series
                    nextSeries = (currentIndex == allSeries.length - 1) ? allSeries[0] : allSeries[currentIndex + 1];
                }

                if (nextSeries != null) {
                    this.replaceSelectedSeries(nextSeries);

                    var loadedSeries = this.selectedSeriesContainer.read();
                    loadedSeries.init();
                    loadedSeries.arrow('next-series');
                }
            }
        };

        /**
        * Replace the selected series with the provided series
        */
        Application.prototype.replaceSelectedSeries = function (series) {
            var views = this.seriesViews.value;
            var selectedSeriesContainer = this.selectedSeriesContainer.read();

            if (!selectedSeriesContainer) {
                this.selectedSeriesKey.write(this.seriesViews.read()[0].viewKey);
                selectedSeriesContainer = this.selectedSeriesContainer.read();
            }

            if (selectedSeriesContainer) {
                if (this.isMagnified()) {
                    this.magnifyMinify(selectedSeriesContainer);
                }

                selectedSeriesContainer.unload();

                for (var i = 0; i < views.length; i++) {
                    if (views[i].viewKey.value === selectedSeriesContainer.viewKey.value) {
                        if (this.singleSeriesEnabled.read()) {
                            this.resetSeriesViews();

                            if (this.singleSeries != series) {
                                this.singleSeriesIndex = 0;
                            }

                            this.render(series);
                            this.selectedSeriesKey.write(this.seriesViews.read()[0].viewKey);
                        } else {
                            var seriesView = this.loadSeriesAt(series, i);
                            this.selectedSeriesKey.write(seriesView.viewKey);
                        }

                        this.seriesViews.raiseChangedEvent(views);

                        break;
                    }
                }

                if (this.selectedTool.read() === 9 /* Localization */) {
                    this.loadVisibleImageAttributes(this.imagePreloadQueue, -1 /* ImageAttributeBackgroundPreloading */);
                }
            }
        };

        /**
        * Load a series at a particular location
        */
        Application.prototype.loadSeriesAt = function (series, index) {
            var _this = this;
            var $series = $('<div>').addClass('series');

            var seriesView = new Views.Series(this, this.sessionId, $series, series, this.settings, this.accountSettings, this.selectedSeriesKey, this.selectedTool, this.selectedTool2, this.selectedToolWheel, this.selectedInstanceGeometry, this.referenceLinesActive, this.planeLocalizationCursor, this.linkSeries, this.imagePreloadQueue, this.terminology, this.cineSpeed, this.ignoreFrameTimeTag, this.startCinePlaybackAutomatically, this.noKeyboardShortcuts, Subjects.map(this.recorder, function (r) {
                return _this.modifyRecorder(r, index);
            }), this.playbackMode);

            this.seriesViews.value[index] = seriesView;
            seriesView.init();
            if (series) {
                seriesView.applyDisplayOptions(series.displayOptions);
            } else {
                seriesView.applyDisplayOptions(null);
            }

            this.recorder.read().append({
                type: 5 /* SeriesChanged */,
                seriesContainerIndex: index,
                seriesInstanceUid: series === null || typeof series == "undefined" ? null : series.seriesAttributes.seriesUid
            });

            return seriesView;
        };

        /**
        * Add a new series
        */
        Application.prototype.addSeries = function (series) {
            var study = _.find(this.studies, function (study) {
                return study.studyAttributes.queryObject.studyUid === series.studyAttributes.queryObject.studyUid;
            });

            if (!study) {
                debugger;
            }

            study.series.push(series);

            this.render();
        };

        /**
        * Open the print view in a new window
        */
        Application.prototype.print = function () {
            var study = this.studies[0];
            var id = study.studyAttributes.queryObject.toString();
            var sid = this.sessionId.value;
            window.open("/viewer/print.html#print/{0}?sid={1}".replace("{0}", encodeURIComponent(id)).replace("{1}", encodeURIComponent(sid)));
        };

        /**
        * Open the metadata editor
        */
        Application.prototype.editMetadata = function () {
            var selectedSeriesContainer = this.selectedSeriesContainer.read();

            if (selectedSeriesContainer && selectedSeriesContainer.series) {
                var id = selectedSeriesContainer.series.studyAttributes.queryObject;
                var selectedInstanceIndex = selectedSeriesContainer.selectedInstanceIndex.read();

                var instance = selectedSeriesContainer.series.instances[selectedInstanceIndex];

                window.open("/studies/metadata.html?namespace_id={0}&study_uid={1}&phi_namespace={2}&image_uid={3}&version={4}".replace("{0}", encodeURIComponent(id.storageNamespace.value)).replace("{1}", encodeURIComponent(id.studyUid.value)).replace("{2}", encodeURIComponent(id.phiNamespace.value)).replace("{3}", encodeURIComponent(instance.id.value)).replace("{4}", encodeURIComponent(instance.instanceAttributes.version.value)));
            }
        };

        /**
        * Open the MPR view in a new window
        */
        Application.prototype.mpr = function () {
            var study = this.studies[0];
            var id = study.studyAttributes.queryObject.toString();
            var sid = this.sessionId.value;
            var series = this.selectedSeriesContainer.read();

            if (series) {
                var seriesUid = series.series.seriesAttributes.seriesUid;

                window.open("/viewer/mpr.html#mpr/{0}?sid={1}&seriesUid={2}".replace("{0}", encodeURIComponent(id)).replace("{1}", encodeURIComponent(sid)).replace("{2}", encodeURIComponent(seriesUid.value)));
            }
        };

        /**
        * Open the viewer in diagnostic quality mode
        */
        Application.prototype.useDiagosticQualityViewer = function () {
            var study = this.studies[0];
            var id = study.studyAttributes.queryObject.toString();
            var sid = this.sessionId.value;
            window.open("/viewer/#study/{0}?sid={1}&viewer_diagnostic_quality=1&viewer_diagnostic_quality_always=1".replace("{0}", encodeURIComponent(id)).replace("{1}", encodeURIComponent(sid)));
        };

        /**
        * Open the recorder in a new window
        */
        Application.prototype.recordAudio = function () {
            var study = this.studies[0];
            var id = study.studyAttributes.queryObject;
            var sid = this.sessionId.value;
            window.open("/studies/recorder.html?namespace_id={0}&study_uid={1}&phi_namespace={2}".replace("{0}", encodeURIComponent(id.storageNamespace.value)).replace("{1}", encodeURIComponent(id.studyUid.value)).replace("{2}", encodeURIComponent(id.phiNamespace.value)));
        };

        /**
        * Open the settings editor in a new window
        */
        Application.prototype.editSettings = function () {
            window.open("/viewer/settings/#?sid=" + encodeURIComponent(this.sessionId.value));
        };

        Application.prototype.getWorklistData = function (byMRN) {
            if (window.sessionStorage) {
                var data = window.sessionStorage.getItem(byMRN ? "worklist:mrn-data" : "worklist:data");
                if (data) {
                    var unstringify = JSON.parse(data);
                    if (unstringify) {
                        return JSON.parse(unstringify);
                    }
                }
            }

            return null;
        };

        Application.prototype.getWorklistFilter = function () {
            var worklistFilter = null;

            if (window.sessionStorage) {
                var data = window.sessionStorage.getItem("worklist:currentSearch");
                if (data) {
                    var unstringify = JSON.parse(data);
                    if (unstringify) {
                        worklistFilter = _.pairs(JSON.parse(unstringify));
                    }
                }
            }

            return worklistFilter;
        };

        Application.prototype.getCurrentWorklistIndex = function (worklist, byMRN) {
            var currentIndex = -1;
            for (var ctr = 0; ctr < worklist.length; ctr += 1) {
                if ((this.queryObject.studyUid.value == worklist[ctr].studyUID) || (byMRN && (this.studyStorage.patientid == worklist[ctr].patientId))) {
                    currentIndex = ctr;
                    break;
                }
            }

            return currentIndex;
        };

        Application.prototype.hasNextWorklistStudy = function (byMRN) {
            var worklist = this.getWorklistData(byMRN);
            if (worklist) {
                var currentIndex = this.getCurrentWorklistIndex(worklist, byMRN);

                if (currentIndex != -1) {
                    return currentIndex < (worklist.length - 1);
                }
            }

            return false;
        };

        Application.prototype.hasPreviousWorklistStudy = function (byMRN) {
            var worklist = this.getWorklistData(byMRN);
            if (worklist) {
                var currentIndex = this.getCurrentWorklistIndex(worklist, byMRN);

                if (currentIndex != -1) {
                    return currentIndex > 0;
                }
            }

            return false;
        };

        Application.prototype.nextWorklistStudy = function (byMRN) {
            this.changeWorklistStudy(false, byMRN);
        };

        Application.prototype.previousWorklistStudy = function (byMRN) {
            this.changeWorklistStudy(true, byMRN);
        };

        Application.prototype.isWorklistAccelerated = function () {
            if (window.sessionStorage) {
                var data = window.sessionStorage.getItem("worklist:acceleration");
                if (data) {
                    var unstringify = JSON.parse(data);
                    if (unstringify) {
                        return JSON.parse(unstringify);
                    }
                }
            }

            return false;
        };

        Application.prototype.changeWorklistStudy = function (decrement, byMRN) {
            var worklist = this.getWorklistData(byMRN);
            if (worklist) {
                var currentIndex = this.getCurrentWorklistIndex(worklist, byMRN);

                if (currentIndex != -1) {
                    var nextIndex = currentIndex + (decrement ? -1 : 1);
                    if ((nextIndex >= 0) && (nextIndex < worklist.length)) {
                        var params = "";

                        if (worklist[nextIndex].accelerated) {
                            params += "&viewer_local_accelerator=1";
                        }

                        if (!this.keyImagesOnly && this.keyImageSeriesEnabled.read()) {
                            params += "&showKeyOnly=true";
                        }

                        window.location.href = "https://" + window.location.hostname + worklist[nextIndex].studyURL + params;
                        window.location.reload();
                    }
                }
            }
        };

        Application.prototype.hasPreviousSingleSeriesImage = function () {
            if (this.singleSeriesEnabled.read() && (this.singleSeries != null) && this.canDisplaySingleSeries(this.layout.read())) {
                return this.singleSeriesIndex > 0;
            }

            return false;
        };

        Application.prototype.previousSingleSeriesImage = function () {
            if (this.hasPreviousSingleSeriesImage()) {
                var selected = this.selectedSeriesKey.read().value;
                var allSeriesViews = this.seriesViews.value;

                var foundSelected = 0;
                for (var ctr = 0; ctr < allSeriesViews.length; ctr += 1) {
                    if (allSeriesViews[ctr].viewKey.value == selected) {
                        foundSelected = ctr;
                        break;
                    }
                }

                allSeriesViews.unshift(allSeriesViews.splice(allSeriesViews.length - 1, 1)[0]);

                var series = Study.splitSeries(this.singleSeries)[this.singleSeriesIndex - 1];
                this.singleSeriesIndex -= 1;

                var seriesToUnload = this.seriesViews.value[0];
                var seriesNew = this.loadSeriesAt(series, 0);

                if (seriesToUnload) {
                    seriesToUnload.unload();
                }

                this.seriesViews.raiseChangedEvent(allSeriesViews);
                this.selectedSeriesKey.write(this.seriesViews.read()[foundSelected].viewKey);
                seriesNew.selectedInstanceIndex.write(0);
            }
        };

        Application.prototype.hasNextSingleSeriesImage = function () {
            if (this.singleSeriesEnabled.read() && (this.singleSeries != null) && this.canDisplaySingleSeries(this.layout.read())) {
                var layout = this.layout.read();
                var available = layout.columns * layout.rows;
                return (this.singleSeriesIndex + available) < this.singleSeries.instances.length;
            }

            return false;
        };

        Application.prototype.nextSingleSeriesImage = function () {
            if (this.hasNextSingleSeriesImage()) {
                var selected = this.selectedSeriesKey.read().value;
                var allSeriesViews = this.seriesViews.value;
                var layout = this.layout.read();
                var available = layout.columns * layout.rows;

                var foundSelected = 0;
                for (var ctr = 0; ctr < allSeriesViews.length; ctr += 1) {
                    if (allSeriesViews[ctr].viewKey.value == selected) {
                        foundSelected = ctr;
                        break;
                    }
                }

                allSeriesViews.push(allSeriesViews.shift());
                var series = Study.splitSeries(this.singleSeries)[this.singleSeriesIndex + available];
                this.singleSeriesIndex += 1;

                var seriesToUnload = this.seriesViews.value[available - 1];
                var seriesNew = this.loadSeriesAt(series, available - 1);

                if (seriesToUnload) {
                    seriesToUnload.unload();
                }

                this.seriesViews.raiseChangedEvent(allSeriesViews);
                this.selectedSeriesKey.write(this.seriesViews.read()[foundSelected].viewKey);
                seriesNew.selectedInstanceIndex.write(0);
            }
        };

        /**
        * Run a study action, aka routing rule
        */
        Application.prototype.invokeStudyAction = function (study, action) {
            var _this = this;
            var studyStorage = Services.getStudyStorageInfo(this.sessionId, study.studyAttributes.queryObject);

            var run = function (task) {
                $('.overlay').addClass('application-loading');

                Observable._finally(task, function () {
                    $('.overlay').removeClass('application-loading');
                }).subscribe({
                    next: function (_) {
                        // Allow 200 ms for processing on the server before requesting the updated study
                        setTimeout(function () {
                            // On completion of routing rule, update study from /get/study and re-render
                            Observable.bind(studyStorage, function (storage) {
                                study = Study.updateStudyFromStorage(study, storage);
                                _this.renderAllFrames();

                                return Observable.ret({});
                            }).subscribe({
                                done: function () {
                                },
                                next: function (_) {
                                },
                                fail: function (err) {
                                    _this.recordError("Unable to update study: " + err);
                                }
                            });
                        }, 200);
                    },
                    done: function () {
                    },
                    fail: function (err) {
                        _this.recordError("Unable to invoke study action: " + err);
                        window.alert("Unable to run the selected study action.");
                    }
                });
            };

            if (action.requiresEmailAddress) {
                var email = window.prompt(this.terminology.lookup(Terminology.Terms.EnterEmailAddress));

                if (email) {
                    var message = window.prompt(this.terminology.lookup(Terminology.Terms.EnterShareMessage));

                    run(Services.runRoutingRule(this.sessionId, study.studyAttributes.uuid, action.id, new Classes.EmailAddress(email), message));
                }
            } else {
                run(Services.runRoutingRule(this.sessionId, study.studyAttributes.uuid, action.id));
            }
        };

        /**
        * Magnify/minify the series with the specified index
        */
        Application.prototype.magnifyMinify = function (series) {
            var layout = this.layout.read();

            if (layout.rows > 1 || layout.columns > 1) {
                var magnifiedSeries = this.magnifiedSeries.read();

                if (magnifiedSeries !== null) {
                    this.magnifiedSeries.write(null);
                    this.updateSeriesHandles(true);
                } else {
                    this.magnifiedSeries.write(series);
                    this.updateSeriesHandles(false);
                }
            }
        };

        Application.prototype.isMagnified = function () {
            return this.magnifiedSeries.read() != null;
        };

        Application.prototype.canDisplaySingleSeries = function (layout) {
            if (layout.rows > 1 || layout.columns > 1) {
                return this.magnifiedSeries.read() == null;
            }

            return false;
        };

        /**
        * Save the current window level setting as a preset
        */
        Application.prototype.saveWindowLevelPreset = function () {
            var _this = this;
            var container = this.selectedSeriesContainer.read();

            if (container) {
                var modality = container.series.seriesAttributes.modality;

                var presetName = window.prompt(this.terminology.lookup(Terminology.Terms.EnterPresetName), this.terminology.lookup(Terminology.Terms.Untitled));

                if (presetName) {
                    Subjects.modify(this.settings, function (settings) {
                        var copy = $.extend(true, settings, {
                            modalities: []
                        });

                        var modalitySettings = _.find(copy.modalities, function (m) {
                            return m.modality === modality;
                        });

                        if (!modalitySettings) {
                            copy.modalities.push(modalitySettings = { modality: modality });
                        }

                        if (!modalitySettings.presets || (modalitySettings.presets.length === 0)) {
                            modalitySettings.presets = WindowLevelPresets.defaults(modality, _this.terminology);
                        }

                        var windowLevel = container.combineWindowLevelFlags().read();

                        if (modalitySettings.presets.length >= 9) {
                            var presetNumberString = window.prompt(_this.terminology.lookup(Terminology.Terms.SelectPresetToReplace), "1");

                            var presetNumber;

                            if (!presetNumberString || presetNumberString.length != 1 || !_.contains(['1', '2', '3', '4', '5', '6', '7', '8', '9'], presetNumberString[0])) {
                                presetNumber = 1;
                            } else {
                                presetNumber = parseInt(presetNumberString);
                            }

                            modalitySettings.presets[presetNumber - 1] = {
                                name: presetName,
                                windowLevel: windowLevel
                            };
                        } else {
                            modalitySettings.presets.push({
                                name: presetName,
                                windowLevel: windowLevel
                            });
                        }

                        return copy;
                    });
                }
            }
        };

        /**
        * Display a new collection of series
        */
        Application.prototype.loadSeriesSet = function (forwards) {
            var layout = this.layout.read();
            var count = layout.columns * layout.rows;

            var seriesViews = this.seriesViews.read();

            var allSeries = _.toArray(_.map(_.flatten(_.map(this.studies, function (study) {
                return study.series;
            })), function (series, index, list) {
                return {
                    index: index,
                    series: series
                };
            }));

            var visibleSeries = _.filter(allSeries, function (pair) {
                return _.any(seriesViews, function (series, index, list) {
                    return series.series === pair.series && index < count;
                });
            });

            var invisibleSeries = _.filter(allSeries, function (pair) {
                return !_.any(seriesViews, function (series, index, list) {
                    return series.series === pair.series && index < count;
                });
            });

            var visibleSeriesNumbers = _.map(visibleSeries, function (pair) {
                return pair.index;
            });
            var minSeriesNumber = _.min(visibleSeriesNumbers);
            var maxSeriesNumber = _.max(visibleSeriesNumbers);

            var validSeries = _.filter(invisibleSeries, function (pair) {
                if (forwards) {
                    return pair.index > maxSeriesNumber;
                } else {
                    return pair.index < minSeriesNumber;
                }
            });

            if (!_.any(validSeries)) {
                validSeries = _.any(invisibleSeries) ? invisibleSeries : allSeries;
            }

            if (forwards) {
                validSeries = _.take(validSeries, count);
            } else {
                validSeries = _.take(validSeries.reverse(), count).reverse();
            }

            for (var i = 0; i < seriesViews.length; i++) {
                var seriesView = seriesViews[i];
                var series = i < validSeries.length ? validSeries[i].series : null;

                this.loadSeriesAt(series, i);
            }

            this.seriesViews.raiseChangedEvent(seriesViews);
            this.selectedSeriesKey.write(seriesViews[0].viewKey);

            if (this.selectedTool.read() === 9 /* Localization */) {
                this.loadVisibleImageAttributes(this.imagePreloadQueue, -1 /* ImageAttributeBackgroundPreloading */);
            }
        };

        /**
        * Toggle cine for all visible series
        */
        Application.prototype.toggleCineAll = function () {
            var layout = this.layout.read();
            var count = layout.columns * layout.rows;

            var currentSeriesViews = this.seriesViews.read();

            var cineIsActive = _.any(currentSeriesViews, function (series, index, list) {
                return series.cineActive.read() && index < count;
            });

            for (var i = 0; i < currentSeriesViews.length; i++) {
                var seriesView = currentSeriesViews[i];

                if (i < count) {
                    seriesView.cineActive.write(!cineIsActive);
                }
            }
        };

        Application.prototype.loadScript = function (studyStorage, queryObject, attachment) {
            var _this = this;
            var $overlay = $('.overlay');

            $overlay.addClass('application-loading');

            Observable._finally(V3Storage.GetAttachment(this.sessionId, studyStorage, queryObject, attachment.id, attachment.phiNamespace, attachment.version), function () {
                $overlay.removeClass('application-loading');
            }).subscribe({
                done: function () {
                },
                next: function (script) {
                    _this.playScript(script);
                },
                fail: function (err) {
                    _this.recordError("Unable to download attachment: " + err);
                    window.alert("Unable to download the attachment.");
                }
            });
        };

        /**
        * Download an attachment
        */
        Application.prototype.loadAttachment = function (study, attachment) {
            var uri = Routes.Attachment(this.sessionId, study.studyAttributes.studyStorage, study.studyAttributes.queryObject, attachment.id, attachment.phiNamespace, attachment.version);

            window.open(uri);
        };

        /**
        * View a report
        */
        Application.prototype.loadHL7Report = function (study, report) {
            var uri = Routes.HL7Report(this.sessionId, study.studyAttributes.uuid, report.id);

            window.open(uri);
        };

        /**
        * View a structured report
        */
        Application.prototype.loadStructuredReport = function (study, report) {
            var id = study.studyAttributes.queryObject.toString();
            var sid = this.sessionId.value;
            window.open("/viewer/report.html#report/{0}?sid={1}&uuid={2}&version={3}".replace("{0}", encodeURIComponent(id)).replace("{1}", encodeURIComponent(sid)).replace("{2}", encodeURIComponent(report.id.value)).replace("{3}", encodeURIComponent(report.version.value)));
        };

        Application.prototype.loadReport = function (uuid, newWin) {
            if (typeof newWin === "undefined") { newWin = false; }
            var id = uuid.value;
            var sid = this.sessionId.value;

            if (newWin) {
                window.open("/?route=study_reports_with_sid&sid={0}&study_uuid={1}".replace("{0}", encodeURIComponent(sid)).replace("{1}", encodeURIComponent(id)), "_blank", "width=800,height=600");
            } else {
                window.open("/?route=study_reports_with_sid&sid={0}&study_uuid={1}".replace("{0}", encodeURIComponent(sid)).replace("{1}", encodeURIComponent(id)));
            }
        };

        /**
        * Duplicate the selected measurement
        */
        Application.prototype.duplicateShape = function () {
            var series = this.selectedSeriesContainer.read();

            if (series) {
                var measurement = series.selectedMeasurement.read();

                if (measurement) {
                    var copy = measurement.clone();
                    copy.editable = true;
                    copy.hideMeasurement = measurement.hideMeasurement;
                    series.addMeasurementToSelectedInstance(copy);

                    series.renderAll();
                }
            }
        };

        /**
        * Copy the selected measurement to the clipboard
        */
        Application.prototype.copyShape = function () {
            var series = this.selectedSeriesContainer.read();

            if (series) {
                var measurement = series.selectedMeasurement.read();

                if (measurement) {
                    this.clipboard.write(measurement.toAnnotationData());
                }
            }
        };

        /**
        * Paste the copied shape as a new measurement
        */
        Application.prototype.pasteShape = function () {
            var series = this.selectedSeriesContainer.read();

            if (series) {
                var data = this.clipboard.read();

                var measurement = Annotations.createMeasurement(data, "", this.user.name, this.user.uuid, true);

                if (measurement) {
                    var copy = measurement.clone();
                    copy.editable = true;
                    copy.hideMeasurement = measurement.hideMeasurement;

                    var instance = series.currentInstance().read();
                    if (this.settings.read().ultrasoundMeasurements && Dicom.Ultrasound.isUltrasound(instance)) {
                        var result = Dicom.Ultrasound.findSpacing(copy, instance);
                        if (result && result.valid) {
                            copy.ultrasoundPixelSpacing = [result.spacingX, result.spacingY];
                        }
                    }

                    measurement.calibration = false;
                    measurement.calibrationValue = 0;

                    series.addMeasurementToSelectedInstance(copy);

                    series.renderAll();
                }
            }
        };

        Application.prototype.geometryMatches = function (instanceFrom, instanceTo) {
            var pixelSpacingFrom = instanceFrom.instanceAttributes.pixelSpacing;
            var pixelSpacingTo = instanceTo.instanceAttributes.pixelSpacing;

            if (pixelSpacingFrom && pixelSpacingFrom[0] && pixelSpacingTo && pixelSpacingTo[0]) {
                var epsilon = Math.max(pixelSpacingFrom[0], pixelSpacingTo[0]) / 2.0;
                var widthFrom = instanceFrom.instanceAttributes.columns * pixelSpacingFrom[0];
                var heightFrom = instanceFrom.instanceAttributes.rows * pixelSpacingFrom[0];

                var widthTo = instanceTo.instanceAttributes.columns * pixelSpacingTo[0];
                var heightTo = instanceTo.instanceAttributes.rows * pixelSpacingTo[0];

                return (Math.abs(widthFrom - widthTo) < epsilon) && (Math.abs(heightFrom - heightTo) < epsilon);
            }

            return false;
        };

        /**
        * Handles co-located annotations across series.
        */
        Application.prototype.copyShapeToAllSeries = function () {
            var currentSeries = this.selectedSeriesContainer.read();

            if (currentSeries) {
                var measurement = currentSeries.selectedMeasurement.read();
                var currentSeriesInstance = currentSeries.currentInstance().read();
                var pixelSpacingFrom = currentSeriesInstance.instanceAttributes.pixelSpacing;
                var modality = currentSeriesInstance.seriesAttributes.modality;

                if (measurement) {
                    var app = this;
                    var userName = this.user.name;
                    var userId = this.user.uuid;
                    var colocationId = measurement.colocationId;
                    var foundColocationId = false;

                    if (colocationId) {
                        foundColocationId = true;
                    } else {
                        colocationId = Annotations.newRandomId();
                        measurement.colocationId = colocationId;
                    }

                    _.each(this.seriesViews.value, function (series) {
                        if (currentSeries != series) {
                            var currentInstance = series.currentInstance().read();

                            if (series.isActive() && currentInstance) {
                                var pixelSpacingTo = currentInstance.instanceAttributes.pixelSpacing;
                                var currentModality = currentInstance.seriesAttributes.modality;

                                var measurements = currentInstance.instanceAttributes.measurements;
                                var measurementCopy = _.find(measurements, function (item) {
                                    return (item.colocationId && (item.colocationId.value == colocationId.value));
                                });
                                var data = jQuery.extend(true, {}, measurement.toAnnotationData());

                                // images must match geometry; pixel location copy is also supported for US modality
                                var geometryMatch = false;
                                var modalityMatch = false;

                                if (pixelSpacingFrom && pixelSpacingFrom[0] && pixelSpacingTo && pixelSpacingTo[0]) {
                                    geometryMatch = app.geometryMatches(currentSeriesInstance, currentInstance);
                                } else if (!pixelSpacingFrom && !pixelSpacingTo) {
                                    modalityMatch = (modality == currentModality) && (modality == "US");
                                }

                                if (geometryMatch || modalityMatch) {
                                    if (geometryMatch) {
                                        Annotations.convertAnnotationLocation(data, pixelSpacingFrom[0], pixelSpacingTo[0]);
                                    }

                                    if (measurement.label) {
                                        data.label = measurement.label;
                                    }

                                    if (measurementCopy) {
                                        var dataOriginal = measurementCopy.toAnnotationData();
                                        _.each(data.points, function (item, index) {
                                            dataOriginal.points[index].x = item.x;
                                            dataOriginal.points[index].y = item.y;
                                        });

                                        series.editMeasurement(measurementCopy, currentInstance);
                                        series.renderAll();
                                    } else if (!foundColocationId) {
                                        var measurementCopy = Annotations.createMeasurement(data, "", userName, userId, true);
                                        measurementCopy.colocationId = colocationId;
                                        measurementCopy.hideMeasurement = measurement.hideMeasurement;
                                        measurementCopy.calibration = false;
                                        measurementCopy.calibrationValue = 0;
                                        series.addMeasurementToSelectedInstance(measurementCopy);
                                        series.renderAll();
                                    }
                                }
                            }
                        }
                    });
                }
            }
        };

        Application.prototype.labelAnnotation = function () {
            var currentSeries = this.selectedSeriesContainer.read();

            if (currentSeries) {
                var measurement = currentSeries.selectedMeasurement.read();
                var currentSeriesInstance = currentSeries.currentInstance().read();

                if (measurement && currentSeriesInstance) {
                    var text = window.prompt(this.terminology.lookup(Terminology.Terms.EnterText), measurement.label);

                    if (text != null) {
                        measurement.label = text;

                        if (measurement.volume) {
                            _.each(measurement.volume.slices, function (s) {
                                if (s.measurement.label != text) {
                                    s.measurement.label = text;
                                    currentSeries.editMeasurement(s.measurement, s.instance);
                                }
                            });
                        }
                    }

                    currentSeries.editMeasurement(measurement, currentSeriesInstance);
                    currentSeries.renderAll();
                }
            }
        };

        Application.prototype.assignPixelSpacing = function () {
            var currentSeries = this.selectedSeriesContainer.read();

            if (currentSeries) {
                var measurement = currentSeries.selectedMeasurement.read();
                var currentSeriesInstance = currentSeries.currentInstance().read();

                if (measurement && currentSeriesInstance) {
                    var spacingStr = "";
                    if (measurement.pixelSpacingUser) {
                        spacingStr = String(measurement.pixelSpacingUser);
                    }

                    var text = window.prompt(this.terminology.lookup(Terminology.Terms.AssignPixelSpacingDialog), spacingStr);
                    measurement.pixelSpacingUser = parseFloat(text);
                    var calibrationUser = !!measurement.sliceSpacingUser || !!measurement.pixelSpacingUser;
                    currentSeriesInstance.instanceAttributes.calibrationUser = calibrationUser;

                    if (measurement.volume) {
                        _.each(measurement.volume.slices, function (s) {
                            s.measurement.volume = null;
                            s.measurement.pixelSpacingUser = parseFloat(text);
                            currentSeries.editMeasurement(s.measurement, s.instance);
                            s.instance.instanceAttributes.calibrationUser = calibrationUser;
                        });
                    }

                    currentSeries.editMeasurement(measurement, currentSeriesInstance);
                    currentSeries.renderAll();
                }
            }
        };

        Application.prototype.assignSliceSpacing = function () {
            var currentSeries = this.selectedSeriesContainer.read();

            if (currentSeries) {
                var measurement = currentSeries.selectedMeasurement.read();
                var currentSeriesInstance = currentSeries.currentInstance().read();

                if (measurement && currentSeriesInstance) {
                    var spacingStr = "";
                    if (measurement.sliceSpacingUser) {
                        spacingStr = String(measurement.sliceSpacingUser);
                    }

                    var text = window.prompt(this.terminology.lookup(Terminology.Terms.AssignSliceSpacingDialog), spacingStr);
                    var value = parseFloat(text);
                    measurement.sliceSpacingUser = value;
                    var calibrationUser = !!measurement.sliceSpacingUser || !!measurement.pixelSpacingUser;
                    currentSeriesInstance.instanceAttributes.calibrationUser = calibrationUser;

                    if (measurement.volume) {
                        _.each(measurement.volume.slices, function (s) {
                            s.measurement.volume = null;
                            s.measurement.sliceSpacingUser = value;
                            currentSeries.editMeasurement(s.measurement, s.instance);
                            s.instance.instanceAttributes.calibrationUser = calibrationUser;
                        });
                    }

                    currentSeries.editMeasurement(measurement, currentSeriesInstance);
                    currentSeries.renderAll();
                }
            }
        };

        /**
        * Remove all temporary annotations
        */
        Application.prototype.removeTemporaryAnnotations = function () {
            _.each(this.studies, function (study) {
                _.each(study.series, function (series) {
                    _.each(series.instances, function (instance) {
                        instance.instanceAttributes.measurements = _.filter(instance.instanceAttributes.measurements, function (m) {
                            return !m.temporary;
                        });
                    });
                });
            });

            this.renderAllFrames();
        };

        /**
        * Delete a series
        */
        Application.prototype.deleteSeries = function (series) {
            var _this = this;
            if (series) {
                var message = this.terminology.lookup(Terminology.Terms.DeleteSeriesWarning) + "\n\n" + series.instances[0].instanceAttributes.seriesDescription;

                if (window.confirm(message)) {
                    // Do not pass a version here - delete all versions of each image
                    var deleteSeries = Observable.sequenceM(_.map(series.instances, function (instance, index, list) {
                        return Observable.invoke(V3Storage.deleteImage(_this.sessionId, instance.studyAttributes.studyStorage, instance.studyAttributes.queryObject, instance.id), function (_) {
                            console.log("Deleted image #" + index + " from series" + series.seriesAttributes.seriesUid.value);
                        });
                    }));

                    $('.overlay').addClass('application-loading');

                    Observable._finally(deleteSeries, function () {
                        $('.overlay').removeClass('application-loading');
                    }).subscribe({
                        done: function () {
                            _.each(_this.studies, function (study) {
                                study.series = _.filter(study.series, function (studySeries) {
                                    return studySeries.seriesAttributes.seriesUid.value !== series.seriesAttributes.seriesUid.value;
                                });
                            });

                            for (var i = 0; i < _this.seriesViews.value.length; i++) {
                                var view = _this.seriesViews.value[i];

                                if (view.series && view.series.seriesAttributes.seriesUid.value === series.seriesAttributes.seriesUid.value) {
                                    _this.loadSeriesAt(null, i);
                                }
                            }

                            _this.render();
                        },
                        next: function (_) {
                        },
                        fail: function (err) {
                            _this.recordError("Unable to delete series: " + err);
                            window.alert(_this.terminology.lookup(Terminology.Terms.ErrorDeletingImage));
                        }
                    });
                }
            }
        };

        Application.prototype.parseInstanceSelectionFormat = function (format) {
            var inputs = [];

            format = format.replace(/\s/g, '');
            var items = format.split(",");
            _.each(items, function (s) {
                if (s.indexOf("-") !== -1) {
                    var range = s.split("-");
                    if (range.length === 2) {
                        try  {
                            inputs.push(_.range(parseInt(range[0]), parseInt(range[1]) + 1));
                        } catch (e) {
                            inputs.push(Number.NaN);
                        }
                    }
                } else {
                    inputs.push(parseInt(s));
                }
            });

            return _.uniq(_.flatten(inputs)).sort();
        };

        Application.prototype.removeImages = function () {
            var _this = this;
            var selectedSeries;
            var selectedSeriesView = this.selectedSeriesContainer.read();

            if (this.singleSeriesEnabled.read()) {
                selectedSeries = this.singleSeries;
            } else if (selectedSeriesView) {
                selectedSeries = selectedSeriesView.series;
            }

            if (selectedSeries && selectedSeriesView) {
                var format = window.prompt(this.terminology.lookup(Terminology.Terms.RemoveImagesInstructions) + "\n\n" + selectedSeries.instances[0].instanceAttributes.seriesDescription, "");

                if (format) {
                    var numInstances = selectedSeries.instances.length;
                    var maxIndex = selectedSeries.instances.length;
                    var inputs = [];
                    format = format.replace(/\s/g, '');
                    var items = format.split(",");
                    _.each(items, function (s) {
                        if (s.indexOf("-") !== -1) {
                            var range = s.split("-");
                            if (range.length === 2) {
                                try  {
                                    inputs.push(_.range(parseInt(range[0]), parseInt(range[1]) + 1));
                                } catch (e) {
                                    inputs.push(Number.NaN);
                                }
                            }
                        } else {
                            inputs.push(parseInt(s));
                        }
                    });

                    var indices = _.uniq(_.flatten(inputs)).sort();
                    var valid = _.reduce(indices, function (t, n) {
                        return t + n;
                    });
                    if (isNaN(valid) || indices[0] < 1 || indices[indices.length - 1] > maxIndex) {
                        window.alert(this.terminology.lookup(Terminology.Terms.RemoveImagesWarning));
                    } else {
                        var seriesRemoved = indices.length === numInstances;
                        var deleteAll = Observable.sequenceM(_.map(indices, function (i) {
                            var instance = selectedSeries.instances[i - 1];
                            return V3Storage.deleteImage(_this.sessionId, instance.studyAttributes.studyStorage, instance.studyAttributes.queryObject, instance.id, instance.instanceAttributes.version);
                        }));

                        $('.overlay').addClass('application-loading');

                        Observable._finally(deleteAll, function () {
                            $('.overlay').removeClass('application-loading');
                        }).subscribe({
                            done: function () {
                                selectedSeries.instances = _.filter(selectedSeries.instances, function (i, n) {
                                    return _.indexOf(indices, n + 1) === -1;
                                });

                                if (seriesRemoved) {
                                    _this.removeEmptySeries(selectedSeries);
                                }

                                if (_this.singleSeriesEnabled.read()) {
                                    _this.resetSeriesViews();
                                    _this.render(_this.singleSeries);
                                }

                                selectedSeriesView.selectedInstanceIndex.write(0);
                                window.alert(_this.terminology.lookup(Terminology.Terms.RemoveImagesSuccess));
                            },
                            next: function (_) {
                            },
                            fail: function (err) {
                                _this.recordError("Unable to remove images: " + err);
                                window.alert(_this.terminology.lookup(Terminology.Terms.RemoveImagesError));
                            }
                        });
                    }
                }
            }
        };

        /**
        * Remove empty series (e.g., when series images were deleted)
        */
        Application.prototype.removeEmptySeries = function (deletedSeries) {
            var _this = this;
            var nextSeries;
            var foundSeries;
            var viewIndex = 0;
            var layout = this.layout.read();

            // find index of old series in layout
            _.each(_.range(0, Math.min(this.seriesViews.value.length, layout.rows * layout.columns)), function (index) {
                if (_this.seriesViews.value[index].series == deletedSeries) {
                    viewIndex = index;
                }
            });

            // find index of old series among all series
            _.each(this.studies, function (study) {
                _.each(study.series, function (series) {
                    if (foundSeries) {
                        foundSeries = false;
                        nextSeries = series;
                    }
                    if (deletedSeries == series) {
                        foundSeries = true;
                    }
                });
            });

            // filter out empty study
            _.each(this.studies, function (study) {
                study.series = _.filter(study.series, function (studySeries) {
                    return studySeries.instances.length > 0;
                });
            });

            // render views
            this.resetSeriesViews();
            this.render();

            // replace series with next one
            if (this.seriesViews.value.length) {
                this.selectedSeriesKey.write(this.seriesViews.value[viewIndex].viewKey);

                if (nextSeries) {
                    this.replaceSelectedSeries(nextSeries);
                }
            }
        };

        /**
        * Open a copy of the viewer in another window
        */
        Application.prototype.openNewWindow = function (screenId, numScreens) {
            var _this = this;
            var study = this.studies[0];
            var id = study.studyAttributes.queryObject.toString();
            var sid = this.sessionId.value;

            var url;

            if (location.hostname == Routes.LocalAccelerator.LOCAL_ACCELERATOR_HOSTNAME) {
                url = location.protocol + '//' + location.host + location.pathname + location.search + "&owner={1}&screen={2}&isChildWindow=1".replace("{1}", encodeURIComponent(this.owner.value)).replace("{2}", encodeURIComponent(screenId.toString()));
            } else {
                url = "/viewer/index.html#study/{0}?sid={1}&owner={2}&screen={3}&isChildWindow=1".replace("{0}", encodeURIComponent(id)).replace("{1}", encodeURIComponent(sid)).replace("{2}", encodeURIComponent(this.owner.value)).replace("{3}", encodeURIComponent(screenId.toString()));
            }

            if (this.accountSettings.fullscreen) {
                this.fullscreenOnDisplay(screenId, numScreens, url);
            } else if (!Browser.uiFound()) {
                this.needsDirectFullscreen(function (fullscreen) {
                    if (fullscreen) {
                        _this.fullscreenOnDisplay(screenId, numScreens, url, true);
                    } else {
                        window.open(url);
                    }
                });
            } else {
                window.open(url);
            }
        };

        /**
        * Positions viewer page at a specific display in a multi-monitor environment (see Local.js in v3ui)
        * @param {number} screenId
        * @param {number} numScreens
        * @param {string} url
        * @param {boolean} directFullscreen  to handle fullscreen directly with the PA, without studylist available
        */
        Application.prototype.fullscreenOnDisplay = function (screenId, numScreens, url, directFullscreen) {
            if (typeof directFullscreen === "undefined") { directFullscreen = false; }
            var title = "Viewer (Screen " + (screenId + 1) + ")";
            var specs;
            var name = "ambra-screen-" + screenId;
            var timestamp = (new Date().getTime());

            if (Browser.isIE() || Browser.isIE11()) {
                specs = "top=0,left=0,height=100,width=100,menubar=yes,location=yes,resizable=yes,scrollbars=yes,status=yes";
            } else {
                specs = "top=0,left=0,height=100,width=100";
            }

            if (url && this.localAccelerator) {
                url += "&viewer_local_accelerator=1";
            }

            if (url) {
                url = url.replace("#study", "?foo=" + timestamp + "#study");
            }

            var win = url ? window.open(url, name, specs) : window;

            win.document.title = title;

            var app = this;
            setTimeout(function () {
                win.document.title = title;

                if (directFullscreen) {
                    app.doDirectFullscreen({
                        screen: screenId.toString(),
                        numScreens: numScreens.toString(),
                        name: title,
                        timestamp: timestamp
                    });
                } else {
                    window.localStorage.setItem("viewEvent", JSON.stringify({
                        screen: screenId.toString(),
                        numScreens: numScreens.toString(),
                        name: title,
                        timestamp: timestamp
                    }));
                }

                // Chrome does not always show the changed title on first attempt
                var titleTries = 0;
                var setTitle = setInterval(function () {
                    win.document.title = title;
                    titleTries++;

                    if (titleTries > 10) {
                        clearInterval(setTitle);
                    }
                }, 100);
            }, 100);
        };

        Application.prototype.needsDirectFullscreen = function (cb) {
            this.callDisplay({}, cb);
        };

        Application.prototype.doDirectFullscreen = function (data) {
            if (Browser.isChrome()) {
                data["browser"] = "Google Chrome";
            } else if (Browser.isFirefox()) {
                data["browser"] = "Firefox";
            } else if (Browser.isSafari()) {
                data["browser"] = "Safari";
            } else {
                data["browser"] = "Unknown";
            }

            data["listName"] = "Ambra | Studies";
            this.callDisplay(data);
        };

        Application.prototype.callDisplay = function (data, cb) {
            var params = $.param({ data: JSON.stringify(data) });

            var img = new Image();

            img.onload = function () {
                if (cb) {
                    cb(true);
                }
            };

            img.onerror = function () {
                if (cb) {
                    cb(false);
                }
            };

            img.src = "https://local.ambrahealth.com:" + 8021 + "/" + "display" + "?" + params + "&a=" + +(new Date());
        };

        /**
        * Export the visible images to PNG format
        */
        Application.prototype.exportCurrentImages = function () {
            var $allSeries = $('.study-series');

            var layout = this.layout.read();

            var width = $allSeries.width();
            var height = $allSeries.height();

            var seriesWidth = width / layout.columns;
            var seriesHeight = height / layout.rows;

            var canvas = Images.createCanvas(width, height);
            var ctx = canvas.getContext('2d');

            ctx.fillStyle = "black";
            ctx.fillRect(0, 0, width, height);

            _.each(this.seriesViews.read(), function (series, index, list) {
                if (index < layout.rows * layout.columns) {
                    var seriesCanvas = series.prepareCanvasForExport();

                    var x = (index * seriesWidth) % width;
                    var y = Math.floor(index / layout.columns) * seriesHeight;

                    ctx.drawImage(seriesCanvas, x, y, seriesWidth, seriesHeight);
                }
            });

            var uri = canvas.toDataURL("image/png");

            var html = $('<img>').attr('src', uri)[0].outerHTML;
            $(window.open().document.body).html(html);
        };

        /**
        * Add a visual marker on the screen to indicate a click
        */
        Application.prototype.addMarker = function (series, imageCoords, clazz) {
            if (series.series) {
                var index = series.selectedInstanceIndex.read();

                var instance = series.series.instances[index];

                var $instance = series.el.find('.instance');
                var screenCoords = series.mapFromImage(imageCoords, $instance.width(), $instance.height(), instance.instanceAttributes.columns, instance.instanceAttributes.rows);

                var $marker = $('<span>').addClass(clazz).css({
                    left: Math.round(screenCoords.x - 5) + "px",
                    top: Math.round(screenCoords.y - 5) + "px"
                });

                $instance.append($marker);

                $marker.animate({
                    opacity: 0.0
                }, 1000, function () {
                    $marker.remove();
                });
            }
        };

        /**
        * Handle an event from a recorded script or meeting
        */
        Application.prototype.handleEvent = function (event, timestamp, reverseEvent) {
            try  {
                var seriesView;

                if (event.seriesContainerIndex !== undefined) {
                    seriesView = this.seriesViews.value[event.seriesContainerIndex];
                }

                if (seriesView !== undefined && event.mousePositionImage) {
                    var clazz = event.type === 7 /* MouseMove */ ? "marker mouse-move" : "marker mouse-up";
                    this.addMarker(seriesView, event.mousePositionImage, clazz);
                }

                switch (event.type) {
                    case 0 /* ImageTransformationChanged */:
                         {
                            if (seriesView) {
                                if (reverseEvent) {
                                    reverseEvent.appendWithTimestamp({
                                        type: 0 /* ImageTransformationChanged */,
                                        newTransformation: seriesView.transform.read(),
                                        seriesContainerIndex: event.seriesContainerIndex,
                                        event: event
                                    }, timestamp);
                                }

                                seriesView.transform.write(event.newTransformation);
                            }
                        }
                        break;
                    case 1 /* WindowLevelChanged */:
                         {
                            if (seriesView) {
                                if (reverseEvent) {
                                    reverseEvent.appendWithTimestamp({
                                        type: 1 /* WindowLevelChanged */,
                                        newWindowLevel: seriesView.windowLevel.read(),
                                        seriesContainerIndex: event.seriesContainerIndex,
                                        event: event
                                    }, timestamp);
                                }

                                if (event.newWindowLevel === null) {
                                    seriesView.resetWindowLevel();
                                } else {
                                    seriesView.useOriginalWindowLevel.write(false);
                                    seriesView.windowLevel.write(event.newWindowLevel);
                                }
                            }
                        }
                        break;
                    case 2 /* AnnotationAdded */:
                         {
                            if (seriesView && seriesView.series) {
                                var instance = _.find(seriesView.series.instances, function (instance) {
                                    return instance.id.value === event.instanceUid.value;
                                });
                                if (instance) {
                                    var original = seriesView.selectedInstanceIndex.read();
                                    var annId = Annotations.newRandomId();
                                    seriesView.selectedInstanceIndex.write(event.instanceIndex);
                                    var measurement = Annotations.createMeasurement(event.annotation, annId.value, "", "", false);

                                    if (reverseEvent) {
                                        reverseEvent.appendWithTimestamp({
                                            type: 16 /* AnnotationDeleted */,
                                            measurement: measurement,
                                            seriesContainerIndex: event.seriesContainerIndex,
                                            instanceUid: event.instanceUid,
                                            event: null
                                        }, timestamp);

                                        reverseEvent.appendWithTimestamp({
                                            type: 6 /* SelectedImageChanged */,
                                            instanceIndex: original,
                                            seriesContainerIndex: event.seriesContainerIndex,
                                            event: event
                                        }, timestamp);
                                    }

                                    measurement.temporary = true;
                                    var measurements = instance.instanceAttributes.measurements;
                                    measurements.push(measurement);
                                    seriesView.renderAll();
                                }
                            }
                        }
                        break;
                    case 16 /* AnnotationDeleted */: {
                        if (seriesView && seriesView.series) {
                            var instance = _.find(seriesView.series.instances, function (instance) {
                                return instance.id.value === event.instanceUid.value;
                            });
                            instance.instanceAttributes.measurements = _.without(instance.instanceAttributes.measurements, event.measurement);
                            seriesView.renderAll();
                        }

                        break;
                    }
                    case 3 /* MouseToolChanged */:
                         {
                            if (reverseEvent) {
                                reverseEvent.appendWithTimestamp({
                                    type: 3 /* MouseToolChanged */,
                                    newMouseTool: this.selectedTool.read(),
                                    seriesContainerIndex: event.seriesContainerIndex,
                                    event: event
                                }, timestamp);
                            }
                            this.selectedTool.write(event.newMouseTool);
                        }
                        break;
                    case 4 /* LayoutChanged */:
                         {
                            if (reverseEvent) {
                                reverseEvent.appendWithTimestamp({
                                    type: 4 /* LayoutChanged */,
                                    newLayout: this.layout.read(),
                                    seriesContainerIndex: event.seriesContainerIndex,
                                    event: event
                                }, timestamp);
                            }

                            this.layout.write(event.newLayout);
                        }
                        break;
                    case 5 /* SeriesChanged */:
                         {
                            if (reverseEvent) {
                                reverseEvent.appendWithTimestamp({
                                    type: 5 /* SeriesChanged */,
                                    seriesInstanceUid: this.seriesViews.value[event.seriesContainerIndex].series.seriesAttributes.seriesUid,
                                    seriesContainerIndex: event.seriesContainerIndex,
                                    event: event
                                }, timestamp);
                            }

                            var series = _.find(_.flatten(_.map(this.studies, function (s) {
                                return s.series;
                            })), function (s) {
                                return s.seriesAttributes.seriesUid.value === event.seriesInstanceUid.value;
                            });
                            this.loadSeriesAt(series, event.seriesContainerIndex);
                            this.seriesViews.raiseChangedEvent(this.seriesViews.value);
                        }
                        break;
                    case 6 /* SelectedImageChanged */:
                         {
                            if (seriesView) {
                                if (reverseEvent) {
                                    reverseEvent.appendWithTimestamp({
                                        type: 6 /* SelectedImageChanged */,
                                        instanceIndex: seriesView.selectedInstanceIndex.read(),
                                        seriesContainerIndex: event.seriesContainerIndex,
                                        event: event
                                    }, timestamp);
                                }
                                seriesView.selectedInstanceIndex.write(event.instanceIndex);
                            }
                        }
                        break;
                    case 9 /* InvertChanged */:
                         {
                            if (seriesView) {
                                if (reverseEvent) {
                                    reverseEvent.appendWithTimestamp({
                                        type: 9 /* InvertChanged */,
                                        inverted: seriesView.invertActive.read(),
                                        seriesContainerIndex: event.seriesContainerIndex,
                                        event: event
                                    }, timestamp);
                                }

                                seriesView.invertActive.write(event.inverted);
                            }
                        }
                        break;
                    case 15 /* EnhanceChanged */:
                         {
                            if (seriesView) {
                                if (reverseEvent) {
                                    reverseEvent.appendWithTimestamp({
                                        type: 15 /* EnhanceChanged */,
                                        enhance: seriesView.enhance.read(),
                                        seriesContainerIndex: event.seriesContainerIndex,
                                        event: event
                                    }, timestamp);
                                }
                                seriesView.enhance.write(event.enhance);
                            }
                        }
                        break;
                    case 11 /* ToggleMeasurementsChanged */:
                         {
                            if (seriesView) {
                                if (reverseEvent) {
                                    reverseEvent.appendWithTimestamp({
                                        type: 11 /* ToggleMeasurementsChanged */,
                                        showMeasurements: seriesView.measurementsVisible.read(),
                                        seriesContainerIndex: event.seriesContainerIndex,
                                        event: event
                                    }, timestamp);
                                }

                                seriesView.measurementsVisible.write(event.showMeasurements);
                            }
                        }
                        break;
                    case 10 /* ToggleTextAnnotationsChanged */:
                         {
                            if (seriesView) {
                                if (reverseEvent) {
                                    reverseEvent.appendWithTimestamp({
                                        type: 10 /* ToggleTextAnnotationsChanged */,
                                        showTextAnnotations: seriesView.infoVisible.read(),
                                        seriesContainerIndex: event.seriesContainerIndex,
                                        event: event
                                    }, timestamp);
                                }

                                seriesView.infoVisible.write(event.showTextAnnotations);
                            }
                        }
                        break;
                    case 12 /* MagnifierPositionChanged */:
                         {
                            if (seriesView) {
                                if (reverseEvent) {
                                    reverseEvent.appendWithTimestamp({
                                        type: 12 /* MagnifierPositionChanged */,
                                        magnifierPosition: seriesView.magnifier.read(),
                                        seriesContainerIndex: event.seriesContainerIndex,
                                        event: event
                                    }, timestamp);
                                }

                                seriesView.magnifier.write(event.magnifierPosition);
                            }
                        }
                        break;
                    case 13 /* ProbeToolChanged */:
                         {
                            if (seriesView) {
                                if (reverseEvent) {
                                    reverseEvent.appendWithTimestamp({
                                        type: 13 /* ProbeToolChanged */,
                                        probeTool: seriesView.probeTool.read(),
                                        seriesContainerIndex: event.seriesContainerIndex,
                                        event: event
                                    }, timestamp);
                                }
                                seriesView.probeTool.write(event.probeTool);
                            }
                        }
                        break;
                    case 14 /* PlaneLocalizationChanged */:
                         {
                            if (reverseEvent) {
                                reverseEvent.appendWithTimestamp({
                                    type: 14 /* PlaneLocalizationChanged */,
                                    planeLocalizationData: this.planeLocalizationCursor.read(),
                                    seriesContainerIndex: event.seriesContainerIndex,
                                    event: event
                                }, timestamp);
                            }
                            this.planeLocalizationCursor.write(event.planeLocalizationData);
                        }
                        break;
                }
            } catch (err) {
                if (err instanceof Error) {
                    this.recordError("Error while handling event: " + err.message);
                }
            }
        };

        /**
        * Play back a recorded script
        */
        Application.prototype.playScript = function (script) {
            var _this = this;
            this.playbackMode.write(true);

            this.render();
            this.showPlaybackBanner(this.terminology.lookup(Terminology.Terms.PlaybackInProgress));

            _.each(script.series, function (s, index, context) {
                var series = _.find(_.flatten(_.map(_this.studies, function (s) {
                    return s.series;
                })), function (series) {
                    return series.seriesAttributes.seriesUid.value === s.seriesUid;
                });
                var seriesView = _this.loadSeriesAt(series, index);
                seriesView.selectedInstanceIndex.write(s.instanceIndex);
            });

            this.layout.write(script.layout);

            this.playbackAudio = new Audio(script.audio);
            $(this.playbackAudio).hide().appendTo($(document.body));

            this.playbackAudio.addEventListener("ended", function () {
                _this.playbackStopped();
            });

            var playPromise = this.playbackAudio.play();

            playPromise.then(function (_) {
                var reverseRecorder = new Recording.ArrayScriptRecorder();
                _this.reversiblePlayback = { script: script, recorder: reverseRecorder };

                Recording.play(script.events, function (event, timestamp) {
                    _this.handleEvent(event, timestamp, reverseRecorder);
                }, _this.playbackAudio, _this.reversiblePlayback);
            });
        };

        Application.prototype.rewindPlayback = function () {
            var _this = this;
            var current = this.playbackAudio.currentTime;
            var newTime = current - 5;

            if (current < 5) {
                newTime = 0;
            }

            this.playbackAudio.pause();

            var eventsToReverse = this.reversiblePlayback.recorder.getEvents();

            if (eventsToReverse) {
                for (var ctr = eventsToReverse.length - 1; ctr >= 0; ctr -= 1) {
                    if (eventsToReverse[ctr].timestamp > (newTime * 1000)) {
                        this.handleEvent(eventsToReverse[ctr].event);
                    }
                }
            }

            this.playbackAudio.currentTime = newTime;
            var playPromise = this.playbackAudio.play();

            playPromise.then(function (_) {
                Recording.play(_this.reversiblePlayback.script.events, function (event, timestamp) {
                    _this.handleEvent(event, timestamp, _this.reversiblePlayback.recorder);
                }, _this.playbackAudio, _this.reversiblePlayback, newTime);
            });
        };

        Application.prototype.fastForwardPlayback = function () {
            var newTime = this.playbackAudio.currentTime + 5;
            if (newTime < this.playbackAudio.duration) {
                this.playbackAudio.currentTime = newTime;
            } else {
                this.playbackStopped();
            }
        };

        Application.prototype.playbackChanged = function (playing) {
            if (this.playbackAudio) {
                if (playing) {
                    this.playbackAudio.play();
                } else {
                    this.playbackAudio.pause();
                }
            }

            this.toolbar.render();
        };

        Application.prototype.playbackStopped = function () {
            $(this.playbackAudio).remove();
            this.hidePlaybackBanner();
            this.removeTemporaryAnnotations();
            this.playbackMode.write(false);
            this.resetSeriesViews();
            this.render();
            this.selectedSeriesKey.write(this.seriesViews.value[0].viewKey);
        };

        Application.prototype.showGlassPane = function (playback) {
            if (typeof playback === "undefined") { playback = false; }
            $('<div>').addClass('overlay' + (playback ? " playback" : "")).addClass('glasspane').appendTo($(document.body));
        };

        Application.prototype.hideGlassPane = function () {
            $('.overlay').remove();
        };

        Application.prototype.showMeetingGlassPane = function () {
            $('.applicationFrame').addClass('toolbar-hidden');

            $('<div>').addClass('overlay').addClass('glasspane').appendTo($('.study-series'));
        };

        Application.prototype.hideMeetingGlassPane = function () {
            $('.applicationFrame').removeClass('toolbar-hidden');
            $('.study-series > .overlay').remove();
        };

        Application.prototype.showPlaybackBanner = function (message) {
            this.hidePlaybackBanner();
            this.showGlassPane(true);
            $(".toolbar-scroll").css({ width: '275px', margin: "auto" });
            this.showWarningBanner(message);
        };

        Application.prototype.hidePlaybackBanner = function () {
            this.hideGlassPane();
            $(".toolbar-scroll").css({ width: '', margin: "" });
            this.hideWarningBanner();
        };

        Application.prototype.showRecordingBanner = function (message) {
            this.showWarningBanner(message);
        };

        Application.prototype.hideRecordingBanner = function () {
            this.hideWarningBanner();
        };

        Application.prototype.showWarningBanner = function (message) {
            if ($("#series-warning-banner").length) {
                $('#series-warning-banner').remove();
            }

            $('<div>').attr('id', 'series-warning-banner').addClass('banner').append(message).prependTo(".applicationFrame-inner");

            $('.study-series').css({ top: '12px' });
            var $body = $("body");

            if ($body.is(".thumbs-on-left") || $body.is(".thumbs-on-top")) {
                $('.thumbs').css({ top: '12px' });
            }

            this.renderAllFrames();
        };

        Application.prototype.updateWarningBanner = function (message) {
            $('#series-warning-banner').html(message);
        };

        Application.prototype.hideWarningBanner = function () {
            $('#series-warning-banner').remove();
            $('.study-series').css({ top: '0' });
            var $body = $("body");

            if ($body.is(".thumbs-on-left") || $body.is(".thumbs-on-top")) {
                $('.thumbs').css({ top: '0' });
            }

            this.renderAllFrames();
        };

        /**
        * Start sending events to the recorder
        */
        Application.prototype.startRecording = function () {
            var _this = this;
            if (this.$applicationFrameInner.hasClass("withrecording")) {
                this.toggleRecordingsInfo();
            }

            Recording.startAudioRecording(function (audioRecorder) {
                _this.showRecordingBanner(_this.terminology.lookup(Terminology.Terms.RecordingInProgress));

                var layout = _this.layout.read();
                var series = _.map(_this.seriesViews.value, function (s) {
                    return {
                        seriesUid: s.series ? s.series.seriesAttributes.seriesUid.value : null,
                        instanceIndex: s.selectedInstanceIndex.read()
                    };
                });

                var recorder = new Recording.ArrayScriptRecorder();
                recorder.setZeroTimestamp(new Date().getTime());
                _this.recorder.write(recorder);

                var stopRecordingSubscription = Subjects.listen(_this.recordingMode, function (recording) {
                    if (!recording) {
                        audioRecorder.stop();

                        var $overlay = $('.overlay');
                        $overlay.addClass('application-loading');

                        var exportData = Observable._finally(Observable.timeout(Recording.exportWaveAsBase64(audioRecorder), 5000), function () {
                            _this.hideRecordingBanner();
                            _this.removeTemporaryAnnotations();
                            stopRecordingSubscription.cancel();
                        });

                        var recordingName = window.prompt(_this.terminology.lookup(Terminology.Terms.RecordingName), _this.terminology.lookup(Terminology.Terms.RecordingUntitled));

                        if (recordingName) {
                            var exportAndUploadScript = Observable._finally(Observable.bind(exportData, function (base64Data) {
                                var script = {
                                    layout: layout,
                                    series: series,
                                    audio: base64Data,
                                    events: _this.recorder.read().getEvents()
                                };

                                return V3Storage.PostAttachment(_this.sessionId, _this.studyStorage, _this.queryObject, Recording.MIME_TYPE, _this.user.name, recordingName + ".json", script);
                            }), function () {
                                _this.recorder.write(new Recording.NullRecorder());
                                $overlay.removeClass('application-loading');
                            });

                            exportAndUploadScript.subscribe({
                                done: function () {
                                },
                                next: function (attachmentResult) {
                                    var attachment = {
                                        id: new Classes.AttachmentID(attachmentResult.attachment_guid),
                                        stored: 0,
                                        mime: Recording.MIME_TYPE,
                                        version: new Classes.ImageVersion(attachmentResult.attachment_version),
                                        phiNamespace: new Classes.PhiNamespace(attachmentResult.namespace),
                                        filename: recordingName + ".json"
                                    };

                                    _this.studies[0].attachments.push(attachment);
                                    window.alert(_this.terminology.lookup(Terminology.Terms.ScriptSaved));
                                },
                                fail: function (err) {
                                    _this.recordError("Unable to save recorded script: " + err);
                                    window.alert(_this.terminology.lookup(Terminology.Terms.ScriptSaveError));
                                }
                            });
                        } else {
                            _this.hideRecordingBanner();
                            _this.removeTemporaryAnnotations();
                            stopRecordingSubscription.cancel();
                            _this.recorder.write(new Recording.NullRecorder());
                            $overlay.removeClass('application-loading');
                        }
                    }
                });
            }, function (err) {
                window.alert(_this.terminology.lookup(Terminology.Terms.AudioRecordingNotSupported));
                _this.recordingMode.write(false);
            });
        };

        Application.prototype.toggleRecordingsInfo = function () {
            var isShowing = this.$applicationFrameInner.hasClass("withrecording");

            if (isShowing) {
                this.updateRecordingsInfoVisibility(false);
            } else {
                this.recordingsInfo.render(this.studies[0], this.studyStorage, this.queryObject);
                this.updateRecordingsInfoVisibility(true);
            }

            this.renderAllFrames();
        };

        Application.prototype.toggleAttachmentInfo = function () {
            var isShowing = this.$applicationFrameInner.hasClass("withattachments");

            if (isShowing) {
                this.updateAttachmentInfoVisibility(false);
            } else {
                this.attachmentInfo.render(this.studies[0], this.studyStorage, this.queryObject);
                this.updateAttachmentInfoVisibility(true);
            }

            this.renderAllFrames();
        };

        Application.prototype.toggleGSPSInfo = function () {
            var isShowing = this.$applicationFrameInner.hasClass("withgsps");

            if (isShowing) {
                this.updateGSPSInfoVisibility(false);
            } else {
                this.updateGSPSInfoVisibility(true);
                this.gspsInfo.render();
            }

            this.renderAllFrames();
        };

        /**
        * Become the meeting presenter for a meeting
        */
        Application.prototype.becomeMeetingPresenter = function (meetingId) {
            var _this = this;
            this.inMeeting.write(false);
            this.meetingHost.write(true);

            this.recorder.write(new Meeting.MeetingRecorder(this.sessionId, meetingId, function () {
                return _this.getLatestMeetingState();
            }));
        };

        /**
        * Become a regular meeting attendee
        */
        Application.prototype.becomeMeetingAttendee = function () {
            this.inMeeting.write(true);
            this.meetingHost.write(false);

            this.recorder.write(new Recording.NullRecorder());
        };

        /**
        * Apply the current meeting state to the application
        */
        Application.prototype.applyMeetingState = function (state) {
            var _this = this;
            _.each(state.series, function (s, index, context) {
                var series = null;

                if (s !== null) {
                    series = _.find(_.flatten(_.map(_this.studies, function (s) {
                        return s.series;
                    })), function (series) {
                        return series.seriesAttributes.seriesUid.value === s.seriesInstanceUid && _.any(series.instances, function (i) {
                            return i.id.value == s.instanceUid;
                        });
                    });
                }

                var seriesView = _this.loadSeriesAt(series, index);

                if (series) {
                    var instanceIndex = series.instances.indexWhere(function (i) {
                        return i.id.value === s.instanceUid && i.frameNumber.value === s.frameNumber;
                    });

                    seriesView.selectedInstanceIndex.write(instanceIndex);
                    seriesView.applyDisplayOptions(s.displayOptions);
                }
            });

            this.seriesViews.raiseChangedEvent(this.seriesViews.read());
            this.layout.write(state.layout);
        };

        Application.prototype.applyKeyImageLayoutState = function (state) {
            var _this = this;
            var original = this.seriesViews.read();
            var ordered = new Array(original.length);
            var empties = _.filter(original, function (s) {
                return s.series == null;
            });

            _.each(state.series, function (s, index) {
                var series = null;
                var seriesView = null;

                if (s !== null) {
                    series = _.find(_.flatten(_.map(_this.studies, function (s) {
                        return s.keyImageSeries;
                    })), function (series) {
                        return _.any(series.instances, function (i) {
                            return i.id.value == s.instanceUid;
                        });
                    });

                    if (series) {
                        seriesView = _this.findSeriesView(series, ordered);

                        if (seriesView) {
                            ordered[index] = seriesView;
                            seriesView.applyDisplayOptions(s.displayOptions);
                        }
                    }
                }
            });

            for (var ctr = 0; ctr < ordered.length; ctr += 1) {
                if (ordered[ctr] == null) {
                    if (empties.length) {
                        ordered[ctr] = empties.pop();
                    } else {
                        var $series = $('<div>').addClass('series');
                        ordered[ctr] = new Views.Series(this, this.sessionId, $series, null, this.settings, this.accountSettings, this.selectedSeriesKey, this.selectedTool, this.selectedTool2, this.selectedToolWheel, this.selectedInstanceGeometry, this.referenceLinesActive, this.planeLocalizationCursor, this.linkSeries, this.imagePreloadQueue, this.terminology, this.cineSpeed, this.ignoreFrameTimeTag, this.startCinePlaybackAutomatically, this.noKeyboardShortcuts, Subjects.map(this.recorder, function (r) {
                            return _this.modifyRecorder(r, ctr);
                        }), this.playbackMode);
                    }
                }
            }

            this.seriesViews.write(ordered);
            this.layout.write(state.layout);
        };

        /**
        * Enter meeting mode - start the ping to the server, and listen for events
        */
        Application.prototype.enterMeetingMode = function (meetingId) {
            var _this = this;
            var ping = this.startMeetingPing(meetingId).subscribe({
                next: function (_) {
                },
                fail: function (_) {
                },
                done: function () {
                }
            });

            var eventStream = Observable.join(Messaging.channel(Messaging.meeting(this.sessionId, meetingId)));

            var eventSubscription = eventStream.subscribe({
                done: function () {
                },
                fail: function (err) {
                    window.alert(_this.terminology.lookup(Terminology.Terms.CannotJoinMeeting));
                    _this.recordError("Unable to join meeting: " + err);
                },
                next: function (event) {
                    switch (event.event) {
                        case "EVENT":
                             {
                                if (!_this.meetingHost.read()) {
                                    var eventData = JSON.parse(event.data);
                                    _this.handleEvent(eventData);
                                }
                            }
                            break;
                        case "FINISHED":
                             {
                                _this.leaveMeetingMode();
                                window.alert(_this.terminology.lookup(Terminology.Terms.MeetingEnded));
                            }
                            break;
                        case "JOIN":
                        case "LEAVE":
                             {
                                _this.meetingInfo.updateRoster(_this.sessionId, meetingId);
                            }
                            break;
                        case "NEW_PRESENTER":
                             {
                                var presenterData = event.data;

                                if (presenterData.uuid === _this.user.uuid) {
                                    window.alert(_this.terminology.lookup(Terminology.Terms.YouAreThePresenter));
                                    _this.becomeMeetingPresenter(meetingId);
                                }

                                _this.meetingInfo.updateRoster(_this.sessionId, meetingId);
                            }
                            break;
                    }
                }
            });

            return Observable.mappendSubscription(ping, eventSubscription);
        };

        /**
        * Leave meeting mode
        */
        Application.prototype.leaveMeetingMode = function () {
            this.meetingHost.write(false);
            this.inMeeting.write(false);

            if (this.meetingSubscription) {
                this.meetingSubscription.cancel();
                this.meetingSubscription = null;
            }
        };

        /**
        * Join a meeting which is in progress
        */
        Application.prototype.joinMeeting = function (meetingId) {
            var _this = this;
            var joinAndGetMeetingInfo = Observable.zip(Services.JoinMeeting(this.sessionId, meetingId), Services.GetMeeting(this.sessionId, meetingId), function (meeting, roster) {
                return {
                    meeting: meeting,
                    roster: roster.users
                };
            });

            var $overlay = $('.overlay');

            $overlay.addClass('application-loading');

            Observable._finally(joinAndGetMeetingInfo, function () {
                $overlay.removeClass('application-loading');
            }).subscribe({
                done: function () {
                },
                fail: function (err) {
                    _this.recordError("Unable to join meeting: " + err);
                    window.alert(_this.terminology.lookup(Terminology.Terms.CannotJoinMeeting));
                },
                next: function (meeting) {
                    _this.inMeeting.write(true);
                    _this.meetingHost.write(false);

                    _this.meetingInfo.render(_this.queryObject, _this.studies[0].studyAttributes.uuid, meetingId);

                    var state = JSON.parse(meeting.meeting.state);

                    _this.applyMeetingState(state);

                    _this.meetingSubscription = _this.enterMeetingMode(meetingId);
                }
            });
        };

        /**
        * Start a new meeting
        */
        Application.prototype.startMeeting = function () {
            var _this = this;
            var meetingName = window.prompt(this.terminology.lookup(Terminology.Terms.EnterMeetingName), this.terminology.lookup(Terminology.Terms.DefaultMeetingName));

            if (meetingName) {
                var initialState = this.getLatestMeetingState();

                Services.StartMeeting(this.sessionId, this.queryObject, meetingName, initialState).subscribe({
                    done: function () {
                    },
                    fail: function (err) {
                        window.alert(_this.terminology.lookup(Terminology.Terms.CannotStartMeeting));
                        _this.recordError("Unable to start meeting: " + err);
                    },
                    next: function (meeting) {
                        var meetingId = new Classes.MeetingId(meeting.uuid);

                        _this.meetingInfo.render(_this.queryObject, _this.studies[0].studyAttributes.uuid, meetingId, meetingName);

                        _this.meetingSubscription = _this.enterMeetingMode(meetingId);

                        _this.becomeMeetingPresenter(meetingId);
                    }
                });
            }
        };

        /**
        * End the current meeting
        */
        Application.prototype.endMeeting = function (meetingId) {
            var _this = this;
            var $overlay = $('.overlay');

            $overlay.addClass('application-loading');

            this.leaveMeetingMode();

            var closeMeeting = Observable._finally(Services.EndMeeting(this.sessionId, meetingId), function () {
                _this.recorder.write(new Recording.NullRecorder());
                $overlay.removeClass('application-loading');
            });

            closeMeeting.subscribe({
                done: function () {
                },
                next: function (_) {
                },
                fail: function (err) {
                    _this.recordError("Unable to end meeting: " + err);
                    window.alert(_this.terminology.lookup(Terminology.Terms.CannotEndMeeting));
                }
            });
        };

        /**
        * Leave the current meeting
        */
        Application.prototype.leaveMeeting = function (meetingId) {
            var _this = this;
            var $overlay = $('.overlay');

            $overlay.addClass('application-loading');

            var leaveMeeting = Observable._finally(Services.LeaveMeeting(this.sessionId, meetingId), function () {
                _this.leaveMeetingMode();

                $overlay.removeClass('application-loading');
            });

            leaveMeeting.subscribe({
                done: function () {
                },
                next: function (_) {
                },
                fail: function (err) {
                    _this.recordError("Unable to leave meeting: " + err);
                    window.alert(_this.terminology.lookup(Terminology.Terms.CannotLeaveMeeting));
                }
            });
        };

        /**
        * Change the presenter to the selected user
        */
        Application.prototype.changeMeetingPresenter = function (meetingId, userUuid) {
            var _this = this;
            if (window.confirm(this.terminology.lookup(Terminology.Terms.ConfirmChangeOfPresenter))) {
                var $overlay = $('.overlay');

                $overlay.addClass('application-loading');

                var changePresenter = Observable._finally(Services.ChangeMeetingPresenter(this.sessionId, meetingId, userUuid), function () {
                    $overlay.removeClass('application-loading');
                });

                changePresenter.subscribe({
                    done: function () {
                    },
                    next: function (_) {
                        _this.becomeMeetingAttendee();
                    },
                    fail: function (err) {
                        _this.recordError("Unable to change the meeting presenter: " + err);
                        window.alert(_this.terminology.lookup(Terminology.Terms.CannotChangePresenter));
                    }
                });
            }
        };

        /**
        * Start a regular ping to the server to keep a meeting alive
        */
        Application.prototype.startMeetingPing = function (meetingId) {
            var _this = this;
            return Observable.bind(Observable.interval(10000), function (_) {
                return Observable.catchError(Services.PingMeeting(_this.sessionId, meetingId), function (err) {
                    _this.recordError("Unable to ping meeting: " + err);
                    return {};
                });
            });
        };

        /**
        * Get the meeting state
        */
        Application.prototype.getLatestMeetingState = function () {
            return {
                layout: this.layout.read(),
                series: _.map(this.seriesViews.value, function (s) {
                    if (s.series) {
                        var instance = s.series.instances[s.selectedInstanceIndex.read()];

                        return {
                            seriesInstanceUid: instance.seriesAttributes.seriesUid.value,
                            instanceUid: instance.id.value,
                            frameNumber: instance.frameNumber.value,
                            studyUUID: instance.studyAttributes.uuid.value,
                            displayOptions: s.getDisplayOptions()
                        };
                    } else {
                        return null;
                    }
                })
            };
        };

        Application.prototype.getKeyImageLayoutState = function () {
            return {
                layout: this.layout.read(),
                series: _.map(this.seriesViews.value, function (s) {
                    if (s.series) {
                        var instance = s.series.instances[s.selectedInstanceIndex.read()];

                        return {
                            seriesInstanceUid: instance.seriesAttributes.seriesUid.value,
                            instanceUid: instance.id.value,
                            frameNumber: instance.frameNumber.value,
                            studyUUID: instance.studyAttributes.uuid.value,
                            displayOptions: s.getExtendedDisplayOptions()
                        };
                    } else {
                        return null;
                    }
                })
            };
        };

        Application.prototype.saveKeyImageLayoutState = function () {
            var _this = this;
            var current = this.getKeyImageLayoutState();
            current["created"] = new Date().getTime();
            current["patientId"] = this.studies[0].studyAttributes.patientId.value;

            var keyImageLayouts = _.filter(this.keyImageLayout, function (k) {
                return current.patientId != k.patientId;
            });
            keyImageLayouts.push(current);

            var now = (new Date()).getTime();
            keyImageLayouts = _.filter(keyImageLayouts, function (k) {
                return (now - k.created) < 5256000000;
            }); // filter data older than two months

            Services.putKeyImageSettings(this.sessionId, keyImageLayouts).subscribe({
                done: function () {
                },
                next: function (_) {
                    window.alert(_this.terminology.lookup(Terminology.Terms.SaveKeyImageLayoutSuccess));
                },
                fail: function (err) {
                    window.alert(_this.terminology.lookup(Terminology.Terms.SaveKeyImageLayoutError));
                }
            });
        };

        Application.prototype.clearKeyImageLayoutState = function () {
            var _this = this;
            var patientId = this.studies[0].studyAttributes.patientId.value;

            var foundLayout = _.find(this.keyImageLayout, function (k) {
                return patientId == k.patientId;
            });

            if (foundLayout) {
                this.keyImageLayout = _.filter(this.keyImageLayout, function (k) {
                    return patientId != k.patientId;
                });
                this.render();

                Services.putKeyImageSettings(this.sessionId, this.keyImageLayout).subscribe({
                    done: function () {
                    },
                    next: function (_) {
                        window.alert(_this.terminology.lookup(Terminology.Terms.ClearedKeyImageLayout));
                    },
                    fail: function (err) {
                        console.error("Unable to save key image settings to services: " + err);
                    }
                });
            }
        };

        /**
        * Anonymize the study
        */
        Application.prototype.anonymize = function (level) {
            var _this = this;
            var regionData = {};

            var series = this.selectedSeriesContainer.read().series;
            var studyUid = series.studyAttributes.queryObject.studyUid;

            var study = _.find(this.studies, function (s) {
                return s.studyAttributes.queryObject.studyUid.value === studyUid.value;
            });

            if (!study) {
                throw new Error("Unable to determine study to anonymize.");
            }

            var measurementsToRegions = function (ms) {
                return _.chain(ms).map(function (m) {
                    return m.toAnnotationData();
                }).filter(function (a) {
                    return a.type === Classes.MouseTool[6 /* Rectangle */];
                }).map(function (a) {
                    var minX = Math.min(a.points[0].x, a.points[1].x);
                    var minY = Math.min(a.points[0].y, a.points[1].y);
                    var maxX = Math.max(a.points[0].x, a.points[1].x);
                    var maxY = Math.max(a.points[0].y, a.points[1].y);

                    return {
                        x: minX,
                        y: minY,
                        width: maxX - minX,
                        height: maxY - minY
                    };
                }).value();
            };

            var haveRegions = false;

            switch (level) {
                case 0 /* Study */:
                     {
                        regionData.regions = measurementsToRegions(_.chain(series.instances).map(function (i) {
                            return i.instanceAttributes.measurements;
                        }).flatten().value());

                        if (_.any(regionData.regions)) {
                            haveRegions = true;
                        }
                    }
                    break;
                case 1 /* Series */:
                     {
                        var regions = {};

                        _.each(study.originalSeries, function (s) {
                            var seriesRegions = {};

                            if (Multiframe.isMultiframe(s)) {
                                var instanceRegions = {};

                                _.each(s.instances, function (inst) {
                                    var newSeries = _.find(study.series, function (s) {
                                        return s.seriesAttributes.seriesUid.value === inst.seriesAttributes.seriesUid.value && _.any(s.instances, function (i) {
                                            return i.id.value === inst.id.value;
                                        });
                                    });

                                    var instanceRegionData = {};
                                    instanceRegionData.regions = measurementsToRegions(_.chain(newSeries.instances).map(function (i) {
                                        return i.instanceAttributes.measurements;
                                    }).flatten().value());

                                    if (_.any(instanceRegionData.regions)) {
                                        haveRegions = true;
                                    }

                                    instanceRegions[inst.id.value] = instanceRegionData;
                                });

                                seriesRegions.instances = instanceRegions;
                            } else {
                                seriesRegions.regions = measurementsToRegions(_.chain(s.instances).map(function (i) {
                                    return i.instanceAttributes.measurements;
                                }).flatten().value());

                                if (_.any(seriesRegions.regions)) {
                                    haveRegions = true;
                                }
                            }

                            regions[s.seriesAttributes.seriesUid.value] = seriesRegions;
                        });

                        regionData.series = regions;
                    }
                    break;
                case 2 /* Image */:
                     {
                        var regions = {};

                        var selectedInstance = this.selectedInstance().read().instance;

                        _.each(study.originalSeries, function (s) {
                            var instanceRegions = {};

                            _.each(s.instances, function (inst) {
                                var measurements = _.chain(study.series).filter(function (s) {
                                    return s.seriesAttributes.seriesUid.value === inst.seriesAttributes.seriesUid.value;
                                }).filter(function (s) {
                                    return s.seriesAttributes.seriesUid.value === selectedInstance.seriesAttributes.seriesUid.value;
                                }).map(function (s) {
                                    return s.instances;
                                }).flatten().filter(function (i) {
                                    return i.id.value === inst.id.value;
                                }).filter(function (i) {
                                    return i.id.value === selectedInstance.id.value;
                                }).map(function (i) {
                                    return i.instanceAttributes.measurements;
                                }).flatten().value();

                                var instanceRegionData = {
                                    regions: measurementsToRegions(measurements)
                                };

                                if (_.any(instanceRegionData.regions)) {
                                    haveRegions = true;
                                }

                                instanceRegions[inst.id.value] = instanceRegionData;
                            });

                            regions[s.seriesAttributes.seriesUid.value] = {
                                instances: instanceRegions
                            };
                        });

                        regionData.series = regions;
                    }
                    break;
            }

            if (!haveRegions) {
                window.alert(this.terminology.lookup(Terminology.Terms.CannotAnonymize));
            } else {
                if (window.confirm(this.terminology.lookup(Terminology.Terms.AnonymizeWarning))) {
                    var anonymize = Modify.anonymize(this.sessionId, study, regionData);

                    anonymize.subscribe({
                        next: function (_) {
                            window.alert(_this.terminology.lookup(Terminology.Terms.AnonymizeSuccess));
                        },
                        done: function () {
                        },
                        fail: function (err) {
                            _this.recordError("Unable to anonymize study: " + err);
                            window.alert(_this.terminology.lookup(Terminology.Terms.AnonymizeFailed));
                        }
                    });
                }
            }
        };

        Application.prototype.splitStudy = function (study, seriesUIDs) {
            var _this = this;
            if (seriesUIDs && seriesUIDs.length) {
                if (study.series.length == seriesUIDs.length) {
                    window.alert(this.terminology.lookup(Terminology.Terms.SplitWarningAllSeries));
                } else if (window.confirm(this.terminology.lookup(Terminology.Terms.SplitWarning))) {
                    var anonymize = Modify.split(this.sessionId, study, seriesUIDs);

                    anonymize.subscribe({
                        next: function (_) {
                            _this.splitStudyEnd(seriesUIDs);
                            window.alert(_this.terminology.lookup(Terminology.Terms.SplitSuccess));
                        },
                        done: function () {
                        },
                        fail: function (err) {
                            _this.recordError("Unable to split study: " + err);
                            window.alert(_this.terminology.lookup(Terminology.Terms.SplitFailed));
                        }
                    });
                }
            } else {
                window.alert(this.terminology.lookup(Terminology.Terms.SplitWarningNoSeries));
            }
        };

        Application.prototype.cropStudy = function () {
            var _this = this;
            var regionData = {};
            regionData.series = {};
            var seriesDescriptions = "\n";

            var foundData = false;
            var selectedSeriesView = this.selectedSeriesContainer.read();
            var selectedSeries = selectedSeriesView.series;
            var studyUid = selectedSeries.studyAttributes.queryObject.studyUid;

            var study = _.find(this.studies, function (s) {
                return s.studyAttributes.queryObject.studyUid.value === studyUid.value;
            });

            if (!study) {
                throw new Error("Unable to determine study to crop.");
            }

            for (var i = 0; i < this.seriesViews.value.length; i++) {
                var view = this.seriesViews.value[i];
                if (view.series) {
                    if (view.series.studyAttributes.uuid.value === selectedSeries.studyAttributes.uuid.value) {
                        var selectedMeasurement = view.selectedMeasurement.read();
                        if (selectedMeasurement) {
                            var a = selectedMeasurement.toAnnotationData();
                            if (a.type === Classes.MouseTool[6 /* Rectangle */]) {
                                foundData = true;

                                seriesDescriptions += ("\n" + view.series.instances[0].instanceAttributes.seriesDescription);

                                var minX = Math.min(a.points[0].x, a.points[1].x);
                                var minY = Math.min(a.points[0].y, a.points[1].y);
                                var maxX = Math.max(a.points[0].x, a.points[1].x);
                                var maxY = Math.max(a.points[0].y, a.points[1].y);

                                regionData.series[view.series.seriesAttributes.seriesUid.value] = {
                                    region: {
                                        x: minX,
                                        y: minY,
                                        width: maxX - minX,
                                        height: maxY - minY
                                    }
                                };
                            }
                        }
                    }
                }
            }

            if (foundData) {
                if (window.confirm(this.terminology.lookup(Terminology.Terms.CropWarning) + seriesDescriptions)) {
                    var crop = Modify.crop(this.sessionId, study, regionData);

                    crop.subscribe({
                        next: function (_) {
                            window.alert(_this.terminology.lookup(Terminology.Terms.CropSuccess));
                        },
                        done: function () {
                        },
                        fail: function (err) {
                            _this.recordError("Unable to anonymize study: " + err);
                            window.alert(_this.terminology.lookup(Terminology.Terms.CropFailed));
                        }
                    });
                }
            } else {
                window.alert(this.terminology.lookup(Terminology.Terms.CannotCrop));
            }
        };

        Application.prototype.splitStudyEnd = function (seriesUIDs) {
            this.studies[0].series = _.difference(this.studies[0].series, _.filter(this.studies[0].series, function (series) {
                return _.contains(seriesUIDs, series.seriesAttributes.seriesUid.value);
            }));
            this.resetSeriesViews();
            this.render();
        };

        Application.prototype.reverseSeries = function () {
            var _this = this;
            var selectedSeriesView = this.selectedSeriesContainer.read();
            var selectedSeries = selectedSeriesView.series;
            selectedSeries.instances = selectedSeries.instances.reverse();

            var studyUid = selectedSeries.studyAttributes.queryObject.studyUid;

            var study = _.find(this.studies, function (s) {
                return s.studyAttributes.queryObject.studyUid.value === studyUid.value;
            });

            if (study) {
                var seriesUID = selectedSeries.seriesAttributes.seriesUid.value;
                var reordered = [];

                _.each(selectedSeries.instances, function (i, index) {
                    i.instanceAttributes.instanceIndex = index;
                    i.instanceAttributes.instanceNumber = index;
                    reordered.push(i.id.value);
                });

                var extended = { series: {} };
                extended.series[seriesUID] = {
                    image_order: reordered
                };

                Services.StudyPHIExtended(this.sessionId, study.studyAttributes.uuid, JSON.stringify(extended)).subscribe({
                    done: function () {
                    },
                    next: function (_) {
                        _this.resetSeriesViews();
                        _this.render();
                        _this.replaceSelectedSeries(selectedSeries);

                        window.alert(_this.terminology.lookup(Terminology.Terms.StudyPHIExtendedSuccess));
                    },
                    fail: function (err) {
                        window.alert(_this.terminology.lookup(Terminology.Terms.StudyPHIExtendedError));
                        console.error("Unable to reverse series: " + err);
                    }
                });
            }
        };

        Application.prototype.unweaveSeries = function () {
            var _this = this;
            var selectedSeriesView = this.selectedSeriesContainer.read();
            var selectedSeries = selectedSeriesView.series;
            var studyUid = selectedSeries.studyAttributes.queryObject.studyUid;

            var study = _.find(this.studies, function (s) {
                return s.studyAttributes.queryObject.studyUid.value === studyUid.value;
            });

            if (study) {
                var uidA = this.generateUID();
                var uidB = this.generateUID();
                var instancesA = [];
                var instancesB = [];
                var images = {};

                var extended = { study: { series_order: [uidA, uidB] } };

                _.each(selectedSeries.instances, function (i, index) {
                    if (index % 2 == 0) {
                        instancesA.push(i);
                    } else {
                        instancesB.push(i);
                    }
                });

                var seriesAttributesA = Multiframe.copySeriesAttributes(selectedSeries.seriesAttributes);
                seriesAttributesA.instanceCount = instancesA.length;
                var seriesA = {
                    uuid: uidA,
                    instances: _.map(instancesA, function (i, index) {
                        var copy = new Models.Instance();
                        copy.frameNumber = new Classes.FrameNumber(0);
                        copy.id = i.id;
                        copy.instanceAttributes = Multiframe.copyInstanceAttributes(i.instanceAttributes);
                        copy.instanceAttributes.measurements = [];
                        copy.instanceAttributes.instanceIndex = index;
                        copy.instanceAttributes.instanceNumber = index;
                        copy.seriesAttributes = seriesAttributesA;
                        copy.studyAttributes = study.studyAttributes;
                        images[i.id.value] = { "(0020,000E)": uidA };
                        return copy;
                    }),
                    seriesAttributes: seriesAttributesA,
                    studyAttributes: study.studyAttributes,
                    loadedStatus: 0 /* None */
                };
                study.series.push(seriesA);

                var seriesAttributesB = Multiframe.copySeriesAttributes(selectedSeries.seriesAttributes);
                seriesAttributesB.instanceCount = instancesB.length;
                var seriesB = {
                    uuid: uidB,
                    instances: _.map(instancesB, function (i, index) {
                        var copy = new Models.Instance();
                        copy.frameNumber = new Classes.FrameNumber(0);
                        copy.id = i.id;
                        copy.instanceAttributes = Multiframe.copyInstanceAttributes(i.instanceAttributes);
                        copy.instanceAttributes.measurements = [];
                        copy.instanceAttributes.instanceIndex = index;
                        copy.instanceAttributes.instanceNumber = index;
                        copy.seriesAttributes = seriesAttributesB;
                        copy.studyAttributes = study.studyAttributes;
                        images[i.id.value] = { "(0020,000E)": uidB };
                        return copy;
                    }),
                    seriesAttributes: seriesAttributesB,
                    studyAttributes: study.studyAttributes,
                    loadedStatus: 0 /* None */
                };
                study.series.push(seriesB);

                // set up series order
                extended["series"] = {};
                extended["series"][uidA] = {
                    image_order: _.map(seriesA.instances, function (i) {
                        return i.id.value;
                    })
                };

                extended["series"][uidB] = {
                    image_order: _.map(seriesB.instances, function (i) {
                        return i.id.value;
                    })
                };

                // set up images order
                extended["images"] = images;

                var success = function () {
                    study.series = _.filter(study.series, function (s) {
                        return s !== selectedSeries;
                    });
                    _this.resetSeriesViews();
                    _this.render();
                    _this.replaceSelectedSeries(seriesA);
                };

                Services.StudyPHIExtended(this.sessionId, study.studyAttributes.uuid, JSON.stringify(extended)).subscribe({
                    done: function () {
                    },
                    next: function (_) {
                        success();
                        window.alert(_this.terminology.lookup(Terminology.Terms.StudyPHIExtendedSuccess));
                    },
                    fail: function (err) {
                        window.alert(_this.terminology.lookup(Terminology.Terms.StudyPHIExtendedError));
                        console.error("Unable to unweave series: " + err);
                    }
                });
            }
        };

        Application.prototype.partSeries = function () {
            var _this = this;
            var selectedSeriesView = this.selectedSeriesContainer.read();
            var selectedSeries = selectedSeriesView.series;
            var studyUid = selectedSeries.studyAttributes.queryObject.studyUid;

            var study = _.find(this.studies, function (s) {
                return s.studyAttributes.queryObject.studyUid.value === studyUid.value;
            });

            if (study) {
                var format = window.prompt(this.terminology.lookup(Terminology.Terms.PartImagesInstructions) + "\n\n" + selectedSeries.instances[0].instanceAttributes.seriesDescription, "");

                if (format) {
                    var formats = format.split(/(\s+)/).filter(function (e) {
                        return e.trim().length > 0;
                    });
                    var images = {};
                    var allMoved = [];
                    var partedSeries = [];

                    _.each(formats, function (format) {
                        var indices = _this.parseInstanceSelectionFormat(format);
                        var maxIndex = selectedSeries.instances.length;

                        var valid = _.reduce(indices, function (t, n) {
                            return t + n;
                        });
                        if (isNaN(valid) || indices[0] < 1 || indices[indices.length - 1] > maxIndex) {
                            window.alert(_this.terminology.lookup(Terminology.Terms.RearrangeImagesWarning));
                        } else {
                            var moving = [];

                            _.each(selectedSeries.instances, function (i, index) {
                                if (_.contains(indices, index + 1)) {
                                    moving.push(i);
                                }
                            });

                            var uidNew = _this.generateUID();
                            var seriesAttributesNew = Multiframe.copySeriesAttributes(selectedSeries.seriesAttributes);
                            seriesAttributesNew.instanceCount = moving.length;
                            var seriesNew = {
                                uuid: uidNew,
                                instances: _.map(moving, function (i, index) {
                                    var copy = new Models.Instance();
                                    copy.frameNumber = new Classes.FrameNumber(0);
                                    copy.id = i.id;
                                    copy.instanceAttributes = Multiframe.copyInstanceAttributes(i.instanceAttributes);
                                    copy.instanceAttributes.measurements = [];
                                    copy.instanceAttributes.instanceIndex = index;
                                    copy.instanceAttributes.instanceNumber = index;
                                    copy.seriesAttributes = seriesAttributesNew;
                                    copy.studyAttributes = study.studyAttributes;
                                    images[i.id.value] = { "(0020,000E)": uidNew };
                                    return copy;
                                }),
                                seriesAttributes: seriesAttributesNew,
                                studyAttributes: study.studyAttributes,
                                loadedStatus: 0 /* None */
                            };

                            partedSeries.push(seriesNew);
                            allMoved = allMoved.concat(moving);
                        }
                    });

                    var original = _.difference(selectedSeries.instances, allMoved);
                    _.each(original, function (i, index) {
                        i.instanceAttributes.instanceIndex = index;
                        i.instanceAttributes.instanceNumber = index;
                    });

                    var extended = { study: { series_order: [selectedSeries.seriesAttributes.seriesUid.value] } };

                    _.each(partedSeries, function (parted) {
                        extended["study"]["series_order"].push(parted.uuid);
                        study.series.push(parted);
                    });

                    selectedSeries.instances = original;
                    selectedSeries.seriesAttributes.instanceCount = original.length;
                    extended["images"] = images;

                    Services.StudyPHIExtended(this.sessionId, study.studyAttributes.uuid, JSON.stringify(extended)).subscribe({
                        done: function () {
                        },
                        next: function (_) {
                            var toSelect = selectedSeries;

                            if (original.length == 0) {
                                toSelect = partedSeries[0];
                                _this.removeEmptySeries(selectedSeries);
                            }

                            _this.resetSeriesViews();
                            _this.render();
                            _this.replaceSelectedSeries(toSelect);

                            window.alert(_this.terminology.lookup(Terminology.Terms.StudyPHIExtendedSuccess));
                        },
                        fail: function (err) {
                            window.alert(_this.terminology.lookup(Terminology.Terms.StudyPHIExtendedError));
                            console.error("Unable to part series: " + err);
                        }
                    });
                }
            }
        };

        Application.prototype.rearrangeSeries = function () {
            var _this = this;
            var selectedSeriesView = this.selectedSeriesContainer.read();
            var selectedSeries = selectedSeriesView.series;
            var selectedInstance = selectedSeriesView.currentInstance().read();

            var studyUid = selectedSeries.studyAttributes.queryObject.studyUid;

            var study = _.find(this.studies, function (s) {
                return s.studyAttributes.queryObject.studyUid.value === studyUid.value;
            });

            if (study) {
                var format = window.prompt(this.terminology.lookup(Terminology.Terms.RearrangeImagesInstructions) + "\n\n" + selectedSeries.instances[0].instanceAttributes.seriesDescription, "");

                if (format) {
                    var indices = this.parseInstanceSelectionFormat(format);
                    var maxIndex = selectedSeries.instances.length;

                    var valid = _.reduce(indices, function (t, n) {
                        return t + n;
                    });
                    if (isNaN(valid) || indices[0] < 1 || indices[indices.length - 1] > maxIndex) {
                        window.alert(this.terminology.lookup(Terminology.Terms.RearrangeImagesWarning));
                    } else {
                        var moving = [];
                        var original = [];

                        _.each(selectedSeries.instances, function (i, index) {
                            if (_.contains(indices, index + 1)) {
                                moving.push(i);
                            } else {
                                original.push(i);
                            }
                        });

                        var currentIndex = _.indexOf(original, selectedInstance);
                        var rearranged = original.slice(0, currentIndex).concat(moving).concat(original.slice(currentIndex, original.length));
                        var seriesUID = selectedSeries.seriesAttributes.seriesUid.value;
                        var reordered = [];

                        _.each(rearranged, function (i, index) {
                            i.instanceAttributes.instanceIndex = index;
                            i.instanceAttributes.instanceNumber = index;
                            reordered.push(i.id.value);
                        });

                        selectedSeries.instances = rearranged;

                        var extended = { series: {} };
                        extended.series[seriesUID] = {
                            image_order: reordered
                        };

                        Services.StudyPHIExtended(this.sessionId, study.studyAttributes.uuid, JSON.stringify(extended)).subscribe({
                            done: function () {
                            },
                            next: function (_) {
                                _this.resetSeriesViews();
                                _this.render();
                                _this.replaceSelectedSeries(selectedSeries);

                                window.alert(_this.terminology.lookup(Terminology.Terms.StudyPHIExtendedSuccess));
                            },
                            fail: function (err) {
                                window.alert(_this.terminology.lookup(Terminology.Terms.StudyPHIExtendedError));
                                console.error("Unable to rearrange series: " + err);
                            }
                        });
                    }
                }
            }
        };

        Application.prototype.resetArrangement = function () {
            var _this = this;
            var selectedSeriesView = this.selectedSeriesContainer.read();
            var selectedSeries = selectedSeriesView.series;
            var studyUid = selectedSeries.studyAttributes.queryObject.studyUid;

            var study = _.find(this.studies, function (s) {
                return s.studyAttributes.queryObject.studyUid.value === studyUid.value;
            });

            if (study) {
                Services.StudyPHIExtended(this.sessionId, study.studyAttributes.uuid, "DELETE").subscribe({
                    done: function () {
                    },
                    next: function (_) {
                        _this.resetSeriesViews();
                        _this.render();

                        window.alert(_this.terminology.lookup(Terminology.Terms.StudyPHIExtendedResetSuccess));
                    },
                    fail: function (err) {
                        window.alert(_this.terminology.lookup(Terminology.Terms.StudyPHIExtendedError));
                        console.error("Unable to reset study: " + err);
                    }
                });
            }
        };

        Application.prototype.generateUID = function () {
            var S4 = function () {
                return Math.floor(Math.random() * 0x10000).toString(10);
            };

            return '1.3.6.1.4.1.34692.' + S4() + S4() + S4() + S4() + S4() + S4();
        };

        Application.prototype.mergeSeries = function (study, merging) {
            var _this = this;
            var uidNew = this.generateUID();
            var images = {};
            var extended = { series: {} };

            var seriesAttributesNew = Multiframe.copySeriesAttributes(merging[0].seriesAttributes);
            var instances = [];
            var index = 0;
            var reordered = [];

            _.each(merging, function (series) {
                _.each(series.instances, function (i) {
                    var copy = new Models.Instance();
                    copy.frameNumber = new Classes.FrameNumber(0);
                    copy.id = i.id;
                    copy.instanceAttributes = Multiframe.copyInstanceAttributes(i.instanceAttributes);
                    copy.instanceAttributes.measurements = [];
                    copy.instanceAttributes.instanceIndex = index;
                    copy.instanceAttributes.instanceNumber = index;
                    copy.seriesAttributes = seriesAttributesNew;
                    copy.seriesAttributes.instanceCount = i.instanceAttributes.frameCount;
                    copy.studyAttributes = study.studyAttributes;
                    images[i.id.value] = { "(0020,000E)": uidNew };
                    reordered.push(i.id.value);
                    instances.push(copy);
                    index += 1;
                });
            });

            var seriesNew = {
                uuid: uidNew,
                instances: instances,
                seriesAttributes: seriesAttributesNew,
                studyAttributes: study.studyAttributes,
                loadedStatus: 0 /* None */
            };

            seriesAttributesNew.instanceCount = instances.length;

            study.series = _.difference(study.series, merging);
            study.series.push(seriesNew);

            extended.series[uidNew] = {
                image_order: reordered
            };

            extended["images"] = images;

            Services.StudyPHIExtended(this.sessionId, study.studyAttributes.uuid, JSON.stringify(extended)).subscribe({
                done: function () {
                },
                next: function (_) {
                    _this.resetSeriesViews();
                    _this.render();
                    window.alert(_this.terminology.lookup(Terminology.Terms.StudyPHIExtendedSuccess));
                },
                fail: function (err) {
                    window.alert(_this.terminology.lookup(Terminology.Terms.StudyPHIExtendedError));
                    console.error("Unable to rearrange series: " + err);
                }
            });
        };

        Application.prototype.updateSeriesHandles = function (show) {
            var views = this.seriesViews.value;

            for (var i = 0; i < views.length; i++) {
                if (views[i].handle) {
                    if (show) {
                        views[i].handle.show();
                    } else {
                        views[i].handle.hide();
                    }
                }
            }
        };

        Application.prototype.recordError = function (message) {
            Services.AuditLog(message, "Application", this.sessionId, this.studies[0].studyAttributes.uuid);
        };

        Application.prototype.recordSeriesError = function (seriesUid, message, source) {
            Services.AuditLog(message, source, this.sessionId, this.studies[0].studyAttributes.uuid, seriesUid);
        };

        Application.prototype.findSeriesView = function (series, skip) {
            var views = this.seriesViews.read();

            for (var i = 0; i < views.length; i++) {
                if (views[i].series && (views[i].series == series) && (!skip || !_.contains(skip, views[i]))) {
                    return views[i];
                }
            }

            return null;
        };

        Application.prototype.isDiagnosticQualityAlways = function () {
            return (this.accountSettings.viewer_diagnostic_quality_always == 1) || this.settings.read().diagnosticQualityAlways;
        };

        Application.prototype.seriesLoaded = function (series) {
            var loaded = 0 /* None */;

            if (series) {
                loaded = series.loadedStatus || 0 /* None */;

                var instances = series.instances;
                var diagnosticAlways = this.isDiagnosticQualityAlways();
                var attributesLoaded = true;

                // check attributes
                if (loaded < 1 /* Attributes */) {
                    for (var ctr = 0; ctr < instances.length; ctr += 1) {
                        var instance = instances[ctr];
                        if (instance.instanceAttributes.attributesDownloadState != 3 /* Success */) {
                            attributesLoaded = false;
                            break;
                        }
                    }

                    if (attributesLoaded) {
                        loaded = 1 /* Attributes */;
                    }
                }

                if (attributesLoaded) {
                    var imageElements = this.imagePreloadStore[series.seriesAttributes.seriesUid.value];

                    if (imageElements) {
                        if (!diagnosticAlways) {
                            if (loaded < 2 /* Thumbnail */) {
                                var thumbnailsLoaded = true;

                                for (var ctr = 0; ctr < instances.length; ctr += 1) {
                                    var instance = instances[ctr];
                                    var instanceKey = instance.id.value + ':' + instance.frameNumber.value;
                                    var image = imageElements.get(instanceKey, 0 /* Thumbnail */);

                                    if (!image || (image.imageDownloadState != 3 /* Success */)) {
                                        thumbnailsLoaded = false;
                                        break;
                                    }
                                }

                                if (thumbnailsLoaded) {
                                    loaded = 2 /* Thumbnail */;
                                }
                            }
                        }

                        if (loaded < 3 /* Diagnostic */) {
                            var diagnosticLoaded = true;
                            var useDiagnostic = this.accountSettings.viewer_diagnostic_quality == 1 || this.settings.read().diagnosticQualityAlways;
                            var hiResImageType = useDiagnostic ? 2 /* Diagnostic */ : 1 /* FullResolution */;

                            for (var ctr = 0; ctr < instances.length; ctr += 1) {
                                var instance = instances[ctr];
                                var instanceKey = instance.id.value + ':' + instance.frameNumber.value;
                                var image = imageElements.get(instanceKey, hiResImageType);

                                if (!image || (image.imageDownloadState != 3 /* Success */)) {
                                    diagnosticLoaded = false;
                                    break;
                                }
                            }

                            if (diagnosticLoaded) {
                                loaded = 3 /* Diagnostic */;
                            }
                        }
                    }
                }

                series.loadedStatus = loaded;
            }

            return loaded;
        };
        return Application;
    })();
    Views.Application = Application;
})(Views || (Views = {}));
///<reference path='../typings/jquery/jquery.d.ts' />
///<reference path='../classes/Types.ts' />
///<reference path='../libs/Observable.ts' />
///<reference path='../libs/Query.ts' />
///<reference path='../libs/Date.ts' />
///<reference path='../libs/Number.ts' />
///<reference path='../libs/Services.ts' />
///<reference path='../libs/V3Storage.ts' />
///<reference path='../libs/Study.ts' />
///<reference path='../libs/Array.ts' />
///<reference path='../libs/Subject.ts' />
///<reference path='../models/StudySchema.ts' />
///<reference path='../models/Study.ts' />
var Views;
(function (Views) {
    /**
    * Contains all of the subviews required for the print page
    */
    (function (Print) {
        /**
        * Calculate the True Size dimensions of the image in millimeters
        */
        function calculatePrintDimensions(rows, columns, pixelSpacing) {
            return [columns * pixelSpacing[0], rows * pixelSpacing[0]];
        }

        /**
        * Calculate the Fit To Page dimensions of the image in inches.
        * For letter paper, we reserve an area 7.5x8.5 inches for the image, allowing for 0.5 inch margins and a 1.5 inch header
        */
        function calculateFitDimensions(paper, rows, columns) {
            var height = paper.height - 1.5;
            var width = paper.width;
            var margin = paper.margin;

            var aspect = (width - margin) / (height - margin);

            if (columns / rows >= aspect) {
                // Wide image, fit to width
                return [width - margin, (width - margin) * rows / columns];
            } else {
                // Tall image, fit to height
                return [(height - margin) * columns / rows, height - margin];
            }
        }

        /**
        * Enumerates the ways in which an image can be displayed
        */
        (function (ImageMode) {
            ImageMode[ImageMode["Hidden"] = 0] = "Hidden";
            ImageMode[ImageMode["ShowFullSize"] = 1] = "ShowFullSize";
            ImageMode[ImageMode["ShowTrueSize"] = 2] = "ShowTrueSize";
            ImageMode[ImageMode["ShowFitToPage"] = 3] = "ShowFitToPage";
        })(Print.ImageMode || (Print.ImageMode = {}));
        var ImageMode = Print.ImageMode;

        

        /**
        * The view which shows the list of available images
        */
        var ImageList = (function () {
            function ImageList(el, study, terminology) {
                /**
                * A list of preview images
                */
                this.frames = [];
                this.el = el;
                this.study = study;
                this.terminology = terminology;
            }
            /**
            * Layout the screen, and hook up any event handlers
            */
            ImageList.prototype.render = function () {
                var _this = this;
                $('<a name="top">').appendTo(this.el);

                var framesIndex = 0;

                _.each(this.study.series, function (series) {
                    var seriesDescription = series.instances[0].instanceAttributes.seriesDescription;

                    if (!seriesDescription) {
                        seriesDescription = _this.terminology.lookup(Terminology.Terms.Untitled);
                    }

                    $('<h2>').text(seriesDescription + ' (' + series.seriesAttributes.modality + ')').appendTo(_this.el);

                    $('<a href="#top">').text(_this.terminology.lookup(Terminology.Terms.BackToTop)).appendTo(_this.el);

                    var $instances = $('<ul>').addClass('instances').appendTo(_this.el);

                    _.each(series.instances, function (instance, imageIndex, list) {
                        var imageText = _this.terminology.lookup(Terminology.Terms.Instance) + ' ' + (imageIndex + 1) + ' / ' + series.instances.length;

                        var $frames = $('<ul>').addClass('frames');

                        var $instance = $('<li>').addClass('instance').text(imageText).appendTo($instances);

                        _.each(_.range(0, instance.instanceAttributes.frameCount), function (frameIndex) {
                            var $frame = $('<li>').addClass('frame').appendTo($frames);

                            var mode = new Subjects.ObservableValue(0 /* Hidden */);

                            var previewImage = {
                                mode: mode,
                                instance: instance,
                                frameIndex: frameIndex,
                                imageIndex: imageIndex
                            };

                            Subjects.listen(mode, function (newMode) {
                                switch (newMode) {
                                    case 0 /* Hidden */:
                                        $frame.show();
                                        break;
                                    default:
                                        $frame.hide();
                                        break;
                                }
                            });

                            _this.frames.push(previewImage);

                            framesIndex++;

                            var $linksLocation;

                            if (instance.instanceAttributes.frameCount > 1) {
                                var frameText = _this.terminology.lookup(Terminology.Terms.Frame) + ' ' + (frameIndex + 1) + ' / ' + instance.instanceAttributes.frameCount;

                                $frame.append(frameText);
                                $instance.append($frames);

                                $linksLocation = $frame;
                            } else {
                                $linksLocation = $instance;
                            }

                            $linksLocation.append(' ').append($('<a>').attr('href', '#').text('(' + _this.terminology.lookup(Terminology.Terms.FitToPage) + ')').click(function (e) {
                                previewImage.mode.write(3 /* ShowFitToPage */);
                                e.preventDefault();
                            }));

                            $linksLocation.append(' ').append($('<a>').attr('href', '#').text('(' + _this.terminology.lookup(Terminology.Terms.FullSize) + ')').click(function (e) {
                                previewImage.mode.write(1 /* ShowFullSize */);
                                e.preventDefault();
                            }));

                            $linksLocation.append(' ').append($('<a>').attr('href', '#').text('(' + _this.terminology.lookup(Terminology.Terms.TrueSize) + ')').click(function (e) {
                                previewImage.mode.write(2 /* ShowTrueSize */);
                                e.preventDefault();
                            }));
                        });
                    });
                });
            };
            return ImageList;
        })();
        Print.ImageList = ImageList;

        /**
        * The view which shows a preview of those images which will be printed
        */
        var Preview = (function () {
            function Preview(sessionId, el, study, terminology, settings) {
                this.sessionId = sessionId;
                this.el = el;
                this.study = study;
                this.terminology = terminology;
                this.settings = settings;
            }
            /**
            * Layout the screen, and hook up any event handlers
            */
            Preview.prototype.render = function (frames) {
                var _this = this;
                var $images = $('.images').length ? $('.images') : $('<div>').addClass('images').appendTo(this.el);

                new PHI($images, this.study.studyAttributes, this.terminology, this.settings.dateFormat, "no-print").render();

                _.each(frames, function (frame, index, list) {
                    // $divWrapper makes sure the images are always shown in the right order
                    var $divWrapper = $('<div>').appendTo($images);

                    // Use the first description as a default if one isn't provided per frame
                    if (frame.instance.instanceAttributes.seriesDescription == null || frame.instance.instanceAttributes.seriesDescription.length == 0) {
                        frame.instance.instanceAttributes.seriesDescription = frames[0].instance.instanceAttributes.seriesDescription;
                    }

                    var image = new PreviewImage(_this.sessionId, $divWrapper, frame, index > 0 ? frames[index - 1] : null, index < frames.length - 1 ? frames[index + 1] : null, _this.settings, _this.terminology);

                    image.render();
                });
            };
            return Preview;
        })();
        Print.Preview = Preview;

        /**
        * A view which renders PHI
        */
        var PHI = (function () {
            function PHI(el, studyAttributes, terminology, dateFormat, className) {
                this.el = el;
                this.studyAttributes = studyAttributes;
                this.terminology = terminology;
                this.dateFormat = dateFormat;
                this.className = className;
            }
            /**
            * Render the patient information
            */
            PHI.prototype.render = function () {
                var $div = $('<div>').addClass("phi").addClass(this.className).appendTo(this.el);

                $('<h2>').text(this.studyAttributes.patientName).appendTo($div);

                if (this.studyAttributes.patientBirthDate) {
                    $div.append($('<div>').append(this.terminology.lookup(Terminology.Terms.PatientDOB) + ": ").append(this.studyAttributes.patientBirthDate.toShortDateString(this.dateFormat)));
                }

                $div.append($('<div>').append(this.terminology.lookup(Terminology.Terms.StudyDescription) + ": ").append(this.studyAttributes.studyDescription));

                if (this.studyAttributes.studyCreateDate) {
                    $div.append($('<div>').append(this.terminology.lookup(Terminology.Terms.StudyCreatedOn) + ": ").append(this.studyAttributes.studyCreateDate.toShortDateString(this.dateFormat)));
                }
            };
            return PHI;
        })();
        Print.PHI = PHI;

        /**
        * A view which displays an image in the preview pane
        */
        var PreviewImage = (function () {
            function PreviewImage(sessionId, el, imageCreationParams, previousFrame, nextFrame, settings, terminology) {
                this.loaded = false;
                this.sessionId = sessionId;
                this.el = el;
                this.imageCreationParams = imageCreationParams;
                this.previousFrame = previousFrame;
                this.nextFrame = nextFrame;
                this.settings = settings;
                this.terminology = terminology;
            }
            /**
            * Create elements, add them to the DOM, and hook up event handlers
            */
            PreviewImage.prototype.render = function () {
                var _this = this;
                var $div = $('<div>').addClass('together').appendTo(this.el);

                $div.hide();

                // Ideally, we would use a CSS selector here to say "add a page break after
                // all but the last images", but there is no selector which can do that, given
                // the way the page is laid out to preserve order of images, so we fix up the
                // CSS classes using jQuery after an update.
                var fixPageBreaks = function () {
                    var cousins = _this.el.parent().children().children();
                    cousins.removeClass('break-after');
                    cousins.filter(':visible').filter(':not(:last)').addClass('break-after');
                };

                new PHI($div, this.imageCreationParams.instance.studyAttributes, this.terminology, this.settings.dateFormat, "print-only").render();

                var instance = this.imageCreationParams.instance;

                var frameTitle = (instance.instanceAttributes.seriesDescription || instance.instanceAttributes.seriesDescription) + ' (' + this.terminology.lookup(Terminology.Terms.Instance) + ' ' + (this.imageCreationParams.imageIndex + 1) + '/' + instance.seriesAttributes.instanceCount + ', ' + this.terminology.lookup(Terminology.Terms.Frame) + ' ' + (this.imageCreationParams.frameIndex + 1) + '/' + instance.instanceAttributes.frameCount + ')';

                $('<h3>').addClass('frame-title').text(frameTitle).appendTo($div);

                this.img = $('<img>').appendTo($div);

                var $menu = $('<div>').addClass('no-print').append($('<a>').attr('href', '#').text(this.terminology.lookup(Terminology.Terms.Remove)).click(function (e) {
                    _this.imageCreationParams.mode.write(0 /* Hidden */);
                    e.preventDefault();
                })).appendTo($div);

                if (this.previousFrame) {
                    $menu.append(' | ').append($('<a>').attr('href', '#').text(this.terminology.lookup(Terminology.Terms.Prev)).click(function (e) {
                        _this.previousFrame.mode.write(_this.imageCreationParams.mode.read());
                        _this.imageCreationParams.mode.write(0 /* Hidden */);
                        e.preventDefault();
                    }));
                }

                if (this.nextFrame) {
                    $menu.append(' | ').append($('<a>').attr('href', '#').text(this.terminology.lookup(Terminology.Terms.Next)).click(function (e) {
                        _this.nextFrame.mode.write(_this.imageCreationParams.mode.read());
                        _this.imageCreationParams.mode.write(0 /* Hidden */);
                        e.preventDefault();
                    }));
                }

                Subjects.listen(this.imageCreationParams.mode, function (newMode) {
                    switch (newMode) {
                        case 0 /* Hidden */:
                            $div.hide();
                            fixPageBreaks();
                            break;
                        default:
                            _this.load(newMode).subscribe({
                                next: function (_) {
                                },
                                done: function () {
                                    var src = Routes.ImageData(_this.sessionId, instance.studyAttributes.studyStorage, instance.studyAttributes.queryObject, instance.id, instance.instanceAttributes.version, new Classes.FrameNumber(_this.imageCreationParams.frameIndex), 1 /* FullResolution */, 8, false);

                                    _this.img.attr('src', src);

                                    switch (newMode) {
                                        case 1 /* ShowFullSize */:
                                            _this.img.css({
                                                width: instance.instanceAttributes.columns,
                                                height: instance.instanceAttributes.rows
                                            });
                                            break;
                                        case 3 /* ShowFitToPage */:
                                            if (instance.instanceAttributes.rows && instance.instanceAttributes.columns && instance.instanceAttributes.pixelSpacing) {
                                                var letter = { width: 8.5, height: 11.0, margin: 1.0 };

                                                var dimensions = calculateFitDimensions(_this.settings.paperSize || letter, instance.instanceAttributes.rows, instance.instanceAttributes.columns);

                                                _this.img.css({ width: dimensions[0] + 'in', height: dimensions[1] + 'in' });
                                            }
                                            break;
                                        case 2 /* ShowTrueSize */:
                                            if (instance.instanceAttributes.rows && instance.instanceAttributes.columns && instance.instanceAttributes.pixelSpacing) {
                                                var dimensions = calculatePrintDimensions(instance.instanceAttributes.rows, instance.instanceAttributes.columns, instance.instanceAttributes.pixelSpacing);

                                                _this.img.css({ width: dimensions[0] + 'mm', height: dimensions[1] + 'mm' });
                                            } else {
                                                _this.img.css({
                                                    width: instance.instanceAttributes.columns,
                                                    height: instance.instanceAttributes.rows
                                                });
                                            }
                                            break;
                                    }

                                    $div.show();

                                    fixPageBreaks();
                                },
                                fail: function (err) {
                                    Services.AuditLog("Unable to load image data: " + err, "Print", _this.sessionId);
                                    window.alert('Unable to load image data.');
                                }
                            });
                            break;
                    }
                });
            };

            /**
            * Load image attributes if necessary
            */
            PreviewImage.prototype.load = function (mode) {
                var _this = this;
                if (this.loaded) {
                    return Observable.ret({});
                } else {
                    return Observable.invoke(Observable.forget(Study.loadImageAttributes(this.sessionId, this.imageCreationParams.instance)), function (_) {
                        _this.loaded = true;
                    });
                }
            };
            return PreviewImage;
        })();
        Print.PreviewImage = PreviewImage;

        /**
        * The Print view
        *
        * This view also doubles as the main viewer in browsers which do not support Canvas.
        *
        * Displays a list of all available series with links to individual frames. Images may be displayed at
        * Full Size (one image pixel equals one pixel on screen) or True Size (one cm in the patient geometry
        * equals one cm on the printed or displayed media)
        *
        * Basic image navigation (previous, next) functions are also provided.
        */
        var Application = (function () {
            function Application(el, studyStorage, queryObject, sessionId, terminology, settings, permissions) {
                this.study = null;
                this.imageLists = [];
                this.el = el;
                this.studyStorage = studyStorage;
                this.queryObject = queryObject;
                this.sessionId = sessionId;
                this.terminology = terminology;
                this.settings = settings;
                this.permissions = permissions;
            }
            /**
            * Load the study from storage and call the render method
            */
            Application.prototype.load = function () {
                var _this = this;
                var loadStudy = Study.loadStudy(this.sessionId, this.studyStorage, this.queryObject, false, this.permissions);

                return Observable.invoke(loadStudy, function (study) {
                    _this.study = study;
                    _this.renderInit();
                    _this.renderStudy(_this.el, study);

                    var getPriors = Services.getStudyList(_this.sessionId, _this.queryObject.phiNamespace, study.studyAttributes.patientId, null, null, study.studyAttributes.uuid);

                    Observable.invoke(getPriors, function (list) {
                        _this.renderList(_this.el, list);
                    }).subscribe({
                        done: function () {
                        },
                        next: function (_) {
                        },
                        fail: function (err) {
                            Services.AuditLog("Unable to load priors: " + err, "Print", _this.sessionId);
                        }
                    });
                });
            };

            Application.prototype.createInstructions = function () {
                var $div = $('<div>').addClass('instructions');

                var redirected = Query.findParameter(window.location, 'redirected');

                if (redirected === 'true') {
                    $div.append($('<p>').text(this.terminology.lookup(Terminology.Terms.PrintViewerRedirectMessageLine1)));
                    $div.append($('<p>').append($('<a>').attr({ href: '/requirements.html', target: '_blank' }).text(this.terminology.lookup(Terminology.Terms.PrintViewerRedirectMessageLine2))));
                } else {
                    $div.append($('<p>').text(this.terminology.lookup(Terminology.Terms.PrintViewerInstructions)));
                }

                return $div;
            };

            /**
            * Create the print toolbar
            */
            Application.prototype.createToolbar = function () {
                var _this = this;
                var $toolbar = $('<p>').append($('<a href="#">Print</a>').click(function (e) {
                    window.print();
                    e.preventDefault();
                })).append(' | ').append($('<a>').attr('href', '#').text(this.terminology.lookup(Terminology.Terms.Clear)).click(function (e) {
                    _.each(_this.imageLists, function (list) {
                        _.each(list.frames, function (frame) {
                            return frame.mode.write(0 /* Hidden */);
                        });
                    });
                    e.preventDefault();
                }));

                var totalNumberOfFrames = this.study.series.sum(function (series) {
                    return series.instances.sum(function (instance) {
                        return instance.instanceAttributes.frameCount;
                    });
                });

                if (totalNumberOfFrames <= 20) {
                    var $showAllFitToPage = $('<a>').attr('href', '#').text(this.terminology.lookup(Terminology.Terms.ShowAllFitToPage)).click(function (e) {
                        _.each(_this.imageLists, function (list) {
                            _.each(list.frames, function (frame) {
                                return frame.mode.write(3 /* ShowFitToPage */);
                            });
                        });
                        e.preventDefault();
                    });
                    var $showAllFullSize = $('<a>').attr('href', '#').text(this.terminology.lookup(Terminology.Terms.ShowAllFullSize)).click(function (e) {
                        _.each(_this.imageLists, function (list) {
                            _.each(list.frames, function (frame) {
                                return frame.mode.write(1 /* ShowFullSize */);
                            });
                        });
                        e.preventDefault();
                    });

                    var $showAllTrueSize = $('<a>').attr('href', '#').text(this.terminology.lookup(Terminology.Terms.ShowAllTrueSize)).click(function (e) {
                        _.each(_this.imageLists, function (list) {
                            _.each(list.frames, function (frame) {
                                return frame.mode.write(2 /* ShowTrueSize */);
                            });
                        });
                        e.preventDefault();
                    });

                    $toolbar.append(' | ').append($showAllFitToPage).append(' | ').append($showAllFullSize).append(' | ').append($showAllTrueSize);
                }

                return $toolbar;
            };

            Application.prototype.renderInit = function () {
                if ($('.instructions').length == 0) {
                    var $study = $('<div>').addClass('study');

                    var $column1 = $('<div>').addClass('content with-margin padded full-height scrolls no-print').appendTo($('<div>').addClass('column1').appendTo($study));
                    $('<div>').addClass('content with-margin padded full-height scrolls bordered').appendTo($('<div>').addClass('column2').appendTo($study));

                    this.el.append($study);

                    this.createInstructions().appendTo($column1);

                    this.createToolbar().appendTo($column1);
                }
            };

            /**
            * Layout the screen, and hook up any event handlers
            */
            Application.prototype.renderStudy = function (el, study) {
                var $column1 = $('.column1 .content');
                var $column2 = $('.column2 .content');

                var imageList = new ImageList($column1, study, this.terminology);
                this.imageLists.push(imageList);

                imageList.render();

                var preview = new Preview(this.sessionId, $column2, study, this.terminology, this.settings);

                preview.render(imageList.frames);
            };

            Application.prototype.renderList = function (el, list) {
                var _this = this;
                for (var i = 0; i < list.length; ++i) {
                    var queryObject = new Classes.QueryObject(new Classes.StorageNamespace(list[i].storage_namespace), new Classes.PhiNamespace(list[i].phi_namespace), new Classes.StudyUid(list[i].study_uid));
                    var studyStorage = Services.getStudyStorageInfo(this.sessionId, queryObject);

                    Observable.bind(studyStorage, function (storage) {
                        var loadStudy = Study.loadStudy(_this.sessionId, storage, queryObject, false, _this.permissions);

                        return Observable.invoke(loadStudy, function (study) {
                            _this.study = study;
                            _this.renderStudy(el, study);
                        });
                    }).subscribe({
                        done: function () {
                        },
                        next: function (_) {
                        },
                        fail: function (err) {
                            Services.AuditLog("Unable to load priors: " + err, "Print", _this.sessionId);
                        }
                    });
                }
            };
            return Application;
        })();
        Print.Application = Application;
    })(Views.Print || (Views.Print = {}));
    var Print = Views.Print;
})(Views || (Views = {}));
///<reference path='typings/jquery/jquery.d.ts' />
///<reference path='libs/Query.ts' />
///<reference path='libs/Services.ts' />
///<reference path='libs/Terminology.ts' />
///<reference path='views/Application.ts' />
///<reference path='views/Print.ts' />

/**
* Bootstrapping module which provides the global onload handler
*/
var Main;
(function (_Main) {
    /**
    * The main application class. Responsible for initializing the application and delegating to the appropriate view
    * based on the route in the query string.
    */
    var Main = (function () {
        function Main() {
        }
        /**
        * Try to get the session ID from the session storage. If it is not provided, attempt to load it from the query string.
        */
        Main.ensureSessionId = function () {
            var sessionId = Query.getSessionId(window.location);

            if (sessionId === null) {
                var sid = sessionStorage.getItem('sid');

                if (!sid) {
                    // Go back to domain root to log back in for SID, then redirect back to viewer
                    window.location.assign("/?route=login_redirect&redirect_url=" + encodeURIComponent(window.location.href));

                    throw new Classes.ExpectedSIDException("");
                }

                sessionId = new Classes.SessionId(sid);
            }

            return sessionId;
        };

        Main.recordMetric = function (name, additionalMetrics) {
            if (!Date.now)
                Date.now = function () {
                    return +new Date();
                };

            function detectBrowser() {
                var ua = navigator.userAgent, tem, M = ua.match(/(opera|chrome|safari|firefox|msie|trident(?=\/))\/?\s*(\d+)/i) || [];
                if (/trident/i.test(M[1])) {
                    tem = /\brv[ :]+(\d+)/g.exec(ua) || [];
                    return 'IE ' + (tem[1] || '');
                }
                if (M[1] === 'Chrome') {
                    tem = ua.match(/\b(OPR|Edge)\/(\d+)/);
                    if (tem != null)
                        return tem.slice(1).join(' ').replace('OPR', 'Opera');
                }
                M = M[2] ? [M[1], M[2]] : [navigator.appName, navigator.appVersion, '-?'];
                if ((tem = ua.match(/version\/(\d+)/i)) != null)
                    M.splice(1, 1, tem[1]);
                return { name: M[0], version: M[1] };
            }

            var uuid = (this.studyStorage) ? this.studyStorage.uuid : null;

            var elapsed = Date.now() - this.mainInitTime;

            var userEmail = (this.user && this.user.email) ? this.user.email : '';
            var userName = (this.user && this.user.name) ? this.user.name : '';

            var browser = detectBrowser();

            var renderingTypes = ['canvas', 'webgl', 'simple'];
            var renderMode = renderingTypes[Rendering.getRenderingMode()];

            var metrics = {
                elapsed: elapsed,
                userEmail: userEmail,
                userName: userName,
                hostname: window.location.hostname,
                browserName: browser.name,
                browserVersion: browser.version,
                screenWidth: window.screen.width,
                screenHeight: window.screen.height,
                renderMode: renderMode,
                uuid: uuid
            };

            if (additionalMetrics) {
                metrics["studyUID"] = additionalMetrics.studyUID.value;
                metrics["accelerated"] = additionalMetrics.accelerated;
                metrics["imageSize"] = additionalMetrics.imageSize;
                metrics["imageWidth"] = additionalMetrics.imageWidth;
                metrics["imageHeight"] = additionalMetrics.imageHeight;
                metrics["renderTime"] = additionalMetrics.renderTime;
            }

            Services.AuditMetric(name, metrics);
        };

        /**
        * Determine if the print viewer should be used since Canvas is not supported
        */
        Main.usePrintViewer = function () {
            return document.createElement('canvas').getContext === undefined;
        };

        /**
        * Redirect to the print viewer
        */
        Main.redirectToPrintViewer = function (query, sessionId) {
            window.location.replace("/viewer/print.html#print/" + encodeURIComponent(query.toString()) + "?redirected=true&sid=" + encodeURIComponent(sessionId.value));
        };

        /**
        * Redirect to view just the priors
        */
        Main.redirectToPriorsView = function (query, sessionId) {
            var originalStudy = null;
            var getFirstPrior = Observable.bind(Services.getStudyStorageInfo(sessionId, query), function (study) {
                originalStudy = study;
                return Services.getStudyList(sessionId, query.phiNamespace, new Classes.PatientID(study.patientid), null, null, new Classes.StudyUUID(study.uuid));
            });

            getFirstPrior.subscribe({
                done: function () {
                },
                fail: function (err) {
                    Services.AuditLog("Error with priors view", "Main", sessionId);
                },
                next: function (priors) {
                    if (priors.length > 0) {
                        var priorQuery = new Classes.QueryObject(new Classes.StorageNamespace(priors[0].storage_namespace), new Classes.PhiNamespace(priors[0].phi_namespace), new Classes.StudyUid(priors[0].study_uid));

                        var newLocation = "/viewer/?foo=" + (new Date().getTime()) + "#study/" + priorQuery.toString() + "?hideStudyUUID=" + originalStudy.uuid;

                        var expandPriors = Query.findParameter(window.location, "expandPriors");
                        if (expandPriors) {
                            newLocation += "&expandPriors=true";
                        }

                        var localAcceleration = Query.findParameter(window.location, "viewer_local_accelerator");
                        if (localAcceleration) {
                            newLocation += "&viewer_local_accelerator=1";
                        }

                        window.location.assign(newLocation);
                    } else {
                        window.location.replace("/blank.html");
                    }
                }
            });
        };

        Main.getDateFormat = function () {
            if (LocalViewer.dateFormat) {
                return LocalViewer.dateFormat;
            } else if (window.sessionStorage) {
                var dateFormat = window.sessionStorage.getItem("viewer:dateFormat");
                if (dateFormat) {
                    var parsedDateFormat = JSON.parse(dateFormat);
                    if (parsedDateFormat) {
                        return parsedDateFormat;
                    }
                }
            }

            return 'MM-DD-YYYY';
        };

        /**
        * Initialize the application.
        *
        * Displays the "Application Loading" message and loads the appropriate view.
        */
        Main.initialize = function () {
            // Time the page is downloaded and parsed
            this.mainInitTime = +new Date();

            var $frame = $('<div>').addClass('applicationFrame').appendTo($(document.body));

            var $overlay = $('<div>').addClass('overlay').appendTo($(document.body));
            $('<div>').addClass('fa fa-gear fa-spin fa-5x').appendTo($overlay);

            try  {
                var query;

                if (LocalViewer.isLocalViewer()) {
                    if (LocalViewer.isPersonalAccelerator()) {
                        query = Query.parsePersonalAcceleratorPath(window.location);
                    } else {
                        var storageNamespace = new Classes.StorageNamespace("default");
                        var studyUid = new Classes.StudyUid("default");
                        var phiNamespace = new Classes.PhiNamespace("default");

                        query = {
                            route: "study",
                            queryObject: new Classes.QueryObject(storageNamespace, phiNamespace, studyUid)
                        };
                    }
                } else {
                    query = Query.parseQueryString(window.location);
                }

                var sessionId = LocalViewer.isStandardLocalViewer() ? new Classes.SessionId(null) : Main.ensureSessionId();

                if (query.route !== "print" && Main.usePrintViewer()) {
                    Main.redirectToPrintViewer(query.queryObject, sessionId);
                } else if (Query.findParameter(window.location, "priorsOnly") == "true") {
                    Main.redirectToPriorsView(query.queryObject, sessionId);
                } else {
                    $overlay.addClass('application-loading');

                    var terminology = Terminology.loadAllTerminology(sessionId, query.queryObject);

                    terminology.subscribe({
                        done: function () {
                        },
                        fail: function (err) {
                            Services.AuditLog("Unable to load terminology: " + err, "Main", sessionId);
                            $frame.append("Error loading terminology");
                            $overlay.removeClass('application-loading');
                        },
                        next: function (terminology) {
                            Main.terminologyLoaded(sessionId, query, terminology);
                        }
                    });
                }
            } catch (ex) {
                if (ex instanceof Classes.ExpectedSIDException) {
                    $frame.append("Redirecting... ");
                    setTimeout(function () {
                        return $frame.append("failed.  No SID found.");
                    }, 5000);
                } else if (ex instanceof Classes.InvalidQueryStringException) {
                    $frame.append(ex.message);
                } else {
                    $frame.append("Unknown error");
                }

                $overlay.removeClass('application-loading');
            }
        };

        /**
        * Called after terminology is loaded (so other errors during loading can be
        * customized).
        */
        Main.terminologyLoaded = function (sessionId, query, terminology) {
            var _this = this;
            var acceleratorUsed = Services.AcceleratorUsed(sessionId);
            acceleratorUsed.subscribe({
                done: function () {
                },
                next: function (result) {
                    if (result && result.used === 1) {
                        _this.acceleratorUsed = true;
                    }
                },
                fail: function (err) {
                    Services.AuditLog("accelerator/used failed: " + err, "Main", sessionId);
                }
            });

            var userInfo = Services.getU